'Simple Garage program
'Original version last modified 2021-10-07
'Version 1.1 started 2022-05-29

#include "file.bi"
#ifdef __FB_LINUX__
	#define SLASH "/"
#else
	#define SLASH "\"
#endif

Type Car
	id As String
	file(1 To 8) As String
	numfiles As Byte
	carname As String
End Type

Type Group
	gname As String
	cars As String
End Type

Dim Shared root(1 To 100) As Car, roots As Short
Dim Shared garage(1 To 300) As Car, parked As Short
Dim Shared rootdir As String, garagedir As String
Dim Shared group(1 To 50) As Group, groups As Short
Dim Shared lastcommand(1 To 10) As String, locked As String


Sub LoadConfig
	rootdir = "." : garagedir = "garage"
	locked = "coun"
	If Not FileExists("sgar.cfg") Then Exit Sub
	
	Dim f As Integer, s As String
	Dim p As String, v As String, n As Short
	
	f = FreeFile
	Open "sgar.cfg" For Input As f
	While Not EoF(f)
		Line Input #f, s
		n = InStr(s, "=")
		If n Then
			p = LCase(Trim(Left(s, n - 1)))
			v = Trim(Mid(s, n + 1))
			
			Select Case p
				Case "stunts", "root"
					rootdir = v
				Case "garage"
					garagedir = v
				Case "locked"
					locked = LCase(Trim(v))
				Case "group"
					n = InStr(v, ":")
					If n Then
						groups += 1
						group(groups).gname = LCase(Trim(Left(v, n - 1)))
						group(groups).cars = LCase(Trim(Mid(v, n + 1)))					
					End If
			End Select
		End If
	WEnd
	Close f
End Sub


Sub SaveConfig
	Dim f As Integer
	
	f = FreeFile
	Open "sgar.cfg" For Output As f
	Print #f, "stunts=" & rootdir
	Print #f, "garage=" & garagedir
	Print #f, "locked=" & locked
	For i As Short = 1 To groups
		Print #f, "group=" & group(i).gname & ":" & group(i).cars
	Next i
	Close f
End Sub


Sub ReadDirs
	Dim As String s, t, id
	Dim code As Byte
	
	For collection As Byte = 0 To 1
		#ifdef __FB_LINUX__
			If collection Then
				s = Dir(garagedir & SLASH & "*")
			Else
				s = Dir(rootdir & SLASH & "*")
			End If
		#else
			If collection Then
				s = Dir(garagedir & SLASH & "*.*")
			Else
				s = Dir(rootdir & SLASH & "*.*")
			End If
		#endif
		While Len(s)
			t = LCase(s) : code = 0
			If Left(t, 3) = "car" AndAlso Right(t, 4) = ".res" AndAlso Len(t) = 11 Then
				'Res file
				id = Mid(t, 4, 4)
				code = 1
			ElseIf Left(t, 4) = "stda" AndAlso Right(t, 4) = ".vsh" AndAlso Len(t) = 12 Then
				'Uncompressed stda file
				id = Mid(t, 5, 4)
				code = 3
			ElseIf Left(t, 4) = "stda" AndAlso Right(t, 4) = ".pvs" AndAlso Len(t) = 12 Then
				'Compressed stda file
				id = Mid(t, 5, 4)
				code = 4
			ElseIf Left(t, 4) = "stdb" AndAlso Right(t, 4) = ".vsh" AndAlso Len(t) = 12 Then
				'Uncompressed stdb file
				id = Mid(t, 5, 4)
				code = 5
			ElseIf Left(t, 4) = "stdb" AndAlso Right(t, 4) = ".pvs" AndAlso Len(t) = 12 Then
				'Compressed stdb file
				id = Mid(t, 5, 4)
				code = 6
			ElseIf Left(t, 2) = "st" AndAlso Right(t, 4) = ".3sh" AndAlso Len(t) = 10 Then
				'Uncompressed shape file
				id = Mid(t, 3, 4)
				code = 7
			ElseIf Left(t, 2) = "st" AndAlso Right(t, 4) = ".p3s" AndAlso Len(t) = 10 Then
				'Compressed shape file
				id = Mid(t, 3, 4)
				code = 8
			End If
			
			If code Then
				If collection Then
					Dim q As Byte = 0
					
					For i As Short = 1 To parked
						If garage(i).id = id Then
							q = -1
							garage(i).file(code) = s
							garage(i).numfiles += 1
							Exit For
						End If
					Next i
					
					If q = 0 Then
						parked += 1
						garage(parked).id = id
						garage(parked).numfiles = 1
						For i As Byte = 1 To 8
							garage(parked).file(i) = ""
						Next i
						garage(parked).file(code) = s
					End If
				Else
					Dim q As Byte = 0
					
					For i As Short = 1 To roots
						If root(i).id = id Then
							q = -1
							root(i).file(code) = s
							root(i).numfiles += 1
							Exit For
						End If
					Next i
					
					If q = 0 Then
						roots += 1
						root(roots).id = id
						root(roots).numfiles = 1
						For i As Byte = 1 To 8
							root(roots).file(i) = ""
						Next i
						root(roots).file(code) = s
					End If
				End If
			End If
		
			s = Dir
		WEnd
	Next collection
End Sub


Sub ExtractCarName (c As Car, path As String)
	If Len(c.file(1)) = 0 Then
		c.carname = "Unknown"
		Exit Sub
	End If
	
	Dim f As Integer, s As String, n As Short, l As Long
	
	f = FreeFile
	Open path & SLASH & c.file(1) For Binary Access Read As f
	s = Space(100)
	Get #f, , s
	n = InStr(s, "gnam")
	Get #f, n + 16, l
	l += 39
	s = Space(255)
	Get #f, l, s
	n = InStr(s, Chr(0))
	If n Then s = Left(s, n - 1) Else s = Left(s, 40)
	Close f
	
	c.carname = s
End Sub


Sub Init
	LoadConfig
	If rootdir = garagedir Then
		Print
		Print "Error: Stunts and garage directory are the same or undefined!"
		Print "       Edit sgar.cfg to set them correctly."
		Print
		End
	End If
	
	Dim f As Integer
	
	f = FreeFile
	Open rootdir & SLASH & "test981" For Output As f : Close f
	If FileExists(rootdir & SLASH & "test981") Then
		Kill rootdir & SLASH & "test981"
	Else
		Print
		Print "Error: Stunts directory not properly configured"
		Print "       Edit sgar.cfg to select the right directory for it"
		Print
		End
	End If
	Open garagedir & SLASH & "test981" For Output As f : Close f
	If FileExists(garagedir & SLASH & "test981") Then
		Kill garagedir & SLASH & "test981"
	Else
		MkDir garagedir
		Open garagedir & SLASH & "test981" For Output As f : Close f
		If FileExists(garagedir & SLASH & "test981") Then
			Kill garagedir & SLASH & "test981"
			Print
			Print "Created garage directory at " & garagedir
			Print
		Else
			Print
			Print "Error: The current garage directory is not accessible and could"
			Print "       not be created. Please edit sgar.cfg to select a garage"
			Print "       directory that can be used"
			Print
			End
		End If
	End If
	
	ReadDirs

	For i As Short = 1 To roots
		ExtractCarName root(i), rootdir
	Next i
	For i As Short = 1 To parked
		ExtractCarName garage(i), garagedir
	Next i
End Sub


Function Resolve(s As String) As String
	'Obtain a list of cars from a search string that can
	'be a car, a list of cars, a group or a filter
	Dim n As Short, t As String, ss As String
	Dim result As String
	
	n = InStr(s, ",")
	If n Then	'It's a list
		Dim temp As String
		
		t = s
		Do
			n = InStr(t, ",")
			If n Then
				ss = Left(t, n - 1)
				t = Mid(t, n + 1)
			Else
				ss = t
				t = ""
			End If
			
			temp = Resolve(ss)
			If Len(temp) Then
				If Len(result) Then result &= ","
				result &= temp
			End If
		Loop Until Len(t) = 0
	ElseIf Left(s, 1) = "#" Then	'It's a group
		For i As Short = 1 To groups
			If LCase(group(i).gname) = LCase(Mid(s, 2)) Then Return group(i).cars
		Next i
	ElseIf Len(s) = 4 AndAlso InStr(s, "*") = 0 AndAlso InStr(s, ",") = 0 Then	'It's a car
		Return LCase(s)
	Else	'It's a filter
		Dim filtype As Byte, q As Byte
		
		If s = "*" Then
			filtype = 0
		ElseIf Left(s, 1) = "*" AndAlso Right(s, 1) = "*" Then
			filtype = 1
			ss = LCase(Mid(s, 2, Len(s) - 2))
		ElseIf Left(s, 1) = "*" Then
			filtype = 2
			ss = LCase(Mid(s, 2))
		ElseIf Right(s, 1) = "*" Then
			filtype = 3
			ss = LCase(Left(s, Len(s) - 1))
		Else
			filtype = 1
			ss = LCase(s)
		End If
		
		For i As Short = 1 To roots
			q = 0
			Select Case filtype
				Case 0 : q = -1		'All
				Case 1
					If InStr(LCase(root(i).carname), ss) _
						OrElse InStr(LCase(root(i).id), ss) Then q = -1
				Case 2
					If Right(LCase(root(i).carname), Len(ss)) = ss _
						OrElse Right(LCase(root(i).id), Len(ss)) = ss Then q = -1
				Case Else
					If Left(LCase(root(i).carname), Len(ss)) = ss _
						OrElse Left(LCase(root(i).id), Len(ss)) = ss Then q = -1
			End Select
			
			If q <> 0 AndAlso InStr(result, LCase(root(i).id)) = 0 Then	'Add car to the list
				If Len(result) Then result &= ","
				result &= LCase(root(i).id)
			End If
		Next i
		
		For i As Short = 1 To parked
			q = 0
			Select Case filtype
				Case 0 : q = -1		'All
				Case 1
					If InStr(LCase(garage(i).carname), ss) _
						OrElse InStr(LCase(garage(i).id), ss) Then q = -1
				Case 2
					If Right(LCase(garage(i).carname), Len(ss)) = ss _
						OrElse Right(LCase(garage(i).id), Len(ss)) = ss Then q = -1
				Case Else
					If Left(LCase(garage(i).carname), Len(ss)) = ss _
						OrElse Left(LCase(garage(i).id), Len(ss)) = ss Then q = -1
			End Select
			
			If q <> 0 AndAlso InStr(result, LCase(garage(i).id)) = 0 Then	'Add car to the list
				If Len(result) Then result &= ","
				result &= LCase(garage(i).id)
			End If
		Next i
	End If

	'Get rid of repeated cars
	t = result
	result = ""
	Do
		n = InStr(t, ",")
		If n Then
			ss = Left(t, n - 1)
			t = Mid(t, n + 1)
		Else
			ss = t
			t = ""
		End If
		
		If InStr(result, ss) = 0 Then
			If Len(result) Then result &= ","
			result &= ss
		End If
	Loop Until Len(t) = 0
	
	Return result
End Function


Function MoveCar (id As String, direc As Byte) As Byte
	Dim n As Short, many As Byte
	
	If Left(id, 1) = "#" Then	'It's a referenced group
		many = 1
	ElseIf InStr(id, ",") Then	'It's an ad-hoc group
		many = 2
	Else						'It's an individual car
		many = 0
	End If
	
	If many Then
		'This is a group. Call repeatedly for each car
		Dim s As String, ercount As Short
		Dim thiscar As String, gn As Short = 0
		
		If many = 1 Then
			For i As Short = 1 To groups
				If group(i).gname = LCase(Mid(id, 2)) Then
					gn = i
					Exit For
				End If
			Next i
			
			If gn = 0 Then Return 1	'Group not found
		
			s = group(gn).cars
		Else
			s = id
		End If
		
		Do
			n = InStr(s, ",")
			
			If n Then
				thiscar = LCase(RTrim(Left(s, n - 1)))
				s = LTrim(Mid(s, n + 1))
			Else
				thiscar = LCase(Trim(s))
				s = ""
			End If
			
			ercount += MoveCar(thiscar, direc)
		Loop Until Len(s) = 0
		
		If ercount Then Return 2 Else Return 0
	End If
	
	'If car is locked, it cannot be moved
	If InStr(LCase(locked), LCase(id)) Then Return 3
	
	If direc Then
		'Find car
		n = 0
		For i As Short = 1 To parked
			If LCase(garage(i).id) = LCase(id) Then
				n = i
				Exit For
			End If
		Next i
		
		If n = 0 Then Return 1	'Car not found
		
		'Move files
		For i As Short = 1 To 8
			If Len(garage(n).file(i)) Then
				FileCopy garagedir & SLASH & garage(n).file(i), rootdir & SLASH & garage(n).file(i)
				Kill garagedir & SLASH & garage(n).file(i)
			End If
		Next i
		
		'Move car in lists
		roots += 1
		root(roots) = garage(n)
		For i As Short = n To parked
			garage(i) = garage(i + 1)
		Next i
		parked -= 1
	Else
		'Find car
		n = 0
		For i As Short = 1 To roots
			If LCase(root(i).id) = LCase(id) Then
				n = i
				Exit For
			End If
		Next i
		
		If n = 0 Then Return 1	'Car not found
		
		'Move files
		For i As Short = 1 To 8
			If Len(root(n).file(i)) Then
				FileCopy rootdir & SLASH & root(n).file(i), garagedir & SLASH & root(n).file(i)
				Kill rootdir & SLASH & root(n).file(i)
			End If
		Next i
		
		'Move car in lists
		parked += 1
		garage(parked) = root(n)
		For i As Short = n To roots
			root(i) = root(i + 1)
		Next i
		roots -= 1
	End If
	
	Return 0	'Executed successfully
End Function


Sub RunCommand (s As String)
	Dim As String c, p(1 To 5), t
	Dim n As Short, ps As Byte
	
	n = InStr(s, " ")
	If n Then
		c = LCase(Trim(Left(s, n - 1)))
		t = Trim(Mid(s, n + 1))
		For i As Byte = 1 To 5
			ps += 1
			n = InStr(t, " ")
			If n Then
				p(ps) = RTrim(Left(t, n - 1))
				t = LTrim(Mid(t, n))
			Else
				p(ps) = t
				Exit For
			End If
		Next i
	Else
		c = LCase(Trim(s))
		ps = 0
	End If
	
	Select Case c
		Case ""
		Case "cars"
			Print roots & " cars ready for Stunts"
			Print parked & " cars parked"
			Print roots + parked & " total cars"
			If Len(locked) Then
				Dim t As String, tn As Short, num As Short
				
				t = locked
				Do
					tn = InStr(t, ",")
					If tn Then
						num += 1
						t = Mid(t, tn + 1)
					Else
						num += 1
						Exit Do
					End If
				Loop
				Print num & " locked"
			End If
			Print
		Case "dirs"
			Print "Stunts directory: " & rootdir
			Print "Garage directory: " & garagedir
			Print
		Case "car"
			If ps <> 1 Then
				Print "Use: car <carid>"
				Print
				Exit Sub
			End If
			
			If InStr(p(1), ",") Then
				Print "Only a single car ID allowed for this command"
				Print
				Exit Sub
			End If
			
			For i As Short = 1 To roots
				If LCase(root(i).id) = LCase(p(1)) Then
					Print "Car name: " & root(i).carname
					Print "Car ID:   " & UCase(root(i).id)
					Print "Found in: Stunts"
					Print "Groups:   ";
					Dim q As Short = 0
					For j As Short = 1 To groups
						If InStr(LCase(group(j).cars), LCase(root(i).id)) Then
							If q Then Print ", ";
							q += 1
							Print group(j).gname;
						End If
					Next j
					If q = 0 Then Print "None" Else Print
					Print "Files:    " & root(i).numfiles
					For j As Byte = 1 To 8
						If Len(root(i).file(j)) Then
							Print "  - " & root(i).file(j);
							Select Case Right(LCase(root(i).file(j)), 4)
								Case ".pvs", ".p3s", ".pre" : Print " (packed)"
								Case Else : Print " (unpacked)"
							End Select
						End If
					Next j
					Print
					Exit Sub
				End If
			Next i
			
			For i As Short = 1 To parked
				If LCase(garage(i).id) = LCase(p(1)) Then
					Print "Car name: " & garage(i).carname
					Print "Car ID:   " & UCase(garage(i).id)
					Print "Found in: Garage"
					Print "Groups:   ";
					Dim q As Short = 0
					For j As Short = 1 To groups
						If InStr(LCase(group(j).cars), LCase(garage(i).id)) Then
							If q Then Print ", ";
							q += 1
							Print group(j).gname;
						End If
					Next j
					If q = 0 Then Print "None" Else Print
					Print "Files:    " & garage(i).numfiles
					For j As Byte = 1 To 8
						If Len(garage(i).file(j)) Then
							Print "  - " & garage(i).file(j);
							Select Case Right(LCase(garage(i).file(j)), 4)
								Case ".pvs", ".p3s", ".pre" : Print " (packed)"
								Case Else : Print " (unpacked)"
							End Select
						End If
					Next j
					Print
					Exit Sub
				End If
			Next i
			
			Print "Car not found: " & p(1)
			Print
		Case "list", "ls", "lp", "lg", "l"
			Dim As Byte fstunts, fgarage	'Filter flags
			Dim thislist As String, acar As String, comma As Short
			Dim carcount As Short, found As Byte
			
			If Right(c, 1) = "s" Then
				fstunts = -1 : fgarage = 0
			ElseIf Right(c, 1) = "p" OrElse Right(c, 1) = "g" Then
				fstunts = 0 : fgarage = -1
			Else
				fstunts = -1 : fgarage = -1
			End If
			
			If p(1) = "" Then p(1) = "*"
			thislist = Resolve(p(1)) : carcount = 0
			Do
				comma = InStr(thislist, ",")
				If comma Then
					acar = Left(thislist, comma - 1)
					thislist = Mid(thislist, comma + 1)
				Else
					acar = thislist
					thislist = ""
				End If
				
				found = 0
				If fstunts Then
					For i As Short = 1 To roots
						If LCase(root(i).id) = LCase(acar) Then
							Print root(i).carname & Space(30 - Len(root(i).carname));
							Print UCase(root(i).id);
							If root(i).numfiles <> 4 Then Print "!  "; Else Print "   ";
							Print "Stunts"
							carcount += 1							
							found = -1
							Exit For
						End If
					Next i
				End If
				
				If fgarage AndAlso found = 0 Then
					For i As Short = 1 To parked
						If LCase(garage(i).id) = LCase(acar) Then
							Print garage(i).carname & Space(30 - Len(garage(i).carname));
							Print UCase(garage(i).id);
							If garage(i).numfiles <> 4 Then Print "!  "; Else Print "   ";
							Print "Garage"
							carcount += 1							
							found = -1
							Exit For
						End If
					Next i
				End If
			Loop Until Len(thislist) = 0
			If carcount > 1 Then
				Print carcount & " cars found"
			ElseIf carcount = 1 Then
				Print "1 car found"
			Else
				Print "No cars found"
			End If
			Print
		Case "park"
			Dim n As Short, t As String, tt As String, c As Short
			
			If ps <> 1 Then
				Print "Use: park <carID>"
				Print
			Else
				t = Resolve(p(1))
				Do
					c = InStr(t, ",")
					If c Then
						tt = Left(t, c - 1)
						t = Mid(t, c + 1)
					Else
						tt = t
						t = ""
					End If
					
					n = MoveCar(tt, 0)
					Select Case n
						Case 0 : Print "Successfully parked " & tt
						Case 1 : Print "Car not found: " & tt
						Case 2 : Print "Some cars parked"
						Case 3 : Print "Car is locked and cannot be parked: " & tt
						Case Else : Print "Unknown error"
					End Select
				Loop Until Len(t) = 0
				Print
			End If
		Case "retrieve", "bring"
			Dim n As Short, t As String, tt As String, c As Short
			
			If ps <> 1 Then
				Print "Use: retrieve|bring <carID>"
				Print
			Else
				t = Resolve(p(1))
				Do
					c = InStr(t, ",")
					If c Then
						tt = Left(t, c - 1)
						t = Mid(t, c + 1)
					Else
						tt = t
						t = ""
					End If
				
					n = MoveCar(tt, -1)
					Select Case n
						Case 0 : Print "Successfully retrieved " & tt
						Case 1 : Print "Car not found: " & tt
						Case 2 : Print "Some cars retrieved"
						Case 3 : Print "Car is locked and cannot be retrieved: " & tt
						Case Else : Print "Unknown error"
					End Select
				Loop Until Len(t) = 0
				Print
			End If
		Case "groups", "gs"
			Dim i As Short
			For i = 1 To groups
				Print group(i).gname
			Next i
			Print
			Print i - 1 & " group";
			If i - 1 = 1 Then Print Else Print "s"
			Print
		Case "group", "g"
			If ps <> 1 Then
				Print "Please specify a group name"
				Print
			Else
				Dim n As Short, m As Short
				
				If Left(p(1), 1) = "#" Then p(1) = Mid(p(1), 2)
				
				For i As Short = 1 To groups
					If group(i).gname = LCase(p(1)) Then
						n = i
						Exit For
					End If
				Next i
				
				If n Then
					Dim b As String, thiscar As String
					Dim q As Byte, count As Short
					
					b = group(n).cars
					Do
						m = InStr(b, ",")
						If m Then
							thiscar = LCase(Trim(Left(b, m - 1)))
							b = LTrim(Mid(b, m + 1))
						Else
							thiscar = LCase(Trim(b))
							b = ""
						End If
						
						q = 0
						'Try to find the car in root
						For i As Short = 1 To roots
							If root(i).id = thiscar Then
								Print root(i).carname & " (" & UCase(root(i).id) & ") - Found in Stunts"
								q += 1 : count += 1
								Exit For
							End If
						Next i
						
						'Try to find it in the garage
						For i As Short = 1 To parked
							If garage(i).id = thiscar Then
								Print garage(i).carname & " (" & UCase(garage(i).id) & ") - Found in garage"
								q += 1 : count += 1
								Exit For
							End If
						Next i
						
						If q > 1 Then
							Print "Warning: Car duplicated!!!"
						ElseIf q = 0 Then
							Print UCase(thiscar) & " - Car not found"
						End If
					Loop Until Len(b) = 0
					Print
					Print count & " cars found"
					Print
				Else
					Print "Group not found: " & p(1)
					Print
				End If
			End If
		Case "addto"
			Dim thisgroup As Short, st As String, ss As String
			
			If ps <> 2 Then
				Print "Use: addto <group> <carid>"
				Print
				Exit Sub
			End If
			
			For i As Short = 1 To groups
				If group(i).gname = LCase(p(1)) Then
					thisgroup = i
					Exit For
				End If
			Next i
			
			st = Resolve(p(2))
			
			Do
				n = InStr(st, ",")
				If n Then
					ss = Left(st, n - 1)
					st = Mid(st, n + 1)
				Else
					ss = st
					st = ""
				End If
				
				If Len(ss) <> 4 Then
					Print "invalid car id: " & ss
				ElseIf thisgroup = 0 Then
					groups += 1
					group(groups).gname = LCase(p(1))
					group(groups).cars = LCase(ss)
					Print "Created group " & p(1) & " with car " & ss
					thisgroup = groups
				ElseIf InStr(LCase(group(thisgroup).cars), LCase(ss)) Then
					Print "Car " & ss & " is already present in group " & p(1)
				Else
					group(thisgroup).cars &= "," & LCase(ss)
					Print "Added " & ss & " to " & p(1)
				End If
			Loop Until Len(st) = 0
			Print
			SaveConfig
		Case "removefrom", "rf"
			Dim thisgroup As Short, st As String, ss As String, nn As Short
			
			If ps <> 2 Then
				Print "Use: removefrom/rf <group> <carid>"
				Print
				Exit Sub
			End If
			
			For i As Short = 1 To groups
				If group(i).gname = LCase(p(1)) Then
					thisgroup = i
					Exit For
				End If
			Next i
			
			If thisgroup = 0 Then
				Print "Group not found: " & p(1)
				Print
				Exit Sub
			End If
			
			st = Resolve(p(2))
			Do
				nn = InStr(st, ",")
				If nn Then
					ss = Left(st, nn - 1)
					st = Mid(st, nn + 1)
				Else
					ss = st
					st = ""
				End If
				
				n = InStr(LCase(group(thisgroup).cars), LCase(ss))
				If Len(ss) <> 4 Then
					Print "invalid car id: " & ss
				ElseIf n = 0 Then
					Print "Car " & ss & " was not in that group"
				Else
					Dim As String b1, b2, temp
					b1 = LCase(Trim(group(thisgroup).cars))
					Do
						n = InStr(b1, ",")
						If n Then
							temp = RTrim(Left(b1, n - 1))
							b1 = LTrim(Mid(b1, n + 1))
						Else
							temp = RTrim(b1)
							b1 = ""
						End If
						
						If temp <> LCase(ss) Then
							If b2 = "" Then b2 = temp Else b2 &= "," & temp
						End If
					Loop Until Len(b1) = 0
					If Len(b2) Then
						group(thisgroup).cars = b2
						Print "Removed " & ss & " from group " & p(1)
					Else
						For i As Short = thisgroup To groups - 1
							group(i) = group(i + 1)
						Next i
						groups -= 1
						Print "Removed group " & p(1)
						Exit Do
					End If
				End If
			Loop Until Len(st) = 0
			Print
			SaveConfig
		Case "delgroup", "rmgroup"
			If ps <> 1 Then
				Print "Use: delgroup|rmgroup <group>"
				Print
				Exit Sub
			End If
			
			If Left(p(1), 1) = "#" Then p(1) = Mid(p(1), 2)
			
			Dim q As Byte = 0
			
			For i As Short = 1 To groups
				If LCase(p(1)) = LCase(group(i).gname) Then
					q = -1
					For j As Short = i To groups - 1
						group(j) = group(j + 1)
					Next j
					groups -= 1
					SaveConfig
					Print "Removed group " & p(1)
					Print
					Exit For
				End If
			Next i
			
			If q = 0 Then
				Print "Group not found: " & p(1)
				Print
			End If
		Case "groupthese", "gt"
			If ps <> 1 Then
				Print "Use: groupthese|gs <group>"
				Print
				Exit Sub
			End If
			If roots = 0 Then
				Print "No cars to add to group"
				Print
				Exit Sub
			End If
			For i As Short = 1 To roots
				RunCommand "addto " & p(1) & " " & root(i).id
			Next i
			SaveConfig
		Case "only"
			If ps <> 1 Then
				Print "Use: only <group>"
				Print
				Exit Sub
			End If
			
			If Left(p(1), 1) = "#" Then p(1) = Mid(p(1), 2)
			
			Dim g As Short = 0
			
			For i As Short = 1 To groups
				If LCase(p(1)) = LCase(group(i).gname) Then
					g = i
					Exit For
				End If
			Next i
			
			If g = 0 Then
				Print "Group not found: " & p(1)
				Print
				Exit Sub
			End If
			
			'Move to garage cars that are not in group
			For i As Short = 1 To roots
				If InStr(LCase(group(g).cars), LCase(root(i).id)) = 0 Then
					RunCommand "park " & root(i).id
					i -= 1
				End If
				If i = roots Then Exit For
			Next i
			
			'Move to Stunts cars that are in group
			For i As Short = 1 To parked
				If InStr(LCase(group(g).cars), LCase(garage(i).id)) Then
					RunCommand "bring " & garage(i).id
					i -= 1
				End If
				If i = parked Then Exit For
			Next i
		Case "lock"
			If ps <> 1 Then
				Print "Use: lock <carID>"
				Print
				Exit Sub
			End If
			p(1) = LCase(Trim(p(1)))
			
			If InStr(p(1), ",") Then
				Print "Only one car ID allowed for this command"
				Print
				Exit Sub
			End If
			
			If Len(p(1)) <> 4 Then
				Print "Invalid car ID: " & UCase(p(1))
				Print
				Exit Sub
			End If
			If InStr(locked, p(1)) Then
				Print "Car already locked: " & UCase(p(1))
				Print
				Exit Sub
			End If
			If Len(locked) Then locked &= ","
			locked &= p(1)
			Print "Locked " & UCase(p(1))
			Print
			SaveConfig
		Case "unlock"
			If ps <> 1 Then
				Print "Use: lock <carID>"
				Print
				Exit Sub
			End If
			p(1) = LCase(Trim(p(1)))
			
			If InStr(p(1), ",") Then
				Print "Only one car ID allowed for this command"
				Print
				Exit Sub
			End If
			
			If Len(p(1)) <> 4 Or InStr(p(1), ",") <> 0 Then
				Print "Invalid car ID: " & UCase(p(1))
				Print
				Exit Sub
			End If
			Dim n As Short
			n = InStr(locked, p(1))
			If n = 0 Then
				Print "Car already unlocked: " & UCase(p(1))
				Print
				Exit Sub
			End If
			locked = Left(locked, n - 1) & Mid(locked, n + 4)
			n = InStr(locked, ",,")
			If n Then locked = Left(locked, n) & Mid(locked, n + 2)
			If Right(locked, 1) = "," Then locked = Left(locked, Len(locked) - 1)
			If Left(locked, 1) = "," Then locked = Mid(locked, 2)
			Print "Unlocked car: " & UCase(p(1))
			Print
			SaveConfig
		Case "locked"
			If Len(locked) Then
				Print "Locked cars: " & locked
			Else
				Print "No cars locked"
			End If
			Print
		Case "version", "ver"
			Print "Simple Garage v1.1 - by Cas"
			Print
		Case "resolve"
			Print Resolve(p(1))
			Print
		Case "help"
			If LCase(p(1)) = "filters" Then
				Print "Most commands in which a car o more can be specified,"
				Print "where you see a parameter like <car(s)> can use any of"
				Print "the following:"
				Print
				Print "- A single car ID, e.g.: pmin"
				Print "- A group preceeded by a # sign, e.g.: #mycars"
				Print "- A filter such as: Porsche*"
				Print "- A comma-separated list of any of the above: jagu,*n,#orig,fgto"
				Print
				Print "Comma-separated lists are called ad-hoc groups in contrast"
				Print "to referenced groups, which are preceeded by a #."
				Print "Filters can be of either of the forms:"
				Print
				Print "*xyz - ending in xyz"
				Print "xyz* - beginning with xyz"
				Print "*xyz* - containing xyz"
				Print "xyz - also containing xyz (as long as length is not 4)"
				Print
				Print "Four characters in a row will be assumed to be a car ID"
				Print
				Exit Sub
			End If
			
			Print "addto <group> <car(s)> - Add car(s) to group"
			Print "car <carID> - Display information about a car"
			Print "cars - Display number of cars in Stunts & garage"
			Print "delgroup|rmgroup <group> - Delete a group"
			Print "dirs - Display current directories"
			Print "exit|quit - Leave Simple Garage"
			Print "group|g <group> - List cars in group"
			Print "groups|gs - List groups"
			Print "groupthese|gt <group> - Add all cars currently in Stunts to group"
			Print "lg|lp <car(s)> - List cars among the ones in the garage"
			Print "list|l <car(s)> - List cars among all available"
			Print "lock <carID> - Lock car"
			Print "locked - List locked car IDs"
			Print "ls <car(s)> - List cars among those ready in Stunts"
			Print "only <group> - Bring all cars in group and park all others"
			Print "park <car(s)> - Park car(s) in garage"
			Print "removefrom|rf <group> <car(s)> - Remove car(s) from group"
			Print "retrieve|bring <car(s)> - Bring car(s) back to Stunts"
			Print "unlock <carID> - Unlock car"
			Print
			Print "Type 'help filters' for more information on car lists"
			Print
		Case "exit", "quit" : End
		Case Else
			Print "Uknown command: " & c
			Print
	End Select
End Sub


Sub MyInput (s As String, maxlength As Short)
	Dim akey As String, t As String, cpos As Short = 1
	Dim As Short startx, starty
	Dim update As Byte = -1, lastn As Byte
	
	startx = Pos(0)
	starty = CSRLin
	
	Do
		If update Then
			Locate starty, startx
			Print t & Space(80 - Len(t) - startx);
			Line (8 * (startx - 2 + cpos), 16 * starty - 3)- Step (7, 0)
			update = 0
		End If
		
		akey = InKey
		
		Select Case akey
			Case " " To Chr(126)
				If Len(t) < maxlength Then
					t = Left(t, cpos - 1) & akey & Mid(t, cpos)
					cpos += 1
					update = -1
				End If
			Case Chr(8)
				If cpos > 1 Then
					t = Left(t, cpos - 2) & Mid(t, cpos)
					cpos -= 1
					update = -1
				End If
			Case Chr(255, 77)
				If cpos < Len(t) + 1 Then
					cpos += 1
					update = -1
				End If
			Case Chr(255, 75)
				If cpos > 1 Then
					cpos -= 1
					update = -1
				End If
			Case Chr(255, 83)
				If cpos <= Len(t) Then
					t = Left(t, cpos - 1) & Mid(t, cpos + 1)
					update = -1
				End If
			Case Chr(255, 71)
				cpos = 1
				update = -1
			Case Chr(255, 79)
				cpos = Len(t) + 1
				update = -1
			Case Chr(255, 72)
				If lastn < 10 Then
					lastn += 1
					t = lastcommand(lastn)
					cpos = Len(t) + 1
					update = -1
				End If
			Case Chr(255, 80)
				If lastn > 0 Then
					lastn -= 1
					If lastn Then
						t = lastcommand(lastn)
					Else
						t = ""
					End If
					cpos = Len(t) + 1
					update = -1
				End If
			Case Chr(13)
				s = t
				Exit Do
			Case Chr(27)
				t = ""
				cpos = 1
				update = -1
		End Select
	Loop
	
	Locate starty, startx : Print t & "  "
End Sub


If Len(Command) Then
	Init
	Print "Simple Garage 1.0"
	Print
	RunCommand Command
	End
End If

ScreenRes 640, 480
Width 80, 30

WindowTitle "Simple Garage 1.0"

Dim s As String

Color 15
Print "Simple Garage 1.0"
Print

Init

Do
	Color 10 : Print "@>";
	Color 15 : MyInput s, 70
	Color 7 : RunCommand s

	For i As Short = 10 To 2 Step -1
		lastcommand(i) = lastcommand(i - 1)
	Next i
	lastcommand(1) = s
Loop
