unit SCPU;

(* Information
   

   Program Title : CPU and FPU detection routines.
   External name : SCPU.TPU
   Version       : 2.2.
   Start date    : 16/8/96.
   Last update   : 23/2/97.
   Author        : Rob Anderton.
   Description   : CPU type, speed and make detection.
                   Version 2.0 add Cyrix chip detection based on
                   Cyrix's own CXID program. Now recognises 6x86!
                   Update v2.2 (23/2/97): MMX detection

*)

interface

{******}

const CPU_8088      = 0;    {8088}
      CPU_8086      = 1;    {8086}
      CPU_80188     = 2;    {188}
      CPU_80186     = 3;    {186}
      CPU_80286     = 4;    {286}
      CPU_80386     = 5;    {386 SX/DX}
      CPU_80486     = 6;    {486 SX/DX}
      CPU_80486SX   = 7;    {486 SX/SX2}
      CPU_80486DX   = 8;    {486 DX/DX2/DX4}

      {*** Cyrix chips (do not change) ***}

      CPU_Cx486SLC  = 9;
      CPU_Cx486DLC  = 10;
      CPU_Cx486SLC2 = 11;
      CPU_Cx486DLC2 = 12;
      CPU_Cx486SRx  = 13;
      CPU_Cx486DRx  = 14;
      CPU_Cx486SRx2 = 15;
      CPU_Cx486DRx2 = 16;
      CPU_Cx486S    = 17;   {M5}
      CPU_Cx486S2   = 18;   {M5}
      CPU_Cx486Se   = 19;
      CPU_Cx486S2e  = 20;
      CPU_Cx486DX   = 21;   {M7}
      CPU_Cx486DX2  = 22;   {M7}
      CPU_Cx486DX4  = 23;

      CPU_Cx5x86    = 24;   {Cyrix 5x86 (M1sc)}
      CPU_Cx6x86    = 25;   {Cyrix 6x86 (M1)}

      {*** Other 586/686 ***}
      CPU_80586     = 26;    {Intel Pentium, AMD K5, NexGen Nx586}
      CPU_80686     = 27;    {Intel Pentium Pro, AMD K6(?)}

      CPU_UNKNOWN   = 255;   {Unrecognised CPU}

      FPU_NONE    = 0;    {No FPU}
      FPU_8087    = 1;    {8087}
      FPU_80287   = 2;    {287}
      FPU_80387   = 3;    {387, 487, or internal FPU (DX, 586, 686)}

{******}

type CPUInfoRec = record
                        CPUType    : byte;
                        FPUType    : byte;
                        VendorID   : string[12];
                        Model      : byte;
                        Revision   : byte;
                        OverDrive  : boolean;
                        DualCPU    : boolean;
                        Capability : word;
                        MMX        : boolean;
                  end;

{******}

var CPUInfo : CPUInfoRec;

{******}

function CPU_GetName : string;
function FPU_GetName : string;

{******}

implementation

uses DOS, SCPU_CX;

{******}

var CPUID : boolean;
    Cyrix : boolean;

{******}

function CPU_GetType : byte;

var CPU   : byte;

begin
     CPU:= CPU_UNKNOWN;
     Cyrix:= Cx_Present;

     if not Cyrix then
     asm
        xor    dx, dx
        pushf

        {*** 80286 or lower? ***}

        xor    ax, ax
        push   ax
        popf
        pushf
        pop    ax
        and    ax, $F000
        cmp    ax, $F000
        je     @LOWER286

        {*** 286 or higher? ***}

        mov    dl, CPU_80286
        mov    ax, $7000
        push   ax
        popf
        pushf
        pop    ax
        and    ax, $7000
        je     @EXIT            {286}

        mov    dl, CPU_80386

        {*** 386 or higher ***}

        cli

        db $66, $8B, $DC             {mov    ebx,esp      Mark current SP    }
        db $66, $83, $E4, $FC        {and    esp,0FFFCh   Round to DWORD     }
        db $66, $9C                  {pushfd              Put Flag reg. on   }
        db $66, $58                  {pop    eax          stack from AX,     }
        db $66, $8B, $C8             {mov    ecx,eax      mark in CX         }
        db $66, $35, $00, $0, $4, $0 {xor    eax,1 shl 18 Shift alignment    }
        db $66, $50                  {push   eax          bit, pass it       }
        db $66, $9D                  {popfd               to flag register   }
        db $66, $9C                  {pushfd              Get flag from      }
        db $66, $58                  {pop    eax          stack, AX          }
        db $66, $51                  {push   ecx          Old flag data      }
        db $66, $9D                  {popfd               Restore            }
        db $66, $33, $C1             {xor    eax,ecx      Test AL bit        }
        db $66, $C1, $E8, $12        {xor    eax,12h      AL bit to bit 0    }
        db $66, $83, $E0, $01        {and    eax,1h       Mask all others    }
        db $66, $8B, $E3             {mov    esp,ebx      Restore SP         }

        sti
        mov    [CPUID], 0
        cmp    al, 1
        jne    @EXIT
        mov    [CPUID], 1

        {*** 486 or higher ***}

        mov    dl, CPU_80486

        cli

        db $66, $8B, $DC                            {mov     ebx, esp        }
        db $66, $83, $E4, $FC                       {and     esp, $FFFC      }
        db $66, $9C                                 {pushfd                  }
        db $66, $58                                 {pop     eax             }
        db $66, $8B, $C8                            {mov     ecx, eax        }
        db $66, $35, $00, $0, $20, $0               {xor     eax, 1 shl 21   }
        db $66, $50                                 {push    eax             }
        db $66, $9D                                 {popfd                   }
        db $66, $9C                                 {pushfd                  }
        db $66, $58                                 {pop     eax             }
        db $66, $51                                 {push    ecx             }
        db $66, $9D                                 {popfd                   }
        db $66, $33, $C1                            {xor     eax, ecx        }
        db $66, $C1, $E8, $15                       {ror     eax, $15        }
        db $66, $83, $E0, $01                       {and     eax, $1         }
        db $66, $8B, $E3                            {mov     esp, ebx        }

        sti

        cmp    al, 1
        jne    @EXIT

        {*** Use CPUID to get processor family - 486, 586 or 686 ***}

        db     $67
        mov    ax, $0001
        db     $0F, $A2

        shr    ax, 8
        and    al, $0F

        cmp    al, 4
        je     @486SXorDX

        cmp    al, 5
        je     @P5

        cmp    al, 6
        je     @P6

        mov    dl, CPU_UNKNOWN
        jmp    @EXIT

     @486SXorDX:

        db     $67
        mov    ax, $0001
        db     $0F, $A2

        and    dl, $1
        cmp    dl, 1
        je     @486DX
        mov    dl, CPU_80486SX
        jmp    @EXIT

     @486DX:
        mov    dl, CPU_80486DX
        jmp    @EXIT

     @P5:
        mov    dl, CPU_80586
        jmp    @EXIT

     @P6:
        mov    dl, CPU_80686
        jmp    @EXIT

        {*** Check for 8088, 8086, 80188 or 80186 ***}

     @LOWER286:

        mov    dl, CPU_80188
        mov    al, $FF
        mov    cl, $21
        shr    al, cl
        jne    @T88or86

        mov    dl, CPU_8088

        {*** Distinguish between 80x88 or 80x86 ***}

     @T88or86:

        push   cs
        pop    es
        std
        mov    di, OFFSET @endq2
        mov    al, $FB
        mov    cx, 3
        cli
        rep    stosb
        cld
        nop
        nop
        nop
        inc    dx
        nop

     @endq2:

        sti

     @EXIT:

        popf
        mov    [CPU], dl

     end
     else
     begin
          CPU:= Cx_GetCPUType;
          CPUID:= Cx_CPUID;
     end;

     CPU_GetType:= CPU;
end;

{******}

function CPU_GetVendor : string;

var N1a, N1b, N2a, N2b, N3a, N3b : array[1..2] of char;
    Vendor                       : string;

begin
     if CPUID then
     begin
          asm
             db   $66
             xor  ax, ax
             db   $0F, $A2
             mov  [N1b], bx
             mov  [N2b], dx
             mov  [N3b], cx
             db   $66
             shr  bx, 16
             mov  [N1a], bx
             db   $66
             shr  dx, 16
             mov  [N2a], dx
             db   $66
             shr  cx, 16
             mov  [N3a], cx
          end;
          Vendor:= N1b + N1a + N2b + N2a + N3b + N3a;
     end
     else if Cyrix then Vendor:= 'CyrixInstead'
                   else Vendor:= '';

     CPU_GetVendor:= Vendor;
end;

{******}

function CPU_GetModel : byte;

var Temp : byte;

begin
     if CPUID then
     asm
        db   $67
        mov  ax, $1
        db   $0F, $A2
        and  ax, $F0
        shr  al, $4
        mov  [Temp], al
     end
     else if Cyrix then
          begin
               Temp:= Cx_GetModel;
          end
          else Temp:= 0;

     CPU_GetModel:= Temp;
end;

{******}

function CPU_GetRevision : byte;

var Temp : byte;

begin
     if CPUID then
     asm
        db   $67
        mov  ax, $0001
        db   $0F, $A2
        and  al, $0F
        mov  [Temp], al
     end
     else if Cyrix then
          begin
               Temp:= Cx_GetRevision;
          end
          else Temp:= 0;

     CPU_GetRevision:= Temp;
end;

{******}

function CPU_GetOverdrive : boolean;

var Temp : word;

begin
     if CPUID then
     asm
        db   $67
        mov  ax, $0001
        db   $0F, $A2
        shr  ax, 12
        mov  cx, 0
        cmp  al, 1
        jne  @EXIT
        mov  cx, 1

     @EXIT:

        mov  [Temp], cx

     end
     else Temp:= 0;

     CPU_GetOverdrive:= (Temp = 1);

end;

{******}

function CPU_GetDual : boolean;

var Temp : word;

begin
     if CPUID then
     asm
        db   $67
        mov  ax, $0001
        db   $0F, $A2
        shr  ax, 12
        and  ax, $000F
        mov  cx, 0
        cmp  ax, 2
        jne  @EXIT
        mov  cx, 1

     @EXIT:

        mov  [Temp], cx

     end
     else Temp:= 0;

     CPU_GetDual:= (Temp = 1);

end;


{******}

function CPU_GetCapability : word;

var Temp : word;

begin
     if CPUID then
     asm
        db   $67
        mov  ax, $0001
        db   $0F, $A2
        mov  [Temp], dx
     end
     else Temp:= 0;

     CPU_GetCapability:= Temp;
end;

{******}

function CPU_GetMMX : boolean;

var Temp : word;

begin
     if CPUID then
     asm
        db   $67
        mov  ax, $0001
        db   $0F, $A2
        db   $66
        shr  dx, 16
        mov  [Temp], dx
     end
     else Temp:= 0;

     CPU_GetMMX:= ((Temp shr 7) and 1) = 1;
end;

{******}

function FPU_GetType : byte;

var Temp : word;
    CPZ  : word;

begin
     asm
        mov    dx, FPU_NONE

        finit
        mov    byte ptr CPZ + 1, 0
        fstcw  CPZ
        cmp    byte ptr CPZ + 1, 3
        jne    @EXIT

        mov    dx, FPU_8087
        and    CPZ, $FF7F
        fldcw  CPZ
        fdisi
        fstcw  CPZ
        test   CPZ, $80
        jnz    @EXIT

        mov    dx, FPU_80287
        finit
        fld1
        fldz
        fdiv
        fld    st
        fchs
        fcompp
        fstsw  CPZ
        mov    ah, byte ptr CPZ + 1
        sahf
        je     @EXIT

        mov    dx, FPU_80387

     @EXIT:

        mov   [Temp], dx

     end;

     FPU_GetType:= byte(Temp);
end;

{******}

function CPU_GetName : string;

var s : string;

begin
     case CPUInfo.CpuType of
          CPU_8088    : s:= '8088';
          CPU_8086    : s:= '8086';
          CPU_80188   : s:= '80188';
          CPU_80186   : s:= '80186';
          CPU_80286   : s:= '80286';
          CPU_80386   : s:= '80386';
          CPU_80486   : s:= '80486';
          CPU_80486SX : s:= '80486SX';
          CPU_80486DX : s:= '80486DX';

          {*** Cyrix chips ***}

          CPU_Cx486SLC  : s:= 'Cyrix Cx486SLC';
          CPU_Cx486DLC  : s:= 'Cyrix Cx486DLC';
          CPU_Cx486SLC2 : s:= 'Cyrix Cx486SLC2';
          CPU_Cx486DLC2 : s:= 'Cyrix Cx486DLC2';
          CPU_Cx486SRx  : s:= 'Cyrix Cx486SRx';
          CPU_Cx486DRx  : s:= 'Cyrix Cx486DRx';
          CPU_Cx486SRx2 : s:= 'Cyrix Cx486SRx2';
          CPU_Cx486DRx2 : s:= 'Cyrix Cx486DRx2';
          CPU_Cx486S    : s:= 'Cyrix Cx486S';
          CPU_Cx486S2   : s:= 'Cyrix Cx486S2';
          CPU_Cx486Se   : s:= 'Cyrix Cx486Se';
          CPU_Cx486S2e  : s:= 'Cyrix Cx486S2e';
          CPU_Cx486DX   : s:= 'Cyrix Cx486DX';
          CPU_Cx486DX2  : s:= 'Cyrix Cx486DX2';
          CPU_Cx486DX4  : s:= 'Cyrix Cx486DX4';

          CPU_Cx5x86    : s:= 'Cyrix 5x86';
          CPU_Cx6x86    : s:= 'Cyrix 6x86';

          {*** Pentiums & Compatibles ***}

          CPU_80586   : s:= 'Intel Pentium or compatible';
          CPU_80686   : s:= 'Intel Pentium Pro or compatible';
                 else   s:= 'Unknown.';
     end;
     CPU_GetName:= s;
end;

{******}

function FPU_GetName : string;

var s : string;

begin
     case CPUInfo.FPUType of
          FPU_NONE  : s:= 'No FPU present';
          FPU_8087  : s:= '8087';
          FPU_80287 : s:= '80287';
          FPU_80387 : case CPUInfo.CPUType of
                           CPU_80486   : s:= 'Integrated FPU or 80487';
                           CPU_80486SX : s:= '80487';
                           CPU_80486DX : s:= 'Integrated FPU';
                           CPU_80586   : s:= 'Integrated FPU';
                           CPU_80686   : s:= 'Integrated FPU';
                                else     s:= '80387';
                      end;
               else   s:= 'Unknown.';
     end;
     FPU_GetName:= s;
end;

{******}

begin
     CPUID:= false;
     Cyrix:= false;
     with CPUInfo do
     begin
          CPUType:=    CPU_GetType;
          FPUType:=    FPU_GetType;
          VendorID:=   CPU_GetVendor;
          Model:=      CPU_GetModel;
          Revision:=   CPU_GetRevision;
          OverDrive:=  CPU_GetOverDrive;
          DualCPU:=    CPU_GetDual;
          Capability:= CPU_GetCapability;
          MMX:= CPU_GetMMX;
     end;
end.