unit SGFX32;

(* Information
   

   Program Title : SGFX32.
   External name : SGFX32.TPU
   Version       : 2.0
   Start date    : 28/12/96
   Last update   : 30/12/96
                     1/1/97
   Author        : Rob Anderton.
   Description   : 386+ enhanced graphics library based on SGFX386.
                   Supports standard VGA, X mode variants and S3 enhanced
                   VGA and SVGA modes.

*)

interface

{******}


{******}

type tbitmap = record end;
type PModeRec = ^TModeRec;
     TModeRec = record
                      wMode         : word; {Graphics mode (see below)}
                      wChip         : word; {Graphics chip (e.g. VGA, S3 968}
                      wMemSize      : word; {Video memory in Kb}
                      bMemModel     : byte; {Memory model (see below)}
                      bBitsPerPixel : byte; {Bits per pixel - 8, 15, 16, 24}
                      wPixelWidth   : word; {GFX_Screen width in pixels}
                      wByteWidth    : word; {GFX_Screen width in bytes}
                      wPixelHeight  : word; {GFX_Screen height in pixels}
                      wStartX       : word; {Start of the screen in VRAM}
                      wStartY       : word; {Start of the screen in VRAM}
                      wMinX         : word; {Minimum X coordinate - 0}
                      wMaxX         : word; {Maximum X coordinate - PixelWidth - 1}
                      wMinY         : word; {Minimum Y coordinate}
                      wMaxY         : word; {Maximum Y coordiante}
                      wTotalPages   : word; {Total graphics pages}
                end;

     PScreen = ^TScreen;
     TScreen = record
                     wActivePage          : word;     {Current active page}
                     wVisualPage          : word;     {Current visual page}
                     wMaxPage             : word;     {Maximum page number}

                     ModeInfo             : TModeRec; {Mode information table}

                     {See implementation for procedure/function information}

                     Func_GetVideoName    : function(ModeInfo : TModeRec) : string;

                     Proc_SetActivePage   : procedure(wPage : word);
                     Proc_SetVisualPage   : procedure(wPage : word);

                     Proc_Cls             : procedure(lCol : longint);

                     Proc_PutPixel        : procedure(wx, wy : word; lCol : longint);
                     Proc_Put2Pixel       : procedure(wx, wy : word; lCol : longint);
                     Proc_Put4Pixel       : procedure(wx, wy : word; lCol : longint);
                     Func_GetPixel        : function(wx, wy : word) : longint;

                     Proc_HLine           : procedure(wx1, wx2, wy : word; lCol : longint);
                     Proc_VLine           : procedure(wx, wy1, wy2 : word; lCol : longint);
                     Proc_Line            : procedure(wx1, wy1, wx2, wy2 : word; lCol : longint);

                     Proc_Rectangle       : procedure(wx, wy, ww, wh : word; lCol : longint);
                     Proc_FilledRectangle : procedure(wx, wy, ww, wh : word; lCol : longint);

                     Proc_Circle          : procedure(wx, wy, wr : word; lCol : longint);
                     Proc_FilledCircle    : procedure(wx, wy, wr : word; lCol : longint);

                     Proc_Triangle        : procedure(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);
                     Proc_FilledTriangle  : procedure(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);

                     Proc_Polygon         : procedure(var aPoints : array of word; lCol : longint);
                     Proc_FilledPolygon   : procedure(var aPoints : array of word; lCol : longint);

                     Proc_PutBitmap       : procedure(wx, wy : word; Bitmap : TBitmap);
                     Proc_PutMaskedBitmap : procedure(wx, wy : word; Bitmap : TBitmap);
                     Proc_GetBitmap       : procedure(wx, wy, ww, wh : word; var Bitmap : TBitmap);
               end;

{*** Graphics modes ***}

const GFX_VGA_320x200x8 = $1001; {Standard VGA - 320x200x256 - 0 pages}

      GFX_X_320x200x8   = $2001; {Mode X - 320x200x256 - 4 pages}
      GFX_X_320x240x8   = $2002; {Mode X - 320x240x256 - 3 pages}
      GFX_X_320x400x8   = $2003; {Mode X - 320x400x256 - 2 pages}
      GFX_X_360x200x8   = $2004; {Mode X - 360x200x256 - 3 pages}
      GFX_X_360x240x8   = $2005; {Mode X - 360x240x256 - 2 pages}

      GFX_S3_320x200x8    = $3001; {S3 -   320x200x256 - VRAM/64   pages - enhanced mode 13h}
      GFX_S3_640x480x8    = $3002; {S3 -   640x480x256 - VRAM/300  pages}
      GFX_S3_640x480x15   = $3003; {S3 -   640x480x32K - VRAM/600  pages}
      GFX_S3_640x480x16   = $3004; {S3 -   640x480x64K - VRAM/600  pages}
      GFX_S3_640x480x32   = $3005; {S3 -   640x480x16M - VRAM/1200 pages}
      GFX_S3_800x600x8    = $3006; {S3 -   800x600x256 - VRAM/468  pages}
      GFX_S3_800x600x15   = $3007; {S3 -   800x600x32K - VRAM/936  pages}
      GFX_S3_800x600x16   = $3008; {S3 -   800x600x64K - VRAM/936  pages}
      GFX_S3_800x600x32   = $3009; {S3 -   800x600x16M - VRAM/1872 pages}
      GFX_S3_1024x768x8   = $300A; {S3 -  1024x768x256 - VRAM/768  pages}
      GFX_S3_1024x768x15  = $300B; {S3 -  1024x768x32K - VRAM/1536 pages}
      GFX_S3_1024x768x16  = $300C; {S3 -  1024x768x64K - VRAM/1536 pages}
      GFX_S3_1024x768x32  = $300D; {S3 -  1024x768x16M - VRAM/3072 pages}
      GFX_S3_1280x1024x8  = $300E; {S3 - 1280x1024x256 - VRAM/1280 pages}
      GFX_S3_1280x1024x15 = $300F; {S3 - 1280x1024x32K - VRAM/2560 pages}
      GFX_S3_1280x1024x16 = $3010; {S3 - 1280x1024x64K - VRAM/2560 pages}

{*** Memory models ***}

const MEM_PACKED8 = 1; {one byte per pixel (256 colour mode)}
      MEM_TRUE15  = 2; {15bpp - 32K colours}
      MEM_TRUE16  = 3; {16bpp - 64K colours}
      MEM_TRUE24  = 4; {3 bytes per pixel - RGB - 16.7M colours}
      MEM_TRUE32  = 5; {4 bytes per pixel - RGBa - 16.7M colours}
      MEM_PLANAR8 = 6; {X mode 256 colour planar memory}

{*** General video constants ***}

const VID_INT        = $10;   {Video BIOS interrupt}
      VID_SEG : word = $A000; {Video memory segment}
      VID_OFS : word = $0;    {Video memory offset}
      CRTC_ADDR      = $03D4; {CRTC controller address}
      SEQU_ADDR      = $03C4; {Sequencer controller address}
      MISC_ADDR      = $03C2; {Miscellaneous graphics control register}
      GC_ADDR        = $03CE; {Graphics controller register}

{******}

var GFX_Screen   : TScreen; {This allows direct access to functions, bypassing the
                             wrapper functions below}
    GFX_Active   : boolean; {True when the graphics system is active}

{******}

procedure GFX_Init(wMode : word);
procedure GFX_Done;
procedure GFX_WaitRetrace;
procedure GFX_TextCursor(bVisible : boolean);

function  GFX_GetVideoName(ModeInfo : TModeRec) : string;
procedure GFX_SetActivePage(wPage : word);
function  GFX_GetActivePage : word;
procedure GFX_SetVisualPage(wPage : word);
function  GFX_GetVisualPage : word;

procedure GFX_Cls(lCol : longint);

procedure GFX_Putpixel(wx, wy : word; lCol : longint);
procedure GFX_Put2pixel(wx, wy : word; lCol : longint);
procedure GFX_Put4Pixel(wx, wy : word; lCol : longint);
function  GFX_Getpixel (wx, wy : word) : longint;

procedure GFX_HLine(wx1, wx2, wy : word; lCol : longint);
procedure GFX_VLine(wx, wy1, wy2 : word; lCol : longint);
procedure GFX_Line(wx1, wy1, wx2, wy2 : word; lCol : longint);

procedure GFX_Rectangle(wx, wy, ww, wh : word; lCol : longint);
procedure GFX_FilledRectangle(wx, wy, ww, wh : word; lCol : longint);

procedure GFX_Circle(wx, wy, wr : word; lCol : longint);
procedure GFX_FilledCircle(wx, wy, wr : word; lCol : longint);

procedure GFX_Triangle(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);
procedure GFX_FilledTriangle(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);

procedure GFX_Polygon(var aPoints : array of word; lCol : longint);
procedure GFX_FilledPolygon(var aPoints : array of word; lCol : longint);

procedure GFX_PutBitmap(wx, wy : word; Bitmap : TBitmap);
procedure GFX_PutMaskedBitmap(wx, wy : word; Bitmap : TBitmap);
procedure GFX_GetBitmap(wx, wy, ww, wh : word; var Bitmap : TBitmap);

{******}

implementation

{******}

uses SGFX_VGA, SGFX_S3, CRT;

{******}

var wTextCursor         : word;    {Stores the state of the text cursor}
    bTextCursorVisible  : boolean; {True when the text cursor is set to visible}
    wTextMode           : word;    {Store the text mode active on entry to the
                                    graphics system}

{******}

procedure GFX_ClearInfo;

(* GFX_ClearInfo all fields in GFX_Screen to 0 or nil *)

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := 0;
               wChip         := 0;
               wMemSize      := 0;
               bMemModel     := 0;
               bBitsPerPixel := 0;
               wPixelWidth   := 0;
               wByteWidth    := 0;
               wPixelHeight  := 0;
               wMinX         := 0;
               wMaxX         := 0;
               wMinY         := 0;
               wMaxY         := 0;
               wTotalPages   := 0;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := 0;
          Func_GetVideoName    := nil;
          Proc_SetActivePage   := nil;
          Proc_SetVisualPage   := nil;
          Proc_Cls             := nil;
          Proc_PutPixel        := nil;
          Proc_Put2Pixel       := nil;
          Proc_Put4Pixel       := nil;
          Func_GetPixel        := nil;
          Proc_HLine           := nil;
          Proc_VLine           := nil;
          Proc_Line            := nil;
          Proc_Rectangle       := nil;
          Proc_FilledRectangle := nil;
          Proc_Circle          := nil;
          Proc_FilledCircle    := nil;
          Proc_Triangle        := nil;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := nil;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;
          Proc_PutMaskedBitmap := nil;
          Proc_GetBitmap       := nil;
     end;
end;

{******}

procedure GFX_Init(wMode : word);

(* GFX_Init - Initialises the graphics system. wMode is one of the above
              GFX constants. GFX_Init sets up fields in GFX_Screen record  *)

begin
     case wMode of

          GFX_VGA_320x200x8 : begin
                                   VGA_Init;
                              end;

          GFX_X_320x200x8   : begin
                              end;

          GFX_X_320x240x8   : begin
                              end;

          GFX_X_320x400x8   : begin
                              end;

          GFX_X_360x200x8   : begin
                              end;

          GFX_X_360x240x8   : begin
                              end;


          GFX_S3_320x200x8  : begin
                                   S3_3001_Init;
                              end;

          GFX_S3_640x480x8  : begin
                                   S3_3002_Init;
                              end;

          GFX_S3_640x480x15 : begin
                                   S3_3003_Init;
                              end;

          GFX_S3_640x480x16 : begin
                                   S3_3004_Init;
                              end;

          GFX_S3_640x480x32 : begin
                                   S3_3005_Init;
                              end;

          GFX_S3_800x600x8  : begin
                                   S3_3006_Init;
                              end;

          GFX_S3_800x600x15 : begin
                                   S3_3007_Init;
                              end;

          GFX_S3_800x600x16 : begin
                                   S3_3008_Init;
                              end;

          GFX_S3_800x600x32 : begin
                                   S3_3009_Init;
                              end;

          GFX_S3_1024x768x8 : begin
                                   S3_300A_Init;
                              end;

          GFX_S3_1024x768x15: begin
                                   S3_300B_Init;
                              end;


          GFX_S3_1024x768x16 :begin
                                   S3_300C_Init;
                              end;

          GFX_S3_1024x768x32 :begin
                                   S3_300D_Init;
                              end;

          GFX_S3_1280x1024x8: begin
                                   S3_300E_Init;
                              end;

          GFX_S3_1280x1024x15:begin
                                   S3_300F_Init;
                              end;

          GFX_S3_1280x1024x16:begin
                                   S3_3010_Init;
                              end;

     end;
     GFX_SetActivePage(0);
     GFX_SetVisualPage(0);
     GFX_Active:= true;
end;

{******}

procedure GFX_Done;

(* GFX_Done - shuts down the graphics system and returns to the screen mode
              that was active on startup                                    *)

begin
     asm
        mov  ax, [wTextMode]
        int  VID_INT
     end;

     case GFX_Screen.ModeInfo.wMode of

          GFX_VGA_320x200x8 : begin
                                   VGA_Done;
                              end;

          GFX_X_320x200x8   : begin
                              end;

          GFX_X_320x240x8   : begin
                              end;

          GFX_X_320x400x8   : begin
                              end;

          GFX_X_360x200x8   : begin
                              end;

          GFX_X_360x240x8   : begin
                              end;


          GFX_S3_320x200x8  : begin
                                   S3_3001_Done;
                              end;

          GFX_S3_640x480x8  : begin
                                   S3_3002_Done;
                              end;

          GFX_S3_640x480x15 : begin
                                   S3_3003_Done;
                              end;

          GFX_S3_640x480x16 : begin
                                   S3_3004_Done;
                              end;

          GFX_S3_640x480x32 : begin
                                   S3_3005_Done;
                              end;

          GFX_S3_800x600x8  : begin
                                   S3_3006_Done;
                              end;

          GFX_S3_800x600x15 : begin
                                   S3_3007_Done;
                              end;

          GFX_S3_800x600x16 : begin
                                   S3_3008_Done;
                              end;

          GFX_S3_800x600x32 : begin
                                   S3_3009_Done;
                              end;

          GFX_S3_1024x768x8 : begin
                                   S3_300A_Done;
                              end;

          GFX_S3_1024x768x15: begin
                                   S3_300B_Done;
                              end;


          GFX_S3_1024x768x16 :begin
                                   S3_300C_Done;
                              end;

          GFX_S3_1024x768x32 :begin
                                   S3_300D_Done;
                              end;

          GFX_S3_1280x1024x8: begin
                                   S3_300E_Done;
                              end;

          GFX_S3_1280x1024x15:begin
                                   S3_300F_Done;
                              end;

          GFX_S3_1280x1024x16:begin
                                   S3_3010_Done;
                              end;
     end;

     GFX_ClearInfo;
     GFX_Active:= false;
end;

{******}

procedure GFX_WaitRetrace; assembler;

(* GFX_WaitRetrace - Waits for a vertical retrace. Used to reduce flicker.  *)

label l1, l2;

asm
   mov dx, 3DAh
l1:
   in  al, dx
   and al, 08h
   jnz l1
l2:
   in  al, dx
   and al, 08h
   jz  l2
end;

{******}

procedure GFX_TextCursor(bVisible : boolean);

(* GFX_TextCursor - Used to switch on or off the text mode cursor *)

var bCurCL, bCurCH : byte;

begin
     if bVisible then
     begin
          asm
             mov ah, 01h
             mov cx, [wTextCursor]
             int 10h
          end;
          bTextCursorVisible:= true;
     end
     else
     begin
          asm
             mov ah, $03
             mov bh, 0
             int $10
             mov [bCurCH], ch
             mov [bCurCL], cl

             mov ah, 01h
             or  ch, 20h
             int $10
          end;
          wTextCursor:= (word(bCurCH) shl 8) + bCurCL;
          bTextCursorVisible:= false;
     end;
end;

{******}

function GFX_GetVideoName(ModeInfo : TModeRec) : string;

(* GFX_GetVideoName - returns a string describing the graphics hardware *)

begin
     if (@GFX_Screen.Func_GetVideoName <> nil) then
        GFX_GetVideoName:= GFX_Screen.Func_GetVideoName(GFX_Screen.ModeInfo);
end;

{******}

procedure GFX_SetActivePage(wPage : word);

(* GFX_SetActivePage - sets the page for drawing operations *)

begin
     if (wPage <= GFX_Screen.wMaxPage) and
        (wPage <> GFX_Screen.wActivePage) then
          if (@GFX_Screen.Proc_SetActivePage <> nil) then
          begin
               GFX_Screen.wActivePage:= wPage;
               GFX_Screen.Proc_SetActivePage(wPage);
          end;
end;

{******}

function GFX_GetActivePage : word;

(* GFX_GetActivePage - returns the currently active page number *)

begin
     GFX_GetActivePage:= GFX_Screen.wActivePage;
end;

{******}

procedure GFX_SetVisualPage(wPage : word);

(* GFX_SetVisualPage - sets the page to be displayed *)

begin
     if (wPage <= GFX_Screen.wMaxPage) and
        (wPage <> GFX_Screen.wVisualPage) then
          if (@GFX_Screen.Proc_SetVisualPage <> nil) then
          begin
               GFX_Screen.wVisualPage:= wPage;
               GFX_Screen.Proc_SetVisualPage(wPage);
          end;
end;

{******}

function GFX_GetVisualPage : word;

(* GFX_GetVisualPage - returns the number of the current visual page *)

begin
     GFX_GetVisualPage:= GFX_Screen.wVisualPage;
end;

{******}

procedure GFX_Cls(lCol : longint);

(* GFX_Cls - clears the active GFX_Screen to the specified colour *)

begin
     if (@GFX_Screen.Proc_Cls <> nil) then GFX_Screen.Proc_Cls(lCol);
end;

{******}

procedure GFX_Putpixel(wx, wy : word; lCol : longint);

(* GFX_PutPixel - plots a single pixel at coordinates (wx, wy) in colour lCol *)

begin
     if (wx <= GFX_Screen.ModeInfo.wMaxX) and
        (wy <= GFX_Screen.ModeInfo.wMaxY) then
            if (@GFX_Screen.Proc_PutPixel <> nil) then
               GFX_Screen.Proc_PutPixel(wx, wy, lCol);
end;

{******}

procedure GFX_Put2pixel(wx, wy : word; lCol : longint);

(* GFX_Put2Pixel - same as GFX_PutPixel but plots 2 adjacent pixels *)

begin
     if (wx <= GFX_Screen.ModeInfo.wMaxX) and
        (wy <= GFX_Screen.ModeInfo.wMaxY) then
            if (@GFX_Screen.Proc_Put2Pixel <> nil) then
               GFX_Screen.Proc_Put2Pixel(wx, wy, lCol);
end;

{******}

procedure GFX_Put4Pixel(wx, wy : word; lCol : longint);

(* GFX_Put4Pixel - same as GFX_PutPixel but plots 4 adjacent pixels *)

begin
     if (wx <= GFX_Screen.ModeInfo.wMaxX) and
        (wy <= GFX_Screen.ModeInfo.wMaxY) then
            if (@GFX_Screen.Proc_Put4Pixel <> nil) then
               GFX_Screen.Proc_Put4Pixel(wx, wy, lCol);
end;

{******}

function GFX_Getpixel (wx, wy : word) : longint;

(* GFX_GetPixel - returns the colour of the pixel at coordinates (wx, wy) *)

begin
     if (wx <= GFX_Screen.ModeInfo.wMaxX) and
        (wy <= GFX_Screen.ModeInfo.wMaxY) then
            if (@GFX_Screen.Proc_PutPixel <> nil) then
               GFX_GetPixel:= GFX_Screen.Func_GetPixel(wx, wy);
end;

{******}

procedure GFX_HLine(wx1, wx2, wy : word; lCol : longint);

(* GFX_HLine - draws a horizontal line from (wx1, wy) to (wx2, wy) in colour
               lCol                                                         *)

var wTemp : word;

begin
     if (wx1 > GFX_Screen.ModeInfo.wMaxX) then wx1:= GFX_Screen.ModeInfo.wMaxX;
     if (wx2 > GFX_Screen.ModeInfo.wMaxX) then wx2:= GFX_Screen.ModeInfo.wMaxX;
     if (wy > GFX_Screen.ModeInfo.wMaxY) then wy:= GFX_Screen.ModeInfo.wMaxY;

     if wx2 < wx1 then
     begin
          wTemp:= wx2;
          wx2:= wx1;
          wx1:= wTemp;
     end;

     if wx2 = wx1 then GFX_PutPixel(wx1, wy, lCol)
                  else if (@GFX_Screen.Proc_HLine <> nil) then
                          GFX_Screen.Proc_HLine(wx1, wx2, wy, lCol);
end;

{******}

procedure GFX_VLine(wx, wy1, wy2 : word; lCol : longint);

(* GFX_VLine - draws a vertical line from (wx, wy1) to (wx, wy2) in colour
               lCol                                                         *)

var wTemp : word;

begin
     if (wx > GFX_Screen.ModeInfo.wMaxX) then wx:= GFX_Screen.ModeInfo.wMaxX;
     if (wy1 > GFX_Screen.ModeInfo.wMaxY) then wy1:= GFX_Screen.ModeInfo.wMaxY;
     if (wy2 > GFX_Screen.ModeInfo.wMaxY) then wy2:= GFX_Screen.ModeInfo.wMaxY;

     if wy1 > wy2 then
     begin
          wTemp:= wy1;
          wy1:= wy2;
          wy2:= wTemp;
     end;

     if wy1 = wy2 then GFX_PutPixel(wx, wy1, lCol)
                  else if (@GFX_Screen.Proc_VLine <> nil) then
                          GFX_Screen.Proc_VLine(wx, wy1, wy2, lCol);
end;

{******}

procedure GFX_Line(wx1, wy1, wx2, wy2 : word; lCol : longint);

(* GFX_Line - draws a line from (wx1, wy1) to (wx2, wy2) in colour lCol     *)

begin
     if (wx1 > GFX_Screen.ModeInfo.wMaxX) then wx1:= GFX_Screen.ModeInfo.wMaxX;
     if (wx2 > GFX_Screen.ModeInfo.wMaxX) then wx2:= GFX_Screen.ModeInfo.wMaxX;
     if (wy1 > GFX_Screen.ModeInfo.wMaxY) then wy1:= GFX_Screen.ModeInfo.wMaxY;
     if (wy2 > GFX_Screen.ModeInfo.wMaxY) then wy2:= GFX_Screen.ModeInfo.wMaxY;

     if (wx1 = wx2) and (wy1 <> wy2) then GFX_VLine(wx1, wy1, wy2, lCol)
     else if (wy1 = wy2) and (wx1 <> wx2) then GFX_HLine(wx1, wx2, wy1, lCol)
          else if (wx1 = wx2) and (wy1 = wy2) then GFX_PutPixel(wx1, wy1, lCol)
               else if (@GFX_Screen.Proc_Line <> nil) then
                               GFX_Screen.Proc_Line(wx1, wy1, wx2, wy2, lCol);
end;

{******}

procedure GFX_Rectangle(wx, wy, ww, wh : word; lCol : longint);

(* GFX_Rectangle - Draws a rectangle from (wx, wy) to (wx + ww, wy + wh)
                   in colour lCol                                           *)

var wx2, wy2 : word;

begin
     wx2:= wx + ww;
     wy2:= wy + wh;

     if wx > GFX_Screen.ModeInfo.wMaxX then wx:= GFX_Screen.ModeInfo.wMaxX;
     if wy > GFX_Screen.ModeInfo.wMaxY then wy:= GFX_Screen.ModeInfo.wMaxY;
     if (wx2 > GFX_Screen.ModeInfo.wMaxX) then wx2:= GFX_Screen.ModeInfo.wMaxX;
     if (wy2 > GFX_Screen.ModeInfo.wMaxY) then wy2:= GFX_Screen.ModeInfo.wMaxY;

     ww:= wx2 - wx;
     wh:= wy2 - wy;

     if ((ww = 0) and (wh = 0)) then GFX_PutPixel(wx, wy, lCol)
        else if (ww = 0) then GFX_VLine(wx, wy, wy2, lCol)
             else if (wh = 0) then GFX_HLine(wx, wx2, wy, lCol)
                  else if (@GFX_Screen.Proc_Rectangle <> nil) then
                       GFX_Screen.Proc_Rectangle(wx, wy, ww, wh, lCol);
end;

{******}

procedure GFX_FilledRectangle(wx, wy, ww, wh : word; lCol : longint);

(* GFX_FilledRectangle - Same as GFX_Rectangle but the rectangle is also
                         filled with colour lCol                            *)

var wx2, wy2 : word;

begin
     wx2:= wx + ww;
     wy2:= wy + wh;

     if wx > GFX_Screen.ModeInfo.wMaxX then wx:= GFX_Screen.ModeInfo.wMaxX;
     if wy > GFX_Screen.ModeInfo.wMaxY then wy:= GFX_Screen.ModeInfo.wMaxY;
     if (wx2 > GFX_Screen.ModeInfo.wMaxX) then wx2:= GFX_Screen.ModeInfo.wMaxX;
     if (wy2 > GFX_Screen.ModeInfo.wMaxY) then wy2:= GFX_Screen.ModeInfo.wMaxY;

     ww:= wx2 - wx;
     wh:= wy2 - wy;

     if ((ww = 0) and (wh = 0)) then GFX_PutPixel(wx, wy, lCol)
        else if (ww = 0) then GFX_VLine(wx, wy, wy2, lCol)
             else if (wh = 0) then GFX_HLine(wx, wx2, wy, lCol)
                  else if (@GFX_Screen.Proc_FilledRectangle <> nil) then
                       GFX_Screen.Proc_FilledRectangle(wx, wy, ww, wh, lCol);
end;

{******}

procedure GFX_Circle(wx, wy, wr : word; lCol : longint);

(* GFX_Circle - draws a circle with centre (wx, wy) and radius wr in colour
                lCol                                                        *)

var wxl, wxr, wyt, wyb : integer;

begin
     wxl:= wx - wr;
     wxr:= wx + wr;
     wyt:= wy - wr;
     wyb:= wy + wr;

     if (wxl < GFX_Screen.ModeInfo.wMinX) then exit;
     if (wxr > GFX_Screen.ModeInfo.wMaxX) then exit;
     if (wyt < GFX_Screen.ModeInfo.wMinY) then exit;
     if (wyb > GFX_Screen.ModeInfo.wMaxY) then exit;
     if (wr = 0) then exit;

     if (@GFX_Screen.Proc_Circle <> nil) then
        GFX_Screen.Proc_Circle(wx, wy, wr, lCol);
end;

{******}

procedure GFX_FilledCircle(wx, wy, wr : word; lCol : longint);

(* GFX_FilledCircle - same as GFX_Circle but the circle is filled with colour
                      lCol                                                  *)

var wxl, wxr, wyt, wyb : integer;

begin
     wxl:= wx - wr;
     wxr:= wx + wr;
     wyt:= wy - wr;
     wyb:= wy + wr;

     if (wxl < GFX_Screen.ModeInfo.wMinX) then exit;
     if (wxr > GFX_Screen.ModeInfo.wMaxX) then exit;
     if (wyt < GFX_Screen.ModeInfo.wMinY) then exit;
     if (wyb > GFX_Screen.ModeInfo.wMaxY) then exit;
     if (wr = 0) then exit;

     if (@GFX_Screen.Proc_FilledCircle <> nil) then
        GFX_Screen.Proc_FilledCircle(wx, wy, wr, lCol);
end;

{******}

procedure GFX_Triangle(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);

begin
     if (wx1 > GFX_Screen.ModeInfo.wMaxX) then wx1:= GFX_Screen.ModeInfo.wMaxX;
     if (wy1 > GFX_Screen.ModeInfo.wMaxY) then wy1:= GFX_Screen.ModeInfo.wMaxY;
     if (wx2 > GFX_Screen.ModeInfo.wMaxX) then wx2:= GFX_Screen.ModeInfo.wMaxX;
     if (wy2 > GFX_Screen.ModeInfo.wMaxY) then wy2:= GFX_Screen.ModeInfo.wMaxY;
     if (wx3 > GFX_Screen.ModeInfo.wMaxX) then wx3:= GFX_Screen.ModeInfo.wMaxX;
     if (wy3 > GFX_Screen.ModeInfo.wMaxY) then wy3:= GFX_Screen.ModeInfo.wMaxY;

     if (@GFX_Screen.Proc_Triangle <> nil) then
        GFX_Screen.Proc_Triangle(wx1, wy1, wx2, wy2, wx3, wy3, lCol);
end;

{******}

procedure GFX_FilledTriangle(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);

begin
     if (wx1 > GFX_Screen.ModeInfo.wMaxX) then wx1:= GFX_Screen.ModeInfo.wMaxX;
     if (wy1 > GFX_Screen.ModeInfo.wMaxY) then wy1:= GFX_Screen.ModeInfo.wMaxY;
     if (wx2 > GFX_Screen.ModeInfo.wMaxX) then wx2:= GFX_Screen.ModeInfo.wMaxX;
     if (wy2 > GFX_Screen.ModeInfo.wMaxY) then wy2:= GFX_Screen.ModeInfo.wMaxY;
     if (wx3 > GFX_Screen.ModeInfo.wMaxX) then wx3:= GFX_Screen.ModeInfo.wMaxX;
     if (wy3 > GFX_Screen.ModeInfo.wMaxY) then wy3:= GFX_Screen.ModeInfo.wMaxY;

     if (@GFX_Screen.Proc_FilledTriangle <> nil) then
        GFX_Screen.Proc_FilledTriangle(wx1, wy1, wx2, wy2, wx3, wy3, lCol);
end;

{******}

procedure GFX_Polygon(var aPoints : array of word; lCol : longint);

begin
     if (Odd(SizeOf(aPoints))) or (SizeOf(aPoints) < 4) or
        (SizeOf(aPoints) mod 4 <> 0) then exit;

     case SizeOf(aPoints) of
          4    : GFX_PutPixel(aPoints[0], aPoints[1], lCol);
          8    : GFX_Line(aPoints[0], aPoints[1], aPoints[2], aPoints[3], lCol);
         12    : GFX_Triangle(aPoints[0], aPoints[1], aPoints[2], aPoints[3],
                              aPoints[4], aPoints[5], lCol);
         16    : if (@GFX_Screen.Proc_Polygon <> nil) then
                    GFX_Screen.Proc_Polygon(aPoints, lCol);
     end;
end;

{******}

procedure GFX_FilledPolygon(var aPoints : array of word; lCol : longint);

begin
end;

{******}

procedure GFX_PutBitmap(wx, wy : word; Bitmap : TBitmap);

(* GFX_PutBitmap - puts a bitmap at position (wx, wy)                       *)

begin
     if (@GFX_Screen.Proc_PutBitmap <> nil) then
        GFX_Screen.Proc_PutBitmap(wx, wy, Bitmap);
end;

{******}

procedure GFX_PutMaskedBitmap(wx, wy : word; Bitmap : TBitmap);

(* GFX_PutMaskedBitmap - same as GFX_PutBitmap but pixels of colour zero are
                         not plotted                                        *)

begin
     if (@GFX_Screen.Proc_PutMaskedBitmap <> nil) then
        GFX_Screen.Proc_PutMaskedBitmap(wx, wy, Bitmap);
end;

{******}

procedure GFX_GetBitmap(wx, wy, ww, wh : word; var Bitmap : TBitmap);

(* GFX_GetBitmap - returns a bitmap containing the region from (wx, wy) to
                   (wx + ww, wy + wh).                                      *)

begin
     if (@GFX_Screen.Proc_GetBitmap <> nil) then
        GFX_Screen.Proc_GetBitmap(wx, wy, ww, wh, Bitmap);
end;

{*** Initialise variables ***}

begin
     GFX_ClearInfo;
     GFX_Active:= false;
     wTextMode:= LastMode;
     GFX_TextCursor(false);
     GFX_TextCursor(true);
end.