unit SGFX_S3;

(* Information
   

   Program Title : SGFX S3 Library.
   External name : SGFX_S3.TPU
   Version       : 2.0
   Start date    : 29/12/96
   Last update   : 31/12/96
                    1/1/97
   Author        : Rob Anderton.
   Description   : S3 Enhanced VGA and SVGA graphics support for SGFX32.

*)

interface

{*** S3 CHIPS ***}

const S3_911        = 1000;
      S3_924        = 1001;
      S3_801AB      = 1002;
      S3_805AB      = 1003;
      S3_801C       = 1004;
      S3_805C       = 1005;
      S3_801D       = 1006;
      S3_805D       = 1007;
      S3_801I       = 1008;
      S3_805I       = 1009;
      S3_801P       = 1010;
      S3_805P       = 1011;
      S3_928        = 1012;
      S3_928C       = 1013;
      S3_928D       = 1014;
      S3_928E       = 1015;
      S3_928PCI     = 1016;
      S3_864        = 1017;
      S3_864P       = 1018;
      S3_964        = 1019;
      S3_964P       = 1020;
      S3_732        = 1021;
      S3_764        = 1022;
      S3_765        = 1023;
      S3_866        = 1024;
      S3_868        = 1025;
      S3_968        = 1026;
      S3_325        = 1027;
      S3_Unknown    = 1028;

{******}

procedure S3_3001_Init;
procedure S3_3001_Done;

procedure S3_3002_Init;
procedure S3_3002_Done;

procedure S3_3003_Init;
procedure S3_3003_Done;

procedure S3_3004_Init;
procedure S3_3004_Done;

procedure S3_3005_Init;
procedure S3_3005_Done;

procedure S3_3006_Init;
procedure S3_3006_Done;

procedure S3_3007_Init;
procedure S3_3007_Done;

procedure S3_3008_Init;
procedure S3_3008_Done;

procedure S3_3009_Init;
procedure S3_3009_Done;

procedure S3_300A_Init;
procedure S3_300A_Done;

procedure S3_300B_Init;
procedure S3_300B_Done;

procedure S3_300C_Init;
procedure S3_300C_Done;

procedure S3_300D_Init;
procedure S3_300D_Done;

procedure S3_300E_Init;
procedure S3_300E_Done;

procedure S3_300F_Init;
procedure S3_300F_Done;

procedure S3_3010_Init;
procedure S3_3010_Done;

{******}

implementation

{******}

uses SGFX32, CRT;

{******}

{*** General S3 routines ***}

procedure S3_DetectHardware(var wChip, wMemSize : word);

(* S3_DetectHardware - returns the S3 chip and the amount of video memory   *)

var CID     : byte;
    CID2    : byte;
    CID3    : byte;
    VMS     : byte;

begin
     asm
        xor  ax, ax      {Unlock the S3 registers}
        mov  bx, CRTC_ADDR
        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, $48
        out  dx, al
        mov  dx, bx
        mov  al, $30
        out  dx, al
        inc  dx
        in   al, dx
        mov  [CID], al

        mov  dx, bx
        mov  al, $36
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $E0
        shr  al, 5
        mov  [VMS], al
     end;


     case CID of

          $81 : wChip:= S3_911;
          $82 : wChip:= S3_924;
          $90 : wChip:= S3_928;
          $91 : wChip:= S3_928C;
          $94 : wChip:= S3_928D;
          $95 : wChip:= S3_928E;

          $A0 : begin
                     asm
                        mov  dx, CRTC_ADDR
                        mov  al, $36
                        out  dx, al
                        inc  dx
                        in   al, dx
                        and  al, 3
                        mov  [CID2], al
                     end;
                     if CID2 = 0 then wChip:= S3_805AB
                                 else wChip:= S3_801AB;
                end;

          $A2 : begin
                     asm
                        mov  dx, CRTC_ADDR
                        mov  al, $36
                        out  dx, al
                        inc  dx
                        in   al, dx
                        and  al, 3
                        mov  [CID2], al
                     end;
                     if CID2 = 0 then wChip:= S3_805C
                                 else wChip:= S3_801C;
                end;

          $A5 : begin
                     asm
                        mov  dx, CRTC_ADDR
                        mov  al, $36
                        out  dx, al
                        inc  dx
                        in   al, dx
                        and  al, 3
                        mov  [CID2], al
                     end;
                     if CID2 = 0 then wChip:= S3_805D
                                 else wChip:= S3_801D;
                end;

          $A6 : begin
                     asm
                        mov  dx, CRTC_ADDR
                        mov  al, $36
                        out  dx, al
                        inc  dx
                        in   al, dx
                        and  al, 3
                        mov  [CID2], al
                     end;
                     if CID2 = 0 then wChip:= S3_805P
                                 else wChip:= S3_801P;
                end;

          $A8 : begin
                     asm
                        mov  dx, CRTC_ADDR
                        mov  al, $36
                        out  dx, al
                        inc  dx
                        in   al, dx
                        and  al, 3
                        mov  [CID2], al
                     end;
                     if CID2 = 0 then wChip:= S3_805I
                                 else wChip:= S3_801I;
                end;


          $B0 : wChip:= S3_928PCI;
          $C0 : wChip:= S3_864;
          $C1 : wChip:= S3_864P;
          $D0 : wChip:= S3_964;
          $D1 : wChip:= S3_964P;

     $E0..$E1 : begin   {Extended ChipID}
                     asm
                        mov  dx, CRTC_ADDR
                        mov  al, $2E
                        out  dx, al
                        inc  dx
                        in   al, dx
                        mov  [CID2], al
                     end;

                     case CID2 of
                          $10 : wChip:= S3_732;
                          $11 : begin
                                     asm
                                        mov  dx, CRTC_ADDR
                                        mov  al, $2F
                                        out  dx, al
                                        inc  dx
                                        in   al, dx
                                        and  al, $F0
                                        shr  al, 7
                                        mov  [CID3], al
                                     end;
                                     if CID3 = 1 then wChip:= S3_765
                                                 else wChip:= S3_764;
                                end;
                          $31 : wChip:= S3_325;
                          $80 : wChip:= S3_866;
                          $90 : wChip:= S3_868;
                          $B0 : wChip:= S3_968;
                     end;
                end;

                else wChip:= S3_Unknown; {New chips can be added to the list}
     end;

     if (wChip <> S3_911) and (wChip <> S3_924) then
     case VMS of
          0 : wMemSize:= 4096;
          2 : wMemSize:= 3072;
          3 : wMemSize:= 8192;
          4 : wMemSize:= 2048;
          5 : wMemSize:= 6144;
          6 : wMemSize:= 1024;
          7 : wMemSize:= 512;
     end
     else
     begin
          VMS:= VMS and 1;
          if VMS = 0 then wMemSize:= 1024
                     else wMemSize:= 512;
     end;

     asm
        mov  dx, CRTC_ADDR {Relock the S3 registers}
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al
     end;
end;

{******}

function S3_GetVideoName(ModeInfo : TModeRec) : string;

(* S3_GetVideoName - returns a string describing the current video hardware *)

const S3ChipName : array[1000..1028] of string =
                 (  'S3 911',   'S3 911A / 924',     'S3 801A / 801B',
            'S3 805A / 805B',         'S3 801C',            'S3 805C',
                   'S3 801D',         'S3 805D',            'S3 801I',
                   'S3 805I',         'S3 801P',            'S3 805P',
                    'S3 928',         'S3 928C',            'S3 928D',
                   'S3 928E',       'S3 928PCI',             'S3 864',
                   'S3 864P',          'S3 964',            'S3 964P',
           'S3 732 (Trio32)', 'S3 764 (Trio64)', 'S3 765 (Trio64 V+)',
                    'S3 866',  'S3 868 (video)',     'S3 968 (video)',
            'S3 325 (ViRGE)',      'Unknown S3');

begin
     if (ModeInfo.wMode < $3001) or
        (ModeInfo.wMode > $3007) or
        (ModeInfo.wChip < Low(S3ChipName)) or
        (ModeInfo.wChip > High(S3ChipName)) then S3_GetVideoName:= ''
                                            else S3_GetVideoName:= S3ChipName[ModeInfo.wChip];
end;

{******}

procedure S3_SetBank(bBank : byte);

(* S3_SetBank - sets the active read/write bank of VRAM. A bank is 64K      *)

begin
     asm
        mov  bx, CRTC_ADDR {Unlock S3 registers}
        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $A5
        out  dx, al

        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, $48
        out  dx, al
     end;

     if (GFX_Screen.ModeInfo.wChip >= S3_864) then
     begin
          asm
             mov  dx, CRTC_ADDR
             mov  al, $6A        {Sets the Extended System Control 4 register }
             out  dx, al         {which is the bank register for S3_864+ chips}
             inc  dx
             mov  al, [bBank]
             out  dx, al
          end;
     end
     else
     begin
          asm
             mov  bx, CRTC_ADDR
             mov  dx, bx
             mov  al, $31
             out  dx, al
             inc  dx
             in   al, dx        {Sets Enhanced Memory mapping and bank      }
             or   al, 9         {operation bits in CR31 memory configutarion}
             out  dx, al        {register.                                  }

             mov  dx, bx
             mov  al, $35
             out  dx, al
             inc  dx
             in   al, dx
             and  al, $F0
             mov  cl, [bBank]
             and  cl, $0F
             add  al, cl        {Sets bits 0-3 of the required bank number}
             out  dx, al
          end;
          if GFX_Screen.ModeInfo.wChip > S3_924 then
          begin
               asm
                  mov  dx, CRTC_ADDR
                  mov  al, $51
                  out  dx, al
                  inc  dx
                  in   al, dx
                  and  al, $3F
                  mov  cl, [bBank]
                  shr  cl, 2
                  and  cl, $0C
                  add  al, cl   {Sets bits 4 - 5 of the required bank number}
                  out  dx, al
               end;
          end;
     end;

     asm
        mov  bx, CRTC_ADDR {Relock the registers}
        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $5A
        out  dx, al
     end;
end;

{******}

procedure S3_SetVisualStart(wSX, wSY : word);

(* S3_SetVisualStart - sets the start address in VRAM of the visible region.
                       Used by S3_SetVisualPage, and can be used for scrolling *)

var lL            : longint;
    wL1, wL2, wL3 : word;

begin
     case GFX_Screen.ModeInfo.bMemModel of

          MEM_PACKED8 : lL:= ((longint(GFX_Screen.ModeInfo.wByteWidth) *
                               longint(wSY)) + longint(wSX)) shl 6;

          MEM_TRUE15,
          MEM_TRUE16  : lL:= ((longint(GFX_Screen.ModeInfo.wByteWidth) *
                               longint(wSY)) + longint(wSX) * 2) shl 6;

          MEM_TRUE32  : lL:= ((longint(GFX_Screen.ModeInfo.wByteWidth) *
                               longint(wSY)) + longint(wSX) * 4) shl 6;

     end;

     wL1:= (lL shr 20);
     wL2:= (lL shr 26);

     asm
        mov  bx, CRTC_ADDR {Unlock S3 registers}
        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, $48
        out  dx, al

        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $A5
        out  dx, al

        mov  dx, bx
        mov  al, $31
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $CF
        mov  cx, [wL1]
        and  cx, $30
        add  ax, cx
        out  dx, al     {Sets bits 16-17 of the display start address}
     end;

     if GFX_Screen.ModeInfo.wChip > S3_924 then
     begin
          asm
             mov  dx, CRTC_ADDR
             mov  al, $51
             out  dx, al
             inc  dx
             in   al, dx
             and  al, $FC
             mov  cx, [wL2]
             and  cx, 3
             add  al, cl
             out  dx, al   {Sets bits 18-19 of the display start address}
          end;
     end;

     wL1:= (lL and $FF) shr 5;
     wL2:= (lL shr 8) and $FF;
     wL3:= (lL shr 16) and $FF;

     asm
        mov  bx, CRTC_ADDR
        mov  dx, bx
        mov  al, $11
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $7F
        out  dx, al

        mov  dx, bx
        add  dx, 6
        in   al, dx
        mov  dx, $03C0
        mov  al, $13
        out  dx, al
        mov  ax, [wL1]
        out  dx, al

        mov  dx, $03C0
        mov  al, $33
        out  dx, al

        mov  dx, bx
        mov  al, $0D
        out  dx, al
        inc  dx
        mov  ax, [wL2]
        out  dx, al       {Set bits 0-7 of the start address}

        mov  dx, bx
        mov  al, $0C
        out  dx, al
        inc  dx
        mov  ax, [wL3]
        out  dx, al       {Set bits 8-15 of the start address}

        mov  dx, bx       {Relock registers}
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $5A
        out  dx, al
     end;
end;

{******}

procedure S3_SetVisualPage(wPage : word);

(* S3_SetVisualPage - sets the visual page using the S3_SetVisualStart
                      procedure                                             *)

var wSx, wSy : word;

begin
     wSX:= ((wPage shl GFX_Screen.ModeInfo.bBitsPerPixel)
           mod GFX_Screen.ModeInfo.wPixelWidth);
     wSY:= (GFX_Screen.ModeInfo.wPixelHeight * wPage)
           + ((wPage shl GFX_Screen.ModeInfo.bBitsPerPixel)
           div GFX_Screen.ModeInfo.wPixelWidth);

     S3_SetVisualStart(wSX, wSY);
end;

{******}

procedure S3_SetActivePage(wPage : word);

(* S3_SetActivePage - sets the page on which graphics operations are carried
                      out                                                   *)

begin
     GFX_Screen.ModeInfo.wStartX:= ((wPage shl GFX_Screen.ModeInfo.bBitsPerPixel)
                                     mod GFX_Screen.ModeInfo.wPixelWidth);
     GFX_Screen.ModeInfo.wStartY:= (GFX_Screen.ModeInfo.wPixelHeight * wPage)
                                   + ((wPage shl GFX_Screen.ModeInfo.bBitsPerPixel)
                                   div GFX_Screen.ModeInfo.wPixelWidth);
end;

{******}

function S3_AccelerateOn : boolean;

(* S3_AccelerateOn - activates S3 acceleration features *)

var bTemp : boolean;

begin
     bTemp:= true;
     if not GFX_Active then S3_AccelerateOn:= bTemp;

     if (GFX_Screen.ModeInfo.wChip > S3_924) then
     begin
          asm
             mov  bx, CRTC_ADDR
             mov  dx, bx
             mov  al, $38
             out  dx, al
             inc  dx
             mov  al, $48
             out  dx, al

             mov  dx, bx
             mov  al, $39
             out  dx, al
             inc  dx
             mov  al, $A5
             out  dx, al

             mov  dx, bx
             mov  al, $58
             out  dx, al
             inc  dx
             in   al, dx
             mov  cl, $14
             not  cl
             and  al, cl
             out  dx, al {Disabled linear address window and read ahead cache}

             mov  dx, bx
             mov  al, $40
             out  dx, al
             inc  dx
             in   al, dx
             mov  cl, 9
             mov  ch, 9
             and  ch, 1
             not  cl
             and  al, cl
             add  al, ch
             out  dx, al

             mov  dx, bx
             mov  al, $53
             out  dx, al
             inc  dx
             in   al, dx

             mov  cl, $F
             not  cl
             and  al, cl
             out  dx, al

             mov  dx, bx
             mov  al, $54
             out  dx, al
             inc  dx
             mov  al, $A0
             out  dx, al

             mov  dx, $BEE8
             mov  ax, $E000
             out  dx, ax

             mov  dx, $AAE8
             mov  ax, $FFFF
             out  dx, ax

             mov  dx, $AEE8
             mov  ax, $FFFF
             out  dx, ax
          end;

          if GFX_Screen.ModeInfo.bMemModel >= MEM_TRUE24 then
          asm
             mov  dx, $BEE8
             mov  ax, $E010
             out  dx, ax

             mov  dx, $AAE8
             mov  ax, $FFFF
             out  dx, ax

             mov  dx, $AEE8
             mov  ax, $FFFF
             out  dx, ax
          end;
     end
     else
     begin
          asm
             mov  bx, CRTC_ADDR
             mov  dx, bx
             mov  al, $38
             out  dx, al
             inc  dx
             mov  al, $48
             out  dx, al

             mov  dx, bx
             mov  al, $39
             out  dx, al
             inc  dx
             mov  al, $A5
             out  dx, al

             mov  dx, bx
             mov  al, $40
             out  dx, al
             inc  dx
             in   al, dx
             and  al, 1
             mov  [bTemp], al

             mov  dx, bx
             mov  al, $39
             out  dx, al
             inc  dx
             mov  al, 0
             out  dx, al
          end;
     end;
     S3_AccelerateOn:= bTemp;
end;

{******}

procedure S3_AccelerateOff;

(* S3_AccelerateOff - deactivates S3_Acceleration features *)

begin
     if not GFX_Active then exit;

     if (GFX_Screen.ModeInfo.wChip > S3_924) then
     begin
          asm
             mov  bx, CRTC_ADDR
             mov  dx, bx
             mov  al, $38
             out  dx, al
             inc  dx
             mov  al, $48
             out  dx, al

             mov  dx, bx
             mov  al, $39
             out  dx, al
             inc  dx
             mov  al, $A5
             out  dx, al

             mov  dx, bx
             mov  al, $40
             out  dx, al
             inc  dx
             in   al, dx
             and  al, 1
             jz   @NOLOOP

          @LOOP:

             mov  dx, $9AE8
             in   ax, dx
             and  ax, $200
             jnz  @LOOP

          @NOLOOP:

             mov  dx, bx
             mov  al, $40
             out  dx, al
             inc  dx
             in   al, dx
             mov  cl, 1
             not  cl
             and  al, cl
             out  dx, al

             mov  dx, bx
             mov  al, $40
             out  dx, al
             inc  dx
             in   al, dx
             or   al, 8
             out  dx, al

             mov  dx, bx
             mov  al, $58
             out  dx, al
             inc  dx
             in   al, dx
             or   al, $10
             out  dx, al

             mov  dx, bx
             mov  al, $39
             out  dx, al
             inc  dx
             mov  al, $5A
             out  dx, al

             mov  bx, CRTC_ADDR
             mov  dx, bx
             mov  al, $38
             out  dx, al
             inc  dx
             mov  al, $00
             out  dx, al
          end;
     end;
end;

{******}

procedure S3_Init;

(* S3_Init - Initialises the S3 hardware for enhanced VGA and SVGA modes *)

var wx       : byte;
    bt1, bt2 : byte;

begin
     if GFX_Screen.ModeInfo.wChip < S3_864 then S3_AccelerateOff
                                           else S3_AccelerateOn;

     asm
        mov  bx, CRTC_ADDR
        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, $48
        out  dx, al

        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $A5
        out  dx, al

        mov  dx, bx
        mov  al, $31
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $C6
        mov  cl, 9
        and  cl, $39
        add  al, cl
        out  dx, al

        mov  dx, bx
        mov  al, $4E
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, bx
        mov  al, $4F
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, bx
        mov  al, $45
        out  dx, al
        inc  dx
        in   al, dx
        mov  cl, $3D
        not  cl
        and  al, cl
        out  dx, al
     end;

     if GFX_Screen.ModeInfo.wChip > S3_924 then
     begin
          asm
             mov  dx, CRTC_ADDR
             mov  al, $55
             out  dx, al
             inc  dx
             in   al, dx
             mov  cl, $30
             not  cl
             and  al, cl
             out  dx, al
          end;

          case GFX_Screen.ModeInfo.bMemModel of

                   MEM_TRUE15,
                   MEM_TRUE16   : begin
                                    if GFX_Screen.ModeInfo.wChip < S3_864 then
                                    asm
                                       mov  dx, CRTC_ADDR
                                       mov  al, $45
                                       out  dx, al
                                       inc  dx
                                       in   al, dx
                                       or   al, 4
                                       out  dx, al
                                    end;
                                    wx:= $10;
                                  end;

                   MEM_TRUE24,
                   MEM_TRUE32   : begin
                                       if GFX_Screen.ModeInfo.wChip < S3_864 then
                                       asm
                                          mov  dx, CRTC_ADDR
                                          mov  al, $45
                                          out  dx, al
                                          inc  dx
                                          in   al, dx
                                          or   al, 8
                                          out  dx, al
                                       end
                                       else
                                       asm
                                          mov  dx, CRTC_ADDR
                                          mov  al, $45
                                          out  dx, al
                                          inc  dx
                                          in   al, dx
                                          or   al, 4
                                          out  dx, al
                                       end;
                                       wx:= $30;
                                  end;
                   else wx:= 0;
          end;

          case ((GFX_Screen.ModeInfo.wByteWidth * 8) div GFX_Screen.ModeInfo.bBitsPerPixel) of

                         640 : Inc(wx, $40);
                         800 : Inc(wx, $80);
                        1280 : Inc(wx, $C0);
                        1600 : Inc(wx, $81);
          end;
          asm
             mov  bx, CRTC_ADDR
             mov  dx, bx
             mov  al, $50
             out  dx, al
             inc  dx
             in   al, dx
             mov  cl, $F1
             mov  ch, [wx]
             and  ch, cl
             not  cl
             and  al, cl
             add  al, ch
             out  dx, al

             mov  dx, bx
             mov  al, $55
             out  dx, al
             inc  dx
             in   al, dx
             and  al, 8
             mov  [bt1], al

             mov  dx, bx
             mov  al, $65
             out  dx, al
             inc  dx
             in   al, dx
             and  al, 2
             mov  [bt2], al
          end;

          if (bt1 > 0) or ((GFX_Screen.ModeInfo.wChip >= S3_864) and (bt2 > 0)) then
          begin
               asm
                  mov  bx, CRTC_ADDR
                  mov  dx, bx
                  mov  al, $45
                  out  dx, al
                  inc  dx
                  in   al, dx
                  or   al, $20
                  out  dx, al

                  mov  dx, bx
                  mov  al, $55
                  out  dx, al
                  inc  dx
                  in   al, dx
                  or   al, $20
                  out  dx, al
               end;
          end;
     end;

     asm
        mov  bx, CRTC_ADDR
        mov  dx, bx
        mov  al, $34
        out  dx, al
        inc  dx
        in   al, dx
        mov  cl, $80
        not  cl
        and  al, cl
        out  dx, al

        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $5A
        out  dx, al

        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, SEQU_ADDR
        mov  al, 4
        out  dx, al
        inc  dx
        in   al, dx
        or   al, 2
        out  dx, al
     end;
end;

{******}

procedure S3_PutPixel8(wx, wy : word; lCol : longint); assembler;

asm
   db   $66
   xor  ax, ax
   mov  ax, [wy]
   add  ax, [GFX_Screen.ModeInfo.wStartY]
   db   $66
   xor  bx, bx
   mov  bx, [GFX_Screen.ModeInfo.wByteWidth]
   db   $66
   mul  bx
   mov  cx, [wx]
   add  cx, [GFX_Screen.ModeInfo.wStartX]
   db   $66
   add  ax, cx

   mov  di, ax
   db   $66
   shr  ax, 16
   mov  cx, ax

   mov  si, [GFX_Screen.ModeInfo.wChip]

   {*** Inline SetBank ***}

   mov  bx, CRTC_ADDR
   mov  dx, bx
   mov  al, $39
   out  dx, al
   inc  dx
   mov  al, $A5
   out  dx, al

   mov  dx, bx
   mov  al, $38
   out  dx, al
   inc  dx
   mov  al, $48
   out  dx, al

   cmp  si, S3_864
   jl   @NOT864

   mov  dx, bx
   mov  al, $6A
   out  dx, al
   inc  dx
   mov  al, cl
   out  dx, al
   jmp  @IS864

@NOT864:

   mov  dx, bx
   mov  al, $31
   out  dx, al
   inc  dx
   in   al, dx
   or   al, 9
   out  dx, al
   mov  dx, bx

   mov  al, $35
   out  dx, al
   inc  dx
   in   al, dx
   and  al, $F0
   mov  bl, cl
   and  bl, $0F
   add  al, bl
   out  dx, al

   cmp  si, S3_924
   jle  @IS864

   mov  dx, CRTC_ADDR
   mov  al, $51
   out  dx, al
   inc  dx
   in   al, dx
   and  al, $3F
   mov  bl, cl
   shr  bl, 2
   and  bl, $0C
   add  al, bl
   out  dx, al

@IS864:

   mov  bx, CRTC_ADDR
   mov  dx, bx
   mov  al, $38
   out  dx, al
   inc  dx
   mov  al, 0
   out  dx, al

   mov  dx, bx
   mov  al, $39
   out  dx, al
   inc  dx
   mov  al, $5A
   out  dx, al

   {*** End of SetBank ***}

   mov  ax, VID_SEG
   mov  es, ax

   mov  al, [byte(lCol)]
   mov  es:[di], al
end;

{******}

function S3_GetPixel8(wx, wy : word) : longint;

var bTemp : byte;

begin
     asm
        db   $66
        xor  ax, ax
        mov  ax, [wy]
        add  ax, [GFX_Screen.ModeInfo.wStartY]
        db   $66
        xor  bx, bx
        mov  bx, [GFX_Screen.ModeInfo.wByteWidth]
        db   $66
        mul  bx
        mov  cx, [wx]
        add  cx, [GFX_Screen.ModeInfo.wStartX]
        db   $66
        add  ax, cx

        mov  di, ax
        db   $66
        shr  ax, 16
        mov  cx, ax

        mov  si, [GFX_Screen.ModeInfo.wChip]

        {*** Inline SetBank ***}

        mov  bx, CRTC_ADDR
        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $A5
        out  dx, al

        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, $48
        out  dx, al

        cmp  si, S3_864
        jl   @NOT864

        mov  dx, bx
        mov  al, $6A
        out  dx, al
        inc  dx
        mov  al, cl
        out  dx, al
        jmp  @IS864

     @NOT864:

        mov  dx, bx
        mov  al, $31
        out  dx, al
        inc  dx
        in   al, dx
        or   al, 9
        out  dx, al
        mov  dx, bx

        mov  al, $35
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $F0
        mov  bl, cl
        and  bl, $0F
        add  al, bl
        out  dx, al

        cmp  si, S3_924
        jle  @IS864

        mov  dx, CRTC_ADDR
        mov  al, $51
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $3F
        mov  bl, cl
        shr  bl, 2
        and  bl, $0C
        add  al, bl
        out  dx, al

     @IS864:

        mov  bx, CRTC_ADDR
        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $5A
        out  dx, al

        {*** End of SetBank ***}

        mov  ax, VID_SEG
        mov  es, ax

        mov  al, es:[di]
        mov  [bTemp], al
     end;

     S3_GetPixel8:= bTemp;
end;

{******}

procedure S3_PutPixel16(wx, wy : word; lCol : longint); assembler;

asm
   db   $66
   xor  ax, ax
   mov  ax, [wy]
   add  ax, [GFX_Screen.ModeInfo.wStartY]
   db   $66
   xor  bx, bx
   mov  bx, [GFX_Screen.ModeInfo.wByteWidth]
   db   $66
   mul  bx
   mov  cx, [wx]
   add  cx, [GFX_Screen.ModeInfo.wStartX]
   shl  cx, 1
   db   $66
   add  ax, cx

   mov  di, ax
   db   $66
   shr  ax, 16
   mov  cx, ax

   mov  si, [GFX_Screen.ModeInfo.wChip]

   {*** Inline SetBank ***}

   mov  bx, CRTC_ADDR
   mov  dx, bx
   mov  al, $39
   out  dx, al
   inc  dx
   mov  al, $A5
   out  dx, al

   mov  dx, bx
   mov  al, $38
   out  dx, al
   inc  dx
   mov  al, $48
   out  dx, al

   cmp  si, S3_864
   jl   @NOT864

   mov  dx, bx
   mov  al, $6A
   out  dx, al
   inc  dx
   mov  al, cl
   out  dx, al
   jmp  @IS864

@NOT864:

   mov  dx, bx
   mov  al, $31
   out  dx, al
   inc  dx
   in   al, dx
   or   al, 9
   out  dx, al
   mov  dx, bx

   mov  al, $35
   out  dx, al
   inc  dx
   in   al, dx
   and  al, $F0
   mov  bl, cl
   and  bl, $0F
   add  al, bl
   out  dx, al

   cmp  si, S3_924
   jle  @IS864

   mov  dx, CRTC_ADDR
   mov  al, $51
   out  dx, al
   inc  dx
   in   al, dx
   and  al, $3F
   mov  bl, cl
   shr  bl, 2
   and  bl, $0C
   add  al, bl
   out  dx, al

@IS864:

   mov  bx, CRTC_ADDR
   mov  dx, bx
   mov  al, $38
   out  dx, al
   inc  dx
   mov  al, 0
   out  dx, al

   mov  dx, bx
   mov  al, $39
   out  dx, al
   inc  dx
   mov  al, $5A
   out  dx, al

   {*** End of SetBank ***}

   mov  ax, VID_SEG
   mov  es, ax

   mov  ax, [word(lCol)]
   mov  es:[di], ax
end;

{******}

function S3_GetPixel16(wx, wy : word) : longint;

var wTemp : word;

begin
  asm
    db   $66
    xor  ax, ax
    mov  ax, [wy]
    add  ax, [GFX_Screen.ModeInfo.wStartY]
    db   $66
    xor  bx, bx
    mov  bx, [GFX_Screen.ModeInfo.wByteWidth]
    db   $66
    mul  bx
    mov  cx, [wx]
    add  cx, [GFX_Screen.ModeInfo.wStartX]
    shl  cx, 1
    db   $66
    add  ax, cx

    mov  di, ax
    db   $66
    shr  ax, 16
    mov  cx, ax

    mov  si, [GFX_Screen.ModeInfo.wChip]

    {*** Inline SetBank ***}

    mov  bx, CRTC_ADDR
    mov  dx, bx
    mov  al, $39
    out  dx, al
    inc  dx
    mov  al, $A5
    out  dx, al

    mov  dx, bx
    mov  al, $38
    out  dx, al
    inc  dx
    mov  al, $48
    out  dx, al

    cmp  si, S3_864
    jl   @NOT864

    mov  dx, bx
    mov  al, $6A
    out  dx, al
    inc  dx
    mov  al, cl
    out  dx, al
    jmp  @IS864

 @NOT864:

    mov  dx, bx
    mov  al, $31
    out  dx, al
    inc  dx
    in   al, dx
    or   al, 9
    out  dx, al
    mov  dx, bx

    mov  al, $35
    out  dx, al
    inc  dx
    in   al, dx
    and  al, $F0
    mov  bl, cl
    and  bl, $0F
    add  al, bl
    out  dx, al

    cmp  si, S3_924
    jle  @IS864

    mov  dx, CRTC_ADDR
    mov  al, $51
    out  dx, al
    inc  dx
    in   al, dx
    and  al, $3F
    mov  bl, cl
    shr  bl, 2
    and  bl, $0C
    add  al, bl
    out  dx, al

 @IS864:

    mov  bx, CRTC_ADDR
    mov  dx, bx
    mov  al, $38
    out  dx, al
    inc  dx
    mov  al, 0
    out  dx, al

    mov  dx, bx
    mov  al, $39
    out  dx, al
    inc  dx
    mov  al, $5A
    out  dx, al

    {*** End of SetBank ***}

    mov  ax, VID_SEG
    mov  es, ax

    mov  ax, es:[di]
    mov  [wTemp], ax
 end;
 S3_GetPixel16:= wTemp;
end;

{******}

{procedure S3_PutPixel24(wx, wy : word; lCol : longint);

var l   : longint;
    z,m : word;

begin
     Inc(wx, GFX_Screen.ModeInfo.wStartX);
     Inc(wy, GFX_Screen.ModeInfo.wStartY);
     l:= (longint(wy) * GFX_Screen.ModeInfo.wByteWidth) + (wx * 3);
     z:= word(l);
     m:= l shr 16;
     S3_SetBank(m);
     if z < $FFFE then Move(lCol, Mem[VID_SEG:z], 3)
     else
     begin
          Mem[VID_SEG:z]:= lo(lCol);
          if z = $ffff then S3_SetBank(m + 1);
          Mem[VID_SEG:z + 1]:= lo(lCol shr 8);
          if z = $fffe then S3_SetBank(m + 1);
          Mem[VID_SEG:z + 2]:= lCol shr 16;
     end;
end;}

{******}

procedure S3_PutPixel32(wx, wy : word; lCol : longint); assembler;

type TLongRec = record
                      Lo, Hi : word;
                end;

asm
   db   $66
   xor  ax, ax
   mov  ax, [wy]
   add  ax, [GFX_Screen.ModeInfo.wStartY]
   db   $66
   xor  bx, bx
   mov  bx, [GFX_Screen.ModeInfo.wByteWidth]
   db   $66
   mul  bx
   mov  cx, [wx]
   add  cx, [GFX_Screen.ModeInfo.wStartX]
   shl  cx, 2
   db   $66
   add  ax, cx

   mov  di, ax
   db   $66
   shr  ax, 16
   mov  cx, ax

   mov  si, [GFX_Screen.ModeInfo.wChip]

   {*** Inline SetBank ***}

   mov  bx, CRTC_ADDR
   mov  dx, bx
   mov  al, $39
   out  dx, al
   inc  dx
   mov  al, $A5
   out  dx, al

   mov  dx, bx
   mov  al, $38
   out  dx, al
   inc  dx
   mov  al, $48
   out  dx, al

   cmp  si, S3_864
   jl   @NOT864

   mov  dx, bx
   mov  al, $6A
   out  dx, al
   inc  dx
   mov  al, cl
   out  dx, al
   jmp  @IS864

@NOT864:

   mov  dx, bx
   mov  al, $31
   out  dx, al
   inc  dx
   in   al, dx
   or   al, 9
   out  dx, al
   mov  dx, bx

   mov  al, $35
   out  dx, al
   inc  dx
   in   al, dx
   and  al, $F0
   mov  bl, cl
   and  bl, $0F
   add  al, bl
   out  dx, al

   cmp  si, S3_924
   jle  @IS864

   mov  dx, CRTC_ADDR
   mov  al, $51
   out  dx, al
   inc  dx
   in   al, dx
   and  al, $3F
   mov  bl, cl
   shr  bl, 2
   and  bl, $0C
   add  al, bl
   out  dx, al

@IS864:

   mov  bx, CRTC_ADDR
   mov  dx, bx
   mov  al, $38
   out  dx, al
   inc  dx
   mov  al, 0
   out  dx, al

   mov  dx, bx
   mov  al, $39
   out  dx, al
   inc  dx
   mov  al, $5A
   out  dx, al

   {*** End of SetBank ***}

   mov  ax, VID_SEG
   mov  es, ax

   mov  ax, [TLongRec(lCol).Hi]
   db   $66
   shl  ax, 16
   mov  ax, [word(lCol)]
   db   $66
   mov  es:[di], ax
end;

{******}

function S3_GetPixel32(wx, wy : word) : longint;

type TLongRec = record
                      Lo, Hi : word;
                end;

var lTemp : longint;

begin
  asm
    db   $66
    xor  ax, ax
    mov  ax, [wy]
    add  ax, [GFX_Screen.ModeInfo.wStartY]
    db   $66
    xor  bx, bx
    mov  bx, [GFX_Screen.ModeInfo.wByteWidth]
    db   $66
    mul  bx
    mov  cx, [wx]
    add  cx, [GFX_Screen.ModeInfo.wStartX]
    shl  cx, 2
    db   $66
    add  ax, cx

    mov  di, ax
    db   $66
    shr  ax, 16
    mov  cx, ax

    mov  si, [GFX_Screen.ModeInfo.wChip]

    {*** Inline SetBank ***}

    mov  bx, CRTC_ADDR
    mov  dx, bx
    mov  al, $39
    out  dx, al
    inc  dx
    mov  al, $A5
    out  dx, al

    mov  dx, bx
    mov  al, $38
    out  dx, al
    inc  dx
    mov  al, $48
    out  dx, al

    cmp  si, S3_864
    jl   @NOT864

    mov  dx, bx
    mov  al, $6A
    out  dx, al
    inc  dx
    mov  al, cl
    out  dx, al
    jmp  @IS864

 @NOT864:

    mov  dx, bx
    mov  al, $31
    out  dx, al
    inc  dx
    in   al, dx
    or   al, 9
    out  dx, al
    mov  dx, bx

    mov  al, $35
    out  dx, al
    inc  dx
    in   al, dx
    and  al, $F0
    mov  bl, cl
    and  bl, $0F
    add  al, bl
    out  dx, al

    cmp  si, S3_924
    jle  @IS864

    mov  dx, CRTC_ADDR
    mov  al, $51
    out  dx, al
    inc  dx
    in   al, dx
    and  al, $3F
    mov  bl, cl
    shr  bl, 2
    and  bl, $0C
    add  al, bl
    out  dx, al

 @IS864:

    mov  bx, CRTC_ADDR
    mov  dx, bx
    mov  al, $38
    out  dx, al
    inc  dx
    mov  al, 0
    out  dx, al

    mov  dx, bx
    mov  al, $39
    out  dx, al
    inc  dx
    mov  al, $5A
    out  dx, al

    {*** End of SetBank ***}

    mov  ax, VID_SEG
    mov  es, ax

    db   $66
    mov  ax, es:[di]
    mov  [TLongRec(lTemp).Lo], ax
    db   $66
    shr  ax, 16
    mov  [TLongRec(lTemp).Hi], ax
  end;
  S3_GetPixel32:= lTemp;
end;

{******}

procedure S3_Put2Pixel(wx, wy : word; lCol : longint);

begin
     GFX_PutPixel(wx, wy, lCol);
     GFX_PutPixel(wx + 1, wy, lCol);
end;

{******}

procedure S3_Put4Pixel(wx, wy : word; lCol : longint);

begin
     GFX_PutPixel(wx, wy, lCol);
     GFX_PutPixel(wx + 1, wy, lCol);
     GFX_PutPixel(wx + 2, wy, lCol);
     GFX_PutPixel(wx + 3, wy, lCol);
end;

{******}

procedure S3_Circle(wx, wy, wr : word; lCol : longint);

{******}

    procedure CirclePoint(X, Y, Xc, Yc : integer; bCol : byte);

    var XXCp, XXCm, XYCp, XYCm, YXCp, YXCm, YYCp, YYCm : integer;

    begin
        asm
           mov  cx, X    { store in registers for faster access }
           mov  dx, Y    { instead of accessing memory each time }

           mov  bx, XC   { store XC in bx - cut down on mem access }

           mov  ax, bx   { XXCp := XC+X; }
           add  ax, cx
           mov  XXCp, ax

           mov  ax, bx   { XXCm := XC-X; }
           sub  ax, cx
           mov  XXCm, ax

           mov  ax, bx   { XYCp := XC+Y; }
           add  ax, dx
           mov  XYCp, ax

           mov  ax, bx   { XYCm := XC-Y; }
           sub  ax, dx
           mov  XYCm, ax

           mov  bx, YC   { store YC in bx - cut down on mem access }

           mov  ax, bx   { YXCp := YC+X; }
           add  ax, cx
           mov  YXCp, ax

           mov  ax, bx   { YXCm := YC-X; }
           sub  ax, cx
           mov  YXCm, ax

           mov  ax, bx   { YYCp := YC+Y; }
           add  ax, dx
           mov  YYCp, ax

           mov  ax, bx   { YYCm := YC-Y; }
           sub  ax, dx
           mov  YYCm, ax

        end;

        GFX_PutPixel(XXCp, YYCp, lCol);
        GFX_PutPixel(XXCp, YYCm, lCol);
        GFX_PutPixel(XXCm, YYCp, lCol);
        GFX_PutPixel(XXCm, YYCm, lCol);
        GFX_PutPixel(XYCp, YXCp, lCol);
        GFX_PutPixel(XYCp, YXCm, lCol);
        GFX_PutPixel(XYCm, YXCp, lCol);
        GFX_PutPixel(XYCm, YXCm, lCol);
    end;

{******}

var wXt, wYt, wD : integer;

begin
     asm
        sub  ax, ax         { X := 0; }
        mov  wXt, ax
        mov  ax, wr         { Y := Radius }
        mov  wYt, ax

        shl  ax, 1          { (2 * Radius) }
        mov  bx, 03H
        sub  bx, ax         { D := 3 - (2 * Radius) }
        mov  wD, bx
     end;

     while (wXt < wYt) do
     begin
          CirclePoint(wXt, wYt, wx, wy, byte(lCol));

          if (wD < 0) then
          asm
             mov  ax, wXt  { 4 * X }
             shl  ax, 2
             add  ax, 6  { (4*X)+6 }
             mov  bx, wD
             add  bx, ax
             mov  wD, bx
          end {D := D + (4 * X) + 6}
          else
          asm
             mov  ax, wXt    { (X-Y) }
             mov  bx, wYt
             sub  ax, bx
             shl  ax, 2      { 4*(X-Y) }
             add  ax, 10
             mov  bx, wD
             add  bx, ax
             mov  wD, bx     { D := D + 4*(X-Y) + 10; }
             dec  wYt
          end;

          Inc(wXt);
     end;

     if (wXt = wYt) then CirclePoint(wXt, wYt, wx, wy, byte(lCol));
end;

{******}


{*** S3 8514/A graphics engine support                     ***}
{*** Uses hardware lines, fills and bitmaps for SVGA modes ***}

procedure S3_Line(wx1, wy1, wx2, wy2 : word; lCol : longint);

var ix, iy : integer;
    itemp  : integer;
    ww, wz : word;
    wCol   : word;

begin
     if (GFX_Screen.ModeInfo.wPixelWidth > 1024) and
        (GFX_Screen.ModeInfo.bMemModel >= MEM_PACKED8) and
        (GFX_Screen.ModeInfo.wChip >= S3_928) and
        (GFX_Screen.ModeInfo.wChip < S3_864) then
     begin
          wy1:= wy1 shl 1;
          wy2:= wy2 shl 1;
     end;

     Inc(wx1, GFX_Screen.ModeInfo.wStartX);
     Inc(wx2, GFX_Screen.ModeInfo.wStartX);
     Inc(wy1, GFX_Screen.ModeInfo.wStartY);
     Inc(wy2, GFX_Screen.ModeInfo.wStartY);

     asm

     @LOOP:

           mov  dx, $9AE8
           in   ax, dx
           and  ax, $FF
           jnz  @LOOP

           mov  dx, $82E8
           mov  ax, [wy1]
           out  dx, ax

           mov  dx, $86E8
           mov  ax, [wx1]
           out  dx, ax
     end;

     ww:= 0;
     wz:= 0;

     ix:= wx1 - wx2;
     if ix < 0 then
     begin
          ix:= -ix;
          ww:= ww or $20;
          wz:= 1;
     end;

     iy:= wy1 - wy2;
     if iy < 0 then
     begin
          iy:= -iy;
          ww:= ww or $80;
     end;
     if ix < iy then
     begin
          itemp:= ix;
          ix:= iy;
          iy:= itemp;
          ww:= ww or $40;
     end;

     asm
        mov  bx, [iy]
        mov  cx, [ix]

        mov  dx, $8AE8
        mov  ax, bx
        shl  ax, 1
        out  dx, ax

        mov  dx, $8EE8
        mov  ax, bx
        sub  ax, cx
        shl  ax, 1
        out  dx, ax

        mov  dx, $92E8
        mov  ax, bx
        shl  ax, 1
        sub  ax, cx
        sub  ax, [wz]
        out  dx, ax

        mov  dx, $96E8
        mov  ax, cx
        out  dx, ax

        mov  dx, $BEE8
        mov  ax, $E000
        out  dx, ax

        mov  dx, $A6E8
        mov  ax, [word(lCol)]
        out  dx, ax

     @LOOP:

        mov  dx, $9AE8
        in   ax, dx
        and  ax, $FF
        jnz  @LOOP

     end;

     wCol:= word(lCol shr 16);

     if (GFX_Screen.ModeInfo.bMemModel > MEM_TRUE16) then
     asm
        mov  dx, $BEE8
        mov  ax, $E010
        out  dx, ax

        mov  dx, $A6E8
        mov  ax, [wCol]
        out  dx, ax
     end;

     asm
        mov  dx, $BAE8
        mov  ax, $27
        out  dx, ax

        mov  dx, $BEE8
        mov  ax, $A000
        mov  dx, ax

        mov  dx, $9AE8
        mov  ax, $2017
        add  ax, [ww]
        out  dx, ax
     end;
end;

{******}

procedure S3_HLine(wx1, wx2, wy : word; lCol : longint);

begin
     S3_Line(wx1, wy, wx2, wy, lCol);
end;

{******}

procedure S3_VLine(wx, wy1, wy2 : word; lCol : longint);

begin
     S3_Line(wx, wy1, wx, wy2, lCol);
end;

{******}

procedure S3_Rectangle(wx, wy, ww, wh : word; lCol : longint);

var wx2, wy2 : word;

begin
     wx2:= wx + ww;
     wy2:= wy + wh;

     S3_Line(wx, wy, wx2, wy, lCol);
     S3_Line(wx, wy2, wx2 + 1, wy2, lCol);
     S3_Line(wx, wy, wx, wy2, lCol);
     S3_Line(wx2, wy, wx2, wy2, lCol);
end;

{******}

procedure S3_FilledRectangle(wx, wy, ww, wh : word; lCol : longint);

var wCol : word;

begin
     if (GFX_Screen.ModeInfo.wPixelWidth > 1024) and
        (GFX_Screen.ModeInfo.bMemModel >= MEM_PACKED8) and
        (GFX_Screen.ModeInfo.wChip >= S3_928) and
        (GFX_Screen.ModeInfo.wChip < S3_864) then
     begin
          wy:= wy shl 2;
          wh:= wh shl 2;
     end;

     Inc(wx, GFX_Screen.ModeInfo.wStartX);
     Inc(wy, GFX_Screen.ModeInfo.wStartY);

     asm

     @LOOP:

        mov  dx, $9AE8
        in   ax, dx
        and  ax, $FF
        jnz  @LOOP

        mov  dx, $82E8
        mov  ax, [wy]
        out  dx, ax

        mov  dx, $86E8
        mov  ax, [wx]
        out  dx, ax

        mov  dx, $96E8
        mov  ax, [ww]
        out  dx, ax

        mov  dx, $BEE8
        mov  ax, $E000
        out  dx, ax

        mov  dx, $A6E8
        mov  ax, [word(lCol)]
        out  dx, ax
     end;

     wCol:= word(lCol shr 16);
     if (GFX_Screen.ModeInfo.bMemModel >= MEM_TRUE24) then
     asm
        mov  dx, $BEE8
        mov  ax, $E010
        out  dx, ax

        mov  dx, $A6E8
        mov  ax, [wCol]
        out  dx, ax
     end;

     asm

     @LOOP:

        mov  dx, $9AE8
        in   ax, dx
        and  ax, $FF
        jnz  @LOOP

        mov  dx, $BAE8
        mov  ax, $27
        out  dx, ax

        mov  dx, $BEE8
        mov  ax, [wh]
        dec  ax
        out  dx, ax

        mov  dx, $BEE8
        mov  ax, $A000
        out  dx, ax

        mov  dx, $9AE8
        mov  ax, $40F1
        out  dx, ax

        mov  dx, CRTC_ADDR
        mov  al, $40
        out  dx, al
        inc  dx
        in   al, dx
        mov  cl, 1
        not  cl
        and  al, cl
        out  dx, al
     end;
end;

{******}

procedure S3_Cls(lCol : longint);

begin
     S3_FilledRectangle(GFX_Screen.ModeInfo.wMinX, GFX_Screen.ModeInfo.wMinY,
                        GFX_Screen.ModeInfo.wPixelWidth,
                        GFX_Screen.ModeInfo.wPixelHeight, lCol);
end;

{******}

procedure S3_FilledCircle(wx, wy, wr : word; lCol : longint);

{******}

     procedure CirclePoint(X, Y, Xc, Yc : integer; lCol : longint);

     var XXCp, XXCm, XYCp, XYCm, YXCp, YXCm, YYCp, YYCm : integer;

     begin
        asm
           mov  cx, X    { store in registers for faster access }
           mov  dx, Y    { instead of accessing memory each time }

           mov  bx, XC   { store XC in bx - cut down on mem access }

           mov  ax, bx   { XXCp := XC+X; }
           add  ax, cx
           mov  XXCp, ax

           mov  ax, bx   { XXCm := XC-X; }
           sub  ax, cx
           mov  XXCm, ax

           mov  ax, bx   { XYCp := XC+Y; }
           add  ax, dx
           mov  XYCp, ax

           mov  ax, bx   { XYCm := XC-Y; }
           sub  ax, dx
           mov  XYCm, ax

           mov  bx, YC   { store YC in bx - cut down on mem access }

           mov  ax, bx   { YXCp := YC+X; }
           add  ax, cx
           mov  YXCp, ax

           mov  ax, bx   { YXCm := YC-X; }
           sub  ax, cx
           mov  YXCm, ax

           mov  ax, bx   { YYCp := YC+Y; }
           add  ax, dx
           mov  YYCp, ax

           mov  ax, bx   { YYCm := YC-Y; }
           sub  ax, dx
           mov  YYCm, ax
        end;

        S3_Line(XXCp, YYCp, XXCm, YYCp, lCol);
        S3_Line(XXCp, YYCm, XXCm, YYCm, lCol);
        S3_Line(XYCp, YXCp, XYCm, YXCp, lCol);
        S3_Line(XYCp, YXCm, XYCm, YXCm, lCol);
     end;

{******}

var wXt, wYt, wD : integer;

begin
     asm
        sub  ax, ax         { X := 0; }
        mov  wXt, ax
        mov  ax, wr         { Y := Radius }
        mov  wYt, ax

        shl  ax, 1          { (2 * Radius) }
        mov  bx, 03H
        sub  bx, ax         { D := 3 - (2 * Radius) }
        mov  wD, bx
     end;

     while (wXt < wYt) do
     begin
          CirclePoint(wXt, wYt, wx, wy, lCol);

          if (wD < 0) then
          asm
             mov  ax, wXt  { 4 * X }
             shl  ax, 2
             add  ax, 6  { (4*X)+6 }
             mov  bx, wD
             add  bx, ax
             mov  wD, bx
          end {D := D + (4 * X) + 6}
          else
          asm
             mov  ax, wXt    { (X-Y) }
             mov  bx, wYt
             sub  ax, bx
             shl  ax, 2      { 4*(X-Y) }
             add  ax, 10
             mov  bx, wD
             add  bx, ax
             mov  wD, bx     { D := D + 4*(X-Y) + 10; }
             dec  wYt
          end;

          Inc(wXt);
     end;

     if (wXt = wYt) then CirclePoint(wXt, wYt, wx, wy, lCol);
end;

{******}

procedure S3_Triangle(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);

begin
     GFX_Line(wx1, wy1, wx2, wy2, lCol);
     GFX_Line(wx2, wy2, wx3, wy3, lCol);
     GFX_Line(wx1, wy1, wx3, wy3, lCol);
end;

{******}

procedure S3_Polygon(var aPoints : array of word; lCol : longint);

var n    : word;
    Flag : boolean;
    Loop : word;
    Temp : word;

const X = 1;
      Y = 2;

type tPtsArray = array[0..16382, X..Y] of word;

var wArrayCount : word;
    wLoop       : word;
    aPts        : tPtsArray absolute aPoints;

begin
     wArrayCount:= High(aPoints);

     GFX_Line(aPts[wArrayCount, X], aPts[wArrayCount, Y], aPts[0, X],
              aPts[0, Y], lCol);
     readkey;
     for wLoop:= 1 to wArrayCount do
     begin
         GFX_Line(aPts[wLoop - 1, X], aPts[wLoop - 1, Y], aPts[wLoop, X],
                  aPts[wLoop, Y], lCol);
         readkey;
     end;

end;

{*** MODE $3001 - enhanced 320x200x256 mode ***}

procedure S3_3001_SetVisualPage(wPage : word);

var wSx, wSy : word;

begin
     wSX:= ((wPage shl GFX_Screen.ModeInfo.bBitsPerPixel)
           mod GFX_Screen.ModeInfo.wPixelWidth);
     wSY:= ((GFX_Screen.ModeInfo.wPixelHeight + 4) * wPage)
           + ((wPage shl GFX_Screen.ModeInfo.bBitsPerPixel)
           div GFX_Screen.ModeInfo.wPixelWidth);

     S3_SetVisualStart(wSX, wSY);
end;

{******}

procedure S3_3001_SetActivePage(wPage : word);

begin
     S3_SetBank(wPage);
end;

{******}

procedure S3_3001_Cls(lCol : longint); assembler;

asm
   mov  ax, VID_SEG
   mov  es, ax
   xor  di, di
   mov  al, [byte(lCol)]
   mov  ah, al
   mov  bx, ax
   db   $66
   shl  ax, 16
   mov  ax, bx
   mov  cx, 16000
   db   $66
   rep  stosw
end;

{******}

procedure S3_3001_PutPixel(wx, wy : word; lCol : longint); assembler;

asm
   mov     ax, VID_SEG
   mov     es, ax
   mov     ax, [wy]
   mov     di, ax
   shl     di, 8
   shl     ax, 6
   add     di, ax
   add     di, [wx]
   mov     al, [byte(lCol)]
   mov     es:[di], al
end;

{******}

procedure S3_3001_Put2Pixel(wx, wy : word; lCol : longint); assembler;

asm
   mov     ax, VID_SEG
   mov     es, ax
   mov     ax, [wy]
   mov     di, ax
   shl     di, 8
   shl     ax, 6
   add     di, ax
   add     di, [wx]
   mov     al, [byte(lCol)]
   mov     ah, al
   mov     es:[di], ax
end;

{******}

procedure S3_3001_Put4Pixel(wx, wy : word; lCol : longint); assembler;

asm
   mov     ax, VID_SEG
   mov     es, ax
   mov     ax, [wy]
   mov     di, ax
   shl     di, 8
   shl     ax, 6
   add     di, ax
   add     di, [wx]
   mov     al, [byte(lCol)]
   mov     ah, al
   mov     dx, ax
   db      $66
   shl     ax, 16
   mov     ax, dx
   db      $66
   mov     es:[di], ax
end;

{******}

function S3_3001_GetPixel(wx, wy : word) : longint; assembler;

asm
   mov     ax, VID_SEG
   mov     es, ax
   mov     ax, [wy]
   mov     di, ax
   shl     di, 8
   shl     ax, 6
   add     di, ax
   add     di, [wx]
   mov     al, es:[di]
end;

{******}

procedure S3_3001_HLine(wx1, wx2, wy : word; lCol : longint); assembler;

asm
   mov   ax, VID_SEG
   mov   es, ax
   mov   ax, [wy]
   mov   di, ax
   shl   ax, 8
   shl   di, 6
   add   di, ax
   mov   bx, [wx1]
   add   di, bx
   mov   al, [byte(lCol)]
   mov   ah, al
   mov   cx, [wx2]
   sub   cx, bx
   test  cx, 1
   jz    @EXIT
   stosb

@EXIT:

   shr cx, 1
   rep stosw

end;

{******}

procedure S3_3001_VLine(wx, wy1, wy2 : word; lCol : longint); assembler;

asm
   mov     ax, VID_SEG
   mov     es, ax
   mov     bx, [wy1]
   mov     ax, bx
   mov     di, ax
   shl     ax, 8
   shl     di, 6
   add     di, ax
   add     di, [wx]
   mov     al, [byte(lCol)]
   mov     cx, [wy2]
   sub     cx, bx

@@1:

   mov     es:[di], al
   add     di, 320
   dec     cx
   jnz     @@1

end;

{******}

procedure S3_3001_Line(wx1, wy1, wx2, wy2 : word; lCol : longint); assembler;

asm
   mov     ax, [wx1]
   mov     bx, [wx2]
   mov     cx, [wy1]
   mov     dx, [wy2]

   cmp     cx, dx
   jbe     @1
   xchg    cx, dx
   xchg    ax, bx

@1:

   mov     di, cx
   shl     di, 2
   add     di, cx
   mov     si, bx
   mov     bx, dx
   sub     bx, cx
   shl     di, 6
   add     di, ax
   mov     dx, si
   sub     dx, ax

   mov     ax, VID_SEG
   mov     es, ax
   mov     al, [byte(lCol)]
   or      dx, 0
   jge     @jmp1
   neg     dx
   cmp     dx, bx
   jbe     @jmp3

   mov     cx, dx
   inc     cx
   mov     si, dx
   shr     si, 1
   std

@1c:

   stosb

@1b:

   or      si,si
   jge     @1a
   add     di, 320
   add     si, dx
   jmp     @1b

@1a:

   sub     si, bx
   dec     cx
   jnz     @1c
   jmp     @EXIT

@jmp3:

   mov     cx, bx
   inc     cx
   mov     si, bx
   neg     si
   sar     si, 1
   cld

@2c:

   stosb

@2b:

   or      si, si
   jl      @2a
   sub     si, bx
   dec     di
   jmp     @2b

@2a:

   add     di, 319
   add     si, dx
   dec     cx
   jnz     @2c
   jmp     @EXIT

@jmp1:

   cmp     dx, bx
   jbe     @jmp4
   mov     cx, dx
   inc     cx
   mov     si, dx
   shr     si, 1
   cld

@3c:

   stosb

@3b:

   or      si, si
   jge     @3a
   add     di, 320
   add     si, dx
   jmp     @3b

@3a:

   sub     si, bx
   dec     cx
   jnz     @3c
   jmp     @EXIT

@jmp4:

   mov     cx, bx
   inc     cx
   mov     si, bx
   neg     si
   sar     si, 1
   std

@4c:

   stosb

@4b:

   or      si, si
   jl      @4a
   sub     si, bx
   inc     di
   jmp     @4b

@4a:

   add     di, 321
   add     si, dx
   dec     cx
   jnz     @4c

@EXIT:

   cld

end;

{******}

procedure S3_3001_Rectangle(wx, wy, ww, wh : word; lCol : longint);

var wx2, wy2 : word;

begin
     wx2:= wx + ww;
     wy2:= wy + wh;
     GFX_HLine(wx, wx2, wy, lCol);
     GFX_HLine(wx, wx2, wy2, lCol);
     GFX_VLine(wx, wy, wy2, lCol);
     GFX_VLine(wx2, wy, wy2, lCol);
end;

{******}

procedure S3_3001_FilledRectangle(wx, wy, ww, wh : word; lCol : longint); assembler;

asm
   mov   ax, VID_SEG
   mov   es, ax
   mov   ax, [wy]
   mov   di, ax
   shl   ax, 8
   shl   di, 6
   add   di, ax
   add   di, [wx]
   mov   al, [byte(lCol)]
   mov   ah, al

   mov   dx, [wh]
   mov   si, [ww]
   mov   bx, 319
   sub   bx, si

@DRAWLOOP:

   mov   cx, si
   inc   cx
   shr   cx, 1
   jnc   @P24
   stosb

@P24:

   rep   stosw

   add   di, bx
   dec   dx
   jnz   @DRAWLOOP

end;

{******}

procedure S3_3001_Circle(wx, wy, wr : word; lCol : longint);

{******}

        procedure CirclePoint(X, Y, Xc, Yc : integer; bCol : byte); assembler;

        var XXCp, XXCm, XYCp, XYCm, YXCp, YXCm, YYCp, YYCm : integer;

        asm
           mov  cx, X    { store in registers for faster access }
           mov  dx, Y    { instead of accessing memory each time }

           mov  bx, XC   { store XC in bx - cut down on mem access }

           mov  ax, bx   { XXCp := XC+X; }
           add  ax, cx
           mov  XXCp, ax

           mov  ax, bx   { XXCm := XC-X; }
           sub  ax, cx
           mov  XXCm, ax

           mov  ax, bx   { XYCp := XC+Y; }
           add  ax, dx
           mov  XYCp, ax

           mov  ax, bx   { XYCm := XC-Y; }
           sub  ax, dx
           mov  XYCm, ax

           mov  bx, YC   { store YC in bx - cut down on mem access }

           mov  ax, bx   { YXCp := YC+X; }
           add  ax, cx
           mov  YXCp, ax

           mov  ax, bx   { YXCm := YC-X; }
           sub  ax, cx
           mov  YXCm, ax

           mov  ax, bx   { YYCp := YC+Y; }
           add  ax, dx
           mov  YYCp, ax

           mov  ax, bx   { YYCm := YC-Y; }
           sub  ax, dx
           mov  YYCm, ax

           mov  bl, bCol
           xor  bh, bh
           mov  dx, XXCp

           mov  ax, VID_SEG
           mov  es, ax
           mov  di, dx
           mov  ax, YYCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb

           sub  di, di
           add  di, dx
           mov  ax, YYCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb

           mov  dx, XXCm

           sub  di, di
           add  di, dx
           mov  ax, YYCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           sub  di, di
           add  di, dx
           mov  ax, YYCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           mov  dx, XYCp

           sub  di, di
           add  di, dx
           mov  ax, YXCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           sub  di, di
           add  di, dx
           mov  ax, YXCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           mov  dx, XYCm
           sub  di, di
           add  di, dx
           mov  ax, YXCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           sub  di, di
           add  di, dx
           mov  ax, YXCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }
        end;

{******}

var wXt, wYt, wD : integer;

begin
     asm
        sub  ax, ax         { X := 0; }
        mov  wXt, ax
        mov  ax, wr         { Y := Radius }
        mov  wYt, ax

        shl  ax, 1          { (2 * Radius) }
        mov  bx, 03H
        sub  bx, ax         { D := 3 - (2 * Radius) }
        mov  wD, bx
     end;

     while (wXt < wYt) do
     begin
          CirclePoint(wXt, wYt, wx, wy, byte(lCol));

          if (wD < 0) then
          asm
             mov  ax, wXt  { 4 * X }
             shl  ax, 2
             add  ax, 6  { (4*X)+6 }
             mov  bx, wD
             add  bx, ax
             mov  wD, bx
          end {D := D + (4 * X) + 6}
          else
          asm
             mov  ax, wXt    { (X-Y) }
             mov  bx, wYt
             sub  ax, bx
             shl  ax, 2      { 4*(X-Y) }
             add  ax, 10
             mov  bx, wD
             add  bx, ax
             mov  wD, bx     { D := D + 4*(X-Y) + 10; }
             dec  wYt
          end;

          Inc(wXt);
     end;

     if (wXt = wYt) then CirclePoint(wXt, wYt, wx, wy, byte(lCol));
end;

{******}

procedure S3_3001_FilledCircle(wx, wy, wr : word; lCol : longint);

{******}

     procedure CirclePoint(X, Y, Xc, Yc : integer; lCol : longint);

     var XXCp, XXCm, XYCp, XYCm, YXCp, YXCm, YYCp, YYCm : integer;

     begin
        asm
           mov  cx, X    { store in registers for faster access }
           mov  dx, Y    { instead of accessing memory each time }

           mov  bx, XC   { store XC in bx - cut down on mem access }

           mov  ax, bx   { XXCp := XC+X; }
           add  ax, cx
           mov  XXCp, ax

           mov  ax, bx   { XXCm := XC-X; }
           sub  ax, cx
           mov  XXCm, ax

           mov  ax, bx   { XYCp := XC+Y; }
           add  ax, dx
           mov  XYCp, ax

           mov  ax, bx   { XYCm := XC-Y; }
           sub  ax, dx
           mov  XYCm, ax

           mov  bx, YC   { store YC in bx - cut down on mem access }

           mov  ax, bx   { YXCp := YC+X; }
           add  ax, cx
           mov  YXCp, ax

           mov  ax, bx   { YXCm := YC-X; }
           sub  ax, cx
           mov  YXCm, ax

           mov  ax, bx   { YYCp := YC+Y; }
           add  ax, dx
           mov  YYCp, ax

           mov  ax, bx   { YYCm := YC-Y; }
           sub  ax, dx
           mov  YYCm, ax
        end;

        GFX_HLine(XXCp, XXCm, YYCp, lCol);
        GFX_HLine(XXCp, XXCm, YYCm, lCol);
        GFX_HLine(XYCp, XYCm, YXCp, lCol);
        GFX_HLine(XYCp, XYCm, YXCm, lCol);
     end;

{******}

var wXt, wYt, wD : integer;

begin
     asm
        sub  ax, ax         { X := 0; }
        mov  wXt, ax
        mov  ax, wr         { Y := Radius }
        mov  wYt, ax

        shl  ax, 1          { (2 * Radius) }
        mov  bx, 03H
        sub  bx, ax         { D := 3 - (2 * Radius) }
        mov  wD, bx
     end;

     while (wXt < wYt) do
     begin
          CirclePoint(wXt, wYt, wx, wy, lCol);

          if (wD < 0) then
          asm
             mov  ax, wXt  { 4 * X }
             shl  ax, 2
             add  ax, 6  { (4*X)+6 }
             mov  bx, wD
             add  bx, ax
             mov  wD, bx
          end {D := D + (4 * X) + 6}
          else
          asm
             mov  ax, wXt    { (X-Y) }
             mov  bx, wYt
             sub  ax, bx
             shl  ax, 2      { 4*(X-Y) }
             add  ax, 10
             mov  bx, wD
             add  bx, ax
             mov  wD, bx     { D := D + 4*(X-Y) + 10; }
             dec  wYt
          end;

          Inc(wXt);
     end;

     if (wXt = wYt) then CirclePoint(wXt, wYt, wx, wy, lCol);
end;

{******}

procedure S3_3001_Triangle(wx1, wy1, wx2, wy2, wx3, wy3 : word; lCol : longint);

begin
     S3_3001_Line(wx1, wy1, wx2, wy2, lCol);
     S3_3001_Line(wx2, wy2, wx3, wy3, lCol);
     S3_3001_Line(wx3, wy3, wx1, wy3, lCol);
end;

{******}

procedure S3_3001_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_320x200x8;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_PACKED8;
               bBitsPerPixel := 8;
               wPixelWidth   := 320;
               wByteWidth    := 320;
               wPixelHeight  := 200;
               wMinX         := 0;
               wMaxX         := 319;
               wMinY         := 0;
               wMaxY         := 199;
               wTotalPages   := wMemSize div 64;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_3001_SetActivePage;
          Proc_SetVisualPage   := S3_3001_SetVisualPage;

          Proc_Cls             := S3_3001_Cls;
          Proc_PutPixel        := S3_3001_PutPixel;
          Proc_Put2Pixel       := S3_3001_Put2Pixel;
          Proc_Put4Pixel       := S3_3001_Put4Pixel;
          Func_GetPixel        := S3_3001_GetPixel;
          Proc_HLine           := S3_3001_HLine;
          Proc_VLine           := S3_3001_VLine;
          Proc_Line            := S3_3001_Line;
          Proc_Rectangle       := S3_3001_Rectangle;
          Proc_FilledRectangle := S3_3001_FilledRectangle;
          Proc_Circle          := S3_3001_Circle;
          Proc_FilledCircle    := S3_3001_FilledCircle;
          Proc_Triangle        := S3_3001_Triangle;
          Proc_FilledTriangle  := nil;

          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $0013
        int  VID_INT

        mov  bx, CRTC_ADDR
        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, $48
        out  dx, al

        mov  dx, bx
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $A5
        out  dx, al

        mov  dx, SEQU_ADDR
        mov  al, $01
        out  dx, al
        inc  dx
        in   al, dx
        or   al, $20
        out  dx, al

        mov  dx, bx
        mov  al, $31
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $C6
        mov  cl, 9
        and  cl, $39
        add  al, cl
        out  dx, al

        mov  dx, bx
        mov  al, $4E
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, bx
        mov  al, $4F
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al

        mov  dx, bx
        mov  al, $34
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $7F
        out  dx, al

        mov  dx, bx
        mov  al, $40
        out  dx, al
        inc  dx
        in   al, dx
        or   al, 1
        out  dx, al

        mov  dx, SEQU_ADDR
        mov  al, 4
        out  dx, al
        inc  dx
        in   al, dx
        or   al, 2
        out  dx, al

        mov  ax, VID_SEG
        mov  es, ax
        xor  di, di
        db   $67
        mov  ax, 0
        mov  cx, 16000
        db   $66
        rep  stosw

        mov  dx, SEQU_ADDR
        mov  al, $01
        out  dx, al
        inc  dx
        in   al, dx
        and  al, $DF
        out  dx, al


        mov  dx, CRTC_ADDR
        mov  al, $39
        out  dx, al
        inc  dx
        mov  al, $5A
        out  dx, al

        mov  dx, bx
        mov  al, $38
        out  dx, al
        inc  dx
        mov  al, 0
        out  dx, al
     end;
end;

{******}

procedure S3_3001_Done;

begin
end;

{*** MODE $3002 - SVGA 640x480x256 mode ***}

procedure S3_3002_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_640x480x8;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_PACKED8;
               bBitsPerPixel := 8;
               wPixelWidth   := 640;
               wByteWidth    := 640;
               wPixelHeight  := 480;
               wMinX         := 0;
               wMaxX         := 639;
               wMinY         := 0;
               wMaxY         := 479;
               wTotalPages   := wMemSize div 300;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel8;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel8;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0101
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3002_Done;

begin
end;

{******}

procedure S3_3003_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_640x480x15;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE15;
               bBitsPerPixel := 16;
               wPixelWidth   := 640;
               wByteWidth    := 1280;
               wPixelHeight  := 480;
               wMinX         := 0;
               wMaxX         := 639;
               wMinY         := 0;
               wMaxY         := 479;
               wTotalPages   := wMemSize div 600;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_FilledTriangle  := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0110
        int  VID_INT
     end;

     S3_Init;
end;

{******}


procedure S3_3003_Done;

begin
end;

{******}

procedure S3_3004_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_640x480x16;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE16;
               bBitsPerPixel := 16;
               wPixelWidth   := 640;
               wByteWidth    := 1280;
               wPixelHeight  := 480;
               wMinX         := 0;
               wMaxX         := 639;
               wMinY         := 0;
               wMaxY         := 479;
               wTotalPages   := wMemSize div 600;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0111
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3004_Done;

begin
end;

{******}

procedure S3_3005_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_640x480x32;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE32;
               bBitsPerPixel := 32;
               wPixelWidth   := 640;
               wByteWidth    := 2560;
               wPixelHeight  := 480;
               wMinX         := 0;
               wMaxX         := 639;
               wMinY         := 0;
               wMaxY         := 479;
               wTotalPages   := wMemSize div 1200;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel32;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel32;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0112
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3005_Done;

begin
end;

{******}

procedure S3_3006_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_800x600x8;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_PACKED8;
               bBitsPerPixel := 8;
               wPixelWidth   := 800;
               wByteWidth    := 800;
               wPixelHeight  := 600;
               wMinX         := 0;
               wMaxX         := 799;
               wMinY         := 0;
               wMaxY         := 599;
               wTotalPages   := wMemSize div 468;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel8;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel8;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0103
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3006_Done;

begin
end;

{******}

procedure S3_3007_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_800x600x15;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE15;
               bBitsPerPixel := 16;
               wPixelWidth   := 800;
               wByteWidth    := 1600;
               wPixelHeight  := 600;
               wMinX         := 0;
               wMaxX         := 799;
               wMinY         := 0;
               wMaxY         := 599;
               wTotalPages   := wMemSize div 936;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0113
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3007_Done;

begin
end;

{******}

procedure S3_3008_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_800x600x16;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE16;
               bBitsPerPixel := 16;
               wPixelWidth   := 800;
               wByteWidth    := 1600;
               wPixelHeight  := 600;
               wMinX         := 0;
               wMaxX         := 799;
               wMinY         := 0;
               wMaxY         := 599;
               wTotalPages   := wMemSize div 936;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0114
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3008_Done;

begin
end;

{******}

procedure S3_3009_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_800x600x32;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE32;
               bBitsPerPixel := 32;
               wPixelWidth   := 800;
               wByteWidth    := 3200;
               wPixelHeight  := 600;
               wMinX         := 0;
               wMaxX         := 799;
               wMinY         := 0;
               wMaxY         := 599;
               wTotalPages   := wMemSize div 1872;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel32;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel32;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0115
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3009_Done;

begin
end;

{******}

procedure S3_300A_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_1024x768x8;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_PACKED8;
               bBitsPerPixel := 8;
               wPixelWidth   := 1024;
               wByteWidth    := 1024;
               wPixelHeight  := 768;
               wMinX         := 0;
               wMaxX         := 1023;
               wMinY         := 0;
               wMaxY         := 767;
               wTotalPages   := wMemSize div 768;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel8;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel8;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0205
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_300A_Done;

begin
end;

{******}

procedure S3_300B_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_1024x768x15;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE15;
               bBitsPerPixel := 15;
               wPixelWidth   := 1024;
               wByteWidth    := 2048;
               wPixelHeight  := 768;
               wMinX         := 0;
               wMaxX         := 1023;
               wMinY         := 0;
               wMaxY         := 767;
               wTotalPages   := wMemSize div 1536;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0116
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_300B_Done;

begin
end;

{******}

procedure S3_300C_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_1024x768x16;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE16;
               bBitsPerPixel := 16;
               wPixelWidth   := 1024;
               wByteWidth    := 2048;
               wPixelHeight  := 768;
               wMinX         := 0;
               wMaxX         := 1023;
               wMinY         := 0;
               wMaxY         := 767;
               wTotalPages   := wMemSize div 1536;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0117
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_300C_Done;

begin
end;

{******}

procedure S3_300D_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_1024x768x32;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE32;
               bBitsPerPixel := 32;
               wPixelWidth   := 1024;
               wByteWidth    := 4096;
               wPixelHeight  := 768;
               wMinX         := 0;
               wMaxX         := 1023;
               wMinY         := 0;
               wMaxY         := 767;
               wTotalPages   := wMemSize div 3072;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel32;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel32;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0118
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_300D_Done;

begin
end;

{******}

procedure S3_300E_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_1280x1024x8;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_PACKED8;
               bBitsPerPixel := 8;
               wPixelWidth   := 1280;
               wByteWidth    := 1280;
               wPixelHeight  := 1024;
               wMinX         := 0;
               wMaxX         := 1279;
               wMinY         := 0;
               wMaxY         := 1023;
               wTotalPages   := wMemSize div 1280;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel8;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel8;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_FilledTriangle  := nil;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0107
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_300E_Done;

begin
end;

{******}

procedure S3_300F_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_1280x1024x15;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE15;
               bBitsPerPixel := 16;
               wPixelWidth   := 1280;
               wByteWidth    := 2560;
               wPixelHeight  := 1024;
               wMinX         := 0;
               wMaxX         := 1279;
               wMinY         := 0;
               wMaxY         := 1023;
               wTotalPages   := wMemSize div 2560;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_FilledTriangle  := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $0119
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_300F_Done;

begin
end;

{******}

procedure S3_3010_Init;

begin
     with GFX_Screen do
     begin
          with ModeInfo do
          begin
               wMode         := GFX_S3_1280x1024x16;
               S3_DetectHardware(wChip, wMemSize);
               bMemModel     := MEM_TRUE16;
               bBitsPerPixel := 16;
               wPixelWidth   := 1280;
               wByteWidth    := 2560;
               wPixelHeight  := 1024;
               wMinX         := 0;
               wMaxX         := 1279;
               wMinY         := 0;
               wMaxY         := 1023;
               wTotalPages   := wMemSize div 2560;
          end;

          wActivePage          := 0;
          wVisualPage          := 0;
          wMaxPage             := ModeInfo.wTotalPages - 1;

          Func_GetVideoName    := S3_GetVideoName;

          Proc_SetActivePage   := S3_SetActivePage;
          Proc_SetVisualPage   := S3_SetVisualPage;

          Proc_Cls             := S3_Cls;
          Proc_PutPixel        := S3_PutPixel16;
          Proc_Put2Pixel       := S3_Put2Pixel;
          Proc_Put4Pixel       := S3_Put4Pixel;
          Func_GetPixel        := S3_GetPixel16;
          Proc_HLine           := S3_HLine;
          Proc_VLine           := S3_VLine;
          Proc_Line            := S3_Line;
          Proc_Rectangle       := S3_Rectangle;
          Proc_FilledRectangle := S3_FilledRectangle;
          Proc_Circle          := S3_Circle;
          Proc_FilledCircle    := S3_FilledCircle;
          Proc_Triangle        := S3_Triangle;
          Proc_Polygon         := S3_Polygon;
          Proc_FilledPolygon   := nil;
          Proc_FilledTriangle  := nil;
          Proc_PutBitmap       := nil;                   {NOT YET IMPLEMENTED}
          Proc_PutMaskedBitmap := nil;                   {NOT YET IMPLEMENTED}
          Proc_GetBitmap       := nil;                   {NOT YET IMPLEMENTED}
     end;

     asm
        mov  ax, $4F02
        mov  bx, $011A
        int  VID_INT
     end;

     S3_Init;
end;

{******}

procedure S3_3010_Done;

begin
end;

{******}

end.