unit SWIN95;

(* Information
   

   Program Title : SWIN95 - Windows95 related functions.
   External name : SWIN95.TPU
   Version       : 1.0
   Start date    : 20/2/97
   Last update   : 22/2/97
   Author        : Rob Anderton.
   Description   : Windows95 support routines including long filename support.

*)

interface

{******}

uses OBJECTS, STRINGS;

{******}

type
     (* Record structure used to store volume information returned by the
        Win95_GetVolumeInfo procedure.                                   *)

     PVolumeInfo95 = ^TVolumeInfo95;
     TVolumeInfo95 = record
                           wFlags           : word;
                           wFileNameLength  : word;
                           wDirectoryLength : word;
                           sFileSystem      : string;
                     end;

     (* TDOSStream95 is a TDosStream descendant which has a name field
        containing the long filename and makes use of the SWIN95 functions
        and procedures.                                                    *)

     PDosStream95 = ^TDOSStream95;
     TDOSStream95 = object(TDosStream)

                          pName : PChar;

                          constructor Init(pFileName : PChar; wMode : word);

                          function    GetPhysicalSize : longint; virtual;
                          function    GetLongFileName : PChar;
                          function    GetShortFileName : string;
                    end;

{******}

const
      (* Constants used for flags word returned by Win95_GetVolumeInfo. *)

      Win95_Volume_CaseSensitive    = 1;     {Searches are case sensitive}
      Win95_Volume_DirPreservesCase = 2;     {Directory entries preserve case}
      Win95_Volume_UniCode          = 4;     {Uses UniCode characters}
      Win95_Volume_DOSLFNSupport    = $4000; {Support DOS long filename access}
      Win95_Volume_Compressed       = $8000; {The volume is compressed}

      (* Constant used to determine if a drive letter created by SUBST is
         used in a long filename, or whether the true drive and pathname is
         returned.                                                          *)

      Win95_TrueName   = true;

{******}

      (* In this version of the library error codes are stored globally in
         this variable.                                                    *)

var   wError : word;

{******}

procedure Win95_GetVolumeInfo(sRootDir : string; var VolumeInfo : TVolumeInfo95);

function  Win95_GetLongFileName(var sShortFileName : string; bTruePath : boolean) : PChar;
function  Win95_GetShortFileName(var pLongFileName : PChar; bTruePath : boolean) : string;

function  Win95_CreateFile(pFileName : PChar; wFlags : word) : word;
function  Win95_OpenFile(pFileName : PChar; wFlags : word) : word;
procedure Win95_RenameFile(pOldFileName, pNewFileName : PChar);
procedure Win95_DeleteFile(pFileName : PChar);

procedure Win95_CompressedFileSize(pFileName : PChar; var lSize : longint);

procedure Win95_GetFileAttr(pFileName : PChar; var wAttr : word);
procedure Win95_SetFileAttr(pFileName : PChar; wAttr : word);

procedure Win95_GetFileWriteTime(pFileName : PChar; var lTime : longint);
procedure Win95_SetFileWriteTime(pFileName : PChar; lTime : longint);

procedure Win95_GetFileAccessTime(pFileName : PChar; var lTime : longint);
procedure Win95_SetFileAccessTime(pFileName : PChar; lTime : longint);

procedure Win95_GetFileCreationTime(pFileName : PChar; var lTime : longint);
procedure Win95_SetFileCreationTime(pFileName : PChar; lTime : longint);

procedure Win95_MakeDirectory(pDirName : PChar);
procedure Win95_RemoveDirectory(pDirName : PChar);
procedure Win95_ChangeDirectory(pDirName : PChar);

{******}

implementation

uses DOS;

{******}

procedure Win95_GetVolumeInfo(sRootDir : string; var VolumeInfo : TVolumeInfo95);

(* Returns information about the volume whose root directory is given by
   sRootDir. The information includes the file system (e.g. NTFS, FAT, CDFS,
   etc.) and whether the volume is compressed.                              *)

const wFileSystemLen : word = 32; {This value is accepted as sufficient by MS}

var pRootDir     : PChar;
    pFileSystem  : PChar;
    wFileNameLen : word;
    wDirNameLen  : word;
    wFlags       : word;

begin
     StrPCopy(pRootDir, sRootDir);
     pFileSystem:= '';
     wError:= 0;                    {No error (yet!)}

     asm
        stc
        mov     cx, wFileSystemLen  {This has to be done before pushing ds}
        push    ds
        mov     ax, $71A0           {Win95 Get Volume Information function}
        lds     dx, pRootDir
        les     di, pFileSystem
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     bx, 0
        mov     cx, 0
        mov     dx, 0
        mov     [wError], 1

     @NOERROR:

        mov     [wFlags], bx
        mov     [wFileNameLen], cx
        mov     [wDirNameLen], dx
        pop     ds
     end;

     VolumeInfo.wFlags:= wFlags;
     VolumeInfo.wFileNameLength:= wFileNameLen;
     VolumeInfo.wDirectoryLength:= wDirNameLen;
     VolumeInfo.sFileSystem:= StrPas(pFileSystem);
end;

{******}

function Win95_GetLongFileName(var sShortFileName : string; bTruePath : boolean) : PChar;

(* Returns the full path and long filename for the given DOS short filename.
   sShortFileName can also include a path, e.g. C:\PROGRA~1\TESTFI~1.TXT
   would be expanded to C:\PROGRAMMING\TESTFILE95.TXT.                         *)

var pShortFileName : PChar;
    pLongFileName  : PChar;
    bTrueName      : byte;

begin
     StrPCopy(pShortFileName, sShortFileName);
     pLongFileName:= '';
     if bTruePath then bTrueName:= $80 else bTrueName:= 0;
     wError:= 0;

     asm
        stc
        mov     ax, $7160
        mov     cl, $02
        mov     ch, [bTrueName]

        push    ds
        lds     si, pShortFileName
        les     di, pLongFileName
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;

     Win95_GetLongFileName:= pLongFileName;
end;

{******}

function Win95_GetShortFileName(var pLongFileName : PChar; bTruePath : boolean) : string;

(* Returns the full path and short filename for the given Win95 long filename.
   sLongFileName can also include a path, e.g. C:\PROGRAMMING\TESTFILE95.TXT
   would be condensed to C:\PROGRA~1\TESTFI~1.TXT.                           *)

var pShortFileName : PChar;
    sShortFileName : string;
    pLongFileLocal : PChar;
    bTrueName      : byte;

begin
     pShortFileName:= '';
     PLongFileLocal:= PLongFileName;
     if bTruePath then bTrueName:= $80 else bTrueName:= 0;
     wError:= 0;

     asm
        stc
        mov     ax, $7160
        mov     cl, $01
        mov     ch, [bTrueName]

        push    ds
        lds     si, pLongFileLocal
        les     di, pShortFileName
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;

     sShortFileName:= StrPas(pShortFileName);
     Win95_GetShortFileName:= sShortFileName;
end;

{******}

function Win95_CreateFile(pFileName : PChar; wFlags : word) : word;

(* Create a new file, returning the DOS handle                              *)

var pFile    : PChar;
    wHandle  : word;

begin
     pFile:= pFileName;
     wError:= 0;

     asm
        stc
        mov     ax, $716C
        mov     bx, [wFlags]
        mov     cx, 0
        mov     dx, $12         {Command = create file/delete if exists}
        push    ds
        lds     si, pFile
        mov     di, 1

        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wHandle], 0
        mov     [wError], ax
        jmp     @EXIT

     @NOERROR:

        mov     [wHandle], ax

     @EXIT:

        pop     ds

     end;
     Win95_CreateFile:= wHandle;
end;

{******}

function  Win95_OpenFile(pFileName : PChar; wFlags : word) : word;

(* Open an existing file, returning the DOS handle                          *)

var pFile    : PChar;
    wHandle  : word;

begin
     pFile:= pFileName;
     wError:= 0;

     asm
        stc
        mov     ax, $716C
        mov     bx, [wFlags]
        mov     cx, 0
        mov     dx, 1          {Command = open file}
        push    ds
        lds     si, pFile

        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wHandle], 0
        mov     [wError], ax
        jmp     @EXIT

     @NOERROR:

        mov     [wHandle], ax

     @EXIT:

        pop     ds

     end;
     Win95_OpenFile:= wHandle;
end;

{******}

procedure Win95_RenameFile(pOldFileName, pNewFileName : PChar);

(* Rename a file using long file names *)

var pOld, pNew : PChar;

begin
     pOld:= pOldFileName;
     pNew:= pNewFileName;
     wError:= 0;

     asm
        stc
        push    ds

        mov     ax, $7156
        lds     dx, pOld
        les     di, pNew
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_DeleteFile(pFileName : PChar);

(* Delete a file using it's long filename *)

var pDelFile : PChar;

begin
     pDelFile:= pFileName;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7141
        lds     dx, pDelFile
        mov     si, 0
        mov     cx, 0
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_CompressedFileSize(pFileName : PChar; var lSize : longint);

(* Return the physical size of a compressed file *)

var pCompFile : PChar;
    wSizeLo   : word;
    wSizeHi   : word;

begin
     pCompFile:= pFileName;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pCompFile
        mov     bl, 2
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax
        mov     [wSizeLo], 0
        mov     [wSizeHi], 0
        jmp     @EXIT

     @NOERROR:

        mov     [wSizeLo], ax
        mov     [wSizeHi], dx

     @EXIT:

        pop     ds

     end;

     lSize:= (longint(wSizeHi) shl 16) + wSizeLo;
end;

{******}

procedure Win95_GetFileAttr(pFileName : PChar; var wAttr : word);

(* Get extended file attributes *)

var pFile  : PChar;
    wTemp  : word;

begin
     pFile:= pFileName;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     bl, 0
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax
        mov     [wTemp], 0
        jmp     @EXIT

     @NOERROR:

        mov     [wTemp], cx

     @EXIT:

        pop     ds

     end;
     wAttr:= wTemp;
end;

{******}

procedure Win95_SetFileAttr(pFileName : PChar; wAttr : word);

(* Set extended file attributes *)

var pFile  : PChar;
    wTemp  : word;

begin
     pFile:= pFileName;
     wTemp:= wAttr;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     cx, [wTemp]
        mov     bl, 1
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_GetFileWriteTime(pFileName : PChar; var lTime : longint);

(* Get time of last write to file *)

var pFile   : PChar;
    wTimeLo : word;
    wTimeHi : word;

begin
     pFile:= pFileName;
     wTimeHi:= 0;
     wTimeLo:= 0;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     bl, 4
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax
        jmp     @EXIT

     @NOERROR:

        mov     [wTimeHi], di
        mov     [wTimeLo], cx

     @EXIT:
        pop     ds

     end;

     lTime:= (longint(wTimeHi) shl 16) + wTimeLo;
end;


{******}

procedure Win95_SetFileWriteTime(pFileName : PChar; lTime : longint);

(* Set time of last write to file *)

var pFile   : PChar;
    wTimeLo : word;
    wTimeHi : word;

begin
     pFile:= pFileName;
     wTimeHi:= longint(lTime shr 16);
     wTimeLo:= word(lTime);
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     bl, 3
        mov     di, [wTimeHi]
        mov     cx, [wTimeLo]
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_GetFileAccessTime(pFileName : PChar; var lTime : longint);

(* Get time of last file access *)

var pFile   : PChar;
    wTimeLo : word;
    wTimeHi : word;

begin
     pFile:= pFileName;
     wTimeHi:= 0;
     wTimeLo:= 0;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     bl, 6
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax
        jmp     @EXIT

     @NOERROR:

        mov     [wTimeHi], di
        mov     [wTimeLo], 0

     @EXIT:
        pop     ds

     end;

     lTime:= (longint(wTimeHi) shl 16) + wTimeLo;
end;

{******}

procedure Win95_SetFileAccessTime(pFileName : PChar; lTime : longint);

(* Set time of last file access *)

var pFile   : PChar;
    wTimeLo : word;
    wTimeHi : word;

begin
     pFile:= pFileName;
     wTimeHi:= longint(lTime shr 16);
     wTimeLo:= 0;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     bl, 5
        mov     di, [wTimeHi]
        mov     cx, [wTimeLo]
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_GetFileCreationTime(pFileName : PChar; var lTime : longint);

(* Get time of file creation - ignoring 100ths of a second accuracy *)

var pFile   : PChar;
    wTimeLo : word;
    wTimeHi : word;

begin
     pFile:= pFileName;
     wTimeHi:= 0;
     wTimeLo:= 0;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     bl, 8
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax
        jmp     @EXIT

     @NOERROR:

        mov     [wTimeHi], di
        mov     [wTimeLo], cx

     @EXIT:
        pop     ds

     end;

     lTime:= (longint(wTimeHi) shl 16) + wTimeLo;
end;


{******}

procedure Win95_SetFileCreationTime(pFileName : PChar; lTime : longint);

(* Set time of file creation - ignoring 100ths of a second accuracy *)

var pFile   : PChar;
    wTimeLo : word;
    wTimeHi : word;

begin
     pFile:= pFileName;
     wTimeHi:= longint(lTime shr 16);
     wTimeLo:= word(lTime);
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7143
        lds     dx, pFile
        mov     bl, 7
        mov     di, [wTimeHi]
        mov     cx, [wTimeLo]
        mov     si, 0
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_MakeDirectory(pDirName : PChar);

(* Make a directory with a long directory name *)

var pDir    : PChar;

begin
     pDir:= pDirName;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $7139
        lds     dx, pDir
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_RemoveDirectory(pDirName : PChar);

(* Remove a directory with a long directory name *)

var pDir    : PChar;

begin
     pDir:= pDirName;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $713A
        lds     dx, pDir
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{******}

procedure Win95_ChangeDirectory(pDirName : PChar);

(* Change current dirctory using long directory name *)

var pDir    : PChar;

begin
     pDir:= pDirName;
     wError:= 0;

     asm
        stc
        push    ds
        mov     ax, $713B
        lds     dx, pDir
        int     $21
        jnc     @NOERROR

     @ERROR:

        mov     [wError], ax

     @NOERROR:

        pop     ds

     end;
end;

{*** TDOSStream95 object methods ***}

constructor TDOSStream95.Init(pFileName : PChar; wMode : word);

(* Create a stream using pFileName as the long filename. wMode is one of the
   stXXXX constants defined in the OBJECTS unit.                            *)

begin
     if not TStream.Init then Fail;
     case wMode of

          stCreate    : Handle:= Win95_CreateFile(pFileName, 2);
          stOpenRead  : Handle:= Win95_OpenFile(pFileName, 0);
          stOpenWrite : Handle:= Win95_OpenFile(pFileName, 1);
          stOpen      : Handle:= Win95_OpenFile(pFileName, 2);

     end;

     if wError <> 0 then Error(stInitError, 0)
                    else pName:= pFileName;
end;

{******}

function TDOSStream95.GetPhysicalSize : longint;

var lTemp : longint;

begin
     Win95_CompressedFileSize(pName, lTemp);
     if wError <> 0 then
     begin
          Error(stError, 0);
          GetPhysicalSize:= 0;
     end
     else GetPhysicalSize:= lTemp;
end;

{******}

function TDOSStream95.GetLongFileName : PChar;

begin
     GetLongFileName:= pName;
end;

{******}

function TDOSStream95.GetShortFileName : string;

begin
     GetShortFileName:= Win95_GetShortFileName(pName, Win95_TrueName);
     if wError <> 0 then
     begin
          Error(stError, 0);
          GetShortFileName:= '';
     end;
end;

{******}

end.