program SNDLIB;

(* Information
   

   Program Title : Picture resource file builder.
   External name : PICLIB.EXE
   Version       : 1.0
   Start date    : 10/11/96
   Last update   : 10/11/96
   Author        : Rob Anderton.
   Description   : Utility to create and modify picture resource files.

*)

uses CRT, OBJECTS, SRES, SUTILS, SSPF, SERROR;

{*** Command constants ***}

const CM_CREATE  = 10;
      CM_ADD     = 11;
      CM_DELETE  = 12;
      CM_EXTRACT = 13;
      CM_PACK    = 14;
      CM_VIEW    = 15;

{******}

type TCommandRec = record
                         case Command : byte of
                              CM_ADD,
                              CM_EXTRACT : (PicName : string;
                                            PicKey  : string);

                              CM_DELETE  : (DelKey : string);
                   end;

{******}

procedure PicLib_StreamError(var S : TStream); far;

begin
     writeln;
     writeln('File I/O error : ', S.Status);
     writeln;
     halt(1);
end;

{******}

procedure PicLib_Usage;

begin
     writeln('Usage : PICLIB <RESOURCEFILE> <OPTIONS>');
     writeln;
     writeln('Options summary : ');
     writeln;
     writeln(' /C                       : create a new resource file.');
     writeln(' /A  <PICTUREFILE> <NAME> : add a new picture resource with name <NAME>.');
     writeln(' /D  <NAME>               : delete the resource called <NAME>.');
     writeln(' /E  <PICTUREFILE> <NAME> : extract picture called <NAME>.');
     writeln(' /P                       : pack the resource file (to reduce file size).');
     writeln(' /V                       : view contents of resource file.');
     writeln;
     halt(1);
end;

{******}

procedure PicLib_Init(var ResName : string; var C : TCommandRec);

var CmdStr : string;
    Ch     : char;

begin
     writeln;
     writeln('---------------------------------');
     writeln(' PICLIB picture resource manager ');
     writeln('          Version 1.0.           ');
     writeln('      (c)1996 Rob Anderton.      ');
     writeln('---------------------------------');
     writeln;

     if ParamCount < 2 then PicLib_Usage;

     ResName:= U_StrUpper(ParamStr(1));
     CmdStr:= U_StrUpper(ParamStr(2));

     if Length(CmdStr) < 2 then PicLib_Usage;

     case CmdStr[2] of
                 'C' : begin
                             C.Command:= CM_CREATE;
                             if U_FileExists(ResName) then
                             begin
                                  writeln('Resource file already exists!');
                                  writeln('Overwrite (Y/N)?');
                                  Ch:= Upcase(readkey);
                                  if Ch = 'N' then halt(0);
                             end;
                             writeln('Creating ', ResName, '...');
                        end;

                 'A' : begin
                             if ParamCount < 4 then PicLib_Usage;
                             C.Command:= CM_ADD;
                             C.PicName:= U_StrUpper(ParamStr(3));
                             C.PicKey:= U_StrUpper(ParamStr(4));
                             if not U_FileExists(ResName) or not U_FileExists(C.PicName) then
                             begin
                                  writeln('File not found!');
                                  halt(1);
                             end;
                        end;

                 'D' : begin
                             if ParamCount < 3 then PicLib_Usage;
                             if not U_FileExists(ResName) then writeln('Resource file not found!');
                             C.Command:= CM_DELETE;
                             C.DelKey:= U_StrUpper(ParamStr(3));
                        end;

                 'E' : begin
                             if ParamCount < 4 then PicLib_Usage;
                             if not U_FileExists(ResName) then
                             begin
                                  writeln('Resource file not found!');
                                  halt(1);
                             end;
                             C.Command:= CM_EXTRACT;
                             C.PicName:= U_StrUpper(ParamStr(3));
                             C.PicKey:= U_StrUpper(ParamStr(4));
                             if U_FileExists(C.PicName) then
                             begin
                                  writeln('Picture file already exists!');
                                  writeln('Overwrite (Y/N)?');
                                  Ch:= Upcase(readkey);
                                  if Ch = 'N' then halt(1);
                             end;

                        end;

                 'P' : begin
                             C.Command:= CM_PACK;
                             if not U_FileExists(ResName) then
                             begin
                                  writeln('Resource file not found!');
                                  halt(1);
                             end;
                        end;

                 'V' : begin
                             C.Command:= CM_VIEW;
                             if not U_FileExists(ResName) then
                             begin
                                  writeln('Resource file not found!');
                                  halt(1);
                             end;
                        end;
                   else PicLib_Usage;
     end;
end;

{******}

procedure PicLib_AddPic(var ResFile : TResourceFile; CommandInfo : TCommandRec);

var Pic  : TBufStream;
    Hdr  : TSPFHeader;
    PR   : TPicRes;
    Data : byte;
    l    : longint;

begin
     writeln('Adding ', CommandInfo.PicName, ':');
     Pic.Init(CommandInfo.PicName, stOpen, 2048);
     Pic.Read(Hdr, sizeof(Hdr));

     if (Hdr.SPFID <> 'SPF') then
     begin
          Pic.Done;
          writeln;
          writeln('Invalid SPF file.');
          writeln;
          halt(1);
     end;

     PR.Init;
{     if PR = nil then halt(1);}
     PR.PicInfo:= Hdr;

     ResFile.Put(@PR, CommandInfo.PicKey);
     ResFile.Stream^.Seek(ResFile.Stream^.GetPos - (PR.PicInfo.Width * PR.PicInfo.Height));

     for l:= 1 to (Hdr.Width * Hdr.Height) do
     begin
          Pic.Read(Data, sizeof(Data));
          ResFile.Stream^.Write(Data, Sizeof(Data));
          if ((Hdr.Width * Hdr.Height) mod l) = 0 then write('');
     end;
     Pic.Done;
     PR.Done;
end;

{******}

procedure PicLib_DelPic(var ResFile : TResourceFile; CommandInfo : TCommandRec);

var ch : char;

begin
     write('Delete ', CommandInfo.DelKey, ' (Y/N)? ');
     ch:= upcase(readkey);
     if ch = 'Y' then
     begin
          writeln('Y');
          writeln('Deleting picture ', CommandInfo.DelKey);
          ResFile.Delete(CommandInfo.DelKey);
     end
     else writeln('N');
end;

{******}

procedure PicLib_ExtPic(var ResFile : TResourceFile; CommandInfo : TCommandRec);

var Pic  : TBufStream;
    Hdr  : TSPFHeader;
    PR   : PPicRes;
    l    : longint;
    Data : byte;

begin
     writeln('Extracting picture ', CommandInfo.PicKey, ' to ', CommandInfo.PicName, ':');
     Pic.Init(CommandInfo.PicName, stCreate, 1024);
     PR:= PPicRes(ResFile.Get(CommandInfo.PicKey));
     if PR = nil then halt(1);
     Hdr:= PR^.PicInfo;
     Pic.Write(Hdr, sizeof(Hdr));
     ResFile.Stream^.Seek(ResFile.Stream^.GetPos - (PR^.PicInfo.Width * PR^.PicInfo.Height));

     for l:= 1 to (Hdr.Width * Hdr.Height) do
     begin
          ResFile.Stream^.Read(Data, Sizeof(Data));
          Pic.Write(Data, sizeof(Data));
          if ((Hdr.Width * Hdr.Height) mod l) = 0 then write('');
     end;

     Pic.Done;
     PR^.Done;
end;

{******}

procedure PicLib_Pack(var ResFile : TResourceFile);

var Tmp : PBufStream;
    Old : PBufStream;
    f   : file;

begin
     writeln('Packing resource file...');
     Tmp^.Init('PICLIB.TMP', stCreate, 2048);
     if Tmp = nil then halt(1);
     Old:= PBufStream(ResFile.SwitchTo(Tmp, true));
     Old^.Seek(0);
     Old^.Truncate;
     ResFile.SwitchTo(Old, false);
     Tmp^.Done;
     Assign(f, 'PICLIB.TMP');
     Erase(f);
end;

{******}

procedure PicLib_View(var ResFile : TResourceFile);

var Loop, MaxRec : integer;
    Key          : string;
    Pic          : PPicRes;

begin
     writeln('Viewing resource file...');
     writeln;

     MaxRec:= ResFile.Count - 1;

     for Loop:= 0 to MaxRec do
     begin
          Key:= ResFile.KeyAt(Loop);
          Pic:= PPicRes(ResFile.Get(Key));
          writeln('Picture ', Loop + 1, ' information.');
          writeln('-----------------------------------');
          writeln;
          if Pic <> nil then
          with Pic^.PicInfo do
          begin
               writeln('Name      : ', Key);
               writeln('Width     : ', Width);
               writeln('Height    : ', Height);
               writeln('Size      : ', Width * Height);
          end
          else writeln('Unable to get data.');
          readkey;
          writeln;
     end;
end;

{******}

{******}

var ResFile     : TResourceFile;
    ResName     : string;
    CommandInfo : TCommandRec;

begin
     StreamError:= @PicLib_StreamError;

     PicLib_Init(ResName, CommandInfo);

     if CommandInfo.Command <> CM_CREATE then
        ResFile.Init(New(PBufStream, Init(ResName, stOpen, 2048)))
     else
        ResFile.Init(New(PBufStream, Init(ResName, stCreate, 2048)));

     if ResFile.Stream = nil then halt(1);

     case CommandInfo.Command of
                  CM_ADD     : PicLib_AddPic(ResFile, CommandInfo);
                  CM_DELETE  : PicLib_DelPic(ResFile, CommandInfo);
                  CM_EXTRACT : PicLib_ExtPic(ResFile, CommandInfo);
                  CM_PACK    : PicLib_Pack(ResFile);
                  CM_VIEW    : PicLib_View(ResFile);
     end;

     ResFile.Done;
end.