program SNDLIB;

(* Information
   

   Program Title : Sound resource file builder.
   External name : SNDLIB.EXE
   Version       : 1.0
   Start date    : 23/10/96
   Last update   : 23/10/96
   Author        : Rob Anderton.
   Description   : Utility to create and modify sound resource files.

*)

uses CRT, OBJECTS, SRES, SUTILS, SBDSP, SERROR;

{*** Command constants ***}

const CM_CREATE  = 10;
      CM_ADD     = 11;
      CM_DELETE  = 12;
      CM_EXTRACT = 13;
      CM_PACK    = 14;
      CM_VIEW    = 15;

{******}

type TCommandRec = record
                         case Command : byte of
                              CM_ADD,
                              CM_EXTRACT : (SoundName : string;
                                            SoundKey  : string);

                              CM_DELETE  : (DelKey : string);
                   end;

{******}

procedure SndLib_StreamError(var S : TStream); far;

begin
     writeln;
     writeln('File I/O error : ', S.Status);
     writeln;
     halt(1);
end;

{******}


procedure SndLib_Usage;

begin
     writeln('Usage : SNDLIB <RESOURCEFILE> <OPTIONS>');
     writeln;
     writeln('Options summary : ');
     writeln;
     writeln(' /C                     : create a new resource file.');
     writeln(' /A  <SOUNDFILE> <NAME> : add a new sound resource with name <NAME>.');
     writeln(' /D  <NAME>             : delete the resource called <NAME>.');
     writeln(' /E  <SOUNDFILE> <NAME> : extract sound called <NAME>.');
     writeln(' /P                     : pack the resource file (to reduce file size).');
     writeln(' /V                     : view contents of resource file.');
     writeln;
     halt(1);
end;

{******}

procedure SndLib_Init(var ResName : string; var C : TCommandRec);

var CmdStr : string;
    Ch     : char;

begin
     writeln;
     writeln('-------------------------------');
     writeln(' SNDLIB sound resource manager ');
     writeln('         Version 1.0.          ');
     writeln('     (c)1996 Rob Anderton.     ');
     writeln('-------------------------------');
     writeln;

     if ParamCount < 2 then SndLib_Usage;

     ResName:= U_StrUpper(ParamStr(1));
     CmdStr:= U_StrUpper(ParamStr(2));

     if Length(CmdStr) < 2 then SndLib_Usage;

     case CmdStr[2] of
                 'C' : begin
                             C.Command:= CM_CREATE;
                             if U_FileExists(ResName) then
                             begin
                                  writeln('Resource file already exists!');
                                  writeln('Overwrite (Y/N)?');
                                  Ch:= Upcase(readkey);
                                  if Ch = 'N' then halt(0);
                             end;
                             writeln('Creating ', ResName, '...');
                        end;

                 'A' : begin
                             if ParamCount < 4 then SndLib_Usage;
                             C.Command:= CM_ADD;
                             C.SoundName:= U_StrUpper(ParamStr(3));
                             C.SoundKey:= U_StrUpper(ParamStr(4));
                             if not U_FileExists(ResName) or not U_FileExists(C.SoundName) then
                             begin
                                  writeln('File not found!');
                                  halt(1);
                             end;
                        end;

                 'D' : begin
                             if ParamCount < 3 then SndLib_Usage;
                             if not U_FileExists(ResName) then writeln('Resource file not found!');
                             C.Command:= CM_DELETE;
                             C.DelKey:= U_StrUpper(ParamStr(3));
                        end;

                 'E' : begin
                             if ParamCount < 4 then SndLib_Usage;
                             if not U_FileExists(ResName) then
                             begin
                                  writeln('Resource file not found!');
                                  halt(1);
                             end;
                             C.Command:= CM_EXTRACT;
                             C.SoundName:= U_StrUpper(ParamStr(3));
                             C.SoundKey:= U_StrUpper(ParamStr(4));
                             if U_FileExists(C.SoundName) then
                             begin
                                  writeln('Sound file already exists!');
                                  writeln('Overwrite (Y/N)?');
                                  Ch:= Upcase(readkey);
                                  if Ch = 'N' then halt(1);
                             end;

                        end;

                 'P' : begin
                             C.Command:= CM_PACK;
                             if not U_FileExists(ResName) then
                             begin
                                  writeln('Resource file not found!');
                                  halt(1);
                             end;
                        end;

                 'V' : begin
                             C.Command:= CM_VIEW;
                             if not U_FileExists(ResName) then
                             begin
                                  writeln('Resource file not found!');
                                  halt(1);
                             end;
                        end;
                   else SndLib_Usage;
     end;
end;

{******}

procedure SndLib_AddSound(var ResFile : TResourceFile; CommandInfo : TCommandRec);

var Snd  : TBufStream;
    Hdr  : TSoundHdr;
    SR   : TSoundRes;
    Data : byte;
    l    : longint;

begin
     writeln('Adding ', CommandInfo.SoundName, ':');
     Snd.Init(CommandInfo.SoundName, stOpen, 2048);
     Snd.Read(Hdr, sizeof(Hdr));

     if (Hdr.SoundID <> 'SWF') then
     begin
          Snd.Done;
          writeln;
          writeln('Invalid SWF file.');
          writeln;
          halt(1);
     end;


     SR.Init;
{     if SR = nil then halt(1);}
     SR.SoundInfo:= Hdr;

     ResFile.Put(@SR, CommandInfo.SoundKey);
     ResFile.Stream^.Seek(ResFile.Stream^.GetPos - SR.SoundInfo.Size);

     for l:= 1 to Hdr.Size do
     begin
          Snd.Read(Data, sizeof(Data));
          ResFile.Stream^.Write(Data, Sizeof(Data));
          if (Hdr.Size mod l) = 0 then write('');
     end;
     Snd.Done;
     SR.Done;
end;

{******}

procedure SndLib_DelSound(var ResFile : TResourceFile; CommandInfo : TCommandRec);

var ch : char;

begin
     write('Delete ', CommandInfo.DelKey, ' (Y/N)? ');
     ch:= upcase(readkey);
     if ch = 'Y' then
     begin
          writeln('Y');
          writeln('Deleting sound ', CommandInfo.DelKey);
          ResFile.Delete(CommandInfo.DelKey);
     end
     else writeln('N');
end;

{******}

procedure SndLib_ExtSound(var ResFile : TResourceFile; CommandInfo : TCommandRec);

var Snd  : TBufStream;
    Hdr  : TSoundHdr;
    SR   : PSoundRes;
    l    : longint;
    Data : byte;

begin
     writeln('Extracting sound ', CommandInfo.SoundKey, ' to ', CommandInfo.SoundName, ':');
     Snd.Init(CommandInfo.SoundName, stCreate, 1024);
     SR:= PSoundRes(ResFile.Get(CommandInfo.SoundKey));
     if SR = nil then halt(1);
     Hdr:= SR^.SoundInfo;
     Snd.Write(Hdr, sizeof(Hdr));
     ResFile.Stream^.Seek(ResFile.Stream^.GetPos - SR^.SoundInfo.Size);

     for l:= 1 to Hdr.Size do
     begin
          ResFile.Stream^.Read(Data, Sizeof(Data));
          Snd.Write(Data, sizeof(Data));
          if (Hdr.Size mod l) = 0 then write('');
     end;

     Snd.Done;
     SR^.Done;
end;

{******}

procedure SndLib_Pack(var ResFile : TResourceFile);

var Tmp : PBufStream;
    Old : PBufStream;
    f   : file;

begin
     writeln('Packing resource file...');
     Tmp^.Init('SNDLIB.TMP', stCreate, 2048);
     if Tmp = nil then halt(1);
     Old:= PBufStream(ResFile.SwitchTo(Tmp, true));
     Old^.Seek(0);
     Old^.Truncate;
     ResFile.SwitchTo(Old, false);
     Tmp^.Done;
     Assign(f, 'SNDLIB.TMP');
     Erase(f);
end;

{******}

procedure SndLib_View(var ResFile : TResourceFile);

var Loop, MaxRec : integer;
    Key          : string;
    Snd          : PSoundRes;

begin
     writeln('Viewing resource file...');
     writeln;

     MaxRec:= ResFile.Count - 1;

     for Loop:= 0 to MaxRec do
     begin
          Key:= ResFile.KeyAt(Loop);
          Snd:= PSoundRes(ResFile.Get(Key));
          writeln('Sound ', Loop + 1, ' information.');
          writeln('-----------------------------');
          writeln;
          if Snd <> nil then
          with Snd^.SoundInfo do
          begin
               writeln('Name      : ', Key);
               writeln('Stereo    : ', Stereo);
               writeln('Bits      : ', Bits);
               writeln('Frequency : ', Freq);
               writeln('Size      : ', Size);
          end
          else writeln('Unable to get data.');
          readkey;
          writeln;
     end;
end;

{******}

{******}

var ResFile     : TResourceFile;
    ResName     : string;
    CommandInfo : TCommandRec;

begin
     StreamError:= @SndLib_StreamError;

     SndLib_Init(ResName, CommandInfo);

     if CommandInfo.Command <> CM_CREATE then
        ResFile.Init(New(PBufStream, Init(ResName, stOpen, 2048)))
     else
        ResFile.Init(New(PBufStream, Init(ResName, stCreate, 2048)));

     if ResFile.Stream = nil then halt(1);

     case CommandInfo.Command of
                  CM_ADD     : SndLib_AddSound(ResFile, CommandInfo);
                  CM_DELETE  : SndLib_DelSound(ResFile, CommandInfo);
                  CM_EXTRACT : SndLib_ExtSound(ResFile, CommandInfo);
                  CM_PACK    : SndLib_Pack(ResFile);
                  CM_VIEW    : SndLib_View(ResFile);
     end;

     ResFile.Done;
end.