unit SAdlib;

(* Information
   

   Program Title : Adlib sound for S-Library.
   External name : SADLIB.TPU
   Version       : 1.1
   Start date    : 23/8/96
   Last update   : 27/8/96
   Author        : Rob Anderton.
   Description   : Adlib support, based on ADLIB.PAS from the PASKAL graphics
                   library v2.0 by Kevin A. Lee. Also works for SoundBlaster
                   OPL2 FM sound.

*)

interface

{******}

type FMInstrument = record
                          Property   : array[1..2] of byte;
                          Level      : array[1..2] of byte;
                          Attack     : array[1..2] of byte;
                          Sustain    : array[1..2] of byte;
                          WaveSelect : array[1..2] of byte;
                          Feedback   : byte;
                    end;

{******}

function  FM_Detect : boolean;
procedure FM_Init;

procedure FM_KeyOn(Voice : byte; Freq : word; Octave : byte);
procedure FM_KeyOff(Voice : byte);
procedure FM_Volume(Voice, Volume : byte);
procedure FM_SetVoice(Voice : byte; var Instrument : FMInstrument);

procedure LoadSBI(FileName : string; var Instrument : FMInstrument);

{*** Notes (octave 4) ***}

const D4b           = $16B;
      D4            = $181;
      E4b           = $198;
      E4            = $1B0;
      F4            = $1CA;
      G4b           = $1E5;
      G4            = $202;
      A4b           = $220;
      A4            = $241;
      B4b           = $263;
      B4            = $287;
      C4            = $2AE;

{******}

const ADLIB_ADDRESS = $388;
      ADLIB_DATA    = $389;

{******}

implementation

uses CRT, SERROR;

{******}

var ShadowReg    : array[1..245] of byte;
    AdlibPresent : boolean;
    OldExit      : pointer;

{******}

procedure FM_Write(Reg, Data : byte);

var i    : integer;
    Temp : byte;

begin
     Port[ADLIB_ADDRESS]:= Reg;
     for i:= 1 to 6 do Temp:= Port[ADLIB_ADDRESS];
     Port[ADLIB_DATA]:= Data;
     for i:= 1 to 35 do Temp:= Port[ADLIB_ADDRESS];
     ShadowReg[Reg]:= Data;
end;

{******}

function FM_Read(Reg : byte) : byte;

begin
     FM_Read:= ShadowReg[Reg];
end;

{******}

function FM_Status : byte;

begin
     FM_Status:= Port[ADLIB_ADDRESS];
end;

{******}

procedure FM_Exit; far;

var i : word;

begin
     ExitProc:= OldExit;
     if AdlibPresent then for i:= $01 to $F5 do FM_Write(i, 0);
end;

{******}

function FM_Detect : boolean;

var Status1, Status2 : byte;

begin
     FM_Write($04, $60);
     FM_Write($04, $80);

     Status1:= FM_Status;

     FM_Write($02, $FF);
     FM_Write($04, $21);
     Delay(800);

     Status2:= FM_Status;

     FM_Write($04, $60);
     FM_Write($04, $80);

     if (((Status1 and $E0) = 0) and
         ((Status2 and $E0) = $C0)) then FM_Detect:= true
                                    else FM_Detect:= false;
end;

{******}

procedure FM_Init;

var i : byte;

begin
     AdlibPresent:= true;
     if not FM_Detect then
     begin
          AdlibPresent:= false;
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_NOADLIB;
          exit;
     end;

     for i:= $01 to $F5 do FM_Write(i, 0);
     FM_Write($01, $20);
     FM_Write($BD, $20);
end;

{******}

procedure FM_KeyOn(Voice : byte; Freq : word; Octave : byte);

var Reg, Temp : byte;

begin
     if not AdlibPresent then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_NOADLIB;
          exit;
     end;

     Reg:= $A0 + Voice mod 11;
     FM_Write(Reg, Freq and $FF);
     Reg:= $B0 + Voice mod 11;
     Temp:= (Freq shr 8) or (Octave shl 2) or $20;
     FM_Write(Reg, Temp);
end;

{******}

procedure FM_KeyOff(Voice : byte);

var Reg : byte;

begin
     if not AdlibPresent then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_NOADLIB;
          exit;
     end;

     Reg:= $B0 + Voice mod 11;
     FM_Write(Reg, 0);
end;

{******}

procedure FM_Volume(Voice, Volume : byte);

var Reg : byte;

begin
     if not AdlibPresent then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_NOADLIB;
          exit;
     end;

     Reg:= $40 + Voice mod 11;
     FM_Write(Reg, Volume);
end;

{******}

procedure FM_SetVoice(Voice : byte; var Instrument : FMInstrument);

var OpCellNum, OpCellOffset, i : byte;

begin
     if not AdlibPresent then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_NOADLIB;
          exit;
     end;

     Voice:= Voice mod 11;
     OpCellOffset:= Voice mod 3 + ((Voice div 3) shl 3);

     OpCellNum:= $20 + OpCellOffset;
     FM_Write(OpCellNum, Instrument.Property[1]);
     OpCellNum:= OpCellNum + 3;
     FM_Write(OpCellNum, Instrument.Property[2]);

     OpCellNum:= $40 + OpCellOffset;
     FM_Write(OpCellNum, Instrument.Level[1]);
     Inc(OpCellNum, 3);
     FM_Write(OpCellNum, Instrument.Level[2]);

     OpCellNum:= $60 + OpCellOffset;
     FM_Write(OpCellNum, Instrument.Attack[1]);
     Inc(OpCellNum, 3);
     FM_Write(OpCellNum, Instrument.Attack[2]);

     OpCellNum:= $80 + OpCellOffset;
     FM_Write(OpCellNum, Instrument.Sustain[1]);
     Inc(OpCellNum, 3);
     FM_Write(OpCellNum, Instrument.Sustain[2]);

     OpCellNum:= $E0 + OpCellOffset;
     FM_Write(OpCellNum, Instrument.WaveSelect[1]);
     Inc(OpCellNum, 3);
     FM_Write(OpCellNum, Instrument.WaveSelect[2]);

     OpCellNum:= $C0 + Voice;
     FM_Write(OpCellNum, Instrument.Feedback);
end;

{******}

procedure LoadSBI(FileName : string; var Instrument : FMInstrument);

var f   : file;
    ID  : array[1..3] of char;
    Hdr : array[1..33] of byte;
    i   : byte;
    NR  : word;

begin
     Assign(f, FileName);
     {$I-}
     Reset(f, 1);
     {$I+}
     if (IOResult <> 0) then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_INVALIDSBI;
          exit;
     end;

     BlockRead(f, ID, sizeof(ID), NR);
     if (NR <> sizeof(ID)) or (ID <> 'SBI') then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_INVALIDSBI;
          exit;
     end;

     BlockRead(f, Hdr, sizeof(Hdr), NR);
     if NR <> sizeof(Hdr) then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_INVALIDSBI;
          exit;
     end;

     BlockRead(f, Instrument, sizeof(FMInstrument), NR);
     if NR <> sizeof(FMInstrument) then
     begin
          SErrorCode:= (word(ERR_ADL) shl 8) + ERR_ADL_INVALIDSBI;
          exit;
     end;

     Close(f);
end;

{******}

begin
     AdlibPresent:= false;
     OldExit:= ExitProc;
     ExitProc:= @FM_Exit;
end.