unit SBDSP;

(* Information
   

   Program Title : SoundBlaster digital sound for S-Library.
   External name : SBDSP.TPU
   Version       : 1.0
   Start date    : 27/8/96
   Last update   : 6/9/96
   Author        : Rob Anderton.
   Description   : Sound Blaster DSP, digital sound support.

*)

interface

uses OBJECTS;

{******}

type SBInfoRec = record
                       DSPAddr    : word;
                       MixerAddr  : word;
                       MPUAddr    : word;
                       DMA8       : word;
                       DMA16      : word;
                       IRQ        : word;
                       DSPVersion : word;
                       DSPName    : string;
                 end;

     DSPUserIRQ  = procedure(l : longint);
     DSP4UserIRQ = procedure(b : byte);

     PlayRec     = record
                         Stereo : boolean;
                         Bits   : word;
                         Freq   : word;
                   end;

     PSoundHdr = ^TSoundHdr;
     TSoundHdr = record
                       SoundID : array[1..3] of char;
                       Stereo  : boolean;
                       Bits    : integer;
                       Freq    : word;
                       Size    : longint;
                 end;

{******}

{*** SoundBlaster cards ***}

const SBCards : array[0..5] of string = ('Sound Blaster compatible',
                                         'Sound Blaster 1.5 or MCV',
                                         'Sound Blaster 2.0',
                                         'Sound Blaster Pro',
                                         'Sound Blaster 16/ASP',
                                         'Sound Blaster AWE32');

{*** DSP versions ***}

const DSP_1XX = $100;
      DSP_200 = $200;
      DSP_201 = $210;
      DSP_3XX = $300;
      DSP_4XX = $400;

{*** SB_ versions ***}

const SB_NONE = 0;
      SB_1XX  = 1;
      SB_2XX  = 2;
      SB_3XX  = 3;
      SB_4XX  = 4;
      SB_XXX  = 5;

{*** DSP registers ***}

const DSP_WRITESTATUS = $0C;
      DSP_WRITEDATA   = $0C;

      DSP_READSTATUS  = $0E;
      DSP_IRQACK      = $0E;
      DSP_READDATA    = $0A;

      DSP_RESET       = $06;
      DSP_GETVERSION  = $E1;

      DSP_DIRECTDAC8  = $10;
      DSP_DMADAC8     = $14;
      DSP_DMAAUTODAC8 = $1C;

      DSP_DIRECTADC8  = $20;
      DSP_DMAADC8     = $24;
      DSP_DMAAUTOADC8 = $2C;

      DSP_SAMPLEFREQ   = $40;
      DSP_TRANSFERSIZE = $48;

      DSP_PAUSEDMA     = $D0;
      DSP_RESUMEDMA    = $D4;

      DSP_SPEAKERON    = $D1;
      DSP_SPEAKEROFF   = $D3;
      DSP_SPEAKERSTAT  = $D8;

      DSP2p_8DMAHIAUTODAC = $90;
      DSP2p_8DMAHIDAC     = $91;

      DSP2p_8DMAHIAUTOADC = $98;
      DSP2p_8DMAHIADC     = $99;

      DSP3_MONOADC        = $A0;
      DSP3_STEREOADC      = $A8;

      DSP4_DACFR          = $41;
      DSP4_ADCFR          = $42;

      DSP4_CMDADC         = $08;
      DSP4_CMDAUTOINIT    = $04;
      DSP4_CMDFIFO        = $02;
      DSP4_CMDDAC         = $00;
      DSP4_CMD8DMA        = $C0;
      DSP4_CMD16DMA       = $B0;

      DSP4_MODESTEREO     =  $20;
      DSP4_MODEMONO       =  $00;
      DSP4_MODESIGNED     =  $10;
      DSP4_MODEUNSIGNED   =  $00;

      DSP4_EXITDMA16      = $D9;
      DSP4_EXITDMA8       = $DA;
      DSP2p_EXITAUTOINIT   = $DA;

      DSP_MONOADC         = $A0;
      DSP_STEREOADC       = $A8;

      DSP_DACFREQ         = $41;
      DSP_ADCFREQ         = $42;

      {*** Maximum playback frequencies ***}

      FRQ1_MONO_DAC    =  23000;
      FRQ2p_MONO_DAC   =  23000;
      FRQ2p_HIMONO_DAC =  44100;
      FRQ3_MONO_DAC    =  23000;
      FRQ3_HIMONO_DAC  =  44100;
      FRQ3_STEREO_DAC  =  22050;
      FRQ4_DAC         =  44100;

      {*** maximum sampling frequencies ***}
      FRQ1_MONO_ADC    =  13000;
      FRQ2p_MONO_ADC   =  13000;
      FRQ2p_HIMONO_ADC =  15000;
      FRQ3_MONO_ADC    =  23000;
      FRQ3_HIMONO_ADC  =  44100;
      FRQ3_STEREO_ADC  =  22050;
      FRQ4_ADC         =  44100;

      {*** Sound file header ID ***}

      DSP_SOUNDID      : array[1..3] of char = 'SWF';

{******}

var SBInfo    : SBInfoRec;
    SBPresent : boolean;

{******}

procedure SB_Init;
procedure SB_GetBlaster(var SB : SBInfoRec);

function  SB_Reset : boolean;

procedure SB_Write(Val : byte);
procedure SB_Read(var Val : byte);

function  SB_Stereo  : boolean;           {DSP 3.00+}
function  SB_MaxBits : byte;

procedure SB_SetSpeaker(SpeakerOn : boolean);
function  SB_GetSpeakerStatus : boolean;    {DSP 2.00+}

function  SB_AdjustFreq(var Freq : word; var Stereo : boolean) : boolean;
procedure SB_SetOutputFreq(Freq : word);  {DSP 4.00+}
procedure SB_SetInputFreq(Freq : word);   {DSP 4.00+}
procedure SB_SetFreq(var Info : PlayRec);
function  SB_HiMonoDACFreq(Freq : word) : boolean;
function  SB_HiMonoADCFreq(Freq : word) : boolean;

procedure SB_SetDMATransferSize(Size : word);

procedure SB_WriteDAC8(Val : byte);
function  SB_ReadADC8 : byte;

procedure SB_IRQHandler; interrupt;
procedure SB_InitIRQHandler;
procedure SB_RestoreIRQHandler;
procedure SB_SetUserIRQ(PFunc : DSPUserIRQ);
procedure SB_SetDSP4UserIRQ(PFunc : DSP4UserIRQ);

function  SB_WaitForIRQ(PFunc : DSPUserIRQ; Param : longint) : boolean;
procedure SB_InitWaitForIRQ;

procedure SB_Play(var Handle : integer; var Info : PlayRec; Buffer : pointer;
                      Size : word; UserProc : DSPUserIRQ);
procedure SB_Record(var Handle : integer; var Info : PlayRec; Buffer : pointer;
                        Size, Secs, Source : word; UserProc : DSPUserIRQ);

{******}

implementation

uses CRT, DOS, SBMIXER, SIRQ, SDMA, SUTILS, SERROR;

{******}

var DSPUserFunc  : DSPUserIRQ;
    DSP4UserFunc : DSP4UserIRQ;
    OldInt       : procedure;
    OldExit      : pointer;
    NewIRQ       : boolean;

const IRQServed    : longint = 0;
      OldIRQServed : longint = 0;
      DMABufCnt    : integer = 0;

{******}

procedure SB_Exit; far;

begin
     ExitProc:= OldExit;

     if SBPresent then
     begin
          if NewIRQ then SB_RestoreIRQHandler;
     end;
end;

{******}

procedure SB_Init;

var Version : array[1..2] of byte;

begin
     SB_GetBlaster(SBInfo);
     if SErrorCode <> 0 then exit;

     if not SB_Reset then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_RESETERROR;
          SBPresent:= false;
          exit;
     end;

     SBPresent:= true;

     SB_Write(DSP_GETVERSION);
     if SErrorCode <> 0 then exit;
     SB_Read(Version[1]);
     if SErrorCode <> 0 then exit;
     SB_Read(Version[2]);
     if SErrorCode <> 0 then exit;

     SBInfo.DSPVersion:= (word(Version[1]) shl 8) + Version[2];
     if (Version[1] >= 1) and
        (Version[1] <= 4) then SBInfo.DSPName:= SBCards[Version[1]]
                          else SBInfo.DSPName:= SBCards[SB_NONE];

     SErrorCode:= 0;
end;

{******}

procedure SB_GetBlaster(var SB : SBInfoRec);

var EnvStr : string;
    Temp   : string;
    Index  : byte;
    Space  : byte;
    Code   : word;

begin
     with SB do
     begin
          EnvStr:= GetEnv('BLASTER');
          if EnvStr = '' then
          begin
               SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOBLASTERENV;
               exit;
          end;

          Index:= Pos('A', EnvStr);
          if Index = 0 then
          begin
               SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOBLASTERENV;
               exit;
          end;

          Temp:= Copy(EnvStr, Index + 1, Length(EnvStr) - Index - 1);
          Space:= Pos(' ', Temp);
          Delete(Temp, Space, Length(Temp) - Space + 1);

          DSPAddr:= U_HexToDec(Temp);

          Index:= Pos('M', EnvStr);
          if Index = 0 then
          begin
               MixerAddr:= DSPAddr;
          end
          else
          begin
               Temp:= Copy(EnvStr, Index + 1, Length(EnvStr) - Index - 1);
               Space:= Pos(' ', Temp);
               Delete(Temp, Space, Length(Temp) - Space + 1);

               MixerAddr:= U_HexToDec(Temp);
          end;

          Index:= Pos('P', EnvStr);
          if Index = 0 then
          begin
               MPUAddr:= $330;
          end
          else
          begin
               Temp:= Copy(EnvStr, Index + 1, Length(EnvStr) - Index - 1);
               Space:= Pos(' ', Temp);
               Delete(Temp, Space, Length(Temp) - Space + 1);

               MixerAddr:= U_HexToDec(Temp);
          end;

          Index:= Pos('I', EnvStr);
          if Index = 0 then
          begin
               IRQ:= 65535;
          end
          else
          begin
               Temp:= Copy(EnvStr, Index + 1, Length(EnvStr) - Index - 1);
               Space:= Pos(' ', Temp);
               Delete(Temp, Space, Length(Temp) - Space + 1);

               Val(Temp, IRQ, Code);
          end;

          Index:= Pos('D', EnvStr);
          if Index = 0 then
          begin
               DMA8:= 65535;
          end
          else
          begin
               Temp:= Copy(EnvStr, Index + 1, Length(EnvStr) - Index - 1);
               Space:= Pos(' ', Temp);
               Delete(Temp, Space, Length(Temp) - Space + 1);

               Val(Temp, DMA8, Code);
          end;

          Index:= Pos('H', EnvStr);
          if Index = 0 then
          begin
               DMA16:= 65535;
          end
          else
          begin
               Temp:= Copy(EnvStr, Index + 1, Length(EnvStr) - Index - 1);
               Space:= Pos(' ', Temp);
               Delete(Temp, Space, Length(Temp) - Space + 1);

               Val(Temp, DMA16, Code);
          end;

     end;
     SErrorCode:= 0;
end;

{******}

function SB_Reset : boolean;

var Val, Temp : byte;

begin
     SBPresent:= true;

     Port[SBInfo.DSPAddr + DSP_RESET]:= 1;
     for Val:= 1 to 3 do Temp:= Port[SBInfo.DSPAddr + DSP_RESET];
     Port[SBInfo.DSPAddr + DSP_RESET]:= 0;
     for Val:= 1 to 3 do Temp:= Port[SBInfo.DSPAddr + DSP_RESET];
     Val:= 0;
     SB_Read(Val);
     if Val = $AA then
     begin
          SErrorCode:= 0;
          SB_Reset:= true;
     end
     else
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_RESETERROR;
          SB_Reset:= false;
     end;
end;

{******}

procedure SB_Write(Val : byte);

const TimeOut : word = 5535;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;
     SErrorCode:= 0;

     while (((Port[SBInfo.DSPAddr + DSP_WRITESTATUS] and $80) <> 0) and
             (TimeOut > 0)) do
     begin
          Delay(2);
          Dec(TimeOut);
     end;

     if TimeOut > 0 then Port[SBInfo.DSPAddr + DSP_WRITEDATA]:= Val
                    else SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_TIMEOUT;
end;

{******}

procedure SB_Read(var Val : byte);

const TimeOut : word = 65535;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;
     SErrorCode:= 0;

     while (not (((Port[SBInfo.DSPAddr + DSP_READSTATUS] and $80) <> 0) and
                  (TimeOut <> 0))) do Dec(TimeOut);

     if TimeOut > 0 then Val:= byte(Port[SBInfo.DSPAddr + DSP_READDATA])
                    else
                    begin
                         SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_TIMEOUT;
                         Val:= 0;
                    end;
end;

{******}

function SB_Stereo  : boolean;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;
     SErrorCode:= 0;

     SB_Stereo:= (SBInfo.DSPVersion >= DSP_3XX);
end;

{******}

function SB_MaxBits : byte;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;
     SErrorCode:= 0;

     if SBInfo.DSPVersion >= DSP_4XX then SB_MaxBits:= 16
                                     else SB_MaxBits:= 8;
end;

{******}

procedure SB_SetSpeaker(SpeakerOn : boolean);

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;
     SErrorCode:= 0;

     if SpeakerOn then SB_Write(DSP_SPEAKERON)
                  else SB_Write(DSP_SPEAKEROFF);

end;

{******}

function SB_GetSpeakerStatus : boolean;

var Temp : byte;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;

     if SBInfo.DSPVersion = DSP_1XX then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_OLDDSP;
          exit;
     end;

     SErrorCode:= 0;

     SB_Write(DSP_SPEAKERSTAT);
     if SErrorCode <> 0 then exit;
     SB_Read(Temp);
     if SErrorCode <> 0 then exit;
     if Temp = 0 then SB_GetSpeakerStatus:= false
                 else SB_GetSpeakerStatus:= true;
end;

{******}

function SB_AdjustFreq(var Freq : word; var Stereo : boolean) : boolean;

begin
     if SBInfo.DSPVersion >= DSP_4XX then
     begin
          if Freq > FRQ4_DAC then Freq:= FRQ4_DAC;
          SB_AdjustFreq:= true;
          exit;
     end;

     if SBInfo.DspVersion >= DSP_3XX then
     begin
          if Stereo then
             if Freq > FRQ3_STEREO_DAC then Freq:= FRQ3_STEREO_DAC
                    else
             if Freq > FRQ3_HIMONO_DAC then Freq:= FRQ3_HIMONO_DAC;

          SB_AdjustFreq:= true;
          exit;
     end;

     if SBInfo.DSPVersion >= DSP_201 then
     begin
          if Freq > FRQ2p_HIMONO_DAC then Freq:= FRQ2p_HIMONO_DAC;
          if Stereo then
          begin
               Stereo:= false;
               SB_AdjustFreq:= false;
               exit;
          end;
          SB_AdjustFreq:= true;
          exit;
     end;

     if SBInfo.DSPVersion >= DSP_1XX then
     begin
          if Freq > FRQ1_MONO_DAC then Freq:= FRQ1_MONO_DAC;
          if Stereo then
          begin
               Stereo:= false;
               SB_AdjustFreq:= false;
               exit;
          end;
          SB_AdjustFreq:= true;
          exit;
     end;

     SB_AdjustFreq:= false;
end;

{******}

procedure SB_SetOutputFreq(Freq : word);

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;

     if SBInfo.DSPVersion < DSP_4XX then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_OLDDSP;
          exit;
     end;
     SErrorCode:= 0;

     SB_Write(DSP_DACFREQ);
     if SErrorCode <> 0 then exit;
     SB_Write(Hi(Freq));
     if SErrorCode <> 0 then exit;
     SB_Write(Lo(Freq));
     if SErrorCode <> 0 then exit;

     SErrorCode:= 0;
end;

{******}

procedure SB_SetInputFreq(Freq : word);

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;

     if SBInfo.DSPVersion < DSP_4XX then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_OLDDSP;
          exit;
     end;

     SErrorCode:= 0;

     SB_Write(DSP_ADCFREQ);
     if SErrorCode <> 0 then exit;
     SB_Write(Hi(Freq));
     if SErrorCode <> 0 then exit;
     SB_Write(Lo(Freq));
     if SErrorCode <> 0 then exit;

     SErrorCode:= 0;
end;

{******}

procedure SB_SetFreq(var Info : PlayRec);

var TC : byte;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;
     SErrorCode:= 0;

     case Info.Bits of
          8 : begin
                  case Info.Stereo of
                           false : TC:= 256 - (1000000 div (Info.Freq));
                           true  : TC:= 256 - (1000000 div  (2 * Info.Freq));
                  end;
                  Info.Freq:= Word(1000000 div (256 - TC));
              end;

         16 : begin
                  case Info.Stereo of
                           false : TC:= 65536 - (256000000 div (Info.Freq));
                           true  : TC:= 65536 - (256000000 div  (2 * Info.Freq));
                  end;
                  Info.Freq:= Word(256000000 div (65536 - TC));
              end;
     end;
     SB_Write(DSP_SAMPLEFREQ);
     SB_Write(TC);
end;

{******}

function SB_HiMonoDACFreq(Freq : word) : boolean;

begin
     if SBInfo.DSPVersion <= DSP_1XX then SB_HiMonoDACFreq:= false
     else
         if SBInfo.DSPVersion <= DSP_201 then
               SB_HiMonoDACFreq:= (Freq <= FRQ2p_HIMONO_DAC)
         else
             if SBInfo.DSPVersion <= DSP_3XX then
                  SB_HiMonoDACFreq:= (Freq <= FRQ3_HIMONO_DAC)
             else
                 if SBInfo.DSPVersion <= DSP_4XX then
                      SB_HiMonoDACFreq:= (Freq <= FRQ4_DAC)
                 else SB_HiMonoDACFreq:= TRUE;
end;

{******}

function SB_HiMonoADCFreq(Freq : word) : boolean;

begin
     if SBInfo.DSPVersion <= DSP_1XX
        then SB_HiMonoADCFreq:= false
        else if SBInfo.DSPVersion <= DSP_201
                then SB_HiMonoADCFreq:= (Freq <= FRQ2p_HIMONO_ADC)
                else if SBInfo.DSPVersion <= DSP_3XX
                     then SB_HiMonoADCFreq:= (Freq <= FRQ3_HIMONO_ADC)
                     else if SBInfo.DspVersion <= DSP_4XX
                     then SB_HiMonoADCFreq:= (Freq <= FRQ4_ADC)
                     else SB_HiMonoADCFreq:= true;
End;

{******}

procedure SB_SetDMATransferSize(Size : word);

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;

     if SBInfo.DSPVersion = DSP_1XX then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_OLDDSP;
          exit;
     end;

     if Size = 0 then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_INVALIDSIZE;
          exit;
     end;

     SErrorCode:= 0;

     Dec(Size);
     SB_Write(DSP_TRANSFERSIZE);
     if SErrorCode <> 0 then exit;
     SB_Write(Lo(Size));
     if SErrorCode <> 0 then exit;
     SB_Write(Hi(Size));
     if SErrorCode <> 0 then exit;
end;

{******}

procedure SB_WriteDAC8(Val : byte);

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;
     SErrorCode:= 0;

     SB_Write(DSP_DIRECTDAC8);
     if SErrorCode <> 0 then exit;
     SB_Write(Val);
     if SErrorCode <> 0 then exit;
end;

{******}

function SB_ReadADC8 : byte;

var Temp : byte;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_SB) shl 8) + ERR_SB_NOINIT;
          exit;
     end;

     SErrorCode:= 0;

     SB_Write(DSP_DIRECTADC8);
     if SErrorCode <> 0 then exit;
     SB_Read(Temp);
     if SErrorCode <> 0 then exit;
     SB_ReadADC8:= Temp;
end;

{******}

procedure SB_IRQHandler;

var who, i : byte;

begin
     Inc(IRQServed);
     if SBInfo.DSPVersion >= DSP_4XX then
     begin
          who:= byte(Mix_Read(MIX4_IRQSOURCE));

          if(@DSP4UserFunc <> nil) then DSP4UserFunc(who);

          {*** 8 bit DMA or Midi transfer ***}
          if (who and MIX4_IRQ8DMA) <> 0 then i:= Port[SBInfo.DSPAddr + $0E];

          {*** 16 bit DMA ***}
          if (who and MIX4_IRQ16DMA) <> 0 then i:= Port[SBInfo.DSPAddr + $0F];

          {** MPU-401 UART ***}
          if (who and MIX4_IRQMPU) <> 0 then i:= Port[SBInfo.MPUAddr];
     end
     else i:= Port[SBInfo.DSPAddr + $0E];

     if @DSPUserFunc <> nil then DSPUserFunc(IRQServed);

     IRQ_SendEOI(SBInfo.IRQ);
end;

{******}

procedure SB_InitIRQHandler;

begin
     @OldInt:= IRQ_SetHandler(SBInfo.IRQ, @SB_IRQHandler);
     NewIRQ:= true;
end;

{******}

procedure SB_RestoreIRQHandler;

begin
     IRQ_SetHandler(SBInfo.IRQ, @OldInt);
     NewIRQ:= false;
end;

{******}

procedure SB_SetUserIRQ(PFunc : DSPUserIRQ);

begin
     DSPUserFunc:= PFunc;
end;

{******}

procedure SB_SetDSP4UserIRQ(PFunc : DSP4UserIRQ);

begin
     DSP4UserFunc:= PFunc;
end;

{******}

function  SB_WaitForIRQ(PFunc : DSPUserIRQ; Param : longint) : boolean;

var iRet : boolean;

begin
     iRet:= ((IRQServed - OldIRQServed) > 1);
     OldIRQServed:= IRQServed;

     while OldIRQServed = IRQServed do
           if @PFunc <> nil then PFunc(Param);

     SB_WaitForIRQ:= iRet;
end;

{******}

procedure SB_InitWaitForIRQ;

begin
     OldIRQServed:= IRQServed;
end;

{******}

procedure SB_InitBuffers;

begin
     DMABufCnt := 0;
end;

{******}

function SB_ReadBuffer(var Handle : integer; Buffer : pointer; HalfSize : integer) : word;

var BufPtr : ^byte;

begin
     BufPtr:= Buffer;
     if (DMABufCnt and 1) <> 0 then Inc(BufPtr, HalfSize);
     BufPtr^:= 255;
     SB_ReadBuffer:= U_dos_read(Handle, BufPtr, HalfSize);
     Inc(DMABufCnt);
end;

{******}

function SB_WriteBuffer(var Handle : integer; Buffer : pointer; HalfSize : integer) : word;

var BufPtr : ^byte;

begin
     BufPtr:= Buffer;

     if (DMABufCnt and 1) <> 0 then Inc(BufPtr, HalfSize);
     SB_WriteBuffer:= U_dos_write(Handle, BufPtr, HalfSize);
     Inc(DMABufCnt);
end;

{******}

procedure SB_ClearBuffer(Buffer : pointer; Size : word);

type BufArr = array[0..65534] of byte;

var i      : word;
    l      : word;
    BufPtr : ^BufArr;

begin
     BufPtr:= Buffer;
     for l:= 0 to Size - 1 do BufPtr^[l]:= 0;
end;

{******}

procedure SB_Play(var Handle : integer; var Info : PlayRec; Buffer : pointer;
                     Size : word; UserProc : DSPUserIRQ);

type BufArr = array[0..65534] of byte;

const DSPMode : byte = 0;
      Divisor : word = 2;

var Samples   : longint;
    Command   : byte;
    Backup    : byte;
    Mode      : byte;
    BufPtr    : ^BufArr;
    MoreSound : boolean;

begin
     BufPtr:= Buffer;
     if Info.Stereo then Info.Stereo:= SB_Stereo;
     if Info.Bits = 16 then Info.Bits:= SB_MaxBits else Info.Bits:= 8;

     SB_AdjustFreq(Info.Freq, Info.Stereo);
     SB_InitIRQHandler;
     SB_InitBuffers;
     SB_ClearBuffer(Buffer, Size);

     Samples:= U_dos_FileSize(Handle);
     SB_ReadBuffer(Handle, Buffer, Size div 2);
     SB_ReadBuffer(Handle, Buffer, Size div 2);

     if SBInfo.DSPVersion < DSP_4XX then
     begin
          SB_SetSpeaker(true);
          if Info.Stereo then Mix3_PrepareForStereo(DAC);
          SB_Write(DSP3_MONOADC);

          if Info.Stereo then
          begin
               Backup:= BufPtr^[0];
               BufPtr^[0]:= $80;
               DMA_SetChannel(SBInfo.DMA8, @Buffer, 1, MODE_SINGLE or
                              MODE_READ);
               SB_Write(DSP_DMADAC8);
               SB_Write(0);
               SB_Write(0);
               SB_WaitForIRQ(UserProc, 0);

               BufPtr^[0]:= Backup;
               Command:= DSP2p_8DMAHIAUTODAC;
          end
          else
          begin
               Command:= byte(DSP_DMAAUTODAC8);
          end;

          DMA_SetChannel(SBInfo.DMA8, Buffer, Size,
                         byte(MODE_SINGLE or MODE_AUTOINIT or MODE_READ));

          SB_SetFreq(Info);
          SB_SetDMATransferSize(Size div 2);
          SB_Write(Command);
     end
     else
     begin
          Mix4_SetVolume(VOICE, 255, 255);
          Mode:= byte(MODE_SINGLE or MODE_AUTOINIT or MODE_READ);

          if Info.Bits = 16 then DMA_SetChannel(SBInfo.DMA16, Buffer, Size, Mode)
                            else DMA_SetChannel(SBInfo.DMA8, Buffer, Size, Mode);

          SB_SetOutputFreq(Info.Freq);
          Command:= DSP4_CMDDAC or DSP4_CMDAUTOINIT or DSP4_CMDFIFO;

          if Info.Bits = 16 then Command:= Command or DSP4_CMD16DMA
                            else Command:= Command or DSP4_CMD8DMA;

          if Info.Stereo then DSPMode:= DSP4_MODESTEREO
                         else DSPMode:= DSP4_MODEMONO;

          DSPMode:= DSPMode or DSP4_MODESIGNED;

          SB_Write(Command);
          SB_Write(DSPMode);

          if Info.Bits = 16 then Divisor:= Divisor * 2;
          SB_Write(Lo((Size div Divisor) - 1));
          SB_Write(Hi((Size div Divisor) - 1));
     end;

     SB_InitWaitForIRQ;

     repeat
           SB_WaitForIRQ(UserProc, 0);
           MoreSound:= SB_ReadBuffer(Handle, Buffer, Size div 2) <> 0;
     until not MoreSound;

     SB_ClearBuffer(Buffer, Size);

     if (Command = DSP2p_8DMAHIAUTODAC) then
     begin
          SB_RestoreIRQHandler;
          SB_Reset;
     end
     else
     begin
          if (not Info.Stereo and (SBInfo.DSPVersion >= DSP_4XX)) then
                  Mix4_PrepareForMonoADC;

          if ((Command and $F0) = DSP4_CMD8DMA) then
                  SB_Write(DSP4_EXITDMA8)
          else
              if ((Command and $F0) = DSP4_CMD16DMA) then
                      SB_Write(DSP4_EXITDMA16)
              else
              begin
                   if (Info.Stereo and (SBInfo.DSPVersion <= DSP_3XX)) then
                           Mix3_RestoreFromStereo;
                   SB_SetSpeaker(false);
                   SB_Write(DSP2p_EXITAUTOINIT);
              end;
          SB_WaitForIRQ(UserProc, 0);
          SB_RestoreIRQHandler;
     end;
end;

{******}

procedure SB_Record(var Handle : integer; var Info : PlayRec; Buffer : pointer;
                        Size, Secs, Source : word; UserProc : DSPUserIRQ);

type BufArr = array[0..65534] of byte;

const DSPMode : byte = 0;
      Divisor : word = 2;

var Samples   : longint;
    Command   : byte;
    Backup    : byte;
    Mode      : byte;
    BufPtr    : ^BufArr;
    MoreSound : boolean;

begin
     BufPtr:= Buffer;
     if Info.Stereo then Info.Stereo:= SB_Stereo;
     if Info.Bits = 16 then Info.Bits:= SB_MaxBits else Info.Bits:= 8;

     SB_AdjustFreq(Info.Freq, Info.Stereo);
     SB_InitIRQHandler;
     SB_InitBuffers;
     SB_ClearBuffer(Buffer, Size);

     Samples:= Info.Freq * Secs;

     if SBInfo.DSPVersion < DSP_4XX then
     begin
          if Info.Stereo then Mix3_PrepareForStereo(ADC);
          Mix3_SetADCSource(Source);
          SB_SetSpeaker(false);

          if SBInfo.DSPVersion >= DSP_3XX then
             if Info.Stereo then SB_Write(DSP3_STEREOADC)
                            else SB_Write(DSP3_MONOADC);

          Command:= DSP_DMAAUTOADC8;

          if SBInfo.DSPVersion >= DSP_201 then
             if (Info.Stereo) or (SB_HiMonoADCFreq(Info.Freq)) then Command:= DSP2p_8DMAHIAUTOADC;


          DMA_SetChannel(SBInfo.DMA8, Buffer, Size,
                         byte(MODE_SINGLE or MODE_AUTOINIT or MODE_WRITE));

          SB_SetFreq(Info);
          SB_SetDMATransferSize(Size div 2);
          SB_Write(Command);
     end
     else
     begin
          case Source of

               CD : begin
                         Mix4_SetADCSourceL(CD_L, true);
                         Mix4_SetADCSourceR(CD_R, true);
                    end;

               LINE : begin
                          Mix4_SetADCSourceL(LINE_L, true);
                          Mix4_SetADCSourceR(LINE_R, true);
                      end;

               MIC : begin
                         Mix4_SetADCSourceL(MIC, true);
                         Mix4_SetADCSourceR(MIC, true);
                     end;
          end;

          if not Info.Stereo then Mix4_PrepareForMonoADC;

          Mode:= byte(MODE_SINGLE or MODE_AUTOINIT or MODE_WRITE);

          if Info.Bits = 16 then DMA_SetChannel(SBInfo.DMA16, Buffer, Size, Mode)
                            else DMA_SetChannel(SBInfo.DMA8, Buffer, Size, Mode);

          SB_SetOutputFreq(Info.Freq);
          Command:= DSP4_CMDADC or DSP4_CMDAUTOINIT or DSP4_CMDFIFO;

          if Info.Bits = 16 then Command:= Command or DSP4_CMD16DMA
                            else Command:= Command or DSP4_CMD8DMA;

          if Info.Stereo then DSPMode:= DSP4_MODESTEREO
                         else DSPMode:= DSP4_MODEMONO;

          DSPMode:= DSPMode or DSP4_MODESIGNED;

          SB_Write(Command);
          SB_Write(DSPMode);

          if Info.Bits = 16 then Divisor:= Divisor * 2;
          SB_Write(Lo((Size div Divisor) - 1));
          SB_Write(Hi((Size div Divisor) - 1));
     end;

     SB_InitWaitForIRQ;

     repeat
           SB_WaitForIRQ(UserProc, 0);
           MoreSound:= false;
           if SB_WriteBuffer(Handle, Buffer, Size div 2) <> 0 then
              if Samples > 0 then
              begin
                   MoreSound:= true;
                   Samples:= Samples - Size div 2;
              end;
     until not MoreSound;

     SB_ClearBuffer(Buffer, Size);

     if (Command = DSP2p_8DMAHIAUTODAC) then
     begin
          SB_RestoreIRQHandler;
          SB_Reset;
     end
     else
     begin
          if (not Info.Stereo and (SBInfo.DSPVersion >= DSP_4XX)) then
                  Mix4_PrepareForMonoADC;

          if ((Command and $F0) = DSP4_CMD8DMA) then
                  SB_Write(DSP4_EXITDMA8)
          else
              if ((Command and $F0) = DSP4_CMD16DMA) then
                      SB_Write(DSP4_EXITDMA16)
              else
              begin
                   if (Info.Stereo and (SBInfo.DSPVersion <= DSP_3XX)) then
                           Mix3_RestoreFromStereo;
                   SB_SetSpeaker(false);
                   SB_Write(DSP2p_EXITAUTOINIT);
              end;
          SB_WaitForIRQ(UserProc, 0);
          SB_RestoreIRQHandler;
     end;
end;


{******}

begin
     SBPresent:= false;
     SErrorCode:= 0;
     DSPUserFunc:= nil;
     DSP4UserFunc:= nil;
     NewIRQ:= false;
     OldExit:= ExitProc;
     ExitProc:= @SB_Exit;
end.