unit SBMixer;

(* Information
   

   Program Title : SoundBlaster digital sound for S-Library.
   External name : SBMIXER.TPU
   Version       : 1.0
   Start date    : 5/9/96
   Last update   : 5/9/96
   Author        : Rob Anderton.
   Description   : Sound Blaster Mixer control.

                   Mix_XXXXX  = routines for all mixer versions.
                   Mix2_XXXXX = routines for CT1335 (DSP 2.00+) mixer.
                   Mix3_XXXXX = routines for CT1345 (DSP 3.00+) mixer.
                   Mix4_XXXXX = routines for CT1745 (DSP 4.00+) mixer.

                   Unbelievably closely based on MIXUTIL.PAS by
                   Michael Tischer and Bruno Jennrich.

*)

interface

{******}

const ERROR    = $ffff;
      NO_ERROR = 0;

      {*** Mixer port offsets ***}
      MIX_REGISTERPORT  =  $04;      {Setting register}
      MIX_DATAPORT      =  $05;      {Read/write data}

      {*** CT1335 mixer (DSP 2.00+) ***}
      MIX_RSET          =  $00;      {Reset mixer}
      MIX2_MASTERVOL    =  $02;      {Total-Volume (BITS 3,2,1)}
      MIX2_MIDIVOL      =  $06;      {MIDI(FM) Volume (BITS 3,2,1)}
      MIX2_CDVOL        =  $08;      {CD Volume (BITS 3,2,1)}
      MIX2_VOICEVOL     =  $0A;      {VOICE(DSP) Volume (BITS 2,1)}

      {*** CT1345 mixer (DSP 3.00+) ***}
      MIX3_VOICEVOL     =   $04;     {VOICE-Vol. (Bits 7,6,5  3,2,1)}
      MIX3_MICVOL       =   $0A;     {Microphone Volume (BITS 2,1)}

      MIX3_ADCSTATE     =   $0C;     {DSP input and filter}

      {*** Input filter ***}
      MIX3_ADCFILTEROFF =   $20;     {Bit 5: Low pass filter on}
      MIX3_LOWPASS88    =   $08;     {Bit 3: 8.8 kHz Low pass filter}

      {*** Input source ***}
      MIX3_MICSRC       =   $00;     {BIT 2,1 = 00: Microphone}
      MIX3_MICSRC_      =   $04;     {BIT 2,1 = 10: also Microphone}
      MIX3_CDSRC        =   $02;     {BIT 2,1 = 01: CD}
      MIX3_LINESRC      =   $06;     {BIT 2,1 = 11: LINE IN}
      MIX3_SRCMSK       =   $06;     {Bit mask of source bits}

      MIX3_DACSTATE     = $0E;       {Output filter and stereo switch}
      MIX3_DACFILTEROFF = $20;       {Bit 5: Low-Pass Filter}
      MIX3_STEREOON     = $02;       {Bit 1: Stereo output}

      MIX3_MASTERVOL    =  $22;      {Totalvol. (BITS 7,6,5 and 3,2,1)}
      MIX3_MIDIVOL      =  $26;      {MIDI-Vol. (BITS 7,6,5 and 3,2,1)}
      MIX3_CDVOL        =  $28;      {CD-Vol (BITS 7,6,5 and 3,2,1)}
      MIX3_LINEVOL      =  $2E;      {LINE-Vol. (BITS 7,6,5 and 3,2,1)}

      {*** CT1745 Mixer (DSP 4.00+ and ASP) ***}

      {*** Volumes ***}
      MIX4_MASTERVOL_L  = $30;       {Total left (BITS 7,6,5,4,3)}
      MIX4_MASTERVOL_R  = $31;       {Total right (BITS 7,6,5,4,3)}
      MIX4_VOICEVOL_L   = $32;       {VOICE(DSP) left (BITS 7,6,5,4,3)}
      MIX4_VOICEVOL_R   = $33;       {VOICE(DSP) right (BITS 7,6,5,4,3)}
      MIX4_MIDIVOL_L    = $34;       {MIDI left (BITS 7,6,5,4,3)}
      MIX4_MIDIVOL_R    = $35;       {MIDI right (BITS 7,6,5,4,3)}
      MIX4_CDVOL_L      = $36;       {CD left (BITS 7,6,5,4,3)}
      MIX4_CDVOL_R      = $37;       {CD right (BITS 7,6,5,4,3)}
      MIX4_LINEVOL_L    = $38;       {LINE left (BITS 7,6,5,4,3)}
      MIX4_LINEVOL_R    = $39;       {LINE right (BITS 7,6,5,4,3)}
      MIX4_MICVOL       = $3A;       {Microphone (BITS 7,6,5,4,3)}
      MIX4_PCSPEAKERVOL = $3B;       {PC-Speaker (BITS 7,6 )}

      MIX4_OUTSOURCE    = $3C;       {Output sources}
      MIX4_ADCSOURCE_L  = $3D;       {Sample sources L/R}
      MIX4_ADCSOURCE_R  = $3E;

      MIX4_MIDI_L       = $40;       {Active recording sources}
      MIX4_MIDI_R       = $20;
      MIX4_LINE_L       = $10;
      MIX4_LINE_R       = $08;
      MIX4_CD_L         = $04;
      MIX4_CD_R         = $02;
      MIX4_MIC          = $01;

      {*** Preamplifier for output(OUT) and sampling(ADC) (Bits 7 and 6 ) ***}
      MIX4_ADCGAIN_L    = $3F;
      MIX4_ADCGAIN_R    = $40;
      MIX4_OUTGAIN_L    = $41;
      MIX4_OUTGAIN_R    = $42;

      MIX4_AGC          = $43;       {Microphone preamplifier (20dB)}
      MIX4_AGCON        = $01;       {Microphone preamplifier on}

      {*** Treble and bass of preamplifier (BITS 7,6,5,4) ***}
      MIX4_TREBLE_L  =     $44;
      MIX4_TREBLE_R  =     $45;
      MIX4_BASS_L    =     $46;
      MIX4_BASS_R    =     $47;

      {*** Which interrupts and DMA lines are being used? ***}
      MIX4_IRQ       = $80;
      MIX4_IRQ2      = $01;          {4 possible interrupt lines}
      MIX4_IRQ5      = $02;
      MIX4_IRQ7      = $04;
      MIX4_IRQ10     = $08;

      {*** Which DMA line is being used? ***}
      MIX4_DMA       = $81;
      MIX4_DMA0      = $01;
      MIX4_DMA1      = $02;
      MIX4_DMA3      = $08;
      MIX4_DMA5      = $20;
      MIX4_DMA6      = $40;
      MIX4_DMA7      = $80;

      MIX4_IRQSOURCE = $82;
      MIX4_IRQ8DMA   = $01;          {Interupt of 8 bit DMA and Midi}
      MIX4_IRQ16DMA  = $02;          {Interrupt of 16 bit DMA}
      MIX4_IRQMPU    = $04;          {Interrupt of MPU}


(*** The unit uses constants for accessing volume arrays. These constants
     specify whether a port address refers to the right channel, left channel
     or both channels.                                                        *)

     SBPORT  = 0;       { For array access: 1st element = port         }
     CHANNEL = 1;       {                   2nd element = access code  }
     L = 0;             { left                                         }
     R = 1;             { right                                        }

     CH_LEFT  = 1;      { Code for left channel                        }
     CH_RIGHT = 2;      { right channel                                }
     CH_BOTH  = 3;      { both channels                                }
     CH_NONE  = 0;      { no channel                                   }
     CH_MAX   = 6;      { use maximum of right and left                }


     {*** Number available sources consecutively ***}
     CD          = 0;
     LINE        = 1;
     VOICE       = 2;
     MASTER      = 3;
     MIDI        = 4;
     MIC         = 5;
     PCSPEAKER   = 6;
     NUM_SOURCES = 7;

     CD_L        = 7;
     LINE_L      = 8;
     VOICE_L     = 9;
     MASTER_L    = 10;
     MIDI_L      = 11;
     CD_R        = 12;
     LINE_R      = 13;
     VOICE_R     = 14;
     MASTER_R    = 15;
     MIDI_R      = 16;

     MAX_SRC     = 17;

     DAC         = TRUE;    { for 'mix3_PrepareForStereo' }
     ADC         = FALSE;

     {*** Globale Module Variables ***}

var bMix3DACStereo,         { The program uses these variables }
    bMix3DACFilter,         { to save some mixer values.                    }
    bMix3ADCFilter : byte;

    bMix4SourceL,
    bMix4SourceR   : byte;

(*** The following tables are used for setting the volume.
     The program notes the register and volume to be set (left, right, both)
     for each possible source.                                            ***)

const bMix3VolTable : array[0..MAX_SRC - 1,0..1] of byte =
                                     ((MIX3_CDVOL, CH_BOTH),
                                      (MIX3_LINEVOL, CH_BOTH),
                                      (MIX3_VOICEVOL, CH_BOTH),
                                      (MIX3_MASTERVOL, CH_BOTH),
                                      (MIX3_MIDIVOL, CH_BOTH),
                                      (MIX3_MICVOL, CH_MAX),
                                      (0, 0), {Speaker}

                                      (MIX3_CDVOL, CH_LEFT),
                                      (MIX3_LINEVOL, CH_LEFT),
                                      (MIX3_VOICEVOL, CH_LEFT),
                                      (MIX3_MASTERVOL, CH_LEFT),
                                      (MIX3_MIDIVOL, CH_LEFT),

                                      (MIX3_CDVOL, CH_RIGHT),
                                      (MIX3_LINEVOL, CH_RIGHT),
                                      (MIX3_VOICEVOL, CH_RIGHT),
                                      (MIX3_MASTERVOL, CH_RIGHT),
                                      (MIX3_MIDIVOL, CH_RIGHT));

      bMix4VolTable : array[0..MAX_SRC - 1,0..1] of byte =
                                     ((MIX4_CDVOL_L, CH_BOTH),
                                      (MIX4_LINEVOL_L, CH_BOTH),
                                      (MIX4_VOICEVOL_L, CH_BOTH),
                                      (MIX4_MASTERVOL_L, CH_BOTH),
                                      (MIX4_MIDIVOL_L, CH_BOTH),
                                      (MIX4_MICVOL, CH_LEFT),
                                      (MIX4_PCSPEAKERVOL, CH_LEFT),

                                      (MIX4_CDVOL_L, CH_LEFT),
                                      (MIX4_LINEVOL_L, CH_LEFT),
                                      (MIX4_VOICEVOL_L, CH_LEFT),
                                      (MIX4_MASTERVOL_L, CH_LEFT),
                                      (MIX4_MIDIVOL_L, CH_LEFT),

                                      (MIX4_CDVOL_L, CH_RIGHT),
                                      (MIX4_LINEVOL_L, CH_RIGHT),
                                      (MIX4_VOICEVOL_L, CH_RIGHT),
                                      (MIX4_MASTERVOL_L, CH_RIGHT),
                                      (MIX4_MIDIVOL_L, CH_RIGHT));

(*** The CT1745 mixer allows users to enable and disable several
     recording and playback sources. The following array notes
     the appropriate switch bits for each potential source.              ***)

       bMix4SourceBits : array[0..MAX_SRC - 1] of byte =
{CD}                             (MIX4_CD_L or MIX4_CD_R,
{LINE}                            MIX4_LINE_L or MIX4_LINE_R,
{VOICE (not switchable)}          0,
{MASTER (not switchable)}         0,
{MIDI}                            MIX4_MIDI_L or MIX4_MIDI_R,
{MIC}                             MIX4_MIC,
{PCSPEAKER (not switchable)}      0,

{CD_L}                            MIX4_CD_L,
{LINE_L}                          MIX4_LINE_L,
{VOICE_L}                         0,
{MASTER_L}                        0,
{MIDI_L}                          MIX4_MIDI_L,

{CD_R}                            MIX4_CD_R,
{LINE_R}                          MIX4_LINE_R,
{VOICE_R}                         0,
{MASTER_R}                        0,
{MIDI_R}                          MIX4_MIDI_R);

{******}

procedure Mix_Reset;
procedure Mix_Write(iReg, iData : word);
function  Mix_Read(iReg : word) : word;

procedure Mix3_SetADCFilter(iState : boolean);
function  Mix3_GetADCFilter : boolean;

procedure Mix3_SetDACFilter(iState : boolean);
function  Mix3_GetDACFilter : boolean;
procedure Mix3_SetDACStereo(iState : boolean);
function  Mix3_GetDACStereo : boolean;

procedure Mix3_SetADDACLowPass(iState : boolean);
function  Mix3_GetADDACLowPass : boolean;

procedure Mix3_PrepareForStereo(iMode : boolean);
procedure Mix3_RestoreFromStereo;

procedure Mix3_SetVolume(iSource, iVolL, iVolR : word);
function  Mix3_GetVolume(iSource : word) : word;

procedure Mix3_SetADCSource(iSource : word);
function  Mix3_GetADCSource : word;

procedure Mix4_PrepareForMonoADC;
procedure Mix4_RestoreFromMonoADC;

procedure Mix4_SetVolume(iSource, iVolL, iVolR : word);
function  Mix4_GetVolume(iSource : word) : word;

procedure Mix4_SetADCSourceL(iSource : word; iState : boolean);
procedure Mix4_SetADCSourceR(iSource : word; iState : boolean);
function  Mix4_GetADCSourceL(iSource : word) : word;
function  Mix4_GetADCSourceR(iSource : word) : word;

procedure Mix4_SetOUTSource(iSource : word; iState : boolean);
function  Mix4_GetOUTSource(iSource : word) : word;

procedure Mix4_SetADCGain(iGainL, iGainR  : word);
function  Mix4_GetADCGain(iChannel : word) : word;

procedure Mix4_SetOUTGain(iGainL, iGainR  : word);
function  Mix4_GetOUTGain(iChannel : word) : word;

procedure Mix4_SetAGC(iState : word);
function  Mix4_GetAGC : boolean;

procedure Mix4_SetTreble(iTrebleL, iTrebleR : word);
function  Mix4_GetTreble(iChannel : word) : word;

procedure Mix4_SetBass(iBassL, iBassR : word);
function  Mix4_GetBass(iChannel : word) : word;

{******}

implementation

uses SBDSP, SERROR;

{******}

procedure Mix_Reset;

begin
     Mix_Write(MIX_RSET, 0);
end;

{******}

procedure Mix_Write(iReg, iData : word);

begin
     Port[SBInfo.MixerAddr + MIX_REGISTERPORT]:= byte(iReg);
     Port[SBInfo.MixerAddr + MIX_DATAPORT]:= byte(iData);
end;

{******}

function Mix_Read(iReg : word) : word;

begin
     Port[SBInfo.MixerAddr + MIX_REGISTERPORT]:= byte(iReg);
     Mix_Read:= Port[SBInfo.MixerAddr + MIX_DATAPORT];
end;

{******}

procedure Mix3_SetADCFilter(iState : boolean);

var bIn : byte;

begin
     bIn:= byte(Mix_Read(MIX3_ADCSTATE));

     if iState then Mix_Write(MIX3_ADCSTATE, bIn and not MIX3_ADCFILTEROFF)
               else Mix_Write(MIX3_ADCSTATE, bIn or MIX3_ADCFILTEROFF);
end;

{******}

function Mix3_GetADCFilter : boolean;

begin
     Mix3_GetADCFilter:= ((Mix_Read(MIX3_ADCSTATE) and MIX3_ADCFILTEROFF) = 0);
end;

{******}

procedure Mix3_SetDACFilter(iState : boolean);

var Bin : byte;

begin
     bIn:= byte(Mix_Read(Mix3_DACSTATE));

     if iState then Mix_Write(MIX3_DACSTATE, bIn and not MIX3_DACFILTEROFF)
               else Mix_Write(MIX3_DACSTATE, bIn or MIX3_DACFILTEROFF);
end;

{******}

function Mix3_GetDACFilter : boolean;

begin
     Mix3_GetDACFilter:= ((Mix_Read(MIX3_DACSTATE) and MIX3_DACFILTEROFF) = 0);
end;

{******}

procedure Mix3_SetDACStereo(iState : boolean);

var bIn : byte;

begin
     bIn:= byte(Mix_Read(MIX3_DACSTATE));

     if iState then Mix_Write(MIX3_DACSTATE,bIn or MIX3_STEREOON)
               else Mix_Write(MIX3_DACSTATE,bIn and not MIX3_STEREOON);
end;

{******}

function Mix3_GetDACStereo : boolean;

begin
     Mix3_GetDACStereo:= ((Mix_Read(MIX3_DACSTATE) and MIX3_STEREOON) <> 0);
end;

{******}

procedure Mix3_SetADDACLowPass(iState : boolean);

var Bin : byte;

begin
   bIn:= byte (Mix_Read(MIX3_ADCSTATE));
   if iState then Mix_Write(MIX3_ADCSTATE, bIn or MIX3_LOWPASS88)
             else Mix_Write(MIX3_ADCSTATE, bIn and not MIX3_LOWPASS88);
end;

{******}

function Mix3_GetADDACLowPass : boolean;

begin
     Mix3_GetADDACLowPass:= ((Mix_Read(MIX3_ADCSTATE) and MIX3_LOWPASS88) <> 0);
end;

{******}

procedure Mix3_PrepareForStereo(iMode : boolean);

begin
     bMix3DACStereo:= byte(Mix3_GetDACStereo);
     bMix3DACFilter:= byte(Mix3_GetDACFilter);
     bMix3ADCFilter:= byte(Mix3_GetADCFilter);

     Mix3_SetDACStereo(iMode);
     Mix3_SetDACFilter(FALSE);
     Mix3_SetADCFilter(FALSE);
end;

{******}

procedure Mix3_RestoreFromStereo;

begin
     Mix3_SetDACStereo(boolean(bMix3DACStereo));
     Mix3_SetDACFilter(boolean(bMix3DACFilter));
     Mix3_SetADCFilter(boolean(bMix3ADCFilter));
end;

{******}

procedure Mix3_SetVolume(iSource, iVolL, iVolR : word);

var bIn, bVolR, bVolL : byte;

begin
     bVolL:= byte(iVolL shr 5);
     bVolR:= byte(iVolR shr 5);

     bIn:= byte(Mix_Read(bMix3VolTable[iSource, SBPORT]));

     case bMix3VolTable[iSource, CHANNEL] of

          CH_LEFT  : bIn:= byte((bIn and $0F) or (bVolL shl 5));

          CH_RIGHT : bIn:= byte ((bIn and $F0) or (bVolR shl 1));

          CH_BOTH  : bIn:= byte ((bVolL shl 5) or (bVolR shl 1));

          CH_MAX   : begin
                          if (bVolL shl 1) > (bVolR shl 1)
                          then bIn:= byte(bVolL shl 1)
                          else bIn:= byte(bVolR shl 1);
                     end;
     end;

     Mix_Write(bMix3VolTable[iSource, SBPORT], bIn);
end;

{******}

function Mix3_GetVolume(iSource : word) : word;

var bIn, bLeft, bRight : byte;

begin
     bIn:= byte(Mix_Read(bMix3VolTable[iSource, SBPORT]));
     bLeft:= byte(((bIn and $F0) shr 5) shl 5);
     bRight:= byte(((bIn and $0F) shr 1) shl 5);

     case bMix3VolTable[iSource, CHANNEL] of
          CH_LEFT  : Mix3_GetVolume:= bLeft;
          CH_MAX,
          CH_RIGHT : Mix3_GetVolume:= bRight;
          CH_BOTH  : Mix3_GetVolume:= bLeft * 256 + bRight;
     end;
end;

{******}

procedure Mix3_SetADCSource(iSource : word);

var bIn : byte;

begin
     bIn:= byte(Mix_Read(Mix3_ADCSTATE));

     case iSource of
          MIC  : begin
                      bIn:= bin and not MIX3_SRCMSK;
                      bIn:= bin or MIX3_MICSRC;
                 end;

          CD   : begin
                      bIn:= bIn and not MIX3_SRCMSK;
                      bIn:= bIn or MIX3_CDSRC;
                 end;

          LINE : begin
                      bIn:= bIn and not MIX3_SRCMSK;
                      bIn:= bIn or MIX3_LINESRC;
                 end;
     end;
     Mix_Write(MIX3_ADCSTATE, bIn);
end;

{******}

function Mix3_GetADCSource : word;

var bIn : byte;

begin
     bIn:= byte(Mix_Read(MIX3_ADCSTATE) and MIX3_SRCMSK);

     case bIn of
          MIX3_MICSRC_,
          MIX3_MICSRC   : Mix3_GetADCSource:= MIC;
          MIX3_CDSRC    : Mix3_GetADCSource:= CD;
          MIX3_LINESRC  : Mix3_GetADCSource:= LINE;
                   else   Mix3_GetADCSource:= $FFFF;
     end;
end;

{******}

procedure Mix4_PrepareForMonoADC;

begin
     bMix4SourceL:= byte(Mix_Read(MIX4_ADCSOURCE_L));
     bMix4SourceR:= byte(Mix_Read(MIX4_ADCSOURCE_R));
     Mix_Write(MIX4_ADCSOURCE_L, bMix4SourceL or bMix4SourceR);
end;

{******}

procedure Mix4_RestoreFromMonoADC;

begin
     Mix_Write(Mix4_ADCSOURCE_L, bMix4SourceL);
     Mix_Write(Mix4_ADCSOURCE_L, bMix4SourceR);
end;

{******}

procedure Mix4_SetVolume(iSource, iVolL, iVolR  : word);

begin
     iVolL:= iVolL and not $07;
     iVolR:= iVolR and not $07;

     case bMix4VolTable[iSource, CHANNEL] of

          CH_LEFT  : Mix_Write(bMix4VolTable[iSource, SBPORT], iVolL);

          CH_RIGHT : Mix_Write(bMix4VolTable[iSource, SBPORT] + 1, iVolR);

          CH_BOTH  : begin
                          Mix_Write(bMix4VolTable[iSource, SBPORT], iVolL);
                          Mix_Write(bMix4VolTable[iSource, SBPORT] + 1, iVolR);
                     end;

          CH_MAX:    begin
                          if iVolL > iVolR
                          then Mix_Write(bMix4VolTable[iSource,SBPORT], iVolL)
                          else Mix_Write(bMix4VolTable[iSource,SBPORT], iVolR)
                     end;
     end;
end;

{******}

function Mix4_GetVolume(iSource : word) : word;

var bLeft, bRight : byte;

begin
     bLeft:= byte (Mix_Read(bMix4VolTable[iSource, SBPORT]));
     bLeft:= bLeft and not $07;
     bRight:= byte (Mix_Read(bMix4VolTable[iSource , SBPORT] + 1));
     bRight:= bRight and not $07;

     case bMix4VolTable[iSource , CHANNEL] of
          CH_LEFT  : Mix4_GetVolume:= bLeft;

          CH_MAX   : if(bLeft > bRight) then Mix4_GetVolume:= bLeft
                                        else Mix4_GetVolume:= bRight;

          CH_RIGHT : Mix4_GetVolume:= bRight;

          CH_BOTH  : Mix4_GetVolume:= bLeft * 256 + bRight;

              else   Mix4_GetVolume:= $FFFF;
     end;
end;

{******}

procedure Mix4_SetADCSourceL(iSource : word; iState : boolean);

var bInL : byte;

begin
     bInL:= byte(Mix_Read(MIX4_ADCSOURCE_L));

     if iState then bInL:= bInL or bMix4SourceBits[iSource]
               else bInL:= bInL and not bMix4SourceBits[iSource];

     Mix_Write(MIX4_ADCSOURCE_L, bInL);
end;

{******}

procedure Mix4_SetADCSourceR(iSource : word; iState : boolean);

var bInR : byte;

begin
     bInR:= byte(Mix_Read(MIX4_ADCSOURCE_R));

     if iState then bInR:= bInR or bMix4SourceBits[iSource]
               else bInR:= bInR and not bMix4SourceBits[iSource];

     Mix_Write(MIX4_ADCSOURCE_R, bInR);
end;

{******}

function Mix4_GetADCSourceL(iSource : word) : word;

var bInL : byte;

begin
     bInL:= byte (Mix_Read(Mix4_ADCSOURCE_L));

     case iSource of
          MIDI,
          CD,
          LINE,
          MIC      : if (bInL and bMix4SourceBits[iSource]) =
                                                    bMix4SourceBits[iSource]
                     then Mix4_GetADCSourceL:= CH_BOTH
                     else Mix4_GetADCSourceL:= CH_NONE;


          MIDI_L,
          CD_L,
          LINE_L   : if (bInL and bMix4SourceBits[iSource] <> 0)
                     then Mix4_GetADCSourceL:= CH_LEFT
                     else Mix4_GetADCSourceL:= CH_NONE;


          MIDI_R,
          CD_R,
          LINE_R   : if (bInL and bMix4SourceBits[iSource] <> 0)
                     then Mix4_GetADCSourceL:= CH_RIGHT
                     else Mix4_GetADCSourceL:= CH_NONE;

              else   Mix4_GetADCSourceL:= CH_NONE;
     end;
end;

{******}

function Mix4_GetADCSourceR(iSource : word) : word;

var bInR : byte;

begin
     bInR:= byte(Mix_Read(MIX4_ADCSOURCE_R));

     case iSource of
          MIDI,
          CD,
          LINE,
          MIC     : if (bInR and bMix4SourceBits[iSource]) =
                                                   bMix4SourceBits[iSource]
                    then Mix4_GetADCSourceR:= CH_BOTH
                    else Mix4_GetADCSourceR:= CH_NONE;


          MIDI_L,
          CD_L,
          LINE_L  : if (bInR and bMix4SourceBits[iSource]) <> 0
                    then Mix4_GetADCSourceR:= CH_LEFT
                    else Mix4_GetADCSourceR:= CH_NONE;


          MIDI_R,
          CD_R,
          LINE_R  : if (bInR and bMix4SourceBits[iSource]) <> 0
                    then Mix4_GetADCSourceR:= CH_RIGHT
                    else Mix4_GetADCSourceR:= CH_NONE;

             else   Mix4_GetADCSourceR:= CH_NONE;
     end;
end;

{******}

procedure Mix4_SetOUTSource(iSource : word; iState : boolean);

var bIn : byte;

begin
     bIn:= byte (Mix_Read(Mix4_OUTSOURCE));

     case iSource of
          CD,
          CD_L,
          CD_R,
          LINE,
          LINE_L,
          LINE_R,
          MIC      : if iState then bIn:= bIn or bMix4SourceBits[iSource]
                               else bIn:= bIn and not bMix4SourceBits[iSource];
     end;
     Mix_Write(MIX4_OUTSOURCE, bIn);
end;

{******}

function Mix4_GetOUTSource(iSource : word) : word;

var bIn : byte;

begin
  bIn:= byte (Mix_Read(Mix4_OUTSOURCE));

  case iSource of
    CD,LINE,MIC:
      if (bIn and bMix4SourceBits[iSource]) <> 0
        then Mix4_GetOUTSource:= CH_BOTH
        else Mix4_GetOUTSource:= CH_NONE;

    CD_L,LINE_L:
      if (bIn and bMix4SourceBits[iSource]) <> 0
        then Mix4_GetOUTSource:= CH_LEFT
        else Mix4_GetOUTSource:= CH_NONE;

    CD_R,LINE_R:
      if (bIn and bMix4SourceBits[iSource]) <> 0
        then Mix4_GetOUTSource:= CH_RIGHT
        else Mix4_GetOUTSource:= CH_NONE;

    else { case else }
      Mix4_GetOUTSource:= CH_NONE;
  end;
end;

{******}

procedure Mix4_SetADCGain(iGainL, iGainR :word);

var bIn, bGainL, bGainR : byte;

begin
  bGainL:= byte (iGainL and $03);
  bGainL:= bGainl shl 6;
  bGainR:= byte (iGainR and $03);
  bGainR:= bGainR shl 6;

  bIn:= byte ((Mix_Read(Mix4_ADCGAIN_L) and $3F) or bGainL);
  Mix_Write(Mix4_ADCGAIN_L, bIn);

  bIn:= byte ((Mix_Read(Mix4_ADCGAIN_R) and $3F) or bGainR);
  Mix_Write(Mix4_ADCGAIN_R, bIn);
end;

{******}

function Mix4_GetADCGain(iChannel : word) : word;

var bGainL, bGainR : byte;

begin
  bGainL:= byte (Mix_Read(Mix4_ADCGAIN_L));
  bGainL:= bGainL and $C0;
  bGainL:= bGainL shr 6;

  bGainR:= byte (Mix_Read(Mix4_ADCGAIN_R));
  bGainR:= bGainR and $C0;
  bGainR:= bGainR shr 6;

  case iChannel of
    CH_LEFT:
      Mix4_GetADCGain:= bGainL;

    CH_RIGHT:
      Mix4_GetADCGain:= bGainR;

    else { case else }
      Mix4_GetADCGain:= bGainL * 256 + bGainR;
  end;
end;

{******}

procedure Mix4_SetOUTGain(iGainL, iGainR : word);

var bIn, bGainL, bGainR : byte;

begin
  bGainL:= byte ((iGainL and $03) shl 6);
  bGainR:= byte ((iGainR and $03) shl 6);

  bIn:= byte ((Mix_Read(Mix4_OUTGAIN_L) and $3F) or bGainL);
  Mix_Write(Mix4_OUTGAIN_L, bIn);

  bIn:= byte ((Mix_Read(Mix4_OUTGAIN_R) and $3F) or bGainR);
  Mix_Write(Mix4_OUTGAIN_R, bIn);
end;

{******}

function Mix4_GetOUTGain(iChannel:word) : word;

var bGainL, bGainR : byte;
begin
  bGainL:= byte (Mix_Read(Mix4_OUTGAIN_L));
  bGainL:= bGainL and $C0;
  bGainL:= bGainL shr 6;
  bGainR:= byte (Mix_Read(Mix4_OUTGAIN_R));
  bGainR:= bGainR and $C0;
  bGainR:= bGainR shr 6;

  case iChannel of
    CH_LEFT:
      Mix4_GetOUTGain:= bGainL;

    CH_RIGHT:
      Mix4_GetOUTGain:= bGainR;

    CH_BOTH:
      Mix4_GetOUTGain:= bGainL * 256 + bGainR;

    else { case else }
      Mix4_GetOUTGain:= 0;
  end;
end;

{******}

procedure Mix4_SetAGC(iState : word);

var bIn : byte;

begin
   bIn:= byte (Mix_Read(Mix4_AGC));
   if iState = 0 then
     Mix_Write(Mix4_AGC, bIn and not Mix4_AGCON)
   else
     Mix_Write(Mix4_AGC, bIn or Mix4_AGCON);
end;

{******}

function Mix4_GetAGC : boolean;

begin
   Mix4_GetAGC:= ((Mix_Read(Mix4_AGC) and Mix4_AGCON) <> 0);
end;

{******}

procedure Mix4_SetTreble(iTrebleL, iTrebleR : word);

begin
  Mix_Write(Mix4_TREBLE_L, byte(iTrebleL shl 4));
  Mix_Write(Mix4_TREBLE_R, byte(iTrebleR shl 4));
end;

{******}

function Mix4_GetTreble(iChannel : word) : word;

var bTrebleL, bTrebleR : byte;

begin
  bTrebleL:= Mix_Read(Mix4_TREBLE_L) shr 4;
  bTrebleR:= Mix_Read(Mix4_TREBLE_R) shr 4;

  case iChannel of
    CH_LEFT:  Mix4_GetTreble:= bTrebleL;
    CH_RIGHT: Mix4_GetTreble:= bTrebleR;
    CH_BOTH:  Mix4_GetTreble:= bTrebleL * 256 + bTrebleR;
    else      Mix4_GetTreble:= 0
  end;
end;

{******}

procedure Mix4_SetBass(iBassL, iBassR : word);

begin
  Mix_Write(Mix4_BASS_L, iBassL shl 4);
  Mix_Write(Mix4_BASS_R, iBassR shl 4);
end;

{******}

function Mix4_GetBass(iChannel : word) : word;

var bBassL, bBassR : byte;

begin
  bBassL:= byte(Mix_Read(Mix4_BASS_L) shr 4);
  bBassR:= byte(Mix_Read(Mix4_BASS_R) shr 4);

  case iChannel of
    CH_LEFT:  Mix4_GetBass:= bBassL;
    CH_RIGHT: Mix4_GetBass:= bBassR;
    CH_BOTH:  Mix4_GetBass:= bBassL * 256 + bBassR;
    else      Mix4_GetBass:= 0;
    end;
end;

{******}

end.