unit SCPU_CX;

(* Information
   

   Program Title : Cyrix CPU and FPU detection routines.
   External name : SCPU_CX.TPU
   Version       : 1.0.
   Start date    : 5/11/96.
   Last update   : 5/11/96.
   Author        : Rob Anderton.
   Description   : Cyrix CPU/FPU detection routines for SCPU unit.
                   CPU type constants are in SCPU. This unit contains
                   special Cyrix CPU routines, based on CXID.
*)

interface

{******}

function  Cx_Present : boolean;
function  Cx_Read(Index : byte) : byte;
procedure Cx_Write(Index, Value : byte);
function  Cx_GetCPUType : byte;
function  Cx_CPUID : boolean;
function  Cx_GetModel : byte;
function  Cx_GetRevision : byte;

{******}

implementation

uses SCPU;

{*** CPU types for internal use ***}

const Cx486_SLC    = $00;
      Cx486_DLC    = $01;
      Cx486_SLC2   = $02;
      Cx486_DLC2   = $03;
      Cx486_SRx    = $04;    {Retail Upgrade Cx486SLC}
      Cx486_DRx    = $05;    {Retail Upgrade Cx486DLC}
      Cx486_SRx2   = $06;    {Retail Upgrade 2x Cx486SLC}
      Cx486_DRx2   = $07;    {Retail Upgrade 2x Cx486DLC}
      Cx486S       = $10;    {B step}
      Cx486S2      = $11;    {B step}
      Cx486Se      = $12;    {B step}
      Cx486S2e     = $13;    {B step}
      Cx486DX      = $1A;
      Cx486DX2     = $1B;
      Cx486DX4     = $1F;

      Cx5x86_1xs   = $28;    {5x86 1x Core/Bus Clock}
      Cx5x86_1xp   = $2A;    {5x86 1x Core/Bus Clock}
      Cx5x86_2xs   = $29;    {5x86 2x Core/Bus Clock}
      Cx5x86_2xp   = $2B;    {5x86 2x Core/Bus Clock}
      Cx5x86_3xs   = $2D;    {5x86 3x Core/Bus Clock}
      Cx5x86_3xp   = $2F;    {5x86 3x Core/Bus Clock}
      Cx5x86_4xs   = $2C;    {5x86 4x Core/Bus Clock}
      Cx5x86_4xp   = $2E;    {5x86 4x Core/Bus Clock}

      Cx6x86_1xs   = $30;    {6x86 1x Core/Bus Clock}
      Cx6x86_1xp   = $32;    {6x86 1x Core/Bus Clock}
      Cx6x86_2xs   = $31;    {6x86 2x Core/Bus Clock}
      Cx6x86_2xp   = $33;    {6x86 2x Core/Bus Clock}
      Cx6x86_3xs   = $35;    {6x86 3x Core/Bus Clock}
      Cx6x86_3xp   = $37;    {6x86 3x Core/Bus Clock}
      Cx6x86_4xs   = $34;    {6x86 4x Core/Bus Clock}
      Cx6x86_4xp   = $36;    {6x86 4x Core/Bus Clock}

{******}

var CR2_RW, CR3_RW : boolean;

{******}

function Cx_Present : boolean;

var Temp : byte;

begin
     asm
        xor   ax, ax
        sahf
        mov   ax, 5
        mov   bx, 2
        div   bl
        lahf
        cmp   ah, 2
        jne   @NOT_CYRIX
        mov   [Temp], 1
        jmp   @DONE

     @NOT_CYRIX:

        mov   [Temp], 0

     @DONE:

     end;

     Cx_Present:= boolean(Temp);
end;

{******}

function Cx_Read(Index : byte) : byte;

var Temp : byte;

begin
     asm
        pushf
        cli
        mov   al, [Index]
        out   $22, al
        in    al, $23
        mov   [Temp], al
        popf
     end;
     Cx_Read:= Temp;
end;

{******}

procedure Cx_Write(Index, Value : byte);

begin
     asm
        pushf
        cli
        mov   al, [Index]
        out   $22, al
        mov   al, [Value]
        out   $23, al
        popf
     end;
end;

{******}

function Cx_GetCPUType : byte;

const OLD      = 1;
      NEW      = 2;

      CR2_MASK = $4;
      CR3_MASK = $80;

var CReg    : array[OLD..NEW] of byte;
    CPUType : byte;

begin
     CPUType:= CPU_UNKNOWN;
     CR2_RW:= false;
     CR3_RW:= false;

     CReg[OLD]:= Cx_Read($C2);
     CReg[NEW]:= CReg[OLD] xor CR2_MASK;
     Cx_Write($C2, CReg[NEW]);
     Cx_Read($C0);
     if (Cx_Read($C2) <> CReg[OLD]) then CR2_RW:= true;
     Cx_Write($C2, CReg[OLD]);

     CReg[OLD]:= Cx_Read($C3);
     CReg[NEW]:= CReg[OLD] xor CR3_MASK;
     Cx_Write($C3, CReg[NEW]);
     Cx_Read($C0);
     if (Cx_Read($C3) <> CReg[OLD]) then CR3_RW:= true;
     Cx_Write($C3, CReg[OLD]);


     if ((CR2_RW and CR3_RW) or (not CR2_RW and CR3_RW)) then
     begin
          CPUType:= Cx_Read($FE);

          case CPUType of
               Cx486_SLC..
               Cx486_DRx2  : Inc(CPUType, 9);
               Cx486S..
               Cx486S2e    : Inc(CPUType);
               Cx486DX..
               Cx486DX2    : Dec(CPUType, 5);

               Cx486DX4    : CPUType:= CPU_Cx486DX4;

               Cx5x86_1xs..
               Cx5x86_3xp  : CPUType:= CPU_Cx5x86;

               Cx6x86_1xs..
               Cx6x86_3xp  : CPUType:= CPU_Cx6x86;

               else CPUType:= CPU_UNKNOWN;
          end;
     end

     else if (CR2_RW and not CR3_RW) then CPUType:= CPU_Cx486S
                                     else if (not CR2_RW and not CR3_RW) then CPUType:= CPU_Cx486SLC;

     Cx_GetCPUType:= CPUType;
end;

{******}

function Cx_CPUID : boolean;

var CPUID : boolean;

begin
     asm
        cli

        db $66, $8B, $DC             {mov    ebx,esp      Mark current SP    }
        db $66, $83, $E4, $FC        {and    esp,0FFFCh   Round to DWORD     }
        db $66, $9C                  {pushfd              Put Flag reg. on   }
        db $66, $58                  {pop    eax          stack from AX,     }
        db $66, $8B, $C8             {mov    ecx,eax      mark in CX         }
        db $66, $35, $00, $0, $4, $0 {xor    eax,1 shl 18 Shift alignment    }
        db $66, $50                  {push   eax          bit, pass it       }
        db $66, $9D                  {popfd               to flag register   }
        db $66, $9C                  {pushfd              Get flag from      }
        db $66, $58                  {pop    eax          stack, AX          }
        db $66, $51                  {push   ecx          Old flag data      }
        db $66, $9D                  {popfd               Restore            }
        db $66, $33, $C1             {xor    eax,ecx      Test AL bit        }
        db $66, $C1, $E8, $12        {xor    eax,12h      AL bit to bit 0    }
        db $66, $83, $E0, $01        {and    eax,1h       Mask all others    }
        db $66, $8B, $E3             {mov    esp,ebx      Restore SP         }

        sti
        mov    [CPUID], 0
        cmp    al, 1
        jne    @EXIT
        mov    [CPUID], 1

     @EXIT:

     end;

     Cx_CPUID:= CPUID;
end;

{******}

function Cx_GetModel : byte;

var Temp : byte;

begin
     if ((CR2_RW and CR3_RW) or (not CR2_RW and CR3_RW)) then Temp:= (Cx_Read($FF) and $F0) shr 4
                                                         else Temp:= 0;

     Cx_GetModel:= Temp;
end;

{******}

function  Cx_GetRevision : byte;

var Temp : byte;

begin
     if ((CR2_RW and CR3_RW) or (not CR2_RW and CR3_RW)) then Temp:= (Cx_Read($FF) and $0F)
                                                         else Temp:= 0;

     Cx_GetRevision:= Temp;
end;

{******}

end.