unit SDMA;

(* Information
   

   Program Title : DMA programming library.
   External name : SDMA.PAS
   Version       : 1.1.
   Start date    : 13/8/96.
   Last update   : 30/9/96.
   Author        : Rob Anderton.
   Description   : DMA code for S-Library, new update based on DMAUTIL.PAS by
                   Michael Tischer and Bruno Jennrich.
*)

interface

{*** Bits of status register ($08, $D0) Read ***}

const STATUS_REQ3 = $80;  {Bit set : specific DMA channel}
      STATUS_REQ2 = $40;  {          gets DMA request.   }
      STATUS_REQ1 = $20;  {          Request             }
      STATUS_REQ0 = $10;
      STATUS_TC3  = $08;  {Bit set : Since the last read of      }
      STATUS_TC2  = $04;  {          status register a DMA       }
      STATUS_TC1  = $02;  {          transfer has been concluded.}
      STATUS_TC0  = $01;  {          Terminal Count              }

{*** Bit of command register ($08, $D0) Write ***}

     COMMAND_DACKLEVEL = $80; {Bit 7 set: DMA acknowledge level}
                              {HIGH active                     }
     COMMAND_DREQLEVEL = $40; {Bit 6 set: REQ acknowledge level}
                              {LOW active                      }
     COMMAND_EXTWRITE  = $20; {Bit 5 set: EXTendED Write,      }
                              {else LATE Write                 }
     COMMAND_FIXEDPRI  = $10; {Bit 4 set: fixed priority       }
     COMMAND_COMPRESS  = $08; {Bit 3 set: compressed cycling   }
     COMMAND_INACTIVE  = $04; {Bit 2 set: controller inactive  }
     COMMAND_ADH0      = $02; {Bit 1 set: address hold for channel}
                              {0/4 inactive                     }
     COMMAND_MEM2MEM   = $01; {Bit 0 set: memory/memory,        }
                              {else memory/periphery            }

{*** Bits of request registers ($09, $D2) Write ***}

     REQUEST_RESERVED = $F8;   {Always set reserved bits = 0      }
     REQUEST_SET      = $04;   {Set DMA request                   }
     REQUEST_CLR      = $00;   {Clear DMA request                 }
     REQUEST_MSK      = $03;   {Specify channel in lower two bits }

{*** Bits of channel masking register ($0A, $D4) Write ***}

     CHANNEL_RESERVED = $F8;   {Always set reserved bits =0       }
     CHANNEL_SET      = $04;   {Mask/lock DMA channel             }
     CHANNEL_CLR      = $00;   {Free DMA channel                  }
     CHANNEL_MSK      = $03;   {Specify channel in lower two bits }

{*** Bits of mode register ($0B, $D6) Write ***}

     MODE_DEMAND     = $00;    {Transfer "On Call"                }
     MODE_SINGLE     = $40;    {Transfer single values            }
     MODE_BLOCK      = $80;    {Block transfer                    }
     MODE_CASCADE    = $C0;    {Transfer cascaded                 }
     MODE_DECREMENT  = $20;    {Decrement                         }
     MODE_AUTOINIT   = $10;    {Auto initialization after end     }
     MODE_VERIFY     = $00;    {Verify                            }
     MODE_WRITE      = $04;    {Write to memory                   }
     MODE_READ       = $08;    {Read from memory                  }
     MODE_INVALID    = $0C;    {Invalid                           }
     MODE_CHANNELMSK = $03;    {Specify channel in lower two bits }

{******}

procedure DMA_MasterClear(DMAChannel : integer);
procedure DMA_SetRequest(DMAChannel : integer);
procedure DMA_ClrRequest(DMAChannel : integer);
procedure DMA_SetMask(DMAChannel : integer);
procedure DMA_ClrMask(DMAChannel : integer);
function  DMA_ReadStatus(DMAChannel : integer) : byte;
procedure DMA_ClrFlipFlop(DMAChannel : integer);
function  DMA_ReadCount(DMAChannel : integer) : word;

procedure DMA_SetChannel(DMAChannel : integer; PMem : pointer; Size : word;
                         DMAMode : byte);

function  DMA_Until64kPage(PMem : pointer; Size : word) : word;
function  DMA_MemAlloc(Size : word) : pointer;
procedure DMA_FreeMem(PMem : pointer);

{******}

implementation

uses DOS, SERROR;

{******}

const DMAPage  : array[0..7] of byte = ($87, $83, $81, $82, $8F, $8B, $89, $8A);
      DMAAddr  : array[0..7] of byte = ($00, $02, $04, $06, $C0, $C4, $C8, $CC);
      DMACount : array[0..7] of byte = ($01, $03, $05, $07, $C2, $C6, $CA, $CE);

      {*** Register offsets for Master and Slave ***}
      DMA_status   : array[0..1] of byte = ($08,$D0);{Status register [Read]   }
      DMA_command  : array[0..1] of byte = ($08,$D0);{Command register[Write]  }
      DMA_request  : array[0..1] of byte = ($09,$D2);{Trigger DMA request      }
      DMA_chmask   : array[0..1] of byte = ($0A,$D4);{Mask channels separately }
      DMA_mode     : array[0..1] of byte = ($0B,$D6);{Transfer mode            }
      DMA_flipflop : array[0..1] of byte = ($0C,$D8);{Address/Counter Flip-flop}
      DMA_masterclr: array[0..1] of byte = ($0D,$DA);{Reset controller         }
      DMA_temp     : array[0..1] of byte = ($0D,$DA);{Temporary register       }
      DMA_maskclr  : array[0..1] of byte = ($0E,$DC);{Free all channels        }
      DMA_mask     : array[0..1] of byte = ($0F,$DE);{Mask channels together   }

{******}

procedure DMA_MasterClear(DMAChannel : integer);

begin
     DMAChannel:= DMAChannel and $0007;
     Port[DMA_masterclr[DMAChannel div 4]]:= 0;
end;

{******}

procedure DMA_SetRequest(DMAChannel : integer);

begin
     DMAChannel:= DMAChannel and $0007;
     Port[DMA_request[DMAChannel div 4]]:= REQUEST_SET or (DMAChannel and $03);
end;

{******}

procedure DMA_ClrRequest(DMAChannel : integer);

begin
     DMAChannel:= DMAChannel and $0007;
     Port[DMA_request[DMAChannel div 4]]:= REQUEST_CLR or (DMAChannel and $03);
end;

{******}

procedure DMA_SetMask(DMAChannel : integer);

begin
     DMAChannel:= DMAChannel and $0007;
     Port[DMA_chmask[DMAChannel div 4]]:= CHANNEL_SET or (DMAChannel and $03);
end;

{******}

procedure DMA_ClrMask(DMAChannel : integer);

begin
     DMAChannel:= DMAChannel and $0007;
     Port[DMA_chmask[DMAChannel div 4]]:= CHANNEL_CLR or (DMAChannel and $03);
end;

{******}

function DMA_ReadStatus(DMAChannel : integer) : byte;

begin
     DMAChannel:= DMAChannel and $0007;
     DMA_ReadStatus:= Port[DMA_status[DMAChannel div 4]];
end;

{******}

procedure DMA_ClrFlipFlop(DMAChannel : integer);

begin
     DMAChannel:=DMAChannel and $0007;
     Port[DMA_flipflop[DMAChannel div 4]]:= 0;
end;

{******}

Function DMA_ReadCount(DMAChannel : integer) : word;

var l, h : byte;

begin
     DMAChannel:= DMAChannel and $0007;

     DMA_ClrFlipFlop(DMAChannel);
     l:= Port[DMACount[DMAChannel]];
     h:= Port[DMACount[DMAChannel]];
     DMA_ReadCount:= (h shl 8) + l;
end;

{******}

procedure DMA_SetChannel(DMAChannel : integer; PMem : pointer; Size : word;
                         DMAMode : byte);

var Address : word;
    Page    : byte;

begin
     DMAChannel:= DMAChannel and $0007;
     DMA_SetMask(DMAChannel);
     if Size <> 0 then Size:= Size - 1;

     if DMAChannel <= 3 then  {8bit DMA}
     begin
          Address:= word((longint(PMem) and $FFFF0000) shr 12) +
                         (longint(PMem) and $FFFF);

          Page:= byte(((((longint(PMem)) and $FFFF0000) shr 12) +
                         (longint(PMem) and $FFFF)) shr 16);
     end
     else
     begin
          Address:= word((longint(PMem) and $FFFF0000) shr 13) +
                        ((longint(PMem) and $FFFF) shr 1);

          Page:= byte((((longint(PMem) and $FFFF0000) shr 12) +
                        (longint(PMem) and $FFFF)) shr 16);

          Page:= Page and $FE;
          Size:= Size div 2;
     end;

     Port[DMA_mode[DMAChannel div 4]]:= DMAMode or (DMAChannel and MODE_CHANNELMSK);

     DMA_ClrFlipFlop(DMAChannel);
     Port[DMAAddr[DMAChannel]]:= Lo(Address);
     Port[DMAAddr[DMAChannel]]:= Hi(Address);
     Port[DMAPage[DMAChannel]]:= Page;

     DMA_ClrFlipFlop(DMAChannel);
     Port[DMACount[DMAChannel]]:= Lo(Size);
     Port[DMACount[DMAChannel]]:= Hi(Size);

     DMA_ClrMask(DMAChannel);
end;

{******}

function DMA_Until64kPage(PMem : pointer; Size : word) : word;

var Adr, PageEnd : longint;

begin
     Adr:= (longint(longint(PMem) shr 16) shl 4) + (longint(PMem) and $ffff);
     PageEnd:= Adr + $FFFF;
     PageEnd:= PageEnd and $FFFF0000;
     PageEnd:= PageEnd - Adr;

     if PageEnd > Size then DMA_Until64kPage:= Size
                       else DMA_Until64kPage:= PageEnd;
end;

{******}

function DMA_MemAlloc(Size : word) : pointer;

const Segment    : word = $FFFF;
      OldSegment : word = $FFFF;

var Test : longint;
    Regs : Registers;

begin
     repeat
           Regs.ah:= $48;
           Regs.bx:= Size div 16 + 1;
           MSDOS(Regs);

           if (Regs.Flags and 1) = 0 then
           begin
                Segment:= Regs.ax;

                if OldSegment <> $FFFF then DMA_FreeMem(pointer(OldSegment shl 16));

                OldSegment:= Segment;

                if DMA_Until64kPage(pointer(longint(Segment) shl 16), Size) = Size then
                begin
                     DMA_MemAlloc:= pointer(longint(Segment) shl 16);
                     exit;
                end;
           end;
     until Regs.Flags > 0;

     if Segment <> $FFFF then DMA_FreeMem(pointer(OldSegment shl 16));
     DMA_MemAlloc:= nil;
end;

{******}

procedure DMA_FreeMem(PMem : Pointer);

var Regs : Registers;

begin
     Regs.ah:= $49;
     Regs.es:= longint(PMem) shr 16;
     MSDOS(regs);
end;

{******}

end.

