unit SGFX386;

(* Information
   

   Program Title : SGFX graphics library.
   External name : SGFX386.TPU
   Version       : 1.11.
   Start date    : 5/7/96.
   Last update   : 11/11/96.
   Author        : Rob Anderton.
   Description   : 386 enhanced multipage graphics library.

*)

interface

uses SGFX_VGA, SGFX_X, SGFX_S3, CRT;

{******}

type TScreen    = array [1..64000] of byte;
     PScreen    = ^TScreen;

     TBitmap    = record
                        Width  : word;
                        Height : word;
                        Size   : word;
                        Data   : pointer;
                  end;
     PBitmap    = ^TBitmap;

     TPalette   = array[0..767] of byte;
     PPalette   = ^TPalette;

     TRGBRec    = record
                        R, G, B : byte;
                  end;

     TRGBPalette = array[0..255] of TRGBRec;
     PRGBPalette = ^TRGBPalette;

     TCircleLUT = array[1..450] of real;
     PCircleLUT = ^TCircleLUT;

     TModeInfoRec = record
                          Chip     : word;
                          ChipID   : word;
                          VRAM     : word;
                          MemModel : byte;
                          MaxPage  : word;
                          CurPage  : word;
                          VisPage  : word;
                          MinX     : word;
                          MaxX     : word;
                          MinY     : word;
                          MaxY     : word;
                    end;

{******}

const VGA_SEG : word  = $A000;
      VGA_OFS : word  = 0;
      CRTC_ADDR       = $03D4;
      SEQU_ADDR       = $03C4;
      MISC_ADDR       = $03C2;
      GC_ADDR         = $03CE;
      VID_INT         = $10;

{*** Set 386 instruction ***}

const USE32 = $66;

{*** SGFX modes ***}

const GFXMODE_VGA_320x200  = $1001;  {320x200x256 standard mode 13h}

      GFXMODE_X_320x200    = $2001;  {320x200x256 unchained mode   }
      GFXMODE_X_320x240    = $2002;  {320x240x256 unchained mode   }
      GFXMODE_X_320x400    = $2003;  {320x400x256 unchained mode   }
      GFXMODE_X_360x200    = $2004;  {320x200x256 unchained mode   }
      GFXMODE_X_360x240    = $2005;  {320x200x256 unchained mode   }

      GFXMODE_S3_320x200   = $3001;  {320x200x256 S3 enhanced mode }

{*** Memory models for VRAM ***}

const GFXMEM_PACKED8 = 0;  {8bit packed pixel}
      GFXMEM_PLANAR8 = 1;  {8bit planar      }

{******}

var GFXInfo   : TModeInfoRec;
    GFXActive : boolean;

{*** Initialisation/hardware interface procedures ***}

function  GFX_DetectVGA : boolean;
procedure GFX_Init(Mode : word);
procedure GFX_Done;
procedure GFX_WaitRetrace;
procedure GFX_TextCursor(Visible : boolean);

{*** Video page management procedures ***}

procedure GFX_SetActivePage(Page : word);
procedure GFX_SetVisualPage(Page : word);
procedure GFX_CopyToVRAM(Source : word);
procedure GFX_CopyFromVRAM(Dest : word);

{*** Basic graphics routines ***}

procedure GFX_Cls(Col : byte); {386+}
procedure GFX_Putpixel (X, Y : word; Col : byte);
procedure GFX_Put2pixel (X, Y : word; Col : byte);
procedure GFX_Put4Pixel(X, Y : word; Col : byte);  {386+}
function  GFX_Getpixel (X, Y : word) : byte;

{*** Palette manipulation routines ***}

procedure GFX_LoadPal(FileName : string; var P : TPalette);
procedure GFX_SetPal(Col, R, G, B : byte);
procedure GFX_GetPal(Col : Byte; var R, G, B : byte);
procedure GFX_SetAllPal(Pal : TPalette);
procedure GFX_SetPartPal(Pal : TPalette; Start, Num : byte);
procedure GFX_GetAllPal(var Pal : TPalette);
procedure GFX_FadeUp(Pal : TPalette; Speed : word);
procedure GFX_StepFadeup(Pal : TPalette; Speed : word);
procedure GFX_FadeDown(Speed : word);
procedure GFX_StepFadeDown(Speed : word);
procedure GFX_FadeToCol(Col : byte; Speed : word);
procedure GFX_FadeToRGB(R, G, B : byte; Speed : word);
procedure GFX_FadeToPal(Pal : TPalette; Speed : word);
procedure GFX_StepFadeToPal(Pal : TPalette; Speed : word);
procedure GFX_BlackOut;

{*** Line/polygon procedures ***}

procedure GFX_HLine(x1, x2, y : word; Col : byte);    {386+}
procedure GFX_VLine(x, y1, y2 : word; Col : byte);
procedure GFX_Line(x1, y1, x2, y2 : word; Col : byte);
procedure GFX_DrawPoly(x1, y1, x2, y2, x3, y3, x4, y4 : word; Col : byte);  {386+}
procedure GFX_Rectangle(x, y, w, h : word; Col : byte); {386+}
procedure GFX_FilledRectangle(x, y, w, h : word; Col : byte); {386+}

{*** Circle procedures (now using Bresenham's routine by John Richardson ***}

function  GFX_Rad(theta : real) : real;
procedure GFX_Circle(xpos, ypos, radius : integer; Col : byte);
procedure GFX_FilledCircle(xpos, ypos, radius : integer; Col : byte);

procedure GFX_CreateCOSFile(FileName : string);
procedure GFX_CreateSINFile(FileName : string);

function  GFX_LoadCOSTable(FileName : string) : boolean;
function  GFX_LoadSINTable(FileName : string) : boolean;

{*** Triangles ***}

procedure GFX_Triangle(x1, y1, x2, y2, x3, y3 : word; Col : byte);

{*** Bitmaps ***}

procedure GFX_LoadBitmap(SPFName : string; var Bitmap : TBitmap; var Palette : TPalette);
procedure GFX_LoadBitmapResource(ResName, Key : string; var Bitmap : TBitmap; var Palette : TPalette);
procedure GFX_PutBitmap(X, Y : word; Bitmap : TBitmap);
procedure GFX_PutMaskedBitmap(X, Y : word; Bitmap : TBitmap);
procedure GFX_GetBitmap(X, Y, W, H : word; var Bitmap : TBitmap);
procedure GFX_SaveBitmap(SPFName : string; Bitmap : TBitmap; var Pal : TPalette; Ovr : boolean);
{procedure GFX_ScaleBitmap(Src : TBitmap; var Dst : TBitmap; NewW, NewH : word);}
procedure GFX_ScaleBitmap(x, y, w, h : word; Src : TBitmap; Dest : byte);
function  GFX_BitmapCollide(X1, Y1 : word; B1 : TBitmap;
                            X2, Y2 : word; B2 : TBitmap) : boolean;

{******}

implementation

uses MEMORY, OBJECTS, SCPU, SUTILS, SERROR, SSPF, SRES;

{******}

var CosTbl, SinTbl : PCircleLUT;
    TextMode       : word;
    OldExit        : pointer;
    TextCursor     : word;
    TextCursorVis  : boolean;

{******}

procedure GFX_Exit; far;

begin
     ExitProc:= OldExit;
     if GFXActive then GFX_Done;
end;

{******}

function GFX_DetectVGA : boolean;

var Temp : word;

begin
     asm
        mov  ax,  $1200
        mov  bl,  $32
        int  $10
        xor  ah, ah
        mov  [Temp], ax
     end;

     if Temp = $12 then GFX_DetectVGA:= true
                   else GFX_DetectVGA:= false;
end;

{******}

procedure GFX_Init(Mode : word);

var CPU  : byte;
    VM   : word;
    Temp : word;
    Deg  : real;
    Loop : word;

begin
     if CPUInfo.CPUType < CPU_80386 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_386REQUIRED;
          exit;
     end;

     if not GFX_DetectVGA then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDMODE;
          exit;
     end;

     case Mode of
          GFXMODE_VGA_320x200 : begin
                                     if not VGA_Detect3x2(GFXInfo.ChipID, GFXInfo.VRAM) then
                                     begin
                                          SErrorCode:= (word(ERR_VGA) shl 8) + ERR_VGA_INITERROR;
                                          exit;
                                     end;
                                     VGA_Init3x2;
                                end;

          GFXMODE_X_320x200   : X_Init(Mode, 320);
          GFXMODE_X_320x240   : X_Init(Mode, 320);
          GFXMODE_X_320x400   : X_Init(Mode, 320);
          GFXMODE_X_360x200   : X_Init(Mode, 360);
          GFXMODE_X_360x240   : X_Init(Mode, 360);


          GFXMODE_S3_320x200  : begin
                                     S3_Detect3x2(GFXInfo.ChipID, GFXInfo.VRAM);
                                     with GFXInfo do
                                     begin
                                          Chip:= GFXMODE_S3_320x200;
                                          MemModel:= GFXMEM_PACKED8;
                                          MinX:= 0;
                                          MaxX:= 319;
                                          MinY:= 0;
                                          MaxY:= 199;
                                          CurPage:= 0;
                                          VisPage:= 0;
                                          MaxPage:= (VRAM div 64) - 1;
                                     end;

                                     S3_Init3x2;
                                end;

     else               begin
                           SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDMODE;
                           exit;
                        end;
     end;

     GetMem(SINTbl, sizeof(TCircleLUT));
     if SINTbl = nil then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_NOSINTABLE;
          exit;
     end;
     GetMem(COSTbl, sizeof(TCircleLUT));
     if COSTbl = nil then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_NOCOSTABLE;
          exit;
     end;

     if not GFX_LoadCOSTable('SGFXCOS.TBL') then
     begin
          Deg:= 0;
          for Loop:= 1 to 450 do
          begin
               Deg:= Deg + 0.1;
               COSTbl^[Loop]:= Cos(GFX_Rad(Deg));
          end;
     end;

     if not GFX_LoadSINTable('SGFXSIN.TBL') then
     begin
          Deg:= 0;
          for Loop:= 1 to 450 do
          begin
               Deg:= Deg + 0.1;
               SINTbl^[Loop]:= Sin(GFX_Rad(Deg));
          end;

     end;

     SErrorCode:= 0;
     GFXActive:= true;
end;

{******}

procedure GFX_Done;

begin
     GFX_SetActivePage(0);
     GFX_SetVisualPage(0);

     asm
        mov ax, [TextMode]
        int VID_INT
     end;

     case GFXInfo.Chip of
                  GFXMODE_VGA_320x200 : VGA_Done3x2;
     end;

     if Assigned(SINTbl) then FreeMem(SINTbl, sizeof(TCircleLUT));
     if Assigned(COSTbl) then FreeMem(COSTbl, sizeof(TCircleLUT));

     GFXActive:= false;
     SErrorCode:= 0;
end;

{******}

procedure GFX_WaitRetrace; assembler;

label l1, l2;

asm
   mov dx, 3DAh
l1:
   in  al, dx
   and al, 08h
   jnz l1
l2:
   in  al, dx
   and al, 08h
   jz  l2
end;

{******}

procedure GFX_TextCursor(Visible : boolean);

var CurCL, CurCH : byte;

begin
     if Visible then
     begin
          asm
             mov ah, 01h
             mov cx, [TextCursor]
             int 10h
          end;
          TextCursorVis:= true;
     end
     else
     begin
          asm
             mov ah, $03
             mov bh, 0
             int $10
             mov [CurCH], ch
             mov [CurCL], cl

             mov ah, 01h
             or  ch, 20h
             int $10
          end;
          TextCursor:= (word(CurCH) shl 8) + CurCL;
          TextCursorVis:= false;
     end;
end;

{******}

procedure GFX_SetActivePage(Page : word);

begin
     if (Page > GFXInfo.MaxPage) then
     begin
          SErrorCode:= ERR_GFX_INVALIDPAGE;
          exit;
     end;

     if (Page = GFXInfo.CurPage) then exit;

     case GFXInfo.Chip of
                   GFXMODE_VGA_320x200  : VGA_SetActivePage3x2(Page);

                   GFXMODE_X_320x200,
                   GFXMODE_X_320x240,
                   GFXMODE_X_320x400,
                   GFXMODE_X_360x200,
                   GFXMODE_X_360x240    : X_SetActivePage(Page);

                   GFXMODE_S3_320x200   : S3_SetActivePage3x2(Page);
     end;

     GFXInfo.CurPage:= Page;
end;

{******}

procedure GFX_SetVisualPage(Page : word);

begin
     if (Page > GFXInfo.MaxPage) then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDPAGE;
          exit;
     end;

     if (Page = GFXInfo.VisPage) then exit;

     case GFXInfo.Chip of
                   GFXMODE_VGA_320x200  : VGA_SetVisualPage3x2(Page);

                   GFXMODE_X_320x200,
                   GFXMODE_X_320x240,
                   GFXMODE_X_320x400,
                   GFXMODE_X_360x200,
                   GFXMODE_X_360x240    : X_SetVisualPage(Page);

                   GFXMODE_S3_320x200   : S3_SetVisualPage3x2(Page);
     end;

     GFXInfo.VisPage:= Page;
end;

{******}

procedure GFX_CopyToVRAM(Source : word); assembler;

asm
   push ds
   mov  ax, $A000
   mov  es, ax
   mov  ax, [Source]
   mov  ds, ax
   xor  si, si
   xor  di, di
   mov  cx, 16000
   db   USE32
   rep  movsw
   pop  ds
end;

{******}

procedure GFX_CopyFromVRAM(Dest : word); assembler;

asm
   push ds
   mov  ax, [Dest]
   mov  es, ax
   mov  ax, $A000
   mov  ds, ax
   xor  si, si
   xor  di, di
   mov  cx, 16000
   db   $66
   rep  movsw
   pop  ds
end;

{******}

procedure GFX_Cls(Col : byte);

var Size : word;

begin
     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov  ax, VGA_SEG
                                      mov  es, ax
                                      xor  di, di
                                      mov  al, [Col]
                                      mov  ah, al
                                      mov  bx, ax
                                      db   USE32
                                      shl  ax, 16    {shl EAX, 16}
                                      mov  ax, bx
                                      mov  cx, 16000
                                      db   USE32
                                      rep  stosw     {rep stosd}
                                   end;

                  GFXMEM_PLANAR8 : begin
                                        Size:= (longint(GFXInfo.MaxX + 1) *
                                                longint(GFXInfo.MaxY + 1)) div 16;
                                        asm
                                           mov  dx, SEQU_ADDR
                                           mov  ax, $0F02
                                           out  dx, ax
                                           mov  ax, VGA_SEG
                                           mov  es, ax
                                           mov  di, VGA_OFS
                                           mov  al, [Col]
                                           mov  ah, al
                                           mov  bx, ax
                                           db   USE32
                                           shl  ax, 16    {shl EAX, 16}
                                           mov  ax, bx
                                           mov  cx, [Size]
                                           db   USE32
                                           rep  stosw     {rep stosd}
                                        end;
                                   end;
     end;
end;

{******}

procedure GFX_PutPixel(X, Y : word; Col : byte);

begin
     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov     ax, VGA_SEG
                                      mov     es, ax
                                      mov     ax, [Y]
                                      mov     di, ax
                                      shl     di, 8
                                      shl     ax, 6
                                      add     di, ax
                                      add     di, [X]
                                      mov     al, [Col]
                                      stosb
                                   end;

                  GFXMEM_PLANAR8 : asm
                                      mov   dx, SEQU_ADDR
                                      mov   cx, [X]
                                      and   cl, 3
                                      mov   ax, $102
                                      shl   ah, cl
                                      out   dx, ax

                                      mov   ax, VGA_SEG
                                      mov   es, ax
                                      mov   ax, [Y]
                                      mov   bx, [XByteWidth]
                                      mul   bx

                                      mov   di, [X]
                                      shr   di, 2
                                      add   di, ax
                                      add   di, VGA_OFS
                                      mov   al, [Col]
                                      mov   es:[di], al
                                   end;
     end;
end;

{******}

procedure GFX_Put2Pixel(X, Y : word; Col : byte);

begin
     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov     ax, VGA_SEG
                                      mov     es, ax
                                      mov     ax, [Y]
                                      mov     di, ax
                                      shl     di, 8
                                      shl     ax, 6
                                      add     di, ax
                                      add     di, [X]
                                      mov     ah, [Col]
                                      mov     al, ah
                                      stosw
                                   end;

                  GFXMEM_PLANAR8 : asm
                                      mov   dx, SEQU_ADDR
                                      mov   cx, [X]
                                      and   cl, 2

                                      mov   ax, $302
                                      shl   ah, cl
                                      out   dx, ax

                                      mov   ax, VGA_SEG
                                      mov   es, ax
                                      mov   ax, [Y]
                                      mov   bx, [XByteWidth]
                                      mul   bx

                                      mov   di, [X]
                                      shr   di, 2
                                      add   di, ax
                                      add   di, VGA_OFS
                                      mov   al, [Col]
                                      mov   es:[di], al
                                   end;
     end;
end;

{******}

procedure GFX_Put4Pixel(X, Y : word; Col : byte);

begin
     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov     ax, VGA_SEG
                                      mov     es, ax
                                      mov     ax, [Y]
                                      mov     di, ax
                                      shl     di, 8
                                      shl     ax, 6
                                      add     di, ax
                                      add     di, [X]
                                      mov     al, [Col]
                                      mov     ah, al
                                      mov     dx, ax
                                      db      USE32
                                      shl     ax, 16
                                      mov     ax, dx
                                      db      USE32
                                      stosw
                                   end;

                  GFXMEM_PLANAR8 : begin
                                        GFX_Put2Pixel(X, Y, Col);
                                        GFX_Put2Pixel(X + 2, Y, Col);
                                   end;
     end;
end;

{******}

function GFX_GetPixel(X, Y : word) : byte;

var Temp : byte;

begin
     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov     ax, VGA_SEG
                                      mov     es, ax
                                      mov     di, VGA_OFS
                                      mov     bx, [X]
                                      mov     dx, [Y]
                                      add     di, bx
                                      mov     bx, dx
                                      shl     dx, 8
                                      shl     bx, 6
                                      add     dx, bx
                                      add     di, dx
                                      mov     al, es:[di]
                                      mov     [Temp], al
                                   end;

                  GFXMEM_PLANAR8 : asm
                                      mov   ax, VGA_SEG
                                      mov   es, ax
                                      mov   ax, [Y]
                                      mov   bx, [XByteWidth]
                                      mul   bx

                                      mov   cx, [X]
                                      mov   di, cx
                                      shr   di, 2
                                      add   di, ax
                                      add   di, VGA_OFS

                                      mov     ax, cx
                                      and     ax, 3
                                      shl     ax, 8
                                      mov     al, 4
                                      mov     dx, GC_ADDR
                                      out     dx, ax
                                      mov     al, es:[di]
                                      mov     [Temp], al
                                   end;
     end;
     GFX_GetPixel:= Temp;
end;

{******}

procedure GFX_LoadPal(FileName : string; var P : TPalette);

var F : TBufStream;

begin
     F.Init(FileName, stOpenRead, 768);
     F.Read(P, sizeof(P));
     F.Done;
end;

{******}

procedure GFX_SetPal(Col, R, G, B : byte); assembler;

asm
   mov dx, 3c8h
   mov al, [Col]
   out dx, al
   inc dx
   mov al, [R]
   out dx, al
   mov al, [G]
   out dx, al
   mov al, [B]
   out dx, al
end;

{******}

procedure GFX_GetPal(Col : byte; var R, G, B : byte);

var rr, gg, bb : byte;

begin
     asm
        mov dx,   3c7h
        mov al,   Col
        out dx,   al
        add dx,   2
        in  al,   dx
        mov [rr], al
        in  al,   dx
        mov [gg], al
        in  al,   dx
        mov [bb], al
     end;
     R:= rr;
     G:= gg;
     B:= bb;
end;

{******}

procedure GFX_Blackout;

var Loop : byte;

begin
     GFX_WaitRetrace;
     for Loop:= 0 to 255 do GFX_SetPal(Loop, 0, 0, 0);
end;

{******}

procedure GFX_SetPartPal(Pal : TPalette; Start, Num : byte);

var Loop   : word;
    PalIdx : byte;

begin
     GFX_WaitRetrace;
     Loop:= Start;
     PalIdx:= Start;
     repeat
           GFX_SetPal(PalIdx, Pal[Loop], Pal[Loop + 1], Pal[Loop + 2]);
           Inc(Loop, 3);
           Inc(PalIdx);
     until PalIdx > (Num - Start + 1);
end;

{******}

procedure GFX_GetAllPal(var Pal : TPalette);

var Loop   : word;
    PalIdx : byte;

begin
     Loop:= 0;
     PalIdx:= 0;
     repeat
           GFX_GetPal(PalIdx, Pal[Loop], Pal[Loop + 1], Pal[Loop + 2]);
           Inc(Loop, 3);
           Inc(PalIdx);
     until Loop > 767;
end;

{******}

procedure GFX_SetAllPal(Pal : TPalette); assembler;

asm
   call  GFX_WaitRetrace
   push  ds
   lds   si, Pal
   mov   dx, 3C8h
   mov   al, 0
   out   dx, al
   inc   dx
   mov   cx, 768
   rep   outsb
   pop   ds
end;

{******}

procedure GFX_Fadeup(Pal : TPalette; Speed : word);

var Loop1, Loop2 : word;
    PalIdx       : byte;
    Tmp          : array [1..3] of byte;

begin
     for Loop1:= 1 to 64 do
     begin
         GFX_WaitRetrace;
         Loop2:= 0;
         PalIdx:= 0;
         repeat
             GFX_GetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
             if Tmp[1] < Pal[Loop2] then Inc(Tmp[1]);
             if Tmp[2] < Pal[Loop2 + 1] then Inc(Tmp[2]);
             if Tmp[3] < Pal[Loop2 + 2] then Inc(Tmp[3]);
             GFX_SetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
             Inc(PalIdx);
             Inc(Loop2, 3);
         until Loop2 > 767;
         Delay(Speed);
     end;
end;

{******}

procedure GFX_StepFadeup(Pal : TPalette; Speed : word);

var Loop1, Loop2 : word;
    Temp1, Temp2 : TRGBPalette;

begin
     Temp1:= TRGBPalette(Pal);
     FillChar(Temp2, sizeof(TPalette), 0);

     for Loop1:= 1 to 64 do
     begin
         for Loop2:= 0 to 255 do
         begin
              Temp2[Loop2].r:= word(Temp1[Loop2].r * Loop1 div 64);
              Temp2[Loop2].g:= word(Temp1[Loop2].g * Loop1 div 64);
              Temp2[Loop2].b:= word(Temp1[Loop2].b * Loop1 div 64);
         end;
         GFX_WaitRetrace;
         GFX_SetAllPal(TPalette(Temp2));
         Delay(Speed);
     end;
end;

{******}

procedure GFX_FadeDown(Speed : word);

var Loop1, Loop2 : word;
    PalIdx       : byte;
    Tmp          : array [1..3] of byte;

begin
     for Loop1:= 1 to 64 do
     begin
         GFX_WaitRetrace;
         Loop2:= 0;
         PalIdx:= 0;

         repeat
               GFX_GetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               if Tmp[1] > 0 then Dec(Tmp[1]);
               if Tmp[2] > 0 then Dec(Tmp[2]);
               if Tmp[3] > 0 then Dec(Tmp[3]);
               GFX_SetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               Inc(PalIdx);
               Inc(Loop2, 3);
         until Loop2 > 767;

         Delay(Speed);
     end;
end;

{******}

procedure GFX_StepFadeDown(Speed : word);

var Loop1, Loop2 : word;
    Temp1, Temp2 : TRGBPalette;
    Pal          : TPalette;

begin
     GFX_GetAllPal(Pal);
     Temp1:= TRGBPalette(Pal);
     FillChar(Temp2, sizeof(TPalette), 0);

     for Loop1:= 64 downto 1 do
     begin
         for Loop2:= 0 to 255 do
         begin
              Temp2[Loop2].r:= word(Temp1[Loop2].r * Loop1 div 64);
              Temp2[Loop2].g:= word(Temp1[Loop2].g * Loop1 div 64);
              Temp2[Loop2].b:= word(Temp1[Loop2].b * Loop1 div 64);
         end;
         GFX_WaitRetrace;
         GFX_SetAllPal(TPalette(Temp2));
         Delay(Speed);
     end;
end;

{******}

procedure GFX_FadeToCol(Col : byte; Speed : word);

var Loop1, Loop2 : word;
    PalIdx       : byte;
    Tmp          : array [1..3] of byte;
    R, G, B      : byte;

begin
     GFX_GetPal(Col, R, G, B);
     for Loop1:= 1 to 64 do
     begin
         GFX_WaitRetrace;
         Loop2:= 0;
         PalIdx:= 0;

         repeat
               GFX_GetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               if Tmp[1] > R then Dec(Tmp[1]);
               if Tmp[1] < R then Inc(Tmp[1]);
               if Tmp[2] > G then Dec(Tmp[2]);
               if Tmp[2] < G then Inc(Tmp[2]);
               if Tmp[3] > B then Dec(Tmp[3]);
               if Tmp[3] < B then Inc(Tmp[3]);

               GFX_SetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               Inc(PalIdx);
               Inc(Loop2, 3);
         until Loop2 > 767;

         Delay(Speed);
     end;
end;

{******}

procedure GFX_FadeToRGB(R, G, B : byte; Speed : word);

var Loop1, Loop2 : word;
    PalIdx       : byte;
    Tmp          : array [1..3] of byte;

begin
     if R > 63 then R:= 63;
     if G > 63 then G:= 63;
     if B > 63 then B:= 63;
     for Loop1:= 1 to 64 do
     begin
         GFX_WaitRetrace;
         Loop2:= 0;
         PalIdx:= 0;

         repeat
               GFX_GetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               if Tmp[1] > R then Dec(Tmp[1]);
               if Tmp[1] < R then Inc(Tmp[1]);
               if Tmp[2] > G then Dec(Tmp[2]);
               if Tmp[2] < G then Inc(Tmp[2]);
               if Tmp[3] > B then Dec(Tmp[3]);
               if Tmp[3] < B then Inc(Tmp[3]);

               GFX_SetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               Inc(PalIdx);
               Inc(Loop2, 3);
         until Loop2 > 767;

         Delay(Speed);
     end;
end;

{******}

procedure GFX_FadeToPal(Pal : TPalette; Speed : word);

var Loop1, Loop2 : word;
    PalTmp       : TRGBPalette;
    PalIdx       : byte;
    Tmp          : array [1..3] of byte;

begin
     PalTmp:= TRGBPalette(Pal);
     for Loop1:= 1 to 64 do
     begin
         GFX_WaitRetrace;
         Loop2:= 0;
         PalIdx:= 0;

         repeat
               GFX_GetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               if Tmp[1] > PalTmp[PalIdx].R then Dec(Tmp[1]);
               if Tmp[1] < PalTmp[PalIdx].R then Inc(Tmp[1]);
               if Tmp[2] > PalTmp[PalIdx].G then Dec(Tmp[2]);
               if Tmp[2] < PalTmp[PalIdx].G then Inc(Tmp[2]);
               if Tmp[3] > PalTmp[PalIdx].B then Dec(Tmp[3]);
               if Tmp[3] < PalTmp[PalIdx].B then Inc(Tmp[3]);

               GFX_SetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
               Inc(PalIdx);
               Inc(Loop2, 3);
         until Loop2 > 767;

         Delay(Speed);
     end;
end;

{******}

procedure GFX_StepFadeToPal(Pal : TPalette; Speed : word);

var Loop1, Loop2 : word;
    PalIdx       : byte;
    Tmp          : array [1..3] of byte;

begin
     for Loop1:= 1 to 64 do
     begin
         GFX_WaitRetrace;
         Loop2:= 0;
         PalIdx:= 0;
         repeat
             GFX_GetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
             Tmp[1]:= byte(Pal[Loop2] * Loop1 div 64);
             Tmp[2]:= byte(Pal[Loop2 + 1] * Loop1 div 64);
             Tmp[3]:= byte(Pal[Loop2 + 2] * Loop1 div 64);
             GFX_SetPal(PalIdx, Tmp[1], Tmp[2], Tmp[3]);
             Inc(PalIdx);
             Inc(Loop2, 3);
         until Loop2 > 767;
         Delay(Speed);
     end;
end;

{******}

procedure GFX_HLine(x1, x2, y : word; Col : byte);

var Temp  : word;

begin
     if X2 < X1 then
     begin
          Temp:= X2;
          X2:= X1;
          X1:= Temp;
     end;

     if X2 = X1 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDXVAL;
          exit;
     end;

     if (x2 > GFXInfo.MaxX) then x2:= GFXInfo.MaxX;
     if (y > GFXInfo.MaxY) then y:= GFXInfo.MaxY;

     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov   ax, VGA_SEG
                                      mov   es, ax
                                      mov   ax, [y]
                                      mov   di, ax
                                      shl   ax, 8
                                      shl   di, 6
                                      add   di, ax
                                      add   di, [x1]

                                      mov   al, [Col]
                                      mov   ah, al
                                      mov   dx, ax
                                      db    USE32
                                      shl   ax, 16
                                      mov   ax, dx

                                      mov   cx, [x2]
                                      sub   cx, [x1]
                                      inc   cx
                                      shr   cx, 1
                                      jnc   @P24
                                      stosb

                                  @P24:

                                      shr   cx, 1
                                      jnc   @P4
                                      stosw

                                  @P4:

                                      db    USE32
                                      rep   stosw
                                   end;

                  GFXMEM_PLANAR8 : asm
                                      mov   ax, VGA_SEG
                                      mov   es, ax

                                      mov   ax, [y]
                                      mov   bx, [x1]
                                      mov   cl, bl

                                      mov   dx, [XByteWidth]
                                      mul   dx

                                      shr   bx, 2
                                      add   bx, ax
                                      add   bx, VGA_OFS
                                      and   cl, 3

                                      mov   di, bx
                                      mov   dl, $0F
                                      shl   dl, cl

                                      mov   cx, [x2]
                                      mov   ax, cx
                                      and   cl, 3
                                      mov   dh, $0E
                                      shl   dh, cl
                                      not   dh

                                      mov   bx, [x1]

                                      shr   ax, 2
                                      shr   bx, 2
                                      mov   cx, ax
                                      sub   cx, bx

                                      mov   ax, dx
                                      mov   dx, SEQU_ADDR
                                      mov   bl, [Col]
                                      or    cx, cx
                                      jnz   @@1
                                      and   ah, al
                                      jmp   @@2

                                  @@1:

                                      mov   si, ax
                                      mov   ah, al
                                      mov   al, $02
                                      out   dx, ax
                                      mov   al, bl
                                      stosb
                                      dec   cx

                                      mov   ah, $0F
                                      mov   al, $02
                                      out   dx, ax
                                      mov   al, bl
                                      rep   stosb
                                      mov   ax, si

                                  @@2:
                                       mov  al, $02
                                       out  dx, ax
                                       mov  byte ptr es:[di], bl
                                  end;
     end;
end;

{******}

procedure GFX_VLine(x, y1, y2 : word; Col : byte);

var t : word;

begin
     if y1 > y2 then
     begin
          t:= y1;
          y1:= y2;
          y2:= t;
     end;

     if y1 = y2 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;

     if y2 > GFXInfo.MaxY then y2:= GFXInfo.MaxY;

     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov     ax, VGA_SEG
                                      mov     es, ax
                                      mov     bx, [y1]
                                      mov     ax, bx
                                      mov     di, ax
                                      shl     ax, 8
                                      shl     di, 6
                                      add     di, ax
                                      add     di, [x]
                                      mov     al, [Col]
                                      xor     ah, ah
                                      mov     cx, [y2]

                                   @@1:

                                      mov     es:[di], al
                                      add     di, 320
                                      dec     cx
                                      cmp     cx, bx
                                      ja      @@1
                                   end;

                  GFXMEM_PLANAR8 : asm
                                      mov     ax, VGA_SEG
                                      mov     es, ax

                                      mov     ax, [y1]
                                      mov     cx, [y2]
                                      sub     cx, ax

                                      inc     cx
                                      mov     bx, [x]
                                      mov     si, cx

                                      mov     cl, bl
                                      mov     dx, [XByteWidth]
                                      mul     dx
                                      shr     bx, 2
                                      add     bx, ax
                                      add     bx, VGA_OFS
                                      and     cl, 3
                                      mov     ah, 1
                                      shl     ah, cl
                                      mov     al, $02
                                      mov     dx, SEQU_ADDR
                                      out     dx, ax

                                      mov     cx, si
                                      mov     al, [Col]
                                      mov     si, [XByteWidth]

                                   @@1:

                                      mov     es:[bx], al
                                      add     bx, si
                                      dec     cx
                                      jnz     @@1

                                   end;
     end;
end;

{******}

procedure GFX_Line(x1, y1, x2, y2 : word; Col : byte);

var VertInc, Inc1, Inc2, Routine : word;


begin
     if (X1 > GFXInfo.MaxX) then X1:= GFXInfo.MaxX;
     if (X2 > GFXInfo.MaxX) then X2:= GFXInfo.MaxX;
     if (Y1 > GFXInfo.MaxY) then Y1:= GFXInfo.MaxY;
     if (Y2 > GFXInfo.MaxY) then Y2:= GFXInfo.MaxY;

     if (X1 = X2) and (Y1 <> Y2) then
     begin
          GFX_VLine(X1, Y1, Y2, Col);
          Exit;
     end;

     if (Y1 = Y2) and (X1 <> X2) then
     begin
          GFX_HLine(X1, X2, Y1, Col);
          Exit;
     end;

     if (X1 = X2) and (Y1 = Y2) then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDXVAL;
          Exit;
     end;

     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov     ax, [x1]
                                      mov     bx, [x2]
                                      mov     cx, [y1]
                                      mov     dx, [y2]

                                      cmp     cx, dx
                                      jbe     @1
                                      xchg    cx, dx
                                      xchg    ax, bx

                                    @1:

                                      mov     di, cx
                                      shl     di, 2
                                      add     di, cx
                                      mov     si, bx
                                      mov     bx, dx
                                      sub     bx, cx
                                      shl     di, 6
                                      add     di, ax
                                      mov     dx, si
                                      sub     dx, ax

                                      mov     ax, VGA_SEG
                                      mov     es, ax
                                      mov     al, [Col]
                                      xor     ah, ah
                                      or      dx, 0
                                      jge     @jmp1
                                      neg     dx
                                      cmp     dx, bx
                                      jbe     @jmp3

                                      mov     cx, dx
                                      inc     cx
                                      mov     si, dx
                                      shr     si, 1
                                      std

                                   @1c:

                                      stosb

                                   @1b:

                                      or      si,si
                                      jge     @1a
                                      add     di, 320
                                      add     si, dx
                                      jmp     @1b

                                   @1a:

                                      sub     si, bx
                                      loop    @1c
                                      jmp     @Exit

                                   @jmp3:

                                      mov     cx, bx
                                      inc     cx
                                      mov     si, bx
                                      neg     si
                                      sar     si, 1
                                      cld

                                   @2c:

                                      stosb

                                   @2b:

                                      or      si, si
                                      jl      @2a
                                      sub     si, bx
                                      dec     di
                                      jmp     @2b

                                   @2a:

                                      add     di, 319
                                      add     si, dx
                                      loop    @2c
                                      jmp     @Exit

                                   @jmp1:

                                      cmp     dx, bx
                                      jbe     @jmp4
                                      mov     cx, dx
                                      inc     cx
                                      mov     si, dx
                                      shr     si, 1
                                      cld

                                   @3c:

                                      stosb

                                   @3b:

                                      or      si, si
                                      jge     @3a
                                      add     di, 320
                                      add     si, dx
                                      jmp     @3b

                                   @3a:

                                      sub     si, bx
                                      loop    @3c
                                      jmp     @Exit

                                   @jmp4:

                                      mov     cx, bx
                                      inc     cx
                                      mov     si, bx
                                      neg     si
                                      sar     si, 1
                                      std

                                   @4c:

                                      stosb

                                   @4b:

                                      or      si, si
                                      jl      @4a
                                      sub     si, bx
                                      inc     di
                                      jmp     @4b

                                   @4a:

                                      add     di, 321
                                      add     si, dx
                                      loop    @4c

                                   @Exit:

                                      cld

                                   end;

                  GFXMEM_PLANAR8 : asm
                                      mov     ax, VGA_SEG
                                      mov     es, ax

                                      mov     si, [XByteWidth]
                                      mov     cx, [x2]
                                      sub     cx, [x1]
                                      jns     @@1

                                      neg     cx

                                   @@1:

                                      mov     bx, [y2]
                                      sub     bx, [y1]
                                      jns     @@2

                                      neg     bx
                                      neg     si

                                   @@2:

                                      mov     [VertInc], si
                                      mov     [Routine], offset @@LoSlopeLine
                                      cmp     bx, cx
                                      jle     @@3
                                      mov     [Routine], offset @@HiSlopeLine
                                      xchg    bx, cx

                                   @@3:

                                      shl     bx, 1
                                      mov     [Inc1], bx
                                      sub     bx, cx
                                      mov     si, bx
                                      sub     bx, cx
                                      mov     [Inc2], bx

                                      mov     ax, [y1]
                                      mov     bx, [x1]

                                      mov     dx, [XByteWidth]
                                      mul     dx
                                      mov     dx, cx
                                      mov     cl, bl

                                      shr     bx, 2
                                      add     bx, ax
                                      add     bx, VGA_OFS
                                      and     cl, 3
                                      mov     di, bx
                                      mov     al, 1
                                      shl     al, cl
                                      mov     ah, al
                                      shl     al, 4
                                      add     ah, al
                                      mov     bl, ah
                                      mov     cx, dx
                                      inc     cx
                                      mov     dx, SEQU_ADDR
                                      jmp     [Routine]

                                   @@LoSlopeLine:

                                      mov     al, $02
                                      mov     bh, [Col]

                                   @@4:

                                      mov     ah, bl

                                   @@5:

                                      or      ah, bl
                                      rol     bl, 1
                                      jc      @@7

                                      or      si, si
                                      jns     @@6
                                      add     si, [Inc1]
                                      dec     cx
                                      jnz     @@5

                                      out     dx, ax
                                      mov     es:[di], bh
                                      jmp     @@EXIT

                                   @@6:

                                      add     si, [Inc2]
                                      out     dx, ax
                                      mov     es:[di], bh
                                      add     di, [VertInc]
                                      dec     cx
                                      jnz     @@4
                                      jmp     @@EXIT

                                   @@7:

                                      out     dx, ax
                                      mov     es:[di], bh
                                      inc     di
                                      or      si, si
                                      jns     @@8
                                      add     si, [Inc1]
                                      dec     cx
                                      jnz     @@4
                                      jmp     @@EXIT

                                   @@8:

                                      add     si, [Inc2]
                                      add     di, [VertInc]
                                      dec     cx
                                      jnz     @@4
                                      jmp     @@EXIT

                                   @@HiSlopeLine:

                                      mov     bx, [VertInc]
                                      mov     al, $02

                                   @@9:

                                      out     dx, ax
                                      mov     dl, [Col]
                                      mov     es:[di],dl
                                      mov     dx, SEQU_ADDR
                                      add     di,bx

                                   @@10:

	                              or      si, si
                                      jns     @@11

                                      add     si, [Inc1]
                                      dec     cx
                                      jnz     @@9

                                      jmp @@EXIT

                                   @@11:

                                      add     si, [Inc2]
                                      rol     ah, 1
                                      adc     di, 0

                                   @@12:

                                      dec     cx
                                      jnz     @@9

                                   @@EXIT:

                                   end;
     end;
end;

{******}

procedure GFX_DrawPoly(x1, y1, x2, y2, x3, y3, x4, y4 : word; Col : byte);

var x            : integer;
    mny, mxy     : integer;
    mnx, mxx, yc : integer;
    mul1, div1,
    mul2, div2,
    mul3, div3,
    mul4, div4   : integer;

begin
     mny:= y1;
     mxy:= y1;

     if y2 < mny then mny:= y2;
     if y2 > mxy then mxy:= y2;

     if y3 < mny then mny:= y3;
     if y3 > mxy then mxy:= y3;

     if y4 < mny then mny:= y4;
     if y4 > mxy then mxy:= y4;

     if mny < GFXInfo.MinY then mny:= GFXInfo.MinY;
     if mxy > GFXInfo.MaxY then mxy:= GFXInfo.MaxY;
     if mny > GFXInfo.MaxY then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;
     if mxy < GFXInfo.MinY then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;

     mul1:= x1 - x4;
     div1:= y1 - y4;

     mul2:=x2 - x1;
     div2:=y2 - y1;

     mul3:= x3 - x2;
     div3:= y3 - y2;

     mul4:= x4 - x3;
     div4:= y4 - y3;

     for yc:= mny to mxy do
     begin
          mnx:= 320;
          mxx:= -1;

          if (y4 >= yc) or (y1 >= yc) then
             if (y4 <= yc) or (y1 <= yc) then
                if not(y4 = y1) then
                begin
                     x:= (yc - y4) * mul1 div div1 + x4;
                     if x < mnx then mnx:= x;
                     if x > mxx then mxx:= x;
                end;

          if (y1 >= yc) or (y2 >= yc) then
             if (y1 <= yc) or (y2 <= yc) then
                if not(y1 = y2) then
                begin
                     x:= (yc - y1) * mul2 div div2 + x1;
                     if x < mnx then mnx:= x;
                     if x > mxx then mxx:= x;
                end;

          if (y2 >= yc) or (y3 >= yc) then
             if (y2 <= yc) or (y3 <= yc) then
                if not(y2 = y3) then
                begin
                     x:= (yc - y2) * mul3 div div3 + x2;
                     if x < mnx then mnx:= x;
                     if x > mxx then mxx:= x;
                end;

          if (y3 >= yc) or (y4 >= yc) then
             if (y3 <= yc) or (y4 <= yc) then
                if not(y3 = y4) then
                begin
                     x:= (yc - y3) * mul4 div div4 + x3;
                     if x < mnx then mnx:= x;
                     if x > mxx then mxx:= x;
                end;

          if mnx < GFXInfo.MinX then mnx:= GFXInfo.MinX;
          if mxx > GFXInfo.MaxX then mxx:= GFXInfo.MaxX;
          if mnx <= mxx then GFX_HLine(mnx, mxx, yc, Col);
     end;
end;

{******}

procedure GFX_Rectangle(x, y, w, h : word; Col : byte);

var x2, y2 : word;

begin
     x2:= x + w;
     y2:= y + h;

     if w = 0 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDWVAL;
          exit;
     end;

     if h = 0 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDHVAL;
          exit;
     end;

     if x > GFXInfo.MaxX then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDXVAL;
          exit;
     end;

     if y > GFXInfo.MaxY then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;

     if (x2 > GFXInfo.MaxX) then x2:= GFXInfo.MaxX;
     if (y2 > GFXInfo.MaxY) then y2:= GFXInfo.MaxY;

     GFX_HLine( x, x2,  y, Col);
     GFX_HLine( x, x2, y2, Col);
     GFX_VLine( x,  y, y2, Col);
     GFX_VLine(x2,  y, y2, Col);
end;

{******}

procedure GFX_FilledRectangle(x, y, w, h : word; Col : byte);

const XLeftPlane  : array[0..3] of byte = ($0F, $0E, $0C, $08);
      XRightPlane : array[0..3] of byte = ($0F, $01, $03, $07);

var x2, y2 : word;
    Loop   : word;

begin
     x2:= x + w;
     y2:= y + h;

     if w = 0 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDWVAL;
          exit;
     end;

     if h = 0 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDHVAL;
          exit;
     end;

     if x > GFXInfo.MaxX then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDXVAL;
          exit;
     end;

     if y > GFXInfo.MaxY then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;

     if (x2 > GFXInfo.MaxX) then x2:= GFXInfo.MaxX;
     if (y2 > GFXInfo.MaxY) then y2:= GFXInfo.MaxY;

     if (x + w > GFXInfo.MaxX) then w:= GFXInfo.MaxX - x;
     if (y + h > GFXInfo.MaxY) then h:= GFXInfo.MaxY - y;

     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : asm
                                      mov   ax, VGA_SEG
                                      mov   es, ax
                                      mov   ax, [y]
                                      mov   di, ax
                                      shl   ax, 8
                                      shl   di, 6
                                      add   di, ax
                                      add   di, [x]

                                      mov   al, [Col]
                                      mov   ah, al
                                      mov   dx, ax
                                      db    USE32
                                      shl   ax, 16
                                      mov   ax, dx
                                      mov   dx, [h]

                                  @LOOPY:

                                      mov   cx, [w]
                                      inc   cx
                                      shr   cx, 1
                                      jnc   @P24
                                      stosb

                                  @P24:

                                      shr   cx, 1
                                      jnc   @P4
                                      stosw

                                  @P4:

                                      db    USE32
                                      rep   stosw

                                      add   di, 319
                                      sub   di, [w]
                                      dec   dx
                                      jnz   @LOOPY
                                   end;

                  GFXMEM_PLANAR8 : asm
                                      push   bp
                                      cld
                                      mov    ax, [XByteWidth]
                                      mul    [y]
                                      mov    di, [x]
                                      shr    di, 2
                                      add    di, ax
                                      add    di, VGA_OFS
                                      mov    ax, VGA_SEG
                                      mov    es, ax
                                      mov    dx, SEQU_ADDR
                                      mov    al, $02
                                      out    dx, al
                                      inc    dx
                                      mov    si, [x]
                                      and    si, 3
                                      mov    bh, byte ptr XLeftPlane[si]
                                      mov    si, [x2]
                                      and    si, 3
                                      mov    bl, byte ptr XRightPlane[si]
                                      mov    cx, [x2]
                                      mov    si, [x]
                                      cmp    cx, si
                                      jle    @FILLDONE

                                      dec    cx
                                      and    si, not 011b
                                      sub    cx, si
                                      shr    cx, 2
                                      jnz    @MaskSet
                                      and    bh, bl

                                  @MaskSet:

                                      mov    si, [y2]
                                      sub    si, [y]
                                      jle    @FILLDONE
                                      mov    ah, [Col]
                                      mov    bp, [XByteWidth]
                                      sub    bp, cx
                                      dec    bp

                                  @FillRowsLoop:

                                      push   cx
                                      mov    al, bh
                                      out    dx, al
                                      mov    al, ah
                                      stosb
                                      dec    cx
                                      js     @FillLoopBottom
                                      jz     @DoRightEdge
                                      mov    al, $0F
                                      out    dx, al
                                      mov    al, ah
                                      rep    stosb

                                  @DoRightEdge:

	                              mov    al, bl
                                      out    dx, al
                                      mov    al, ah
                                      stosb

                                  @FillLoopBottom:

	                              add    di, bp
                                      pop    cx
                                      dec    si
                                      jnz    @FillRowsLoop

                                  @FillDone:

                                      pop    bp

                                  end;
     end;
end;

{******}

function GFX_Rad(theta : real) : real;

begin
     GFX_Rad:= theta * pi / 180
end;

{******}

procedure GFX_Circle(xpos, ypos, radius : integer; Col : byte);

{******}

        procedure GFX_OldCircle(xpos, ypos, radius : integer; Col : byte);

        var x, y, pos : integer;

        begin
             pos:= 1;
             repeat
                   x:= round(radius * CosTbl^[pos]);
                   y:= round(radius * SinTbl^[pos]);
                   GFX_PutPixel(xpos + X, ypos + Y, Col);
                   GFX_PutPixel(xpos + X, ypos - Y, Col);
                   GFX_PutPixel(xpos - X, ypos + Y, Col);
                   GFX_PutPixel(xpos - X, ypos - Y, Col);
                   GFX_PutPixel(xpos + Y, ypos + X, Col);
                   GFX_PutPixel(xpos + Y, ypos - X, Col);
                   GFX_PutPixel(xpos - Y, ypos + X, Col);
                   GFX_PutPixel(xpos - Y, ypos - X, Col);
                   Inc(Pos);
             until (Pos > 450);
        end;

{******}

        procedure GFX_XCircle(xpos, ypos, radius : integer; Col : byte);

        var offset0, offset1, offset2, offset3, offset4, offset5, offset6,
            offset7, mask0n1, mask2n3, mask4n5, mask6n7, shrunkradius,
            diametereven, error, Left, Top, Diameter, VSeg, VOfs, XW : word;

        begin
             Left:= xpos - radius;
             Top:= ypos - radius;
             Diameter:= radius shl 1;
             shrunkradius:= radius;
             VSeg:= VGA_SEG;
             VOfs:= VGA_OFS;
             XW:= XByteWidth;

             asm
                push ds
	        mov  di, [XW]
	        xor  dx, dx
                mov  ax, Diameter
                dec  ax
                shr  ax, 1
                adc  dx, 0
                mov  shrunkradius, ax
                mov  diametereven, dx
                add  ax, Top
                mul  di
                add  ax, VOfs

                mov  bx, Left
                mov  cx, bx
                mov  si, bx
                shr  si, 2
                add  si, ax
                mov  offset6, si
                and  bx, 3
                mov  bl, byte ptr XColumnMask[bx]
                mov  mask6n7, bx

                add  cx, Diameter
                dec  cx
                mov  bx, cx
                shr  cx, 2
                add  cx, ax
                mov  offset2, cx
                and  bx, 3
                mov  bl, byte ptr XColumnMask[bx]
                mov  mask2n3, bx

                cmp  diametereven, 1
                jne  @@MiddlePlotsOverlap
                add  si, di
                add  cx, di

             @@MiddlePlotsOverlap:

	        mov  offset7, si
                mov  offset3, cx

                mov  bx, Left
                add  bx, shrunkradius

                mov  ax, Top
                mul  di
                add  ax, VOfs
                mov  si, ax

                mov  ax, Diameter
                dec  ax
                mul  di
                add  ax, si

                mov  di, bx
                shr  di, 2
                add  si, di
                mov  offset4, si
                add  di, ax
                mov  offset5, di
                and  bx, 3
                mov  bl, byte ptr XColumnMask[bx]
                mov  mask4n5, bx

                cmp  diametereven, 1
                jne  @@TopAndBottomPlotsOverlap
                rol  bl, 1
                jnc  @@TopAndBottomPlotsOverlap
                inc  si
                inc  di

             @@TopAndBottomPlotsOverlap:

                mov  offset0, si
                mov  offset1, di
                mov  mask0n1, bx
                mov  bx, XW

                mov  dx, VSeg
                mov  ds, dx

                mov  dx, SEQU_ADDR
                mov  al, $02
                out  dx, al
                inc  dx

                mov  si, Diameter
                inc  si

                mov  cx, si
                neg  cx
                add  cx, 2
                mov  error, cx

                xor  cx, cx
                mov  ah, byte ptr Col
                jmp  @@CircleCalc

             @@NoAdvance:

                mov  al, byte ptr mask0n1
                out  dx, al
                mov  di, offset0
                mov  [di], ah
                rol  al, 1
                mov  byte ptr mask0n1, al
                adc  di, 0
                mov  offset0, di
                mov  di, offset1
                mov  [di], ah
                ror  al, 1
                adc  di, 0
                mov  offset1, di

                mov  al, byte ptr mask2n3
                out  dx, al
                mov  di, offset2
                mov  [di], ah
                sub  di, bx
                mov  offset2, di
                mov  di, offset3
                mov  [di], ah
                add  di, bx
                mov  offset3, di

                mov  al, byte ptr mask4n5
                out  dx, al
                mov  di, offset4
                mov  [di], ah
                ror  al, 1
                mov  byte ptr mask4n5, al
                sbb  di, 0
                mov  offset4, di
                mov  di, offset5
                mov  [di], ah
                rol  al, 1
                sbb  di, 0
                mov  offset5, di

                mov  al, byte ptr mask6n7
                out  dx, al
                mov  di, offset6
                mov  [di], ah
                sub  di, bx
                mov  offset6, di
                mov  di, offset7
                mov  [di], ah
                add  di, bx
                mov  offset7, di

                jmp  @@CircleCalc

             @@Advance:

                mov  al, byte ptr mask0n1
                out  dx, al
                mov  di, offset0
                mov  [di], ah
                rol  al, 1
                mov  byte ptr mask0n1, al
                adc  di, bx
                mov  offset0, di
                mov  di, offset1
                mov  [di], ah
                ror  al, 1
                adc  di, 0
                sub  di, bx
                mov  offset1, di

                mov  al, byte ptr mask2n3
                out  dx, al
                mov  di, offset2
                mov  [di], ah
                ror  al, 1
                mov  byte ptr mask2n3, al
                sbb  di, bx
                mov  offset2, di
                mov  di, offset3
                mov  [di], ah
                rol  al, 1
                sbb  di, 0
                add  di, bx
                mov  offset3, di

                mov  al, byte ptr mask4n5
                out  dx, al
                mov  di, offset4
                mov  [di], ah
                ror  al, 1
                mov  byte ptr mask4n5, al
                sbb  di, 0
                add  di, bx
                mov  offset4, di
                mov  di, offset5
                mov  [di], ah
                rol  al, 1
                sbb  di, bx
                mov  offset5, di

	        mov  al, byte ptr mask6n7
                out  dx, al
                mov  di, offset6
                mov  [di], ah
                rol  al, 1
                mov  byte ptr mask6n7, al
                adc  di, 0
                sub  di, bx
                mov  offset6, di
                mov  di, offset7
                mov  [di], ah
                ror  al, 1
                adc  di, bx
                mov  offset7, di

             @@CircleCalc:

                add  cx, 2
                mov  di, error
                add  di, cx
                inc  di
                jl   @@CircleNoError
                cmp  cx, si
                ja   @@FleeFlyFlowFum
                sub  si, 2
                sub  di, si
                mov  error, di
                jmp  @@Advance

             @@CircleNoError:

                mov  error, di
                jmp  @@NoAdvance

             @@FleeFlyFlowFum:

                pop  ds

             end;
        end;

{******}

        procedure GFX_CirclePoint(X, Y, Xc, Yc : integer; Col : byte); assembler;

        var XXCp, XXCm, XYCp, XYCm, YXCp, YXCm, YYCp, YYCm : integer;

        asm
           mov  cx, X    { store in registers for faster access }
           mov  dx, Y    { instead of accessing memory each time }

           mov  bx, XC   { store XC in bx - cut down on mem access }

           mov  ax, bx   { XXCp := XC+X; }
           add  ax, cx
           mov  XXCp, ax

           mov  ax, bx   { XXCm := XC-X; }
           sub  ax, cx
           mov  XXCm, ax

           mov  ax, bx   { XYCp := XC+Y; }
           add  ax, dx
           mov  XYCp, ax

           mov  ax, bx   { XYCm := XC-Y; }
           sub  ax, dx
           mov  XYCm, ax

           mov  bx, YC   { store YC in bx - cut down on mem access }

           mov  ax, bx   { YXCp := YC+X; }
           add  ax, cx
           mov  YXCp, ax

           mov  ax, bx   { YXCm := YC-X; }
           sub  ax, cx
           mov  YXCm, ax

           mov  ax, bx   { YYCp := YC+Y; }
           add  ax, dx
           mov  YYCp, ax

           mov  ax, bx   { YYCm := YC-Y; }
           sub  ax, dx
           mov  YYCm, ax

           mov  bl, Col
           xor  bh, bh
           mov  dx, XXCp

           mov  ax, VGA_SEG
           mov  es, ax
           mov  di, VGA_OFS
           add  di, dx
           mov  ax, YYCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb

           sub  di, di
           add  di, dx
           mov  ax, YYCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb

           mov  dx, XXCm

           sub  di, di
           add  di, dx
           mov  ax, YYCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           sub  di, di
           add  di, dx
           mov  ax, YYCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           mov  dx, XYCp

           sub  di, di
           add  di, dx
           mov  ax, YXCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           sub  di, di
           add  di, dx
           mov  ax, YXCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           mov  dx, XYCm
           sub  di, di
           add  di, dx
           mov  ax, YXCp
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }

           sub  di, di
           add  di, dx
           mov  ax, YXCm
           xchg al, ah
           add  di, ax
           shr  ax, 2
           add  di, ax
           mov  ax, bx
           stosb              { Plot pixel }
        end;

{******}

        procedure GFX_LCircle(xpos, ypos, radius : integer; Col : byte);

        var X, Y, D : integer;

        begin
             asm
                sub  ax, ax         { X := 0; }
                mov  X, ax
                mov  ax, Radius     { Y := Radius }
                mov  Y, ax

                shl  ax, 1          { (2 * Radius) }
                mov  bx, 03H
                sub  bx, ax         { D := 3 - (2 * Radius) }
                mov  D, bx
             end;

             while (X < Y) Do
             begin
                  GFX_CirclePoint(X, Y, xpos, ypos, Col);

                  if (D < 0) then
                  asm
                     mov  ax, X  { 4 * X }
                     shl  ax, 2
                     add  ax, 6  { (4*X)+6 }
                     mov  bx, D
                     add  bx, ax
                     mov  D, bx
                  end {D := D + (4 * X) + 6}
                  else
                  begin
                       asm
                          mov  ax, X    { (X-Y) }
                          mov  bx, Y
                          sub  ax, bx
                          shl  ax, 2    { 4*(X-Y) }
                          add  ax, 10
                          mov  bx, D
                          add  bx, ax
                          mov  D, bx { D := D + 4*(X-Y) + 10; }
                          dec  Y
                       end;
                  end;

                  Inc(X);
             end;

             if (X = Y) then GFX_CirclePoint(X, Y, xpos, ypos, Col);

        end;

{******}

begin
     case GFXInfo.MemModel of

                  GFXMEM_PACKED8 : GFX_LCircle(xpos, ypos, radius, Col);
                  GFXMEM_PLANAR8 : GFX_XCircle(xpos, ypos, radius, Col);
     end;
end;

{******}

procedure GFX_FilledCircle(xpos, ypos, radius : integer; Col : byte);

var x, y, pos      : integer;
    x1, x2, y1, y2 : integer;

begin
     pos:= 1;

     repeat
          x:= round(radius * CosTbl^[pos]);
          y:= round(radius * SinTbl^[pos]);

          GFX_HLine(xpos - X, xpos + X, ypos + Y, Col);
          GFX_HLine(xpos - X, xpos + X, ypos - Y, Col);
          GFX_HLine(xpos - Y, xpos + Y, ypos + X, Col);
          GFX_HLine(xpos - Y, xpos + Y, ypos - X, Col);

          Inc(Pos);
     until (Pos > 450);
end;


{******}

procedure GFX_CreateCOSFile(FileName : string);

var Temp    : PCircleLUT;
    COSFile : TBufStream;
    Deg     : real;
    Loop    : word;

begin
     GetMem(Temp, sizeof(TCircleLUT));
     if Temp = nil then exit;
     Deg:= 0;
     for Loop:= 1 to 450 do
     begin
          Deg:= Deg + 0.1;
          Temp^[Loop]:= Cos(GFX_Rad(Deg));
     end;

     COSFile.Init(FileName, stCreate, 2048);
     COSFile.Write(Temp^, sizeof(TCircleLUT));
     COSFile.Done;

     FreeMem(Temp, sizeof(TCircleLUT));
end;

{******}

procedure GFX_CreateSINFile(FileName : string);

var Temp    : PCircleLUT;
    SINFile : TBufStream;
    Deg     : real;
    Loop    : word;

begin
     GetMem(Temp, sizeof(TCircleLUT));
     if Temp = nil then exit;

     Deg:= 0;
     for Loop:= 1 to 450 do
     begin
          Deg:= Deg + 0.1;
          Temp^[Loop]:= Sin(GFX_Rad(Deg));
     end;

     SINFile.Init(FileName, stCreate, 2048);
     SINFile.Write(Temp^, sizeof(TCircleLUT));
     SINFile.Done;

     FreeMem(Temp, sizeof(TCircleLUT));
end;

{******}

function GFX_LoadCOSTable(FileName : string) : boolean;

var COSFile : TBufStream;

begin
     if (Assigned(COSTbl)) and (U_FileExists(FileName)) then
     begin
          COSFile.Init(FileName, stOpenRead, 2048);
          if COSFile.Status <> stOK then
          begin
               GFX_LoadCOSTable:= false;
               exit;
          end;
          COSFile.Read(COSTbl^, sizeof(TCircleLUT));
          if COSFile.Status <> stOK then
          begin
               COSFile.Done;
               GFX_LoadCOSTable:= false;
               exit;
          end;
          COSFile.Done;
          GFX_LoadCOSTable:= true;
     end
     else GFX_LoadCOSTable:= false;
end;

{******}

function GFX_LoadSINTable(FileName : string) : boolean;

var SINFile : TBufStream;

begin
     if (Assigned(SINTbl)) and (U_FileExists(FileName)) then
     begin
          SINFile.Init(FileName, stOpenRead, 2048);
          if SINFile.Status <> stOK then
          begin
               GFX_LoadSINTable:= false;
               exit;
          end;
          SINFile.Read(SINTbl^, sizeof(TCircleLUT));
          if SINFile.Status <> stOK then
          begin
               GFX_LoadSINTable:= false;
               exit;
          end;
          SINFile.Done;
          GFX_LoadSINTable:= true;
     end
     else GFX_LoadSINTable:= false;
end;

{******}

procedure GFX_Triangle(x1, y1, x2, y2, x3, y3 : word; Col : byte);

var slope1,
    slope2,
    slope3,
    aux,
    bendCheck,
    z1,
    first,
    z2,
    last,
    z3,
    z4          : word;
    sign1,
    sign2,
    sign3,
    firstSign,
    lastSign,
    auxSign,
    bend : byte;

begin
     z1:= 0;
     z2:= 0;
     z3:= 0;
     z4:= 0;

     if (y2 < y1) then
     begin
	  asm
             mov  ax, y1
	     mov  bx, y2
	     mov  y1, bx
	     mov  y2, ax
	     mov  ax, x1
	     mov  bx, x2
	     mov  x1, bx
	     mov  x2, ax
	  end;
     end;

     if (y3 < y2) then
     begin
	  asm
             mov  ax, y3
	     mov  bx, y2
	     mov  y3, bx
	     mov  y2, ax
	     mov  ax, x3
	     mov  bx, x2
	     mov  x3, bx
	     mov  x2, ax
	  end;
     end;

     if (y2 < y1) then
     begin
	  asm
             mov  ax, y1
             mov  bx, y2
             mov  y1, bx
             mov  y2, ax
             mov  ax, x1
             mov  bx, x2
             mov  x1, bx
             mov  x2, ax
          end;
     end;


     if (y1 = y2) then
     begin
	  Inc(y2);
	  Inc(y3);
     end;

     if (y2 = y3) then Inc(y3);

     if (x1 < x2) then
     begin
          asm
             mov  dx, 0
	     mov  ax, x2
             sub  ax, x1
             mov  ah, al
             mov  al, 0
             mov  bx, y2
             sub  bx, y1
             div  bx
             mov  slope2, ax
             mov  byte ptr sign2, 0
          end;
     end
     else
     begin
          asm
             mov  dx, 0
             mov  ax, x1
             sub  ax, x2
             mov  ah, al
             mov  al, 0
             mov  bx, y2
             sub  bx, y1
             div  bx
             mov  slope2, ax
             mov  byte ptr sign2, 1
          end;
     end;

     if (x3 < x1) then
     begin
	  asm
             mov  dx, 0
             mov  ax, x1
             sub  ax, x3
             mov  ah, al
             mov  al, 0
             mov  bx, y3
             sub  bx, y1
             div  bx
             mov  slope1, ax
             mov  byte ptr sign1, 1
          end;
     end
     else
     begin
          asm
             mov  dx, 0
             mov  ax, x3
             sub  ax, x1
             mov  ah, al
             mov  al, 0
             mov  bx, y3
             sub  bx, y1
             div  bx
             mov  slope1, ax
             mov  byte ptr sign1, 0
          end;
     end;

     if (x3 < x2) then
     begin
	  asm
             mov  dx, 0
             mov  ax, x2
             sub  ax, x3
             mov  ah, al
             mov  al, 0
             mov  bx, y3
             sub  bx, y2
             div  bx
             mov  slope3, ax
             mov  aux, ax
             mov  byte ptr sign3, 1
             mov  byte ptr auxSign, 1
          end;
     end
     else
     begin
	  asm
             mov  dx, 0
             mov  ax, x3
             sub  ax, x2
             mov  ah, al
             mov  al, 0
             mov  bx, y3
             sub  bx, y2
             div  bx
             mov  slope3, ax
             mov  aux, ax
             mov  byte ptr sign3, 0
             mov  byte ptr auxSign, 0
          end;
     end;

     if (sign1 <> 0) then
     begin
	  if (sign2 <> 0) then
          begin
               if (slope1 > slope2) then
               begin
                    first:= slope1;
                    firstSign:= sign1;
                    last:= slope2;
                    lastSign:= sign2;
                    bend:=2;
               end
               else
               begin
                    first:= slope2;
                    firstSign:= sign2;
                    last:= slope1;
                    lastSign:= sign1;
                    bend:=1;
               end;
          end
          else
          begin
               first:= slope1;
               firstSign:= sign1;
               bend:= 2;
	       last:= slope2;
               lastSign:= sign2;
          end;
     end
     else
     begin
	  if (sign2 <> 0) then
          begin
               first:= slope2;
               firstSign:= sign2;
               bend:= 1;
	       last:= slope1;
               lastSign:= sign1;
	  end
          else
          begin
	       if (slope1 > slope2) then
               begin
	            bend:= 1;
                    first:= slope2;
                    firstSign:= sign2;
                    last:= slope1;
                    lastSign:= sign1;
               end
               else
               begin
	            bend:= 2;
                    first:= slope1;
                    firstSign:= sign1;
                    last:= slope2;
                    lastSign:= sign2;
               end;
	  end;
     end;

     bendCheck:= y3 - y2 + 1;
     x2:= 320 * y2 + x2;

     (* bx = screenpos of first line
        si = screenpos of last line
        dh = bx remainder
        dl = si remainder
        al = color                      *)

     asm
        mov  ax, VGA_SEG
        mov  es, ax

        mov  ax, [y1]
        mov  bx, ax
        shl  ax, 8
        shl  bx, 6
        add  bx, ax
        add  bx, [x1]

        mov  si, bx
        mov  cx, [y3]
        sub  cx, [y1]
        mov  al, [Col]
        mov  ah, al
        mov  dx, 0

        cmp  byte ptr firstSign, 1
        je   @l1out

        cmp  byte ptr lastSign, 1
        je   @l2out

        jmp  @pospos

@l2out:
        jmp  @negpos

@l1out:
        cmp  byte ptr lastSign, 1
        je   @l3out
        jmp  @negpos

@l3out:
        jmp  @negneg

@negpos:
        push  cx
        mov   cx, si
        sub   cx, bx
        mov   di, bx
        test  di, 01
        je    @l1np
        stosb
        dec   cx

@l1np:
        cmp  cx, 0
        jl   @l9np
        shr  cx,1
        jc   @l2np
        rep  stosw
        jmp  @l9np

@l2np:
        rep   stosw
        stosb

@l9np:
        add bx, 320
        sub dh, byte ptr first
        sbb bx, [first+1]
        add si, 320
        add dl, byte ptr last
        adc si, [last+1]
        pop cx
        cmp cx, bendCheck
        je  @l1npo

@negPosEntry:
        loop  @negpos
        jmp   @Exit

@l1npo:
        jmp   @switchdirs

@posneg:
        push  cx
        mov   cx, si
        sub   cx, bx
        mov   di, bx
        test  di, 01
        je    @l1pn
        stosb
        dec   cx

@l1pn:
        cmp  cx, 0
        jl   @l9pn
        shr  cx, 1
        jc   @l2pn
        rep  stosw
        jmp  @l9pn

@l2pn:
        rep stosw
        stosb

@l9pn:
        add  bx, 320
        add  dh, byte ptr first
        adc  bx, [first+1]
        add  si, 320
        sub  dl, byte ptr last
        sbb  si, [last+1]

        pop  cx
        cmp  cx, bendCheck
        je   @l1pno

@posnegEntry:
        loop  @posneg
        jmp   @Exit

@l1pno:
        jmp   @switchdirs

@negneg:
        push  cx
        mov   cx, si
        sub   cx, bx
        mov   di, bx
        test  di, 01
        je    @l1nn
        stosb
        dec   cx

@l1nn:
        cmp   cx, 0
        jl    @l9nn
        shr   cx, 1
        jc    @l2nn
        rep   stosw
        jmp   @l9nn

@l2nn:
        rep   stosw
        stosb

@l9nn:
        add  bx, 320
        sub  dh, byte ptr first
        sbb  bx, [first+1]
        add  si, 320
        sub  dl, byte ptr last
        sbb  si, [last+1]

        pop  cx
        cmp  cx, bendCheck
        je   @l1nno

@negnegEntry:
        loop @negneg
        jmp  @Exit

@l1nno:
        jmp  @switchdirs

@pospos:
        push  cx
        mov   cx, si
        sub   cx, bx
        mov   di, bx
        test  di, 01
        je    @l1pp
        stosb
        dec   cx

@l1pp:
        cmp  cx, 0
        jl   @l9pp
        shr  cx, 1
        jc   @l2pp
        rep  stosw
        jmp  @l9pp

@l2pp:
        rep   stosw
        stosb

@l9pp:
        add  bx, 320
        add  dh, byte ptr first
        adc  bx, [first+1]
        add  si, 320
        add  dl, byte ptr last
        adc  si, [last+1]

        pop  cx
        cmp  cx, bendCheck
        je   @switchdirs

@posposEntry:
        loop  @pospos
        jmp   @Exit

@switchdirs:
        push  ax
        cmp   byte ptr bend, 2
        je    @l9sw

        mov   y2, si
        mov   bx, x2
        mov   ax, aux
        mov   first, ax
        mov   ah, auxSign
        mov   firstSign, ah
        jmp   @l8sw

@l9sw:
        mov   si, x2
        mov   ax, aux
        mov   last, ax
        mov   ah, auxSign
        mov   lastSign, ah

@l8sw:
        pop   ax
        cmp   byte ptr firstSign, 1
        je    @l1sw
        cmp   byte ptr lastSign, 1
        je    @l2sw
        jmp   @posposEntry

@l2sw:
        jmp   @posnegEntry

@l1sw:
        cmp   byte ptr lastSign, 1
        je    @l3sw
        jmp   @posnegEntry

@l3sw:
        jmp   @negnegEntry

@Exit:
     end;
end;

{******}

procedure GFX_LoadBitmap(SPFName : string; var Bitmap : TBitmap; var Palette : TPalette);

var SPFFile : file;
    SPFHdr  : TSPFHeader;
    W, H    : word;
    Temp    : pointer;
    Data    : pointer;

begin
     SErrorCode:= 0;
     with Bitmap do
     begin
          Width:= 0;
          Height:= 0;
          Size:= 0;
          Data:= nil;
     end;
     if not U_FileExists(SPFName) then exit;

     Assign(SPFFile, SPFName);
     Reset(SPFFile, 1);
     BlockRead(SPFFile, SPFHdr, sizeof(TSPFHeader));

     if SPFHdr.Width * SPFHdr.Height > 65535 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPINVALIDSIZE;
          exit;
     end;

     Bitmap.Width:= SPFHdr.Width;
     Bitmap.Height:= SPFHdr.Height;
     Bitmap.Size:= SPFHdr.Width * SPFHdr.Height;

     GetMem(Bitmap.Data, Bitmap.Size);
     if not Assigned(Bitmap.Data) then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPMEMERROR;
          exit;
     end;

     BlockRead(SPFFile, Bitmap.Data^, Bitmap.Size);
     if IOResult <> 0 then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPFILEERROR;
          exit;
     end;

     Close(SPFFile);
     Palette:= TPalette(SPFHdr.Palette);

     if (GFXInfo.MemModel = GFXMEM_PLANAR8) and (Bitmap.Width mod 4 = 0) then
     begin
          GetMem(Temp, Bitmap.Size);
          if not Assigned(Temp) then
          begin
               SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPXCNVERROR;
               exit;
          end;

          Bitmap.Width:= Bitmap.Width shr 2;
          W:= Bitmap.Width;
          H:= Bitmap.Height;
          Data:= Bitmap.Data;

          asm
             push ds
             les  di, Temp
             lds  si, Data
             mov  bx, [W]
             mov  ax, [H]
             mul  bx
             mov  dx, si
             mov  bl, 3

          @PLANES:

             mov  cx, ax

          @PIXELS:

             movsb
             add  si, 3
             dec  cx
             jnz  @PIXELS

             inc  dx
             mov  si, dx
             dec  bl
             jns  @PLANES

             pop  ds
          end;
          FreeMem(Bitmap.Data, Bitmap.Size);
          Bitmap.Data:= Temp;
     end;
end;

{******}

procedure GFX_LoadBitmapResource(ResName, Key : string; var Bitmap : TBitmap; var Palette : TPalette);

var ResFile : TResourceFile;
    PicRes  : PPicRes;
    W, H    : word;
    Temp    : pointer;
    Data    : pointer;

begin
     SErrorCode:= 0;
     with Bitmap do
     begin
          Width:= 0;
          Height:= 0;
          Size:= 0;
          Data:= nil;
     end;
     if not U_FileExists(ResName) then exit;

     ResFile.Init(New(PBufStream, Init(ResName, stOpenRead, 2048)));
     PicRes:= PPicRes(ResFile.Get(Key));
     if PicRes = nil then
     begin
          ResFile.Done;
          exit;
     end;

     ResFile.Stream^.Seek(ResFile.Stream^.GetPos - (PicRes^.PicInfo.Width *
                                                    PicRes^.PicInfo.Height));

     if PicRes^.PicInfo.Width * PicRes^.PicInfo.Height > 65535 then
     begin
          PicRes^.Done;
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPINVALIDSIZE;
          exit;
     end;

     Bitmap.Width:= PicRes^.PicInfo.Width;
     Bitmap.Height:= PicRes^.PicInfo.Height;
     Bitmap.Size:= Bitmap.Width * Bitmap.Height;

     Bitmap.Data:= MemAllocSeg(Bitmap.Size);
     if not Assigned(Bitmap.Data) then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPMEMERROR;
          exit;
     end;

     ResFile.Stream^.Read(Bitmap.Data^, Bitmap.Size);
     ResFile.Done;

     Palette:= TPalette(PicRes^.PicInfo.Palette);
     PicRes^.Done;

     if (GFXInfo.MemModel = GFXMEM_PLANAR8) and (Bitmap.Width mod 4 = 0) then
     begin
          GetMem(Temp, Bitmap.Size);
          if not Assigned(Temp) then
          begin
               SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPXCNVERROR;
               exit;
          end;

          Bitmap.Width:= Bitmap.Width shr 2;
          W:= Bitmap.Width;
          H:= Bitmap.Height;
          Data:= Bitmap.Data;

          asm
             push ds
             les  di, Temp
             lds  si, Data
             mov  bx, [W]
             mov  ax, [H]
             mul  bx
             mov  dx, si
             mov  bl, 3

          @PLANES:

             mov  cx, ax

          @PIXELS:

             movsb
             add  si, 3
             dec  cx
             jnz  @PIXELS

             inc  dx
             mov  si, dx
             dec  bl
             jns  @PLANES

             pop  ds
          end;
          FreeMem(Bitmap.Data, Bitmap.Size);
          Bitmap.Data:= Temp;
     end;
end;

{******}

procedure GFX_PutBitmap(X, Y : word; Bitmap : TBitmap);

var W, H, SWidth, Trim : word;
    TrimH, TrimW       : word;
    Plane              : byte;
    S                  : pointer;
    VSeg, VOfs         : word;

begin
     SErrorCode:= 0;

     if X > GFXInfo.MaxX then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDXVAL;
          exit;
     end;

     if Y > GFXInfo.MaxY then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;

     if Bitmap.Data = nil then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPNODATA;
          exit;
     end;


     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : begin
                                        if X > GFXInfo.MaxX then X:= GFXInfo.MaxX;
                                        W:= Bitmap.Width;
                                        if (W - 1 + X) > GFXInfo.MaxX then
                                        begin
                                             W:= GFXInfo.MaxX - X + 1;
                                             Trim:= Bitmap.Width - W;
                                        end
                                        else Trim:= 0;

                                        if Y > GFXInfo.MaxY then Y:= GFXInfo.MaxY;
                                        H:= Bitmap.Height;
                                        if (H - 1 + Y) > GFXInfo.MaxY then H:= GFXInfo.MaxY - Y + 1;

                                        SWidth:= GFXInfo.MaxX + 1;
                                        S:= Bitmap.Data;

                                        asm
                                           mov  ax, VGA_SEG
                                           mov  es, ax

                                           push ds
                                           lds  si, S

                                        @REDRAW:

                                           mov  ax, [Y]
                                           mov  di, ax
                                           shl  di, 8
                                           shl  ax, 6
                                           add  di, ax
                                           add  di, [X]

                                           mov  dx, [H]
                                           mov  ax, [W]
                                           cld

                                        @DRAWLOOP:

                                           push di
                                           mov  cx, ax

                                        @LINELOOP:

                                           cmp  cx, 4
                                           jle  @ONE
                                           db   $66
                                           movsw
                                           sub  cx, 4
                                           jnz  @LINELOOP

                                        @ONE:

                                           movsb
                                           dec  cx
                                           jnz  @LINELOOP

                                        @NEXTLINE:

                                           pop  di
                                           dec  dx
                                           jz   @EXIT
                                           add  di, [SWidth]
                                           add  si, [Trim]
                                           jmp  @DRAWLOOP

                                        @EXIT:

                                           pop  ds

                                        end;
                                   end;

                  GFXMEM_PLANAR8 : begin
                                        if X > GFXInfo.MaxX then X:= GFXInfo.MaxX;
                                        W:= Bitmap.Width;

                                        X:= X and $FFFC;
                                        if ((W shl 2) - 1 + X) > GFXInfo.MaxX then
                                        begin
                                             W:= (GFXInfo.MaxX - X + 1) div 4;
                                             TrimW:= (Bitmap.Width - W);
                                        end
                                        else TrimW:= 0;
                                        X:= X shr 2;

                                        if Y > GFXInfo.MaxY then Y:= GFXInfo.MaxY;
                                        H:= Bitmap.Height;
                                        if (H - 1 + Y) > GFXInfo.MaxY then
                                        begin
                                             H:= GFXInfo.MaxY - Y + 1;
                                             TrimH:= (Bitmap.Height - H) * Bitmap.Width;
                                        end
                                        else TrimH:= 0;

                                        SWidth:= XByteWidth;
                                        S:= Bitmap.Data;
                                        VSeg:= VGA_SEG;
                                        VOfs:= VGA_OFS;
                                        Plane:= 0;

                                        asm
                                           mov  ax, VGA_SEG
                                           mov  es, ax

                                           push ds
                                           lds  si, S

                                        @PLANELOOP:

                                           mov   ax, [Y]
                                           mov   bx, [SWidth]
                                           mul   bx

                                           mov   di, [X]
                                           add   di, ax
                                           add   di, [VOfs]

                                           cld

                                           mov  dx, SEQU_ADDR
                                           mov  ah, 1
                                           mov  cl, [Plane]
                                           shl  ah, cl
                                           mov  al, $02
                                           out  dx, ax

                                           mov  dx, [H]
                                           mov  ax, [W]

                                        @DRAWLOOP:

                                           push di
                                           mov  cx, ax

                                        @LINELOOP:

                                           cmp  cx, 2
                                           jle  @ONE
                                           movsw
                                           sub  cx, 2
                                           jnz  @LINELOOP

                                        @ONE:

                                           movsb
                                           dec  cx
                                           jnz  @LINELOOP

                                        @NEXTLINE:

                                           pop  di
                                           dec  dx
                                           jz   @NEXTPLANE
                                           add  di, [SWidth]
                                           add  si, [TrimW]
                                           jmp  @DRAWLOOP

                                        @NEXTPLANE:

                                           add  si, [TrimW]
                                           add  si, [TrimH]
                                           inc  [Plane]
                                           cmp  [Plane], 4
                                           jl   @PLANELOOP

                                        @EXIT:

                                           pop  ds

                                        end;
                                   end;

   end;
end;

{******}

procedure GFX_PutMaskedBitmap(X, Y : word; Bitmap : TBitmap);

var W, H, SWidth, Trim : word;
    TrimH, TrimW       : word;
    Plane              : byte;
    S                  : pointer;
    VSeg, VOfs         : word;

begin
     SErrorCode:= 0;
     if X > GFXInfo.MaxX then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDXVAL;
          exit;
     end;

     if Y > GFXInfo.MaxY then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;

     if Bitmap.Data = nil then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPNODATA;
          exit;
     end;

     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : begin
                                        W:= Bitmap.Width;
                                        if (W - 1 + X) > GFXInfo.MaxX then
                                        begin
                                             W:= GFXInfo.MaxX - X + 1;
                                             Trim:= Bitmap.Width - W;
                                        end
                                        else Trim:= 0;

                                        H:= Bitmap.Height;
                                        if (H - 1 + Y) > GFXInfo.MaxY then H:= GFXInfo.MaxY - Y + 1;

                                        SWidth:= GFXInfo.MaxX + 1;
                                        S:= Bitmap.Data;

                                        asm
                                           mov  ax, VGA_SEG
                                           mov  es, ax

                                           push ds
                                           lds  si, S

                                        @REDRAW:

                                           mov  ax, [Y]
                                           mov  di, ax
                                           shl  di, 8
                                           shl  ax, 6
                                           add  di, ax
                                           add  di, [X]

                                           mov  dx, [H]
                                           mov  ax, [W]
                                           cld

                                        @DRAWLOOP:

                                           push di
                                           mov  cx, ax

                                        @LINELOOP:

                                           mov  bl, byte ptr [si]
                                           or   bl, bl
                                           jnz  @PAINT

                                        @NOPAINT:

                                           inc  si
                                           inc  di
                                           dec  cx
                                           jnz  @LINELOOP
                                           jmp  @NEXTLINE

                                        @PAINT:

                                           movsb
                                           dec  cx
                                           jnz  @LINELOOP

                                        @NEXTLINE:

                                           pop  di
                                           dec  dx
                                           jz   @EXIT
                                           add  di, [SWidth]
                                           add  si, [Trim]
                                           jmp  @DRAWLOOP

                                        @EXIT:

                                           pop  ds

                                        end;
                                   end;

                  GFXMEM_PLANAR8 : begin
                                        W:= Bitmap.Width;

                                        X:= X and $FFFC;
                                        if ((W shl 2) - 1 + X) > GFXInfo.MaxX then
                                        begin
                                             W:= (GFXInfo.MaxX - X + 1) div 4;
                                             TrimW:= (Bitmap.Width - W);
                                        end
                                        else TrimW:= 0;
                                        X:= X shr 2;

                                        H:= Bitmap.Height;
                                        if (H - 1 + Y) > GFXInfo.MaxY then
                                        begin
                                             H:= GFXInfo.MaxY - Y + 1;
                                             TrimH:= (Bitmap.Height - H) * Bitmap.Width;
                                        end
                                        else TrimH:= 0;

                                        SWidth:= XByteWidth;
                                        S:= Bitmap.Data;
                                        VSeg:= VGA_SEG;
                                        VOfs:= VGA_OFS;
                                        Plane:= 0;

                                        asm
                                           mov  ax, VGA_SEG
                                           mov  es, ax

                                           push ds
                                           lds  si, S

                                        @PLANELOOP:

                                           mov   ax, [Y]
                                           mov   bx, [SWidth]
                                           mul   bx

                                           mov   di, [X]
                                           add   di, ax
                                           add   di, [VOfs]

                                           cld

                                           mov  dx, SEQU_ADDR
                                           mov  ah, 1
                                           mov  cl, [Plane]
                                           shl  ah, cl
                                           mov  al, $02
                                           out  dx, ax

                                           mov  dx, [H]
                                           mov  ax, [W]

                                        @DRAWLOOP:

                                           push di
                                           mov  cx, ax

                                        @LINELOOP:

                                           mov  bl, byte ptr [si]
                                           or   bl, bl
                                           jnz  @PAINT

                                        @NOPAINT:
                                           inc  si
                                           inc  di
                                           dec  cx
                                           jnz  @LINELOOP
                                           jmp  @NEXTLINE

                                        @PAINT:
                                           movsb
                                           dec  cx
                                           jnz  @LINELOOP

                                        @NEXTLINE:

                                           pop  di
                                           dec  dx
                                           jz   @NEXTPLANE
                                           add  di, [SWidth]
                                           add  si, [TrimW]
                                           jmp  @DRAWLOOP

                                        @NEXTPLANE:

                                           add  si, [TrimW]
                                           add  si, [TrimH]
                                           inc  [Plane]
                                           cmp  [Plane], 4
                                           jl   @PLANELOOP

                                        @EXIT:

                                           pop  ds

                                        end;
                                   end;

     end;
end;

{******}

procedure GFX_GetBitmap(X, Y, W, H : word; var Bitmap : TBitmap);

var SWidth             : word;
    Plane              : byte;
    S                  : pointer;
    VSeg, VOfs         : word;

begin
     SErrorCode:= 0;
     if X > GFXInfo.MaxX then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDXVAL;
          exit;
     end;

     if Y > GFXInfo.MaxY then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_INVALIDYVAL;
          exit;
     end;

     case GFXInfo.MemModel of
                  GFXMEM_PACKED8 : begin
                                        if (W - 1 + X) > GFXInfo.MaxX then W:= GFXInfo.MaxX - X + 1;
                                        Bitmap.Width:= W;

                                        if (H - 1 + Y) > GFXInfo.MaxY then H:= GFXInfo.MaxY - Y + 1;
                                        Bitmap.Height:= H;

                                        if W * H > 65535 then
                                        begin
                                             SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPINVALIDSIZE;
                                             exit;
                                        end;

                                        Bitmap.Size:= W * H;

                                        GetMem(Bitmap.Data, Bitmap.Size);
                                        if not Assigned(Bitmap.Data) then
                                        begin
                                             SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPNODATA;
                                             exit;
                                        end;

                                        SWidth:= GFXInfo.MaxX + 1;
                                        S:= Bitmap.Data;

                                        asm
                                           mov  ax, VGA_SEG
                                           push ds
                                           mov  ds, ax

                                           les  di, S

                                        @REDRAW:

                                           mov  ax, [Y]
                                           mov  si, ax
                                           shl  si, 8
                                           shl  ax, 6
                                           add  si, ax
                                           add  si, [X]

                                           mov  dx, [H]
                                           mov  ax, [W]
                                           cld

                                        @DRAWLOOP:

                                           push si
                                           mov  cx, ax

                                        @LINELOOP:

                                           cmp  cx, 4
                                           jle  @ONE
                                           db   $66
                                           movsw
                                           sub  cx, 4
                                           jnz  @LINELOOP

                                        @ONE:
                                           movsb
                                           dec  cx
                                           jnz  @LINELOOP

                                        @NEXTLINE:

                                           pop  si
                                           dec  dx
                                           jz   @EXIT
                                           add  si, [SWidth]
                                           jmp  @DRAWLOOP

                                        @EXIT:

                                           pop  ds

                                        end;

                                   end;

                  GFXMEM_PLANAR8 : begin
                                        X:= X and $FFFC;
                                        if (W - 1 + X) > GFXInfo.MaxX then W:= (GFXInfo.MaxX - X + 1);

                                        Bitmap.Width:= W shr 2;
                                        X:= X shr 2;

                                        if (H - 1 + Y) > GFXInfo.MaxY then H:= GFXInfo.MaxY - Y + 1;

                                        Bitmap.Height:= H;

                                        if W * H > 65535 then
                                        begin
                                             SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPINVALIDSIZE;
                                             exit;
                                        end;

                                        Bitmap.Size:= W * H;

                                        GetMem(Bitmap.Data, Bitmap.Size);
                                        if not Assigned(Bitmap.Data) then
                                        begin
                                             SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPNODATA;
                                             exit;
                                        end;
                                        W:= W shr 2;

                                        SWidth:= XByteWidth;
                                        S:= Bitmap.Data;
                                        VSeg:= VGA_SEG;
                                        VOfs:= VGA_OFS;
                                        Plane:= 0;

                                        asm
                                           mov  ax, VGA_SEG
                                           push ds
                                           mov  ds, ax

                                           les  di, S

                                        @PLANELOOP:

                                           mov   ax, [Y]
                                           mov   bx, [SWidth]
                                           mul   bx

                                           mov   si, [X]
                                           add   si, ax
                                           add   si, [VOfs]

                                           cld

                                           mov  dx, GC_ADDR
                                           mov  ah, [Plane]
                                           mov  al, $04
                                           out  dx, ax

                                           mov  dx, [H]
                                           mov  ax, [W]

                                        @DRAWLOOP:

                                           push si
                                           mov  cx, ax

                                        @LINELOOP:

                                           cmp  cx, 2
                                           jle  @ONE
                                           movsw
                                           sub  cx, 2
                                           jnz  @LINELOOP

                                        @ONE:
                                           movsb
                                           dec  cx
                                           jnz  @LINELOOP

                                        @NEXTLINE:

                                           pop  si
                                           dec  dx
                                           jz   @NEXTPLANE
                                           add  si, [SWidth]
                                           jmp  @DRAWLOOP

                                        @NEXTPLANE:

                                           inc  [Plane]
                                           cmp  [Plane], 4
                                           jl   @PLANELOOP

                                        @EXIT:

                                           pop  ds

                                        end;
                                   end;

     end;
end;


{******}

procedure GFX_SaveBitmap(SPFName : string; Bitmap : TBitmap; var Pal : TPalette; Ovr : boolean);

var SPFFile : file;
    SPFHdr  : TSPFHeader;
    W, H    : word;
    Temp    : pointer;
    Data    : pointer;

begin
     SErrorCode:= 0;
     if Bitmap.Data = nil then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPNODATA;
          exit;
     end;

     if (U_FileExists(SPFName) and not Ovr) then
     begin
          SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPFILEEXISTS;
          exit;
     end;

     if GFXInfo.MemModel = GFXMEM_PACKED8 then SPFHdr.Width:= Bitmap.Width
                                          else SPFHdr.Width:= Bitmap.Width shl 2;
     SPFHdr.Height:= Bitmap.Height;
     TPalette(SPFHdr.Palette):= Pal ;
     SPFHdr.SPFID:= 'SPF';
     SPFHdr.Version:= $0100;

     Assign(SPFFile, SPFName);
     Rewrite(SPFFile, 1);
     BlockWrite(SPFFile, SPFHdr, sizeof(TSPFHeader));

     case GFXInfo.MemModel of

                  GFXMEM_PACKED8 : begin
                                        BlockWrite(SPFFile, Bitmap.Data^, Bitmap.Size);
                                        if IOResult <> 0 then
                                        begin
                                             SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPFILEERROR;
                                             exit;
                                        end;
                                   end;

                  GFXMEM_PLANAR8 : begin
                                        GetMem(Temp, Bitmap.Size);
                                        if not Assigned(Temp) then
                                        begin
                                             SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPXCNVERROR;
                                             exit;
                                        end;

                                        W:= Bitmap.Width;
                                        H:= Bitmap.Height;
                                        Data:= Bitmap.Data;

                                        asm
                                           push ds

                                           les  di, Temp
                                           lds  si, Data

                                           mov  bx, [W]
                                           mov  ax, [H]
                                           mul  bx

                                           mov  dx, di
                                           mov  bl, 3

                                        @PLANES:

                                           mov  cx, ax

                                        @PIXELS:

                                           movsb
                                           add  di, 3
                                           dec  cx
                                           jnz  @PIXELS

                                           inc  dx
                                           mov  di, dx
                                           dec  bl
                                           jns  @PLANES

                                           pop  ds
                                        end;
                                        FreeMem(Bitmap.Data, Bitmap.Size);
                                        Bitmap.Data:= Temp;

                                        BlockWrite(SPFFile, Bitmap.Data^, Bitmap.Size);
                                        if IOResult <> 0 then
                                        begin
                                             SErrorCode:= (word(ERR_GFX) shl 8) + ERR_GFX_BMPFILEERROR;
                                             exit;
                                        end;
                                   end;
     end;
     Close(SPFFile);
end;

{*** TEMPORARY PROCEDURE ***}

procedure GFX_ScaleBitmap(x, y, w, h : word; Src : TBitmap; Dest : byte);

var jx, jy, depth, temp, oldw, oldh, data : word;

begin
     if Src.Data = nil then exit;
     if Dest > GFXInfo.MaxPage then Dest:= GFXInfo.MaxPage;
     GFX_SetActivePage(Dest);
     oldw:= Src.Width;
     oldh:= Src.Height;
     data:= Seg(Src.Data^);

     asm
        mov  ax, VGA_SEG
        mov  es, ax
        push ds
        mov  ax, Data
        mov  ds, ax
        mov  depth, 0
        dec  h

        xor  dx, dx
        mov  ax, oldw
        shl  ax, 6
        mov  bx, w
        div  bx
        shl  ax, 2
        mov  jx, ax

        xor  dx, dx
        mov  ax, oldh
        shl  ax, 6
        mov  bx, h
        div  bx
        shl  ax, 2
        mov  jy, ax

        xor  cx, cx

     @L2:

        push  cx
        mov   ax, depth
        add   ax, jy
        mov   depth, ax

        xor   dx, dx
        mov   ax, depth
        shr   ax, 8
        mov   bx, oldw
        mul   bx
        mov   temp, ax

        mov   di, y
        add   di, cx
        mov   bx, di
        shl   di, 8
        shl   bx, 6
        add   di, bx
        add   di, x

        mov   cx, w
        xor   bx, bx
        mov   dx, jx
        mov   ax, temp

     @L1:

        mov   si, bx
        shr   si, 8
        add   si, ax

        movsb
        add   bx, dx

        dec   cx
        jnz   @L1

        pop   cx
        inc   cx
        cmp   cx, h
        jl    @L2

        pop  ds
     end;
     GFX_SetVisualPage(Dest);
end;

{******}

function GFX_BitmapCollide(X1, Y1 : word; B1 : TBitmap;
                           X2, Y2 : word; B2 : TBitmap) : boolean;

var Bottom1, Bottom2, Right1, Right2 : word;
    Temp                             : boolean;

begin
     Bottom1:= Y1 + B1.Height;
     Bottom2:= Y2 + B2.Height;
     Right1:= X1 + B1.Width;
     Right2:= X2 + B2.Width;

     if ((X1 >= Right2) or (X2 >= Right1) or (Y1 >= Bottom2) or (Y2 >= Bottom1)) then
          Temp:= false
     else
          Temp:= true;

     GFX_BitmapCollide:= Temp;
end;

{******}

begin
     OldExit:= ExitProc;
     ExitProc:= @GFX_Exit;
     GFXActive:= false;
     SErrorCode:= 0;
     with GFXInfo do
     begin
          Chip:= 0;
          ChipID:= 0;
          VRAM:= 0;
          MemModel:= 0;
          MaxPage:= 0;
          CurPage:= 0;
          VisPage:= 0;
          MinX:= 0;
          MaxX:= 0;
          MinY:= 0;
          MaxY:= 0;
     end;
     TextMode:= LastMode;
     GFX_TextCursor(false);
     GFX_TextCursor(true);
end.
