unit SGFXFlic;

(* Information
   

   Program Title : SGFX graphics library FLI/FLC support.
   External name : SGFXFLIC.TPU
   Version       : 1.1.
   Start date    : 21/6/96.
   Last update   : 17/11/96.
   Author        : Rob Anderton.
   Description   : FLI/FLC playing routines (92nd attempt to get it right).
                   NOTE: Only supports 320x200 animations.

*)

interface

uses SGFX386, Objects, Crt, Memory;

{******}

type FLIHeader = record
                       Size      : longint;
                       Magic     : word;
                       Frames    : word;
                       Width     : word;
                       Height    : word;
                       Depth     : word;
                       Flags     : word;
                       Speed     : word;
                       Reserved  : array[1..110] of byte;
                 end;

     FLCHeader = record
                       Size      : longint;
                       Magic     : word;
                       Frames    : word;
                       Width     : word;
                       Height    : word;
                       Depth     : word;
                       Flags     : word;
                       Speed     : longint;
                       Reserved1 : word;
                       Created   : longint;
                       Creator   : longint;
                       Updated   : longint;
                       Updater   : longint;
                       AspectX   : word;
                       AspectY   : word;
                       Reserved2 : array[1..38] of byte;
                       oframe1   : longint;
                       oframe2   : longint;
                       Reserved3 : array[1..40] of byte;
                 end;

     FrameHeader = record
                       Size     : longint;
                       Magic    : word;
                       Chunks   : word;
                       Reserved : array[1..8] of byte;
                   end;

     ChunkHeader = record
                       Size      : longint;
                       ChunkType : word;
                   end;

{******}

const {*** Magic words ***}
      FLI_MAGIC    = $AF11;
      FLC_MAGIC    = $AF12;
      PREFIX_MAGIC = $F100;
      FRAME_MAGIC  = $F1FA;

      {*** Chunk types ***}
      FLI_COLOUR256 = 4;
      FLI_SS2       = 7;
      FLI_COLOUR    = 11;
      FLI_LC        = 12;
      FLI_BLACK     = 13;
      FLI_BRUN      = 15;
      FLI_COPY      = 16;
      FLI_PSTAMP    = 18;


      {*** Postage stamp constants ***}
      FPS_BRUN      = 15;
      FPS_COPY      = 16;
      FPS_XLAT256   = 18;

{*** FLI support ***}
procedure FLI_LoadHeader(var S : TStream; var H : FLIHeader);
procedure FLI_LoadFrame(var S : TStream; Header : FLIHeader; Num : word);
procedure FLI_Play(var S : TStream; Speed, Loop : word);
procedure FLI_PlayBuf(var S : TStream; Speed, Loop : word);

{*** FLC support ***}
procedure FLC_LoadHeader(var S : TStream; var H : FLCHeader);
procedure FLC_Play(var S : TStream; Speed, Loop : word);
procedure FLC_PlayBuf(var S : TStream; Speed, Loop : word);

{******}

implementation

type TData = array[0..64000] of byte;
     PData = ^TData;


{******}

procedure FLI_LoadHeader(var S : TStream; var H : FLIHeader);

var Dummy : byte;

begin
     S.Seek(0);
     S.Read(H, sizeof(FLIHeader));
end;

{******}

procedure LoadFLIColour64(var D : Pointer);

var Packets : word;
    Skip    : byte;
    Count   : byte;
    Pal     : TRGBPalette;
    RGB     : TRGBRec;
    Loop1   : word;
    Loop2   : word;
    Idx     : byte;
    NumCols : word;
    DP      : PData;
    Index   : word;

begin
     DP:= PData(D);
     Index:= 0;

     Idx:= 0;

     WordRec(Packets).Lo:= DP^[Index];
     Inc(Index);
     WordRec(Packets).Hi:= DP^[Index];
     Inc(Index);

     if Packets > 0 then
     begin
        GFX_GetAllPal(TPalette(Pal));
        for Loop1:= 1 to Packets do
        begin
             Skip:= DP^[Index];
             Inc(Index);

             Inc(Idx, (Skip * 3));
             Count:= DP^[Index];
             Inc(Index);

             if Count = 0 then NumCols:= 256
                          else NumCols:= Count;

             for Loop2:= 1 to NumCols do
             begin
                  RGB.R:= DP^[Index];
                  Inc(Index);
                  RGB.G:= DP^[Index];
                  Inc(Index);
                  RGB.B:= DP^[Index];
                  Inc(Index);

                  Pal[Idx]:= RGB;
                  Inc(Idx);
             end;
        end;
        GFX_SetAllPal(TPalette(Pal));
     end;
end;

{******}

procedure DecodeFLILC(var D : pointer);

var SkipLines  : word;
    CountLines : word;
    StartX     : byte;
    Packets    : byte;
    SkipX      : byte;
    Size       : shortint;
    xpos, ypos : word;
    Loop1      : word;
    Loop2      : word;
    Loop3      : word;
    Data       : byte;
    DP         : PData;
    Index      : word;

begin
     Index:= 0;
     DP:= PData(D);

     WordRec(SkipLines).Lo:= DP^[Index];
     Inc(Index);
     WordRec(SkipLines).Hi:= DP^[Index];
     Inc(Index);
     WordRec(CountLines).Lo:= DP^[Index];
     Inc(Index);
     WordRec(CountLines).Hi:= DP^[Index];
     Inc(Index);

     xpos:= 0;
     ypos:= SkipLines;

     for Loop1:= 1 to CountLines do
     begin
        Packets:= DP^[Index];
        Inc(Index);

        if Packets > 0 then
          for Loop2:= 1 to Packets do
          begin
               SkipX:= DP^[Index];
               Inc(Index);
               Inc(xpos, SkipX);
               Size:= Shortint(DP^[Index]);
               Inc(Index);

               if (Size > 0) then
               begin
                    for Loop3:= 1 to Size do
                    begin
                         Data:= DP^[Index];
                         Inc(Index);
                         GFX_PutPixel(xpos, ypos, Data);
                         Inc(xpos);
                         if xpos > GFXInfo.MaxX then Xpos:= 0;
                    end;
               end
               else
               begin
                    Size:= -Size;
                    Data:= DP^[Index];
                    Inc(Index);
                    if (Size and $03) = 0 then
                    begin
                         Size:= Size div 4;
                         for Loop3:= 1 to Size do
                         begin
                            GFX_Put4Pixel(xpos, ypos, Data);
                            Inc(xpos, 4);
                         end;
                         if xpos > GFXInfo.MaxX then Xpos:= 0;
                    end
                    else
                    begin
                         if (Size and $01) = 0 then
                         begin
                              Size:= Size div 2;
                              for Loop3:= 1 to Size do
                              begin
                                   GFX_Put2Pixel(xpos, ypos, Data);
                                   Inc(xpos, 2);
                              end;
                              if xpos > GFXInfo.MaxX then Xpos:= 0;
                         end
                         else
                         begin
                              for Loop3:= 1 to Size do
                              begin
                                   GFX_PutPixel(xpos, ypos, Data);
                                   Inc(xpos);
                              end;
                              if xpos > GFXInfo.MaxX then Xpos:= 0;
                         end;
                    end;
               end;
          end;
          Inc(ypos);
          xpos:= 0;
     end;
end;


{******}

procedure DecodeFLIBRUN(var D : Pointer);

var Packets : byte;
    ypos    : word;
    xpos    : word;
    Size    : shortint;
    Data    : byte;
    Loop1   : byte;
    Loop2   : byte;
    DP      : PData;
    Index   : word;

begin
     Index:= 0;
     DP:= PData(D);
     xpos:= 0;
     for ypos:= 0 to GFXInfo.MaxY do
     begin
          Packets:= DP^[Index];
          Inc(Index);

          if Packets > 0 then
             for Loop1:= 1 to Packets do
             begin
                  Size:= ShortInt(DP^[Index]);
                  Inc(Index);
                  if (Size < 0) then
                  begin
                       Size:= -Size;
                       for Loop2:= 1 to Size do
                       begin
                            Data:= DP^[Index];
                            Inc(Index);
                            GFX_PutPixel(xpos, ypos, Data);
                            Inc(xpos);
                            if xpos > GFXInfo.MaxX then xpos:= 0;
                       end;
                  end
                  else
                  begin
                       Data:= DP^[Index];
                       Inc(Index);

                       if (Size and $01) = 1 then
                       begin
                            GFX_PutPixel(xpos, ypos, Data);
                            Inc(xpos);
                       end;
                       if (Size and $02) = 2 then
                       begin
                            GFX_Put2Pixel(xpos, ypos, Data);
                            Inc(xpos, 2);
                       end;
                       if xpos > GFXInfo.MaxX then Xpos:= 0;

                       if (Size > 0) then
                       begin
                            Size:= Size shr 2;
                            for Loop2:= 1 to Size do
                            begin
                                 GFX_Put4Pixel(xpos, ypos, Data);
                                 Inc(xpos, 4);
                                 if xpos > GFXInfo.MaxX then Xpos:= 0;
                            end;
                       end;
                  end;
             end;
     end;
end;

{******}

procedure FLI_LoadFrame(var S : TStream; Header : FLIHeader; Num : word);

var CurFrame   : word;
    FrameHdr   : FrameHeader;
    ChunkHdr   : ChunkHeader;
    ChunkData  : pointer;
    ChunkSize  : longint;
    CLoop      : word;
    VideoMem   : pointer;

begin
     if Num > Header.Frames then exit;

     S.Seek(sizeof(FLIHeader));

     CurFrame:= 0;
     while (CurFrame < Num) do
     begin
          S.Read(FrameHdr, sizeof(FrameHeader));
          if FrameHdr.Magic = FRAME_MAGIC then Inc(CurFrame);
          if CurFrame < Num then S.Seek(S.GetPos + FrameHdr.Size - 16);
     end;

     if FrameHdr.Chunks = 0 then exit;

     for CLoop:= 1 to FrameHdr.Chunks do
     begin
          S.Read(ChunkHdr, sizeof(ChunkHeader));
          ChunkSize:= ChunkHdr.Size - 6;
          if ChunkSize > 0 then
          begin
               ChunkData:= MemAllocSeg(ChunkSize);
               S.Read(ChunkData^, ChunkSize);
          end;

          case ChunkHdr.ChunkType of

               FLI_COLOUR  : LoadFLIColour64(ChunkData);

               FLI_BLACK   : GFX_Cls(0);

               FLI_LC      : DecodeFLILC(ChunkData);

               FLI_BRUN    : DecodeFLIBRUN(ChunkData);

               FLI_COPY    : begin
                                  GFX_CopyToVRAM(Seg(ChunkData^));
                             end;

          else begin
                    {Ignore unknown chunks}
               end;
          end;
          if ChunkSize > 0 then FreeMem(ChunkData, ChunkSize);
     end;
end;

{******}

procedure FLI_Play(var S : TStream; Speed, Loop : word);

var CurFrame   : word;
    FLIHdr     : FLIHeader;
    FrameHdr   : FrameHeader;
    ChunkHdr   : ChunkHeader;
    ChunkData  : pointer;
    ChunkSize  : longint;
    CLoop      : word;
    StartPos   : longint;
    FLoop      : word;
    VideoMem   : pointer;

begin
   S.Seek(0);
   S.Read(FLIHdr, sizeof(FLIHeader));
   FLoop:= 0;

   repeat
      CurFrame:= 1;
      while (CurFrame <= FLIHdr.Frames) and not Keypressed do
      begin
        S.Read(FrameHdr, sizeof(FrameHeader));
        if FrameHdr.Magic = FRAME_MAGIC then Inc(CurFrame);

        if FrameHdr.Chunks <> 0 then

          for CLoop:= 1 to FrameHdr.Chunks do
          begin
               S.Read(ChunkHdr, sizeof(ChunkHeader));
               ChunkSize:= ChunkHdr.Size - 6;
               if ChunkSize > 0 then
               begin
                    ChunkData:= MemAllocSeg(ChunkSize);
                    S.Read(ChunkData^, ChunkSize);
               end;

               case ChunkHdr.ChunkType of

                    FLI_COLOUR  : LoadFLIColour64(ChunkData);

                    FLI_BLACK   : begin
                                       GFX_Cls(0);
                                  end;

                    FLI_LC      : DecodeFLILC(ChunkData);

                    FLI_BRUN    : DecodeFLIBRUN(ChunkData);

                    FLI_COPY    : begin
                                       GFX_CopyToVRAM(Seg(ChunkData^));
                                   end;

               else begin
                         {Ignore unknown chunks}
                    end;
               end;
               if ChunkSize > 0 then FreeMem(ChunkData, ChunkSize);
          end;
          Delay(Speed);
      end;
      S.Seek(sizeof(FLIHeader));
      Inc(FLoop);
      if Loop = 0 then FLoop:= 1;
   until (FLoop = Loop) or KeyPressed;
end;

{******}

procedure FLI_PlayBuf(var S : TStream; Speed, Loop : word);

var CurFrame   : word;
    FLIHdr     : FLIHeader;
    FrameHdr   : FrameHeader;
    ChunkHdr   : ChunkHeader;
    ChunkData  : pointer;
    ChunkSize  : longint;
    CLoop      : word;
    StartPos   : longint;
    FLoop      : word;
    VideoMem   : pointer;
    Page       : boolean;

begin
   Page:= false;
   S.Seek(0);
   S.Read(FLIHdr, sizeof(FLIHeader));
   FLoop:= 0;

   repeat
      CurFrame:= 1;
      while (CurFrame <= FLIHdr.Frames) and not Keypressed do
      begin
        GFX_SetActivePage(byte(Page));
        S.Read(FrameHdr, sizeof(FrameHeader));
        if FrameHdr.Magic = FRAME_MAGIC then Inc(CurFrame);

        if FrameHdr.Chunks <> 0 then

          for CLoop:= 1 to FrameHdr.Chunks do
          begin
               S.Read(ChunkHdr, sizeof(ChunkHeader));
               ChunkSize:= ChunkHdr.Size - 6;
               if ChunkSize > 0 then
               begin
                    ChunkData:= MemAllocSeg(ChunkSize);
                    S.Read(ChunkData^, ChunkSize);
               end;

               case ChunkHdr.ChunkType of

                    FLI_COLOUR  : LoadFLIColour64(ChunkData);

                    FLI_BLACK   : begin
                                       GFX_Cls(0);
                                  end;

                    FLI_LC      : DecodeFLILC(ChunkData);

                    FLI_BRUN    : DecodeFLIBRUN(ChunkData);

                    FLI_COPY    : begin
                                       GFX_CopyToVRAM(Seg(ChunkData^));
                                   end;

               else begin
                         {Ignore unknown chunks}
                    end;
               end;
               if ChunkSize > 0 then FreeMem(ChunkData, ChunkSize);
          end;
          GFX_WaitRetrace;
          GFX_SetVisualPage(byte(Page));
          Page:= not Page;
          Delay(Speed);
      end;
      S.Seek(sizeof(FLIHeader));
      Inc(FLoop);
      if Loop = 0 then FLoop:= 1;
   until (FLoop = Loop) or KeyPressed;
end;


{******}

procedure FLC_LoadHeader(var S : TStream; var H : FLCHeader);

begin
     S.Seek(0);
     S.Read(H, sizeof(FLCHeader));
end;

{******}

procedure LoadFLCColour256(var D : Pointer);

var Packets : word;
    Skip    : byte;
    Count   : byte;
    Pal     : TRGBPalette;
    RGB     : TRGBRec;
    Loop1   : word;
    Loop2   : word;
    Idx     : byte;
    NumCols : word;
    DP      : PData;
    Index   : word;

begin
     DP:= PData(D);
     Index:= 0;

     Idx:= 0;

     WordRec(Packets).Lo:= DP^[Index];
     Inc(Index);
     WordRec(Packets).Hi:= DP^[Index];
     Inc(Index);

     if Packets > 0 then
     begin
        GFX_GetAllPal(TPalette(Pal));
        for Loop1:= 1 to Packets do
        begin
             Skip:= DP^[Index];
             Inc(Index);

             Inc(Idx, (Skip * 3));
             Count:= DP^[Index];
             Inc(Index);

             if Count = 0 then NumCols:= 256
                          else NumCols:= Count;

             for Loop2:= 1 to NumCols do
             begin
                  RGB.R:= ((DP^[Index] * 63) div 255);
                  Inc(Index);
                  RGB.G:= ((DP^[Index] * 63) div 255);
                  Inc(Index);
                  RGB.B:= ((DP^[Index] * 63) div 255);
                  Inc(Index);

                  Pal[Idx]:= RGB;
                  Inc(Idx);
             end;
        end;
        GFX_SetAllPal(TPalette(Pal));
     end;
end;

{******}

procedure DecodeFLCSS2(var D : pointer);

var DP      : PData;
    Index   : word;
    Change  : word;
    Loop1   : word;
    Loop2   : word;
    Loop3   : word;
    Packets : integer;
    Skip    : byte;
    Size    : shortint;
    DataH   : byte;
    DataL   : byte;
    xpos    : word;
    ypos    : word;

begin
     DP:= PData(D);
     Index:= 0;
     WordRec(Change).Lo:= DP^[Index];
     Inc(Index);
     WordRec(Change).Hi:= DP^[Index];
     Inc(Index);
     xpos:= 0;
     ypos:= 0;

     if Change > 0 then
     begin
          for Loop1:= 1 to Change do
          begin
               WordRec(Packets).Lo:= DP^[Index];
               Inc(Index);
               WordRec(Packets).Hi:= DP^[Index];
               Inc(Index);

               if (Packets < 0) then
               begin
                    Packets:= -Packets - 1;
                    Inc(ypos, Packets);
               end
               else
               begin
                    for Loop2:= 1 to Packets do
                    begin
                         Skip:= DP^[Index];
                         Inc(Index);
                         Inc(xpos, skip);
                         if xpos > GFXInfo.MaxX then xpos:= GFXInfo.MaxX;
                         Size:= shortint(DP^[Index]);
                         Inc(Index);

                         if (Size > 0) then
                         begin
                              for Loop3:= 1 to Size do
                              begin
                                   DataL:= DP^[Index];
                                   Inc(Index);
                                   DataH:= DP^[Index];
                                   Inc(Index);
                                   GFX_PutPixel(xpos, ypos, DataL);
                                   Inc(xpos);
                                   if xpos > GFXInfo.MaxX then xpos:= GFXInfo.MaxX;
                                   GFX_PutPixel(xpos, ypos, DataH);
                                   Inc(xpos);
                                   if xpos > GFXInfo.MaxX then xpos:= GFXInfo.MaxX;
                              end;
                         end
                         else
                         begin
                              DataL:= DP^[Index];
                              Inc(Index);
                              DataH:= DP^[Index];
                              Inc(Index);

                              Size:= -Size;
                              for Loop3:= 1 to Size do
                              begin
                                   GFX_PutPixel(xpos, ypos, DataL);
                                   Inc(xpos);
                                   if xpos > GFXInfo.MaxX then xpos:= GFXInfo.MaxX;
                                   GFX_PutPixel(xpos, ypos, DataH);
                                   Inc(xpos);
                                   if xpos > GFXInfo.MaxX then xpos:= GFXInfo.MaxX;

                              end;
                         end;
                    end;
               end;
               Inc(ypos);
               xpos:= 0;
          end;
     end;
end;

{******}

procedure FLC_Play(var S : TStream; Speed, Loop : word);

var CurFrame   : word;
    FLCHdr     : FLCHeader;
    FrameHdr   : FrameHeader;
    ChunkHdr   : ChunkHeader;
    ChunkData  : pointer;
    ChunkSize  : longint;
    CLoop      : word;
    StartPos   : longint;
    FLoop      : word;

begin
   S.Seek(0);
   S.Read(FLCHdr, sizeof(FLCHeader));
   if (FLCHdr.Width > GFXInfo.MaxX + 1) or (FLCHdr.Height > GFXInfo.MaxY + 1) or
      (FLCHdr.Magic <> FLC_MAGIC) then exit;

   S.Seek(FLCHdr.oframe1);
   FLoop:= 0;

   repeat
      CurFrame:= 1;
      while (CurFrame <= FLCHdr.Frames) and not Keypressed do
      begin
        S.Read(FrameHdr, sizeof(FrameHeader));
        if FrameHdr.Magic = FRAME_MAGIC then Inc(CurFrame);

        if FrameHdr.Chunks <> 0 then

          for CLoop:= 1 to FrameHdr.Chunks do
          begin
               S.Read(ChunkHdr, sizeof(ChunkHeader));
               ChunkSize:= ChunkHdr.Size - 6;
               if ChunkSize > 0 then
               begin
                    ChunkData:= MemAllocSeg(ChunkSize);
                    S.Read(ChunkData^, ChunkSize);
               end;

               case ChunkHdr.ChunkType of

                    FLI_COLOUR256 : LoadFLCColour256(ChunkData);

                    FLI_COLOUR    : LoadFLIColour64(ChunkData);

                    FLI_BLACK     : begin
                                        GFX_Cls(0);
                                    end;

                    FLI_LC        : DecodeFLILC(ChunkData);

                    FLI_BRUN      : DecodeFLIBRUN(ChunkData);

                    FLI_COPY      : begin
                                       GFX_CopyToVRAM(Seg(ChunkData^));
                                    end;

                    FLI_SS2       : begin
                                       DecodeFLCSS2(ChunkData);
                                    end;

               else begin
                         {ignore unknown chunks}
                    end;
               end;
               if ChunkSize > 0 then FreeMem(ChunkData, ChunkSize);
          end;
          Delay(Speed);
      end;
      S.Seek(FLCHdr.oframe2);
      Inc(FLoop);
      if Loop = 0 then FLoop:= 1;
   until (FLoop = Loop) or KeyPressed;
end;

{******}

procedure FLC_PlayBuf(var S : TStream; Speed, Loop : word);

var CurFrame   : word;
    FLCHdr     : FLCHeader;
    FrameHdr   : FrameHeader;
    ChunkHdr   : ChunkHeader;
    ChunkData  : pointer;
    ChunkSize  : longint;
    CLoop      : word;
    StartPos   : longint;
    FLoop      : word;
    Page       : boolean;

begin
   Page:= false;

   S.Seek(0);
   S.Read(FLCHdr, sizeof(FLCHeader));
   if (FLCHdr.Width > GFXInfo.MaxX + 1) or (FLCHdr.Height > GFXInfo.MaxY + 1) or
      (FLCHdr.Magic <> FLC_MAGIC) then exit;

   S.Seek(FLCHdr.oframe1);
   FLoop:= 0;

   repeat
      CurFrame:= 1;
      while (CurFrame <= FLCHdr.Frames) and not Keypressed do
      begin
        GFX_SetActivePage(word(Page));
        S.Read(FrameHdr, sizeof(FrameHeader));
        if FrameHdr.Magic = FRAME_MAGIC then Inc(CurFrame);

        if FrameHdr.Chunks <> 0 then

          for CLoop:= 1 to FrameHdr.Chunks do
          begin
               S.Read(ChunkHdr, sizeof(ChunkHeader));
               ChunkSize:= ChunkHdr.Size - 6;
               if ChunkSize > 0 then
               begin
                    ChunkData:= MemAllocSeg(ChunkSize);
                    S.Read(ChunkData^, ChunkSize);
               end;

               case ChunkHdr.ChunkType of

                    FLI_COLOUR256 : LoadFLCColour256(ChunkData);

                    FLI_COLOUR    : LoadFLIColour64(ChunkData);

                    FLI_BLACK     : begin
                                        GFX_Cls(0);
                                    end;

                    FLI_LC        : DecodeFLILC(ChunkData);

                    FLI_BRUN      : DecodeFLIBRUN(ChunkData);

                    FLI_COPY      : begin
                                       GFX_CopyToVRAM(Seg(ChunkData^));
                                    end;

                    FLI_SS2       : begin
                                       DecodeFLCSS2(ChunkData);
                                    end;

               else begin
                         {ignore unknown chunks}
                    end;
               end;
               if ChunkSize > 0 then FreeMem(ChunkData, ChunkSize);
          end;
          GFX_WaitRetrace;
          GFX_SetVisualPage(word(Page));
          Page:= not Page;
          if Speed <> 0 then Delay(Speed);
      end;
      S.Seek(FLCHdr.oframe2);
      Inc(FLoop);
      if Loop = 0 then FLoop:= 1;
   until (FLoop = Loop) or KeyPressed;
end;


{******}

{procedure LoadPostageStamp(var S : TStream; var B : TBitmap);

begin
end;                                                          }

{******}

end.