unit SGFXFONT;

(* Information
   

   Program Title : SGFX386 font module.
   External name : SGFXFONT.TPU
   Version       : 1.1.
   Start date    : 19/6/96
   Last update   : 16/12/96
   Author        : Rob Anderton.
   Description   : SGFX386 font support for PASKAL graphics library and
                   VGL 2.0 bitmapped fonts. Last update fixed MAJOR bug in
                   VGL_WidthInPixels (it was returning a nonsense value).

*)

interface

{******}

type vglFont = record
                     First  : char; {First character in font}
                     Last   : char; {Last character in font}
                     Flags  : byte; {Special flags}
                     WsWid  : byte; {White space width in pixels}
                     VSpace : byte; {Space between rows in pixels}
                     HSpace : byte; {Space between characters in pixels}

                     ChWid  : array[0..255] of byte;     {Width of each character}
                     ChHit  : array[0..255] of byte;     {Height of each character}
                     ChBas  : array[0..255] of shortint; {Baseline of each character}
                     ChOff  : array[0..255] of word;     {Offset of character data}

                     Data   : word; {Size of bitmap data in bytes}
               end;

      vglFontInfo   = record
                            FontHeader    : vglFont;
                            FontColour    : byte;
                            FontBold      : boolean;
                            FontItalic    : boolean;
                            FontUnderline : boolean;
                            FontULColour  : byte;
                            FontULPos     : shortint;
                            FontULThick   : byte;
                            FontShadow    : boolean;
                            FontShColour  : byte;
                            FontData      : pointer;
                      end;
{******}

const FONT_UPPERCASE = 1; {Flag indicating font contains only upper case}

{******}

procedure VGL_LoadFont(FileName : string; var Font : vglFontInfo);

procedure VGL_SetColour(var Font : vglFontInfo; Col : byte);
procedure VGL_SetBold(var Font : vglFontInfo; Bold : boolean);
procedure VGL_SetItalic(var Font : vglFontInfo; Italic : boolean);

procedure VGL_SetUnderline(var Font : vglFontInfo; Underline : boolean);
procedure VGL_SetUnderlineColour(var Font : vglFontInfo; Col : byte);
procedure VGL_SetUnderlinePos(var Font : vglFontInfo; Pos : shortint);
procedure VGL_SetUnderlineThickness(var Font : vglFontInfo; Thick : byte);

procedure VGL_SetShadow(var Font : vglFontInfo; Shadow : boolean);
procedure VGL_SetShadowColour(var Font : vglFontInfo; Col : byte);

procedure VGL_WriteChar(X, Y : word; C : char; Font : vglFontInfo);
procedure VGL_WriteString(X, Y : word; S : string; Font : vglFontInfo);

function  VGL_WidthInPixels(Font : vglFontInfo; S : string) : word;

{******}

implementation

{******}

uses SGFX386, Objects, Memory;

{******}

procedure VGL_LoadFont(FileName : string; var Font : vglFontInfo);

var F : TBufStream;

begin
     F.Init(FileName, stOpenRead, 1024);
     F.Read(Font.FontHeader, sizeof(Font.FontHeader));

     Font.FontData:= MemAllocSeg(Font.FontHeader.Data);
     if Font.FontData = nil then exit;

     F.Read(Font.FontData^, Font.FontHeader.Data);
     F.Done;

     with Font do
     begin
          FontColour:= 15;
          FontBold:= false;
          FontItalic:= false;
          FontUnderline:= false;
          FontULColour:= 0;
          FontULPos:= 0;
          FontULThick:= 1;
          FontShadow:= false;
          FontShColour:= 0;
     end;
end;

{******}

procedure VGL_SetColour(var Font : vglFontInfo; Col : byte);

begin
     Font.FontColour:= Col;
end;

{******}

procedure VGL_SetBold(var Font : vglFontInfo; Bold : boolean);

begin
     Font.FontBold:= Bold;
end;

{******}

procedure VGL_SetItalic(var Font : vglFontInfo; Italic : boolean);

begin
     Font.FontItalic:= Italic;
end;

{******}

procedure VGL_SetUnderline(var Font : vglFontInfo; Underline : boolean);

begin
     Font.FontUnderline:= Underline;
end;

{******}

procedure VGL_SetUnderlineColour(var Font : vglFontInfo; Col : byte);

begin
     Font.FontULColour:= Col;
end;

{******}

procedure VGL_SetUnderlinePos(var Font : vglFontInfo; Pos : shortint);

begin
     Font.FontULPos:= Pos;
end;

{******}

procedure VGL_SetUnderlineThickness(var Font : vglFontInfo; Thick : byte);

begin
     Font.FontULThick:= Thick;
end;

{******}

procedure VGL_SetShadow(var Font : vglFontInfo; Shadow : boolean);

begin
     Font.FontShadow:= Shadow;
end;

{******}

procedure VGL_SetShadowColour(var Font : vglFontInfo; Col : byte);

begin
     Font.FontShColour:= Col;
end;

{******}

procedure VGL_PutChar(X, Y : word; var C : char; Font : vglFontInfo);

var xpos, ypos, bit, oy, h, w, rw, o, oo, wid, hit, tx : word;
    ChrNum, l, ix                                      : byte;
    Off                                                : ^byte;

begin
     ix:= 8;
     if (Font.FontHeader.Flags and FONT_UPPERCASE <> 0) and
        (C >= 'a') and (C <= 'z') then C:= UpCase(C);

     ChrNum:= Ord(C);

     if (C < Font.FontHeader.First) or (C > Font.FontHeader.Last) then exit;
     if Font.FontHeader.ChWid[ChrNum] = 0 then exit;

     h:= Font.FontHeader.ChHit[ChrNum];
     w:= Font.FontHeader.ChWid[ChrNum];

     if Font.FontItalic = true then oo:= h shr 1
                               else oo:= 0;

     o:= oo;
     rw:= w + o;
     if Font.FontBold then Inc(rw);
     if Font.FontShadow then Inc(rw);

     if (x + rw > GFXInfo.MaxX) then exit;
     if (y > GFXInfo.MaxY) then exit;

     Off:= Ptr(Seg(Font.FontData^), Font.FontHeader.ChOff[ChrNum]);
     oy:= y - Font.FontHeader.ChBas[ChrNum] + 1;

     wid:= rw;
     hit:= h;
     if Font.FontShadow = true then Inc(hit);

     for ypos:= 0 to (h - 1) do
     begin
          bit:= 1;
          for xpos:= 0 to (w - 1) do
          begin
               if bit = 256 then
               begin
                    bit:= 1;
                    Off:= Ptr(Seg(Off^), Ofs(Off^) + 1);
               end;
               if (Off^ and bit) <> 0 then
               begin
                    if Font.FontItalic then
                        tx:= xpos + ix
                    else
                        tx:= xpos;

                    GFX_PutPixel(x + tx, oy + ypos, Font.FontColour);
                    if Font.FontShadow then
                    begin
                        GFX_PutPixel(x + tx + 1, oy + ypos + 1, Font.FontShColour);
                    end;
                    if Font.FontBold then
                    begin
                         GFX_PutPixel(x + tx + 1, oy + ypos, Font.FontColour);
                         if Font.FontShadow then
                            GFX_PutPixel(x + tx + 2, oy + ypos + 1, Font.FontShColour);
                    end;
               end;
               bit:= bit shl 1;
          end;
          Off:= Ptr(Seg(Off^), Ofs(Off^) + 1);
          if (Font.FontItalic) and (ypos and 1 <> 0) then Dec(ix);
     end;
end;

{******}

procedure VGL_WriteChar(X, Y : word; C : char; Font : vglFontInfo);

var P, M, L : word;

begin
     P:= X;
     VGL_PutChar(P, Y, C, Font);
     with Font.FontHeader do
     begin
          if (C > First) and (C < Last)
          then M:= ChWid[Ord(C)] + HSpace
          else M:= WSWid + HSpace;
          if Font.FontBold = true then Inc(M);
          if Font.FontShadow = true then Inc(M);
          if Font.FontItalic = true then Inc(M);
     end;
     Inc(P, M);
     if Font.FontUnderline = true then
     begin
          for L:= 0 to (Font.FontULThick - 1) do
          begin
               GFX_HLine(x, P, y + Font.FontULPos + L, Font.FontULColour);
          end;
     end;
end;

{******}

procedure VGL_WriteString(X, Y : word; S : string; Font : vglFontInfo);

var L : byte;
    P : word;
    M : word;

begin
     P:= X;

     for L:= 1 to Length(S) do
     begin
          VGL_PutChar(P, Y, S[L], Font);
          with Font.FontHeader do
          begin
               if (S[L] > First) and (S[L] < Last)
               then M:= ChWid[Ord(S[L])] + HSpace
               else M:= WSWid + HSpace;
               if Font.FontBold = true then Inc(M);
               if Font.FontShadow = true then Inc(M);
               if Font.FontItalic = true then Inc(M);
          end;
          Inc(P, M);
     end;
     if Font.FontUnderline = true then
     begin
          for L:= 0 to (Font.FontULThick - 1) do
          begin
               GFX_HLine(x, P, y + Font.FontULPos + L, Font.FontULColour);
          end;
     end;
end;

{******}

function VGL_WidthInPixels(Font : vglFontInfo; S : string) : word;

var Temp : word;
    L    : byte;

begin
     Temp:= 0;
     for L:= 1 to Length(S) do
     begin
          if (S[L] > Font.FontHeader.First) and
             (S[L] < Font.FontHeader.Last) and (S[L] <> ' ') then
          begin
               Inc(Temp, Font.FontHeader.ChWid[Ord(S[L])] +
                         Font.FontHeader.HSpace);
          end
          else
          begin
               Inc(Temp, Font.FontHeader.WSWid + Font.FontHeader.HSpace);
          end;
     end;
     VGL_WidthInPixels:= Temp;
end;

{******}

begin
end.