unit SIRQ;

(* Information
   

   Program Title : Programmable interrupt controller (PIC) support.
   External name : SIRQ.TPU
   Version       : 1.0
   Start date    : 28/8/96
   Last update   : 28/8/96
   Author        : Rob Anderton.
   Description   : PIC support for IRQ handling, from PC-Intern IRQUTIL unit.

*)

interface

{*** IRQ controller port addresses ***}

const MASTER_PIC = $20; {Base address of Master-PIC}
      SLAVE_PIC  = $A0; {Base address of Slave-PIC}
      IRQ_MASK   = $01; {Offset to masking port}

{*** IRQ Commands ***}

const EOI                 = $20; {Unspecified End of Interrupt}
      MASTER_FIRST_VECTOR = $08; {Software vectors of the hardware}
      SLAVE_FIRST_VECTOR  = $70; {Interrupts}

{*** IRQ routines ***}

procedure IRQ_Enable(IRQ : integer);
procedure IRQ_Disable(IRQ : integer);

procedure IRQ_SendEOI(IRQ : integer);

function  IRQ_SetHandler(IRQ : integer; Handler : pointer) : pointer;
function  IRQ_ReadMask(Controller : integer) : byte;

function  IRQ_ReadISR(Controller : integer) : byte;
function  IRQ_ReadIRR(Controller : integer) : byte;

{******}

implementation

uses DOS;

{******}

procedure IRQ_Enable(IRQ : integer);

var PortNum : integer;

begin
     if IRQ <= 7 then PortNum:= MASTER_PIC
                 else PortNum:= SLAVE_PIC;

     PortNum:= PortNum + IRQ_MASK;
     IRQ:= IRQ and $0007;

     Port[PortNum]:= Port[PortNum] and not (1 shl IRQ);
end;

{******}

procedure IRQ_Disable(IRQ : integer);

var PortNum : integer;

begin
     if IRQ <= 7 then PortNum:= MASTER_PIC
                 else PortNum:= SLAVE_PIC;

     PortNum:= PortNum + IRQ_MASK;
     IRQ:= IRQ and $0007;

     Port[PortNum]:= Port[PortNum] or (1 shl IRQ);
end;

{******}

procedure IRQ_SendEOI(IRQ : integer);

begin
     if IRQ > 7 then Port[SLAVE_PIC]:= EOI;
     Port[MASTER_PIC]:= EOI;
end;

{******}

function IRQ_SetHandler(IRQ : integer; Handler : pointer) : pointer;

var OldHandler : pointer;
    Vect       : integer;

begin
     if IRQ <= 7 then Vect:= (MASTER_FIRST_VECTOR + IRQ)
                 else Vect:= (SLAVE_FIRST_VECTOR + (IRQ and $7));

     IRQ_Disable(IRQ);
     asm
        cli
     end;

     GetIntVec(Vect, OldHandler);
     SetIntVec(Vect, Handler);

     asm
        sti
     end;

     if Handler <> nil then IRQ_Enable(IRQ);
                            
     IRQ_SetHandler:= OldHandler;
end;

{******}

function IRQ_ReadMask(Controller : integer) : byte;

begin
     IRQ_ReadMask:= Port[Controller + IRQ_MASK];
end;

{******}

function IRQ_ReadISR(Controller : integer ) : byte;

begin
     Port[Controller]:= $0B;
     IRQ_ReadISR:= Port[Controller];
end;

{******}

function IRQ_ReadIRR(Controller : integer) : byte;

begin
     Port[Controller]:= $0A;
     IRQ_ReadIRR:= Port[Controller];
end;

{******}

end.


