unit SJoy;

(* Information
   

   Program Title : Joystick library.
   External name : SJOY.TPU
   Version       : 1.1.
   Start date    : 13/8/96.
   Last update   : 8/9/96.
   Author        : Rob Anderton.
   Description   : Joystick detection and interfacing routines.

*)

interface

{******}

type JoystickInfoRec = record
                             Available : boolean;
                             Direction : byte;
                             Button1   : boolean;
                             Button2   : boolean;
                             XCentre   : integer;
                             YCentre   : integer;
                             XOffset   : integer;
                             YOffset   : integer;
                             XPosition : integer;
                             YPosition : integer;
                             Range     : integer;
                       end;

{*** Joystick directions ***}

const JOY_CENTRE     = 0;
      JOY_UP         = 1;
      JOY_DOWN       = 2;
      JOY_LEFT       = 3;
      JOY_RIGHT      = 4;
      JOY_UPLEFT     = 5;
      JOY_UPRIGHT    = 6;
      JOY_DOWNRIGHT  = 7;
      JOY_DOWNLEFT   = 8;

{******}

var Joysticks  : array[1..2] of JoystickInfoRec;

{******}

function  DetectJoysticks : boolean;
function  GetTotalJoysticks : byte;
function  GetInstalledJoystick(Joystick : byte) : boolean;

procedure InitJoysticks;
procedure CalibrateJoystick(Joystick : byte);
procedure GetJoystickStatus(Joystick : byte);

{******}

implementation

uses Crt, SError;

{******}

const JOY_PORT        = $201;
      JOY_DETECTDELAY = 2;
      JOY_DETECTRETRY = 2;

{******}

var JoyRange : integer;

{******}

function DetectJoysticks : boolean;

var JDelay  : longint;
    JType   : byte;
    Temp    : boolean;

begin
     JDelay:= JOY_DETECTRETRY;
     Port[JOY_PORT]:= $FF;

     repeat
           Delay(JOY_DETECTDELAY);
           Dec(JDelay);
     until ((JDelay = 0) or (Port[JOY_PORT] <> $FF));

     JType:= Port[JOY_PORT] and $0F;
     Temp:= false;

     if (JType = $00) or (JType = $04) or
        (JType = $03) or (JType = $0C) then Temp:= true;

     DetectJoysticks:= Temp;
end;

{******}

function GetTotalJoysticks : byte;

var JDelay  : word;
    JType   : byte;
    Temp    : byte;

begin
     JDelay:= JOY_DETECTRETRY;
     Port[JOY_PORT]:= $FF;

     repeat
           Delay(JOY_DETECTDELAY);
           Dec(JDelay);
     until ((JDelay = 0) or (Port[JOY_PORT] <> $FF));

     JType:= Port[JOY_PORT] and $0F;

     case JType of
          $00, $04 : Temp:= 2;
          $03, $0C : Temp:= 1;

                else Temp:= 0;
     end;

     GetTotalJoysticks:= Temp;
end;

{******}

function GetInstalledJoystick(Joystick : byte) : boolean;

var JDelay  : word;
    JType   : byte;
    Temp    : boolean;

begin
     JDelay:= JOY_DETECTRETRY;
     Port[JOY_PORT]:= $FF;

     repeat
           Delay(JOY_DETECTDELAY);
           Dec(JDelay);
     until ((JDelay = 0) or (Port[JOY_PORT] <> $FF));

     JType:= Port[JOY_PORT] and $0F;

     Temp:= false;

     case JType of
          $00, $04 : if (Joystick = 1) or (Joystick = 2) then Temp:= true;
          $03      : if (Joystick = 2) then Temp:= true;
          $0C      : if (Joystick = 1) then Temp:= true;
          else       Temp:= false;
     end;

     GetInstalledJoystick:= Temp;
end;

{******}

procedure InitJoysticks;

begin
     SErrorCode:= 0;

     if not DetectJoysticks then
     begin
          Joysticks[1].Available:= false;
          Joysticks[2].Available:= false;
          SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_NOJOYSTICKS;
          exit;
     end;

     with Joysticks[1] do
     begin
          Available:= GetInstalledJoystick(1);
          Direction:= JOY_CENTRE;
          Button1:= false;
          Button2:= false;
          if Available then CalibrateJoystick(1);
     end;

     with Joysticks[2] do
     begin
          Available:= GetInstalledJoystick(2);
          Direction:= JOY_CENTRE;
          Button1:= false;
          Button2:= false;
          if Available then CalibrateJoystick(2);
     end;
end;

{******}

procedure GetJoystickStat(Joystick : byte);

var Jx, Jy, Jb, Jr : integer;

begin
     if (Joystick < Low(Joysticks)) or (Joystick > High(Joysticks)) then
     begin
          SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_INVALIDJOYSTICK;
          exit;
     end;

     if (Joysticks[Joystick].Available = false) then
     begin
          SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_INVALIDJOYSTICK;
          exit;
     end;

     SErrorCode:= 0;
     Jr:= JoyRange;

     asm
        mov  dx, JOY_PORT
        cli
        mov  al, $FF
        out  dx, al
        xor  ax, ax
        mov  [Jx], ax
        mov  [Jy], ax
        mov  cx, [Jr]

     @RANGE_LOOP:

        in   al, dx
        shr  al, 1
        adc  [Jx], 0
        shr  al, 1
        adc  [Jy], 0
        dec  cx
        jnz  @RANGE_LOOP

        not  al
        and  ax, $0F
        mov  [Jb], ax
        sti
     end;

     with Joysticks[Joystick] do
     begin
          XPosition:= Jx;
          YPosition:= Jy;
          if Joystick = 1 then Button1:= ((Jb and 1) = 1)
                          else Button1:= (((Jb shr 1) and 1) = 1);
          if Joystick = 1 then Button2:= (((Jb shr 2) and 1) = 1)
                          else Button2:= (((Jb shr 3) and 1) = 1);
     end;
end;

{******}

procedure CalibrateJoystick(Joystick : byte);

{******}

   function MaxVal(a, b : integer) : integer;

   begin
        if (a > b) then MaxVal:= a else MaxVal:= b;
   end;

{******}

var Flag : boolean;

begin
     if (Joystick < Low(Joysticks)) or (Joystick > High(Joysticks)) then
     begin
          SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_INVALIDJOYSTICK;
          exit;
     end;

     if (Joysticks[Joystick].Available = false) then
     begin
          SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_INVALIDJOYSTICK;
          exit;
     end;

     SErrorCode:= 0;

     JoyRange:= 100;
     Flag:= false;

     while (not Flag) do
     begin
          GetJoystickStat(Joystick);
          Delay(JOY_DETECTDELAY);
          if MaxVal(Joysticks[Joystick].XPosition,
                    Joysticks[Joystick].YPosition) < JoyRange then
          begin
               Flag:= true;
          end
          else
          begin
               if (JoyRange < 10000) then
               begin
                    Inc(JoyRange, 100);
               end
               else
               begin
                    Flag:= true;
                    SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_INVALIDJOYSTICK;
                    exit;
               end;
          end;
     end;

     Inc(JoyRange, 100);
     Flag:= false;

     while (not Flag) do
     begin
          Delay(JOY_DETECTDELAY);
          GetJoystickStat(Joystick);
          Joysticks[Joystick].XCentre:= Joysticks[Joystick].XPosition;
          Joysticks[Joystick].YCentre:= Joysticks[Joystick].YPosition;
          Delay(JOY_DETECTDELAY);
          GetJoystickStat(Joystick);
          if ((Joysticks[Joystick].XCentre = Joysticks[Joystick].XPosition) and
              (Joysticks[Joystick].YCentre = Joysticks[Joystick].YPosition)) then Flag:= true;
     end;

     JoyRange:= 2 * (MaxVal(Joysticks[Joystick].XCentre, Joysticks[Joystick].YCentre));
     Joysticks[Joystick].Range:= JoyRange div 4;
     Joysticks[Joystick].XOffset:= Joysticks[Joystick].XPosition - Joysticks[Joystick].XCentre;
     Joysticks[Joystick].YOffset:= Joysticks[Joystick].YPosition - Joysticks[Joystick].YCentre;
end;

{******}

procedure GetJoystickStatus(Joystick : byte);

begin
     if (Joystick < Low(Joysticks)) or (Joystick > High(Joysticks)) then
     begin
          SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_INVALIDJOYSTICK;
          exit;
     end;

     if (Joysticks[Joystick].Available = false) then
     begin
          SErrorCode:= (word(ERR_JOY) shl 8) + ERR_JOY_INVALIDJOYSTICK;
          exit;
     end;

     GetJoystickStat(Joystick);

     with Joysticks[Joystick] do
     begin
          XOffset:= XPosition - XCentre;
          YOffset:= YPosition - YCentre;

          if ((XOffset < Range) and (XOffset > -Range)) then
          begin
               if (YOffset < -Range) then
               begin
                    Direction:= JOY_UP;
               end
               else
               begin
                    if (YOffset > Range) then
                    begin
                         Direction:= JOY_DOWN;
                    end
                    else
                    begin
                         Direction:= JOY_CENTRE;
                    end;
               end;
          end
          else
          begin
               if (XOffset > Range) then
               begin
                    if (YOffset < -Range) then
                    begin
                         Direction:= JOY_UPRIGHT;
                    end
                    else
                    begin
                         if (YOffset > Range) then
                         begin
                              Direction:= JOY_DOWNRIGHT;
                         end
                         else
                         begin
                              Direction:= JOY_RIGHT;
                         end;
                    end;
               end
               else
               begin
                    if (XOffset < -Range) then
                    begin
                         if (YOffset < -Range) then
                         begin
                              Direction:= JOY_UPLEFT;
                         end
                         else
                         begin
                              if (YOffset > Range) then
                              begin
                                   Direction:= JOY_DOWNLEFT;
                              end
                              else
                              begin
                                   Direction:= JOY_LEFT;
                              end;
                         end;
                    end;
               end;
          end;
     end;
end;

{******}

var JLoop : word;

begin
     SErrorCode:= 0;
     for JLoop:= Low(Joysticks) to High(Joysticks) do
         Joysticks[JLoop].Available:= false;
end.