unit SVoice;

(* Information
   

   Program Title : SoundBlaster digital sound for S-Library.
   External name : SVOICE.TPU
   Version       : 1.1
   Start date    : 9/9/96
   Last update   : 23/10/96
   Author        : Rob Anderton.
   Description   : Multiple sound output for S-Library.
                   Based on SMIX version 1.27 (c)1995 Ethan Brodsky.
                   This is a reworked version designed for use with my
                   S-Library units and updated sound library files.
                   Latest edition uses S-Library sound resource files.

*)

{******}

interface

{******}

type PSound = ^TSound;
     TSound = record
                    XMSHandle : word;
                    StartOfs  : longint;
                    SoundSize : longint;
              end;

{******}

const BlockLength   = 512;      {Size of digitized sound block               }
      LoadChunkSize = 8192;     {Chunk size used for loading sounds from disk}
      Voices        = 8;        {Number of available voices                  }
      SamplingRate  = 22050;    {Sampling rate for output                    }

{******}

var IntCount   : longint;  {Number of sound interrupts that have occured   }
    AutoInit   : boolean;  {Tells Auto-initialized DMA transfers are in use}
    SixteenBit : boolean;  {Tells whether 16-bit sound output is occuring  }
    VoiceCount : byte;     {Number of voices currently in use              }

{******}

procedure Voice_Init(ShareXMS : boolean);
procedure Voice_Done;

procedure Voice_InitMixing;
procedure Voice_DoneMixing;

procedure Voice_LoadFile(FileName : string; var Sound : PSound);
procedure Voice_LoadResource(ResName : string; Key : string; var Sound : PSound);
procedure Voice_Free(var Sound : PSound);

procedure Voice_Start(Sound : PSound; Index : byte; Loop : boolean);
procedure Voice_Stop(Index : byte);
function  Voice_Playing(Index : byte) : boolean;
function  Voice_GetFreeIndex : byte;

{******}

implementation

{******}

uses Crt, Objects, SXMS, SDMA, SBDSP, SRES, SUTILS, SERROR;

{******}

const BufferLength = BlockLength * 2;

{******}

var DMAChannel       : word;
    DMAMode          : byte;
    DMALength        : word;

    OldExitProc      : pointer;
    HandlerInstalled : boolean;

    SharedEMB        : boolean;
    SharedHandle     : word;
    SharedSize       : longint;

{******}

procedure Voice_IntHandler(l : longint); forward;
procedure Voice_ExitProc; far; forward;

{******}

procedure Voice_Init(ShareXMS : boolean);

begin
     if not(SBPresent) then
     begin
          SErrorCode:= (word(ERR_VOICE) shl 8) + ERR_VOICE_NOSBCARD;
          exit;
     end;

     AutoInit:= SBInfo.DSPVersion > DSP_201;
     SixteenBit:= (SBInfo.DSPVersion > DSP_4XX) and
                  (SBInfo.DMA16 <> $FFFF) and
                  (SBInfo.DMA16 > 3);


     if SixteenBit then
     begin
          DMAChannel:= SBInfo.DMA16;
          DMAMode:= SBInfo.DMA16 - 4 + $58;
     end
     else
     begin
          DMAChannel:= SBInfo.DMA8;
          if AutoInit then DMAMode:= SBInfo.DMA8 + $58
                      else DMAMode:= SBInfo.DMA8+ $48;
     end;

     if AutoInit then DMALength:= BufferLength
                 else DMALength:= BlockLength;

     SB_InitIRQHandler;
     SB_SetUserIRQ(Voice_IntHandler);
     HandlerInstalled:= true;

{     OldExitProc:= ExitProc;
     ExitProc:= @Voice_ExitProc;}

     if not(XMS_Detect) then
     begin
          SErrorCode:= (word(ERR_VOICE) shl 8) + ERR_VOICE_NOXMS;
          exit;
     end;
     if not XMMPresent then XMS_Init;

     if ShareXMS then
     begin
          SharedEMB:= true;
          SharedSize:= 0;
          SharedHandle:= XMS_EMBGetMem(SharedSize);
     end;
end;

{******}

procedure Voice_Done;

begin
     if HandlerInstalled then
     begin
          SB_SetUserIRQ(nil);
          SB_RestoreIRQHandler;
          HandlerInstalled:= false;
     end;

     SB_Reset;

     if SharedEMB then
     begin
          XMS_EMBFreeMem(SharedHandle);
          SharedEMB := false;
     end;
end;

{*** Voice control ***}

type PVoice = ^TVoice;
     TVoice = record
                    Sound  : PSound;
                    Index  : byte;
                    CurPos : longint;
                    Loop   : boolean;
              end;

var VoiceInUse : array[0..Voices - 1] of boolean;
    Voice      : array[0..Voices - 1] of TVoice;
    CurBlock   : byte;

{*** Sound buffer ***}

var SoundBlock : array[1..BlockLength + 1] of shortint;

{*** Mixing buffers ***}

type PMixingBlock = ^TMixingBlock;
     TMixingBlock = array[1..BlockLength] of integer;

var MixingBlock  : TMixingBlock;

{*** Output buffers ***}

type POut8Block   = ^TOut8Block;
     TOut8Block   = array[1..BlockLength] of byte;
     POut8Buffer  = ^TOut8Buffer;
     TOut8Buffer  = array[1..2] of TOut8Block;

     POut16Block  = ^TOut16Block;
     TOut16Block  = array[1..BlockLength] of integer;
     POut16Buffer = ^TOut16Buffer;
     TOut16Buffer = array[1..2] of TOut16Block;

var OutMemArea  : pointer;
    Out8Buffer  : POut8Buffer;
    Out16Buffer : POut16Buffer;

    BlockPtr    : array[1..2] of pointer;
    CurBlockPtr : pointer;

{*** Clipping for 8-bit output ***}

var Clip8 : array[-128 * Voices..128 * Voices] of byte;

{******}

procedure Voice_StartDAC;

var i : PlayRec;

begin
     DMA_SetChannel(DMAChannel, OutMemArea, DMALength, DMAMode);

     if SixteenBit then
     begin
          SB_SetOutputFreq(SamplingRate);
          SB_Write($B6);
          SB_Write(DSP4_MODESIGNED);
          SB_Write(Lo(BlockLength - 1));
          SB_Write(Hi(BlockLength - 1));
     end
     else
     begin
          SB_SetSpeaker(true);
          i.Stereo:= false;
          i.Bits:= 8;
          i.Freq:= SamplingRate;
          SB_SetFreq(i);
          if AutoInit then
          begin
               SB_SetDMATransferSize(BlockLength);
               SB_Write(DSP_DMAAUTODAC8);
          end
          else
          begin
               SB_Write(DSP_DMADAC8);
               SB_Write(Lo(BlockLength - 1));
               SB_Write(Hi(BlockLength - 1));
          end;
     end;
end;

{******}

procedure Voice_StopDAC;

begin
     if SixteenBit then
     begin
          SB_Write($D5);
     end
     else
     begin
          SB_Write($D0);
          SB_Write($D3);
     end;

     DMA_ClrMask(DMAChannel);
end;

{*** Loading and freeing sounds ***}

var MoveParams : TMoveParams;

{******}

procedure Voice_LoadFile(FileName : string; var Sound : PSound);

var SoundFile : TDosStream;
    SoundHdr  : TSoundHdr;
    InBuffer  : array[1..LoadChunkSize] of byte;
    Remaining : longint;

begin
     if not U_FileExists(FileName) then
     begin
          Sound:= nil;
          exit;
     end;

     SoundFile.Init(FileName, stOpenRead);
     SoundFile.Read(SoundHdr, sizeof(SoundHdr));

     New(Sound);
     Sound^.SoundSize:= SoundHdr.Size;

     if not(SharedEMB) then
     begin
          Sound^.StartOfs:= 0;
          Sound^.XMSHandle:= XMS_EMBGetMem((SoundHdr.Size + 1023) div 1024);
     end
     else
     begin
          Sound^.StartOfs:= SharedSize;
          Sound^.XMSHandle:= SharedHandle;
          SharedSize:= SharedSize + SoundHdr.Size;
          XMS_EMBResize(SharedHandle, (SharedSize + 1023) div 1024);
     end;

     MoveParams.SourceHandle:= 0;
     MoveParams.SourceOffset:= LongInt(Addr(InBuffer));
     MoveParams.DestHandle:= Sound^.XMSHandle;
     MoveParams.DestOffset:= Sound^.StartOfs;

     Remaining:= Sound^.SoundSize;

     repeat
           MoveParams.Length:= U_Min(Remaining, LoadChunkSize);

           SoundFile.Read(InBuffer, MoveParams.Length);
           MoveParams.Length:= ((MoveParams.Length + 1) div 2) * 2;

           XMS_EMBMove(@MoveParams);
           Inc(MoveParams.DestOffset, MoveParams.Length);
           Dec(Remaining, MoveParams.Length);
     until not(Remaining > 0);

     SoundFile.Done;
end;


{******}

procedure Voice_LoadResource(ResName : string; Key : string; var Sound : PSound);

var ResFile   : TResourceFile;
    SoundRes  : PSoundRes;
    InBuffer  : array[1..LoadChunkSize] of byte;
    Remaining : longint;

begin
     if not U_FileExists(ResName) then
     begin
          Sound:= nil;
          exit;
     end;

     ResFile.Init(New(PBufStream, Init(ResName, stOpenRead, 2048)));
     SoundRes:= PSoundRes(ResFile.Get(Key));
     if SoundRes = nil then
     begin
          ResFile.Done;
          Sound:= nil;
          exit;
     end;

     ResFile.Stream^.Seek(ResFile.Stream^.GetPos - SoundRes^.SoundInfo.Size);

     New(Sound);
     Sound^.SoundSize:= SoundRes^.SoundInfo.Size;

     if not(SharedEMB) then
     begin
          Sound^.StartOfs:= 0;
          Sound^.XMSHandle:= XMS_EMBGetMem((SoundRes^.SoundInfo.Size + 1023) div 1024);
     end
     else
     begin
          Sound^.StartOfs:= SharedSize;
          Sound^.XMSHandle:= SharedHandle;
          SharedSize:= SharedSize + SoundRes^.SoundInfo.Size;
          XMS_EMBResize(SharedHandle, (SharedSize + 1023) div 1024);
     end;

     MoveParams.SourceHandle:= 0;
     MoveParams.SourceOffset:= LongInt(Addr(InBuffer));
     MoveParams.DestHandle:= Sound^.XMSHandle;
     MoveParams.DestOffset:= Sound^.StartOfs;

     Remaining:= Sound^.SoundSize;

     repeat
           MoveParams.Length:= U_Min(Remaining, LoadChunkSize);

           ResFile.Stream^.Read(InBuffer, MoveParams.Length);
           MoveParams.Length:= ((MoveParams.Length + 1) div 2) * 2;

           XMS_EMBMove(@MoveParams);
           Inc(MoveParams.DestOffset, MoveParams.Length);
           Dec(Remaining, MoveParams.Length);
     until not(Remaining > 0);

     ResFile.Done;
     SoundRes^.Done;
end;

{******}

procedure Voice_Free(var Sound : PSound);

begin
     if Assigned(Sound) then
     begin
          if not(SharedEMB) then XMS_EMBFreeMem(Sound^.XMSHandle);
          Dispose(Sound);
          Sound:= nil;
     end;
end;

{*** Voice maintainance ***}

procedure Voice_Deallocate(VoiceNum : byte);

begin
     VoiceInUse[VoiceNum] := false;
     with Voice[VoiceNum] do
     begin
          Sound:= nil;
          Index:= 0;
          CurPos:= 0;
          Loop:= false;
     end;
end;

{******}

procedure Voice_Start(Sound : PSound; Index : byte; Loop : boolean);

var i, Slot : byte;

begin
     if Sound = nil then exit;
     Slot:= $FF;
     i:= 0;

     repeat
           if not(VoiceInUse[i]) then Slot:= i;
           Inc(i);
     until ((Slot <> $FF) or (i = Voices));

     if Slot <> $FF then
     begin
          Inc(VoiceCount);
          Voice[Slot].Sound:= Sound;
          Voice[Slot].Index:= Index;
          Voice[Slot].CurPos:= 0;
          Voice[Slot].Loop:= Loop;

          VoiceInUse[Slot]:= true;
     end;
end;

{******}

procedure Voice_Stop(Index : byte);

var i : byte;

begin
     for i := 0 to Voices-1 do
         if Voice[i].Index = Index then
         begin
              Voice_Deallocate(i);
              Dec(VoiceCount);
         end;
     end;

{******}

function Voice_Playing(Index : byte) : boolean;

var i : byte;

begin
     Voice_Playing:= false;

     for i:= 0 to Voices - 1 do
         if Voice[i].Index = Index then Voice_Playing:= true;
end;

{******}

procedure Voice_Update;

var VoiceNum : byte;

begin
     for VoiceNum:= 0 to Voices - 1 do
     begin
          if VoiceInUse[VoiceNum] then
             if Voice[VoiceNum].CurPos >= Voice[VoiceNum].Sound^.SoundSize then
             begin
                  Voice_Deallocate(VoiceNum);
                  Dec(VoiceCount);
             end;
     end;
end;

{*** Utility functions ***}

procedure Voice_SetCurBlock(BlockNum : byte);

begin
     CurBlock:= BlockNum;
     CurBlockPtr:= pointer(BlockPtr[BlockNum]);
end;

{******}

procedure Voice_ToggleBlock;

begin
     if CurBlock = 1 then Voice_SetCurBlock(2)
                     else Voice_SetCurBlock(1);
end;

{******}

procedure Voice_SilenceBlock;

begin
     FillChar(MixingBlock, BlockLength * 2, 0);
end;

{******}

procedure Voice_InitClip8;

var i, Value : integer;

begin
     for i:= -128 * Voices to 128 * Voices do
     begin
          Value:= i;
          if (Value < -128) then Value:= -128;
          if (Value > +127) then Value:= +127;

          Clip8[i]:= Value + 128;
     end;
end;

{******}

procedure Voice_InitMixing;

var i : integer;

begin
     for i:= 0 to Voices - 1 do Voice_Deallocate(i);
     VoiceCount:= 0;

     if SixteenBit then
     begin
          OutMemArea:= DMA_MemAlloc(4 * BufferLength);
          Out16Buffer:= OutMemArea;
          for i := 1 to 2 do BlockPtr[i]:= U_NormalisePtr(Addr(Out16Buffer^[i]));
          FillChar(Out16Buffer^, BufferLength * 2, $00);
     end
     else
     begin
          OutMemArea:= DMA_MemAlloc(2 * BufferLength);
          Out8Buffer:= OutMemArea;
          for i := 1 to 2 do BlockPtr[i] := U_NormalisePtr(Addr(Out8Buffer^[i]));
          FillChar(Out8Buffer^, BufferLength, $80);
          Voice_InitClip8;
     end;

     FillChar(MixingBlock, BlockLength * 2, $00);
     Voice_SetCurBlock(1);
     IntCount:= 0;
     Voice_StartDAC;
end;

{******}

procedure Voice_DoneMixing;

begin
     Voice_StopDAC;

     if SixteenBit then DMA_FreeMem(OutMemArea)
                   else DMA_FreeMem(OutMemArea);
end;

{******}

function Voice_GetFreeIndex : byte;

var Loop : word;

begin
end;

{******}

var IntMoveParams : TMoveParams;

{******}

procedure Voice_CopySound(Sound : PSound; var CurPos : longint;
                          CopyLength : word; Loop : boolean);

var SoundSize : longint;
    DestPtr   : pointer;

begin
     SoundSize:= Sound^.SoundSize;
     DestPtr:= pointer(@SoundBlock);
     IntMoveParams.SourceHandle:= Sound^.XMSHandle;
     IntMoveParams.DestHandle:= 0;
     while CopyLength > 0 do
     begin
          if CopyLength < SoundSize - CurPos then IntMoveParams.Length:= CopyLength
                                             else IntMoveParams.Length:= SoundSize - CurPos;

          IntMoveParams.SourceOffset:= Sound^.StartOfs + CurPos;
          CurPos:= CurPos + IntMoveParams.Length;

          if Loop then CurPos:= CurPos mod SoundSize;

          IntMoveParams.DestOffset:= LongInt(DestPtr);
          DestPtr:= U_NormalisePtr(Ptr(Seg(DestPtr^), Ofs(DestPtr^) + IntMoveParams.Length));

          CopyLength:= CopyLength - IntMoveParams.Length;

          IntMoveParams.Length:= ((IntMoveParams.Length + 1) div 2) * 2;
          XMS_EMBMove(@IntMoveParams);
     end;
end;

{******}

procedure Voice_Mixer(VoiceNum : byte);

var MixLength : word;

begin
     with Voice[VoiceNum] do
          if Loop then MixLength:= BlockLength
                  else if BlockLength < Sound^.SoundSize - CurPos
                       then MixLength:= BlockLength
                       else MixLength:= Sound^.SoundSize - CurPos;

     Voice_CopySound(Voice[VoiceNum].Sound, Voice[VoiceNum].CurPos, MixLength, Voice[VoiceNum].Loop);

     asm
        lea  si, SoundBlock         {DS:SI -> Sound data (Source)          }
        mov  ax, ds                 {ES:DI -> Mixing block (Destination)   }
        mov  es, ax
        lea  di, MixingBlock
        mov  cx, MixLength          {CX = Number of samples to copy        }

     @MixSample:
        mov  al, [si]               {Load a sample from the sound block    }
        inc  si                     { increment pointer                    }
        cbw                         {Convert it to a 16-bit signed sample  }
        add  es:[di], ax            {Add it into the mixing buffer         }
        add  di, 2                  {Next word in mixing buffer            }
        dec  cx                     {Loop for next sample                  }
        jnz  @MixSample
     end;
end;

{******}

procedure Voice_Mix;

var i : word;

begin
     Voice_SilenceBlock;
     for i:= 0 to Voices - 1 do if VoiceInUse[i] then Voice_Mixer(i);
end;

{******}

procedure Voice_CopyData16; assembler;

asm
   lea   si, MixingBlock         {DS:SI -> 16-bit input block           }
   les   di, [CurBlockPtr]       {ES:DI -> 16-bit output block          }
   mov   cx, BlockLength         {CX = Number of samples to copy        }

@CopySample:

   mov   ax, [si]                {Load a sample from the mixing block   }
   add   di, 2                   {Increment destination pointer         }
   sal   ax, 5                   {Shift sample left to fill 16-bit range}
   add   si, 2                   {Increment source pointer              }
   mov   es:[di-2], ax           {Store sample in output block          }
   dec   cx                      {Process the next sample               }
   jnz   @CopySample

end;

{******}

procedure Voice_CopyData8; assembler;

asm
   push  bp
   mov   dx, ss                  {Preserve SS in DX                     }
   pushf
   cli                           {Disable interrupts                    }
   mov   ax, ds                  {Using SS for data                     }
   mov   ss, ax

   lea   si, Clip8               {DS:SI -> 8-bit clipping buffer        }
   add   si, 128*Voices          {DS:SI -> Center of clipping buffer    }

   lea   bp, MixingBlock         {SS:BP -> 16-bit input block           }
   les   di, [CurBlockPtr]       {ES:DI -> 8-bit output block           }
   mov   cx, BlockLength         {CX = Number of samples to copy        }

@CopySample:

   mov   bx, [bp]                {BX = Sample from mixing block         }
   inc   di                      {Increment destination pointer (DI)    }
   add   bp, 2                   {Increment source pointer (BP)         }
   mov   al, [si+bx]             {AL = Clipped sample                   }
   mov   es:[di-1], al           {Store sample in output block          }
   dec   cx                      {Process the next sample               }
   jnz   @CopySample

   mov   ss, dx                  {Restore SS                            }
   popf                          {Restore flags                         }
   pop   bp

end;

{******}

procedure Voice_CopyData;

begin
     if SixteenBit then Voice_CopyData16
                   else Voice_CopyData8;
end;

{******}

procedure Voice_StartDMABlock; {For single mode}

begin
     DMA_SetChannel(DMAChannel, BlockPtr[CurBlock], DMALength, DMAMode);
     SB_Write(DSP_DMADAC8);
     SB_Write(Lo(BlockLength - 1));
     SB_Write(Hi(BlockLength - 1));
end;

{******}

var SaveCPU : byte;

{******}

procedure Voice_IntHandler(l : longint);

var Temp : byte;

begin
     SaveCPU:= Test8086;
     Test8086:= 0;

     Inc(IntCount);

     if not(AutoInit) then
     begin
          Voice_StartDMABlock;
          Voice_CopyData;
          Voice_ToggleBlock;
     end;

     Voice_Update;
     Voice_Mix;

     if (AutoInit) then
     begin
          Voice_CopyData;
          Voice_ToggleBlock;
     end;

     Test8086:= SaveCPU;
end;

{******}

{$F+}
procedure Voice_ExitProc;

begin
    ExitProc:= OldExitProc;
    Voice_StopDAC;
    Voice_Done;
end;
{$F-}
{******}

begin
     HandlerInstalled:= false;
     SharedEMB:= false;
end.