unit SWAVE;

(* Information
   

   Program Title : WAVEAudio support for S-Library.
   External name : SWAVE.TPU
   Version       : 1.0
   Start date    : 28/8/96
   Last update   : 28/8/96
   Author        : Rob Anderton.
   Description   : Windows WAV file playing routines.

*)

interface

uses SBDSP, SERROR;

{*** Wave chunks ***}

const ckRIFF : array[1..4] of char = 'RIFF';
      ckWAVE : array[1..4] of char = 'WAVE';
      ckFMT  : array[1..4] of char = 'fmt ';
      ckDATA : array[1..4] of char = 'data';

{******}

type ChunkHeader = record
                         ChunkID   : array[1..4] of char;
                         ChunkSize : longint;
                   end;

     WaveFormat  = record
                         wFormat         : word;
                         nChannels       : word;
                         nSamplesPerSec  : longint;
                         nAvgBytesPerSec : longint;
                         nBlockAlign     : word;
                         wBitsPerSample  : word;
                     end;

{******}

procedure WAVE_Play(WaveName : string; UserProc : DSPUserIRQ);

{******}

implementation

uses SDMA, Objects;

{******}

procedure WAVE_StreamError(var S : TStream); far;

begin
     writeln('Stream error : ', S.Status);
     writeln;
     halt(1);
end;

{******}

function WAVE_GetFormat(var S : TStream; var WaveFmt : pointer;
                        var WavefmtSize : word) : boolean;

var ChunkHdr    : ChunkHeader;
    Found       : boolean;

begin
     SErrorCode:= 0;
     S.Read(ChunkHdr, sizeof(ChunkHdr));
     if ChunkHdr.ChunkID = ckWAVE then Found:= true else Found:= false;
     WaveFmt:= nil;

     while ((not Found) and
            (S.GetPos + ChunkHdr.ChunkSize + 1 < S.GetSize)) do
     begin
          S.Seek(S.GetPos + ChunkHdr.ChunkSize);
          S.Read(ChunkHdr, sizeof(ChunkHdr));
          if ChunkHdr.ChunkID = ckWAVE then Found:= true else Found:= false;
     end;

     if not Found then
     begin
          WAVE_GetFormat:= false;
          SErrorCode:= (word(ERR_WAVE) shl 8) + ERR_WAVE_INVALIDWAVE;
          exit;
     end;

     S.Seek(S.GetPos - 4);
     S.Read(ChunkHdr, sizeof(ChunkHdr));
     if ChunkHdr.ChunkID = ckFMT then Found:= true else Found:= false;
     while (not Found) do
     begin
          S.Seek(S.GetPos + ChunkHdr.ChunkSize);
          S.Read(ChunkHdr, sizeof(ChunkHdr));
          if ChunkHdr.ChunkID = ckFMT then Found:= true else Found:= false;
     end;

     if not Found then
     begin
          WAVE_GetFormat:= false;
          SErrorCode:= (word(ERR_WAVE) shl 8) + ERR_WAVE_NOFORMATINFO;
          exit;
     end;

     WavefmtSize:= ChunkHdr.ChunkSize;
     GetMem(Wavefmt, WavefmtSize);
     if Wavefmt = nil then
     begin
          WAVE_GetFormat:= false;
          exit;
     end;

     S.Read(Wavefmt^, WavefmtSize);

     with WaveFormat(Wavefmt^) do
     begin
          if wFormat <> 1 then
          begin
               SErrorCode:= (word(ERR_WAVE) shl 8) + ERR_WAVE_NOTPCM;
               S.Done;
               FreeMem(Wavefmt, WavefmtSize);
               exit
          end;
     end;
     WAVE_GetFormat:= true;
end;

{******}

procedure WAVE_Play(WaveName : string; UserProc : DSPUserIRQ);

var OldError    : pointer;
    WaveFile    : TDosStream;
    ChunkHdr    : ChunkHeader;
    WaveFmt     : pointer;
    WaveFmtSize : word;
    DMABuffer   : pointer;
    DMABufSize  : word;
    DataInfo    : PlayRec;
    TempHandle  : integer;
    Found       : boolean;

begin
     if not SBPresent then
     begin
          SErrorCode:= (word(ERR_WAVE) shl 8) + ERR_WAVE_NOSBCARD;
          exit;
     end;
     SErrorCode:= 0;

     OldError:= StreamError;
     StreamError:= @WAVE_StreamError;

     WaveFile.Init(WaveName, stOpenRead);

     WaveFile.Read(ChunkHdr, sizeof(ChunkHeader));

     if ChunkHdr.ChunkID <> ckRIFF then
     begin
          SErrorCode:= (word(ERR_WAVE) shl 8) + ERR_WAVE_INVALIDRIFF;
          WaveFile.Done;
          exit;
     end;

     if not WAVE_GetFormat(WaveFile, WaveFmt, WavefmtSize) then exit;

     WaveFile.Read(ChunkHdr, sizeof(ChunkHdr));
     if (ChunkHdr.ChunkID = ckDATA) then Found:= true else Found:= false;
     while not Found do
     begin
          WaveFile.Seek(WaveFile.GetPos + ChunkHdr.ChunkSize);
          WaveFile.Read(ChunkHdr, sizeof(ChunkHdr));
          if (ChunkHdr.ChunkID = ckDATA) then Found:= true else Found:= false;
     end;

     if not Found then
     begin
          SErrorCode:= (word(ERR_WAVE) shl 8) + ERR_WAVE_NOWAVEDATA;
          exit;
     end;

     with DataInfo do
     begin
          Stereo:= (WaveFormat(WaveFmt^).nChannels = 2);
          Bits:= WaveFormat(WaveFmt^).wBitsPerSample;
          Freq:= WaveFormat(WaveFmt^).nSamplesPerSec;
     end;

     SB_AdjustFreq(DataInfo.Freq, DataInfo.Stereo);
     DMABufSize:= 8192;
     DMABufSize:= DMABufSize and not $0003;
     DMABuffer:= DMA_MemAlloc(DMABufSize);
     if DMABuffer <> nil then
     begin
          TempHandle:= WaveFile.Handle;
          SB_Play(TempHandle, DataInfo, DMABuffer, DMABufSize, UserProc);
          DMA_FreeMem(DMABuffer);
     end
     else
     begin
          SErrorCode:= (word(ERR_WAVE) shl 8) + ERR_WAVE_NODMABUFFER;
          WaveFile.Done;
          StreamError:= OldError;
          exit;
     end;
     FreeMem(Wavefmt, WavefmtSize);
     WaveFile.Done;
     StreamError:= OldError;
end;

{******}

end.