unit SXMS;

(* Information
   

   Program Title : XMS interface for S-Library.
   External name : SXMS.PAS
   Version       : 1.0.
   Start date    : 11/8/96.
   Last update   : 1/10/96.
   Author        : Rob Anderton.
   Description   : XMS 3.0 specification code.

*)

interface

{******}

const AHIGH = 1;
      ALOW  = 2;

{******}

type PMoveParams = ^TMoveParams;
     TMoveParams = record
                        Length       : longint;
                        SourceHandle : word;
                        SourceOffset : longInt;
                        DestHandle   : word;
                        DestOffset   : longInt;
                   end;

{******}

var XMMPresent : boolean;

{******}

function  XMS_Detect : boolean;
procedure XMS_Init;

function  XMS_GetVersion : word;
function  XMS_GetXMMVersion : word;
function  XMS_GetHMAPresent : boolean;

procedure XMS_HMAGetMem(Size : word);
procedure XMS_HMAFreeMem;

procedure XMS_EnableGlobalA20;
procedure XMS_DisableGlobalA20;
procedure XMS_EnableLocalA20;
procedure XMS_DisableLocalA20;
function  XMS_GetA20Status : boolean;

function  XMS_GetTotalFree : word;
function  XMS_GetLargestFree : word;

function  XMS_EMBGetMem(Size : word) : word;
procedure XMS_EMBFreeMem(Handle : word);
procedure XMS_MoveToEMB(var Source; Handle : word; BlockLength : longint);
procedure XMS_MoveFromEMB(Handle : word; var Dest; BlockLength : longint);
procedure XMS_EMBMove(MoveInfo : PMoveParams);
function  XMS_EMBLock(Handle : word) : longint;
procedure XMS_EMBUnlock(Handle : word);
function  XMS_GetEMBSize(Handle : word) : word;
function  XMS_GetEMBLock(Handle : word) : byte;
function  XMS_GetFreeEMBHandles(Handle : word) : byte;
procedure XMS_EMBResize(Handle, Size : word);

function  XMS_GetFreeUMB : word;
function  XMS_UMBGetMem(Size : word; var Segment : word) : word;
procedure XMS_UMBFreeMem(Segment : word);
procedure XMS_UMBResize(Size : word; Segment : word);

{******}

implementation

uses Dos, SError;

{******}

type PXMSMoveRec = ^TXMSMoveRec;
     TXMSMoveRec = record
                        Length    : longint;
                        SrcHandle : word;
                        SrcOffset : array[AHIGH..ALOW] of word;
                        DstHandle : word;
                        DstOffset : array[AHIGH..ALOW] of word;
                  end;

{******}

var XMSAddr    : array[AHIGH..ALOW] of word;
    XMSError   : byte;

{******}

function XMS_Detect : boolean;

var Error : byte;

begin
     asm
        mov  ax, $4300
        int  $2F
        mov  [Error], al
     end;
     XMS_Detect:= (Error = $80);
end;

{******}

procedure XMS_Init;

var Regs : registers;

begin
     if not XMS_Detect then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          XMMPresent:= false;
          exit;
     end;

     with Regs do
     begin
          AX:= $4310;
          Intr($2F, Regs);
          XMSAddr[AHIGH]:= BX;
          XMSAddr[ALOW]:= ES;
     end;

     SErrorCode:= 0;
     XMMPresent:= true;
end;

{******}

function XMS_GetVersion : word;

var Ver : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;

     asm
        mov  ah, 0
        call [XMSAddr]
        mov  [Ver], ax
     end;

     XMS_GetVersion:= Ver;
end;

{******}

function XMS_GetXMMVersion : word;

var Ver : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;

     asm
        mov  ah, 0
        call [XMSAddr]
        mov  [Ver], bx
     end;

     XMS_GetXMMVersion:= Ver;
end;

{******}

function XMS_GetHMAPresent : boolean;

var Temp : wordbool;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     Temp:= false;

     asm
        mov  ah, 0
        call [XMSAddr]
        mov  [Temp], dx
     end;

     XMS_GetHMAPresent:= Temp;
end;

{******}

procedure XMS_HMAGetMem(Size : word);

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ax, 1
        mov  dx, [Size]
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;

     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_HMAFreeMem;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 2
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;

     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_EnableGlobalA20;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 3
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;

     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_DisableGlobalA20;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 4
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_EnableLocalA20;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 5
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_DisableLocalA20;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 6
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

function XMS_GetA20Status : boolean;

var Temp : wordbool;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;
     Temp:= true;

     asm
        mov  ah, 7
        call [XMSAddr]
        mov  [Temp], ax
        cmp  bl, 0
        je   @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_GetA20Status:= Temp;
end;

{******}

function XMS_GetTotalFree : word;

var MemFree : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 8
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        mov  [MemFree], 0
        jmp  @EXIT

     @NO_ERROR:

        mov  [MemFree], dx

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_GetTotalFree:= MemFree;
end;

{******}

function XMS_GetLargestFree : word;

var MemFree : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 8
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        mov  [MemFree], 0
        jmp  @EXIT

     @NO_ERROR:

        mov  [MemFree], ax

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_GetLargestFree:= MemFree;
end;

{******}

function XMS_EMBGetMem(Size : word) : word;

var Handle : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, 9
        mov  dx, [Size]
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        jmp  @EXIT

     @NO_ERROR:

        mov  [Handle], dx

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_EMBGetMem:= Handle;
end;

{******}

procedure XMS_EMBFreeMem(Handle : word);

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $0A
        mov  dx, [Handle]
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_MoveToEMB(var Source; Handle : word; BlockLength : longint);

var XMSMove  : TXMSMoveRec;
    XSeg     : word;
    XOfs     : word;
    XAddrSeg : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     with XMSMove do
     begin
          Length:= BlockLength;
          SrcHandle:= 0;
          SrcOffset[AHIGH]:= Ofs(Source);
          SrcOffset[ALOW]:= Seg(Source);
          DstHandle:= Handle;
          DstOffset[AHIGH]:= 0;
          DstOffset[ALOW]:= 0;
     end;

     XSeg:= Seg(XMSMove);
     XOfs:= Ofs(XMSMove);
     XAddrSeg:= Seg(XMSAddr);

     asm
        push ds
        mov  ah, $0B
        mov  bx, [XSeg]
        mov  ds, bx
        mov  si, [XOfs]
        mov  bx, [XAddrSeg]
        mov  es, bx
        call [es:XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

        pop  ds

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_MoveFromEMB(Handle : word; var Dest; BlockLength : longint);

var XMSMove  : TXMSMoveRec;
    XSeg     : word;
    XOfs     : word;
    XAddrSeg : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     with XMSMove do
     begin
          Length:= BlockLength;
          SrcHandle:= Handle;
          SrcOffset[AHIGH]:= 0;
          SrcOffset[ALOW]:= 0;
          DstHandle:= 0;
          DstOffset[AHIGH]:= Ofs(Dest);
          DstOffset[ALOW]:= Seg(Dest);
     end;

     XSeg:= Seg(XMSMove);
     XOfs:= Ofs(XMSMove);
     XAddrSeg:= Seg(XMSAddr);

     asm
        push ds
        mov  ah, $0B
        mov  bx, [XSeg]
        mov  ds, bx
        mov  si, [XOfs]
        mov  bx, [XAddrSeg]
        mov  es, bx
        call [es:XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

        pop  ds

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_EMBMove(MoveInfo : PMoveParams);

var XAddrSeg : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;
     XAddrSeg:= Seg(XMSAddr);
     asm
        push  ds
        mov   ah, $0B
        lds   si, MoveInfo
        mov  bx, [XAddrSeg]
        mov  es, bx
        call [es:XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

        pop   ds

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

function XMS_EMBLock(Handle : word) : longint;

var Addr1, Addr2 : word;
    Temp         : longint;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $0C
        mov  dx, [Handle]
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        mov  [Addr1], 0
        mov  [Addr2], 0
        jmp  @EXIT

     @NO_ERROR:

        mov  [Addr1], dx
        mov  [Addr2], bx

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     Temp:= Addr1;
     XMS_EMBLock:= (Temp shl 4) + Addr2;
end;

{******}

procedure XMS_EMBUnlock(Handle : word);

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $0D
        mov  dx, [Handle]
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

function XMS_GetEMBSize(Handle : word) : word;

var Temp : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $0E
        mov  dx, [Handle]
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        mov  [Temp], 0
        jmp  @EXIT

     @NO_ERROR:

        mov  [Temp], dx

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_GetEMBSize:= Temp;
end;

{******}

function XMS_GetEMBLock(Handle : word) : byte;

var Temp : byte;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $0E
        mov  dx, [Handle]
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        mov  [Temp], 0
        jmp  @EXIT

     @NO_ERROR:

        mov  [Temp], bh

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_GetEMBLock:= Temp;
end;

{******}

function XMS_GetFreeEMBHandles(Handle : word) : byte;

var Temp : byte;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $0E
        mov  dx, [Handle]
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        mov  [Temp], 0
        jmp  @EXIT

     @NO_ERROR:

        mov  [Temp], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_GetFreeEMBHandles:= Temp;
end;

{******}

procedure XMS_EMBResize(Handle, Size : word);

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $0F
        mov  bx, [Size]
        mov  dx, [Handle]
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

function XMS_GetFreeUMB : word;

var Temp : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $10
        mov  dx, $FFFF
        call [XMSAddr]
        mov  [Temp], dx
        mov  [XMSError], bl
     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     XMS_GetFreeUMB:= Temp;
end;

{******}

function XMS_UMBGetMem(Size : word; var Segment : word) : word;

var Temp1, Temp2 : word;

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $10
        mov  dx, [Size]
        call [XMSAddr]
        cmp  ax, 0
        jne  @NO_ERROR
        mov  [XMSError], bl
        mov  [Temp2], 0
        jmp  @EXIT

     @NO_ERROR:

        mov  [Temp2], bx

     @EXIT:

        mov  [Temp1], dx

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
     Segment:= Temp2;
     XMS_UMBGetMem:= Temp1;
end;

{******}

procedure XMS_UMBFreeMem(Segment : word);

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ax, $11
        mov  dx, [Segment]
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

procedure XMS_UMBResize(Size : word; Segment : word);

begin
     if not XMMPresent then
     begin
          SErrorCode:= (word(ERR_XMS) shl 8) + ERR_XMS_NOXMMPRESENT;
          exit;
     end;
     SErrorCode:= 0;
     XMSError:= 0;

     asm
        mov  ah, $12
        mov  bx, [Size]
        mov  dx, [Segment]
        call [XMSAddr]
        cmp  ax, 0
        jne  @EXIT
        mov  [XMSError], bl

     @EXIT:

     end;
     if XMSError <> 0 then SErrorCode:= (word(ERR_XMS) shl 8) + XMSError;
end;

{******}

begin
     XMMPresent:= false;
end.