unit SPNP;

(* Information
   

   Program Title : Plug and Play support.
   External name : SPNP.TPU
   Version       : 1.0
   Start date    : 16/7/97
   Last update   : 19/9/97
   Author        : Rob Anderton.
   Description   : Routines to detect hardware devices installed in a
                   system and create a system configuration file.

                   Incorporating:

                   Plug and Play BIOS interface routines (SPNPBIOS.PAS)
                   (PnP 1.0A specification)
                   Plug and Play ISA routines            (SPNPISA.PAS)
                   (PnP ISA 1.0A specification)
                   Plug and Play PCI routines            (SPNPPCI.PAS)
                   (PCI 2.0c specification)

*)

interface

{******}

{*** Global constants ***}

const
      {*** File name of device list INI file used to detect devices ***}
      PNP_DEVICELIST_FILENAME = 'SPNP'; {NO .INI EXTENSION}

      {*** File name constant for configuration settings to be stored in ***}
      PNP_CONFIG_FILENAME = 'SCONFIG'; {NO .INI EXTENSION}

      {*** If set to true, the PnP unit will always rescan hardware and
           overwrite the configuration file. If false, the unit will only
           scan hardware if no configuration file exists ***}

      PNP_CONFIG_DETECT   = true;

{*** Functions/procedures ***}

function PnP_WriteConfigFile : byte;

function PnP_GetBaseTypeDesc(bBaseType : byte) : string;
function PnP_GetSubTypeDesc(bBaseType, bSubType : byte) : string;
function PnP_GetIFTypeDesc(bBaseType, bSubType, bIFType : byte) : string;


{******}

implementation

{******}

uses SPNPTYPE, SPNPBIOS, SPNPISA, SPNPPCI, SUTILS, SBITS, SINI, DOS, OBJECTS;

{******}

(* PNP_GetBaseTypeDesc - returns a string describing the given base type *)

function PNP_GetBaseTypeDesc(bBaseType : byte) : string;

const BaseTypes : array[0..$0C] of string[50] = ('Reserved',
                                                 'Mass storage device',
                                                 'Network interface controller',
                                                 'Display controller',
                                                 'Multimedia controller',
                                                 'Memory',
                                                 'Bridge controller',
                                                 'Communications device',
                                                 'System peripherals',
                                                 'Input devices',
                                                 'Docking station',
                                                 'CPU',
                                                 'Serial Bus controller');

var TypeStr : string;

begin
     if bBaseType > HIGH(BaseTypes) then
     begin
        {*** Return base type as part of string ***}
        Str(bBaseType, TypeStr);
        PNP_GetBaseTypeDesc:= 'Unrecognised base type (' + TypeStr + ')';
     end
     else
        PNP_GetBaseTypeDesc:= BaseTypes[bBaseType];
end;

(* PNP_GetSubTypeDesc - returns a string description of the given subtype of
                        a given base type                                   *)

function PNP_GetSubTypeDesc(bBaseType, bSubType : byte) : string;

var DescStr : string;
    TypeStr : string;

begin
     {*** Convert bSubType into string for return if not detected ***}
     Str(bSubType, TypeStr);

     case bBaseType of
          0 : DescStr:= 'Reserved';

          1 : case bSubType of
                   0 : DescStr:= 'SCSI controller';
                   1 : DescStr:= 'IDE controller (Standard ATA compatible)';
                   2 : DescStr:= 'Floppy controller';
                   3 : DescStr:= 'IPI controller';
                   4 : DescStr:= 'RAID controller';
                 else  DescStr:= 'Other mass storage controller (' + TypeStr + ')';
              end;

          2 : case bSubType of
                   0 : DescStr:= 'Ethernet';
                   1 : DescStr:= 'Token ring controller';
                   2 : DescStr:= 'FDDI Controller';
                   3 : DescStr:= 'ATM Controller';
                 else  DescStr:= 'Other network interface controller (' + TypeStr + ')';;
              end;

          3 : case bSubType of
                   0 : DescStr:= 'VGA controller (standard VGA compatible)';
                   1 : DescStr:= 'XGA compatible controller';
                 else  DescStr:= 'Other display controller (' + TypeStr + ')';
              end;

          4 : case bSubType of
                   0 : DescStr:= 'Video controller';
                   1 : DescStr:= 'Audio controller';
                else   DescStr:= 'Other multimedia controller (' + TypeStr + ')';
              end;

          5 : case bSubType of
                   0 : DescStr:= 'RAM';
                   1 : DescStr:= 'FLASH memory';
                else   DescStr:= 'Other memory device (' + TypeStr + ')';
              end;

          6 : case bSubType of
                   0 : DescStr:= 'Host processor bridge';
                   1 : DescStr:= 'ISA bridge';
                   2 : DescStr:= 'EISA bridge';
                   3 : DescStr:= 'MicroChannel bridge';
                   4 : DescStr:= 'PCI bridge';
                   5 : DescStr:= 'PCMCIA bridge';
                   6 : DescStr:= 'NuBus bridge';
                   7 : DescStr:= 'CardBus bridge';
                 else  DescStr:= 'Other bridge device (' + TypeStr + ')';
              end;

          7 : case bSubType of
                   0 : DescStr:= 'RS-232 device (XT compatible)';
                   1 : DescStr:= 'AT compatible parallel port';
                 else  DescStr:= 'Other communications device (' + TypeStr + ')';
              end;

          8 : case bSubType of
                   0 : DescStr:= 'Programmable Interrupt Controller';
                   1 : DescStr:= 'DMA controller (8237 compatible)';
                   2 : DescStr:= 'System timer (8254 compatible)';
                   3 : DescStr:= 'Real time clock';
                 else  DescStr:= 'Other system peripheral (' + TypeStr + ')';
              end;

          9 : case bSubType of
                   0 : DescStr:= 'Keyboard controller';
                   1 : DescStr:= 'Pen digitiser';
                   2 : DescStr:= 'Mouse controller';
                 else  DescStr:= 'Other input controller (' + TypeStr + ')';
              end;

        $0A : case bSubType of
                   0 : DescStr:= 'Generic docking station';
                 else  DescStr:= 'Other docking station (' + TypeStr + ')';
              end;

        $0B : case bSubType of
                   0 : DescStr:= '386 based processor';
                   1 : DescStr:= '486 based processor';
                   2 : DescStr:= 'Pentium based processor';
                   3 : DescStr:= 'Pentium-Pro based processor';
                 $10 : DescStr:= 'DEC Alpha based processor';
                 $40 : DescStr:= 'Coprocessor';
                 else  DescStr:= 'Other processor (' + TypeStr + ')';
              end;

        $0C : case bSubType of
                   0 : DescStr:= 'Firewire (IEEE 1394)';
                   1 : DescStr:= 'ACCESS.bus';
                   2 : DescStr:= 'SSA';
                   3 : DescStr:= 'Universal Serial Bus (USB)';
                 else  DescStr:= 'Other serial bus device';
              end;

       else DescStr:= 'Unknown system device (' + TypeStr + ')';
     end;

     PNP_GetSubTypeDesc:= DescStr;
end;

(* PNP_GetIFTypeDesc - returns a string description of the given interface
                       (IF) type of a given base and sub type              *)

function PNP_GetIFTypeDesc(bBaseType, bSubType, bIFType : byte) : string;

var DescStr : string;
    TypeStr : string;

begin
     DescStr:= '';
     case bBaseType of
          1 : case bSubType of
                   1 : case bIFType of
                            0 : DescStr:= 'Generic IDE';
                       end;

                   2 : case bIFType of
                            0 : DescStr:= 'Generic floppy';
                       end;

                   3 : case bIFType of
                            0 : DescStr:= 'General IPI';
                       end;
              end;

          2 : case bSubType of
                   0 : case bIFType of
                            0 : DescStr:= 'General Ethernet';
                       end;
                   1 : case bIFType of
                            0 : DescStr:= 'General Token Ring';
                       end;
                   2 : case bIFType of
                            0 : DescStr:= 'General FDDI';
                       end;
              end;

          3 : case bSubType of
                   0 : case bIFType of
                            0 : DescStr:= 'Generic VGA compatible';
                            1 : DescStr:= 'VESA SVGA compatible controller';
                       end;
                   1 : case bIFType of
                            0 : DescStr:= 'General XGA compatible controller';
                       end;
              end;

          4 : case bSubType of
                   0 : case bIFType of
                            0 : DescStr:= 'General video controller';
                       end;
                   1 : case bIFType of
                            0 : DescStr:= 'General audio controller';
                       end;
              end;

          5 : case bSubType of
                   0 : case bIFType of
                            0 : DescStr:= 'General RAM';
                       end;
                   1 : case bIFType of
                            0 : DescStr:= 'General FLASH memory';
                       end;
              end;

          6 : case bSubType of
                   0 : case bIFType of
                            0 : DescStr:= 'General host processor bridge';
                       end;
                   1 : case bIFType of
                            0 : DescStr:= 'General ISA bridge';
                       end;
                   2 : case bIFType of
                            0 : DescStr:= 'General EISA bridge';
                       end;
                   3 : case bIFType of
                            0 : DescStr:= 'General MicroChannel bridge';
                       end;
                   4 : case bIFType of
                            0 : DescStr:= 'General PCI bridge';
                       end;
                   5 : case bIFType of
                            0 : DescStr:= 'General PCMCIA bridge';
                       end;
              end;

          7 : case bSubType of
                   0 : case bIFType of
                            0 : DescStr:= 'Generic XT compatible';
                            1 : DescStr:= '16450 compatible';
                            2 : DescStr:= '16550 compatible';
                       end;
                   1 : case bIFType of
                            0 : DescStr:= 'Generic AT parallel port';
                            1 : DescStr:= 'Model-30 bi-directional port';
                            2 : DescStr:= 'ECP 1.x compliant port';
                       end;
              end;

          8 : case bSubType of
                   0 : case bIFType of
                            0 : DescStr:= 'General 8259 PIC';
                            1 : DescStr:= 'ISA PIC (8259 compatible)';
                            2 : DescStr:= 'EISA PIC (8259 compatible)';
                       end;
                   1 : case bIFType of
                            0 : DescStr:= 'Generic DMA controller';
                            1 : DescStr:= 'ISA DMA controller';
                            2 : DescStr:= 'EISA DMA controller';
                       end;
                   2 : case bIFType of
                            0 : DescStr:= 'Generic system timer';
                            1 : DescStr:= 'ISA system timer';
                            2 : DescStr:= 'EISA dual system timers';
                       end;
                   3 : case bIFType of
                            0 : DescStr:= 'Generic RTC controller';
                            1 : DescStr:= 'ISA RTC controller';
                       end;
              end;

          (* Sub-types 9-$0B have no IF types defined
          9 : case bSubType of
              end;

        $0A : case bSubType of
              end;

        $0B : case bSubType of
              end;
           *)

        $0C : case bSubType of
                   3 : case bIFType of
                          $10 : DescStr:= 'OpenHCI Host Controller';
                       end;
              end;
     end;

     {*** if DescStr is still empty then set default message ***}
     if DescStr = '' then
     begin
          Str(bIFType, TypeStr);
          DescStr:= 'Other interface type (' + TypeStr + ')';
     end;

     {*** Return string ***}
     PNP_GetIFTypeDesc:= DescStr;
end;

(* PNP_UnpackID - converts a 32bit EISA ID into a 7 character identifier. *)

function PNP_UnpackID(lID : longint) : string;

{*** Array used to typecast ID number ***}
type TIDByteArray = array[0..3] of byte;

{*** Hex characters ***}
const HexChars : array[0..15] of char = ('0', '1', '2', '3', '4', '5', '6',
                                         '7', '8', '9', 'A', 'B', 'C', 'D',
                                         'E', 'F');

var TempStr   : string[7];      {Used for processing}
    TempArray : TIDByteArray;

begin
     {*** Typecast ID number for processing ***}
     TempArray:= TIDByteArray(lID);

     {*** Blank string ***}
     TempStr:= '       ';

     {*** Get characters ***}

     {*** Bits 6-2 of byte 0 ***}
     TempStr[1]:= Chr(((TempArray[0] and $7C) shr 2) + $40);

     {*** Bits 1-0 of byte 0 and bits 7-5 of byte 1 ***}
     TempStr[2]:= Chr((((TempArray[0] and $03) shl 3) or
                       ((TempArray[1] and $E0) shr 5)) + $40);

     {*** Bits 4-0 of byte 1 ***}
     TempStr[3]:= Chr((TempArray[1] and $1F) + $40);

     {*** Bits 7-4 of byte 2 ***}
     TempStr[4]:= HexChars[(TempArray[2] and $F0) shr 4];

     {*** Bits 3-0 of byte 2 ***}
     TempStr[5]:= HexChars[TempArray[2] and $0F];

     {*** Bits 7-4 of byte 3 ***}
     TempStr[6]:= HexChars[(TempArray[3] and $F0) shr 4];

     {*** Bits 3-0 of byte 3 ***}
     TempStr[7]:= HexChars[TempArray[3] and $0F];

     {*** Return string ***}
     PNP_UnpackID:= TempStr;
end;

(* PnP_WriteConfigFile - creates a configuration file containing all devices
                         detected in the system. Device detection relies
                         on PnP BIOS calls, PnP ISA enumeration and PCI
                         BIOS calls.

                         This function returns an error/success code.
*)

function PnP_WriteConfigFile : byte;

var hFile      : Text;  {The configuration file to write to}
    wDeviceNum : word;  {The number of the device currently being detected}
    y, m, d, s : word;  {Used to get date for file header}
    RetCode    : byte;  {The return code of PnP functions}

    {*** Variables for System devices and ISA PnP ***}
    NodeNum     : byte;     {The number of system nodes}
    NodeSize    : word;     {The size in bytes of the largest node}
    NodeHandle  : byte;     {The handle of the node to be accessed}
    NodeData    : pointer;  {Pointer to untyped variable length node data}
    NodePos     : word;     {Position in node data}

    DataTag     : byte;     {Tag used to identify structure type}
    DataSize    : word;     {Size of structure}

    LargeItem   : boolean;  {True if the structure is a large resource data type}
    EndTagFound : boolean;  {True when End Tag has been reached}
    DFTagFound  : boolean;  {True when in Dependent Function area}

    {*** Used as temporary stores for node data ***}
    PnPVersion    : TPnPVersionNumber;
    LogicalID     : TLogicalDeviceID;
    CompatibleID  : TCompatibleDeviceID;
    IRQ           : TIRQFormat;
    DMA           : TDMAFormat;
    IO            : TIOPortDescriptor;
    Mem           : TMemoryRangeDescriptor;
    ANSI          : TANSIIdentifier;
    Mem32         : TMemoryDescriptor32;

    {*** Used to convert data types ***}
    FixedIODesc    : TFixedIOPortDescriptor;
    FixedMem32     : TFixedMemoryDescriptor32;

    {*** Used to store count of different data types ***}
    PnPVersionCount    : word;
    LogicalIDCount     : word;
    CompatibleIDCount  : word;
    IRQCount           : word;
    DMACount           : word;
    IOCount            : word;
    MemCount           : word;
    ANSICount          : word;
    Mem32Count         : word;


 (* WritePnPVersion - output PnP version information *)

 procedure WritePnPVersion;

 begin
      Move(TByteArray(NodeData^)[NodePos], PnPVersion, DataSize);
      writeln(hFile, 'PNP_VERSION_', PnPVersionCount, '=', PnPVersion.Version);
      writeln(hFile, 'VENDOR_VERSION_', PnPVersionCount, '=', PnPVersion.VendorVersion);
      Inc(PnPVersionCount);
 end;

 (* WriteCompatibleID - output compatible ID *)
 procedure WriteCompatibleID;

 begin
      Move(TByteArray(NodeData^)[NodePos], CompatibleID, DataSize);
      writeln(hFile, 'COMPATIBLE_ID_', CompatibleIDCount, '=',
              PnP_UnpackID(CompatibleID.CompatibleID));
      Inc(CompatibleIDCount);
 end;

 (* WriteIRQ - output IRQ information *)

 procedure WriteIRQ;

 var IRQLoop : word;

 begin
      Move(TByteArray(NodeData^)[NodePos], IRQ, DataSize);
      for IRQLoop:= 0 to 15 do
          if Bit_Test(IRQLoop, IRQ.IRQMaskBits) then
          begin
               writeln(hFile, 'IRQ_', IRQCount, '=', IRQLoop);
               Inc(IRQCount);
          end;
 end;

 (* WriteDMA - output DMA information *)

 procedure WriteDMA;

 var DMALoop : word;

 begin
      Move(TByteArray(NodeData^)[NodePos], DMA, DataSize);
      for DMALoop:= 0 to 7 do
          if Bit_Test(DMALoop, DMA.DMAMaskBits) then
          begin
               writeln(hFile, 'DMA_', DMACount, '=', DMALoop);
               writeln(hFile, 'DMA_TYPE_', DMACount, '=', DMA.DMAInfo);
               Inc(DMACount);
          end;
 end;

 (* WriteIOPort - output IO port information *)

 procedure WriteIOPort;

 begin
      Move(TByteArray(NodeData^)[NodePos], IO, DataSize);
      writeln(hFile, 'IO_BASE_', IOCount, '=', IO.MinimumBaseAddr);
      writeln(hFile, 'IO_RANGE_', IOCount, '=', IO.RangeLength);
      writeln(hFile, 'IO_DECODE_', IOCount, '=', IO.Information);
      Inc(IOCount);
 end;

 (* WriteFixedIOPort - output fixed IO port information *)

 procedure WriteFixedIOPort;

 begin
      Move(TByteArray(NodeData^)[NodePos], FixedIODesc, DataSize);
      writeln(hFile, 'IO_BASE_', IOCount, '=', FixedIODesc.BaseAddr);
      writeln(hFile, 'IO_RANGE_', IOCount, '=', FixedIODesc.RangeLength);
      writeln(hFile, 'IO_DECODE_', IOCount, '=', 0);
      Inc(IOCount);
 end;

 (* WriteMemoryRange - output memory range information *)

 procedure WriteMemoryRange;

 begin
      Move(TByteArray(NodeData^)[NodePos], Mem, DataSize);
      writeln(hFile, 'MEM_BASE_', MemCount, '=', Mem.MinimumBaseAddr);
      writeln(hFile, 'MEM_RANGE_', MemCount, '=', Mem.RangeLength);
      writeln(hFile, 'MEM_INFO_', MemCount, '=', Mem.Information);
      Inc(MemCount);
 end;

 (* WriteMemoryRange32 - output 32 bit memory range information *)

 procedure WriteMemoryRange32;

 begin
      Move(TByteArray(NodeData^)[NodePos], Mem32, DataSize);
      writeln(hFile, 'MEM_BASE32_', Mem32Count, '=', Mem32.MinimumBaseAddr);
      writeln(hFile, 'MEM_RANGE32_', Mem32Count, '=', Mem32.RangeLength);
      writeln(hFile, 'MEM_INFO32_', Mem32Count, '=', Mem32.Information);
      Inc(Mem32Count);
 end;

 (* WriteFixedMemoryRange32 - output fixed 32 bit memory range information *)

 procedure WriteFixedMemoryRange32;

 begin
      Move(TByteArray(NodeData^)[NodePos], FixedMem32, DataSize);
      writeln(hFile, 'MEM_BASE32_', Mem32Count, '=', FixedMem32.BaseAddr);
      writeln(hFile, 'MEM_RANGE32_', Mem32Count, '=', FixedMem32.RangeLength);
      writeln(hFile, 'MEM_INFO32_', Mem32Count, '=', FixedMem32.Information);
      Inc(Mem32Count);
 end;

 (* WriteANSI - output ANSI identifier *)

 procedure WriteANSI;

 var ANSILoop : word;

 begin
      GetMem(ANSI.Identifier, DataSize);
      if ANSI.Identifier <> nil then
      begin
           Move(TByteArray(NodeData^)[NodePos], ANSI.Identifier^, DataSize);
           write(hFile, 'ANSI_ID_', ANSICount, '=');
           for ANSILoop:= 0 to DataSize - 1 do
               write(hFile, TCharArray(ANSI.Identifier^)[ANSILoop]);
           writeln(hFile);
           Inc(ANSICount);
           FreeMem(ANSI.Identifier, DataSize);
           ANSI.Identifier:= nil;
      end;
 end;

 (* WriteSystemDevices - call PnP BIOS to get system devices and output to
                         configuration file.
 *)

 procedure WriteSystemDevices;

 begin
     {*** Setup variables ***}
     EndTagFound:= false;
     DFTagFound:= false;

     {*** Obtain information on the number and size of system nodes ***}
     RetCode:= PnP_BIOS_GetNumSysDevNodes(NodeNum, NodeSize);

     {*** If an error occurs then exit ***}
     if RetCode <> PNP_SUCCESS then Exit;

     {*** Exit if no devices found ***}
     if NodeNum = 0 then Exit;

     {*** Set NodeHandle to zero, system returns first device node ***}
     NodeHandle:= 0;

     repeat
        {*** Allocate memory for node data ***}
        GetMem(NodeData, NodeSize);

        {*** Obtain information about first node ***}
        RetCode:= PnP_BIOS_GetSysDevNode(NodeHandle, NodeData, 1);

        {*** If an error occurs then exit ***}
        if RetCode <> PNP_SUCCESS then
        begin
             {*** Free memory ***}
             if NodeData <> nil then FreeMem(NodeData, NodeSize);
             Exit;
        end;

        {*** Write device header ***}
        writeln(hFile, '[DEVICE_', wDeviceNum, ']');
        writeln(hFile);
        writeln(hFile, ';Device header');
        writeln(hFile, 'DEVICE_TYPE=SYSTEM');
        writeln(hFile, 'DEVICE_NODE=', TSystemDeviceNode(NodeData^).Handle);
        writeln(hFile, 'DEVICE_ID=', PnP_UnpackID(TSystemDeviceNode(NodeData^).ProductID));
        writeln(hFile, 'DEVICE_BASETYPE=', TSystemDeviceNode(NodeData^).DeviceType[1]);
        writeln(hFile, 'DEVICE_SUBTYPE=', TSystemDeviceNode(NodeData^).DeviceType[2]);
        writeln(hFile, 'DEVICE_IFTYPE=', TSystemDeviceNode(NodeData^).DeviceType[3]);
        writeln(hFile, 'DEVICE_ATTRIBUTES=', TSystemDeviceNode(NodeData^).Attributes);
        writeln(hFile);
        writeln(hFile, ';Configuration information');

        {*** Initialise count variables ***}
        PnPVersionCount:= 0;
        LogicalIDCount:= 0;
        CompatibleIDCount:= 0;
        IRQCount:= 0;
        DMACount:= 0;
        IOCount:= 0;
        MemCount:= 0;
        ANSICount:= 0;
        Mem32Count:= 0;

        {*** Skip the system device node header ***}
        NodePos:= sizeof(TSystemDeviceNode);

        {*** Analyse node ***}
        while (NodePos < NodeSize) and (not EndTagFound) do
        begin
           {*** Determine resource data type ***}
           if (TByteArray(NodeData^)[NodePos] and $80) <> $80 then
           begin
              DataTag:= (TByteArray(NodeData^)[NodePos] and $78) shr 3;
              DataSize:= (TByteArray(NodeData^)[NodePos] and $07);
              Inc(NodePos);

              if not DFTagFound then
              begin
                   case DataTag of

                                  PnP_Version_Num : WritePnPVersion;
                             Compatible_Device_ID : WriteCompatibleID;
                                       IRQ_Format : WriteIRQ;
                                       DMA_Format : WriteDMA;
                               IO_Port_Descriptor : WriteIOPort;
                         Fixed_IO_Port_Descriptor : WriteFixedIOPort;
                         Start_Dependent_Function : DFTagFound:= true;
                                          End_Tag : EndTagFound:= true;

                   end;
                   Inc(NodePos, DataSize);
              end
              else
              begin
                  if DataTag = End_Dependent_Function then DFTagFound:= false;
                  Inc(NodePos, DataSize);
              end
           end
           else
           begin
              DataTag:= TByteArray(NodeData^)[NodePos] and $7F;
              Inc(NodePos);
              WordRec(DataSize).Lo:= TByteArray(NodeData^)[NodePos];
              Inc(NodePos);
              WordRec(DataSize).Hi:= TByteArray(NodeData^)[NodePos];
              Inc(NodePos);
              if not DFTagFound then
              begin
                   case DataTag of
                           Memory_Range_Descriptor : WriteMemoryRange;
                                Memory_Range_32Bit : WriteMemoryRange32;
                          Fixed_Memory_Range_32Bit : WriteFixedMemoryRange32;
                            Identifier_String_ANSI : WriteANSI;
                   end;
                   Inc(NodePos, DataSize);
              end;
           end;
        end;

        {*** Free memory ***}
        if NodeData <> nil then FreeMem(NodeData, NodeSize);

        EndTagFound:= false;
        DFTagFound:= false;

        {*** Next device please... ***}
        Inc(wDeviceNum);

        {*** Add spacing in configuration file ***}
        writeln(hFile);
        writeln(hFile, ';****************************************************************************');
        writeln(hFile);

     until (NodeHandle = $FF);
 end;

 (* WriteISARegisters - read ISA PnP configuration registers and output
                        to configuration file.
 *)

 procedure WriteISARegisters(bLDN : byte);

 const IRQ_Regs : array[1..2] of byte = ($70, $72);         {IRQ registers}
       DMA_Regs : array[1..2] of byte = ($74, $75);         {DMA registers}
       IO_Regs  : array[1..8] of byte = ($60, $62, $64, $66,
                                         $68, $6A, $6C, $6E);{I/O port registers}

 var Loop    : word; {Loop control variable}
     Data    : byte; {Data read from register}
     IOAddr  : word; {Used for reading IO port address}

 begin
     {*** Setup logical device ready to access registers ***}
     Port[ISA_ADDRESS_PORT]:= $07;     {Logical device number register}
     Port[ISA_WRITE_DATA_PORT]:= bLDN; {Set logical device}

     {*** Get IRQ channel information (registers 70h-73h) ***}
     for Loop:= 1 to 2 do
     begin
          Port[ISA_ADDRESS_PORT]:= IRQ_Regs[Loop];
          Data:= Port[ISA_READ_DATA_PORT];

          if Data <> 0 then
          begin
               writeln(hFile, 'IRQ_', IRQCount, '=', Data);
               Inc(IRQCount);
          end;
     end;

     {*** Get DMA channel information (registers 74h-75h) ***}
     for Loop:= 1 to 2 do
     begin
          Port[ISA_ADDRESS_PORT]:= DMA_Regs[Loop];
          Data:= Port[ISA_READ_DATA_PORT];

          if Data <> 4 then
          begin
               writeln(hFile, 'DMA_', DMACount, '=', Data);
               Inc(DMACount);
          end;
     end;

     {*** Get I/O configuration information (registers 60h-6Fh) ***}
     for Loop:= 1 to 8 do
     begin
          Port[ISA_ADDRESS_PORT]:= IO_Regs[Loop];
          IOAddr:= Port[ISA_READ_DATA_PORT] shl 8; {Upper 8bits of IO address}
          Port[ISA_ADDRESS_PORT]:= IO_Regs[Loop] + 1;
          Inc(IOAddr, Port[ISA_READ_DATA_PORT]);   {Lower 8bits of IO address}
          if IOAddr <> $0000 then
          begin
               writeln(hFile, 'IO_BASE_', IOCount, '=', IOAddr);
               Inc(IOCount);
          end;
     end;
 end;

 (* WriteLogicalISADevices - call ISA PnP functions to output all logical
                             device information to configuration file.
 *)

 procedure WriteLogicalISADevices;

 var LDN          : byte;    {Logical device number}
     LogicalFound : boolean; {True when next logical device found}

 begin
     {*** Setup variables ***}
     EndTagFound:= false;
     DFTagFound:= false;
     LogicalFound:= false;
     Inc(wDeviceNum);

     {*** Start with first logical device ***}
     LDN:= 0;
     repeat
        {*** Add spacing in configuration file ***}
        writeln(hFile);
        writeln(hFile, ';****************************************************************************');
        writeln(hFile);

        {*** Get logical device info ***}
        Move(TByteArray(NodeData^)[NodePos], LogicalID, DataSize);
        if DataSize = 5 then LogicalID.Flags[2]:= 0;
        Inc(NodePos, DataSize);

        {*** Write device header ***}
        writeln(hFile, '[DEVICE_', wDeviceNum, ']');
        writeln(hFile);
        writeln(hFile, ';Device header');
        writeln(hFile, 'DEVICE_TYPE=ISAPNPLOGICAL');
        writeln(hFile, 'DEVICE_CSN=', NodeHandle);
        writeln(hFile, 'DEVICE_LDN=', LDN);
        writeln(hFile, 'DEVICE_ID=', PnP_UnpackID(LogicalID.LogicalID));
        writeln(hFile);
        writeln(hFile, ';Configuration information');

        {*** Initialise count variables ***}
        PnPVersionCount:= 0;
        LogicalIDCount:= 0;
        CompatibleIDCount:= 0;
        IRQCount:= 0;
        DMACount:= 0;
        IOCount:= 0;
        MemCount:= 0;
        ANSICount:= 0;
        Mem32Count:= 0;

        {*** Analyse node ***}
        while (NodePos < NodeSize) and (not EndTagFound)
              and not (LogicalFound) do
        begin
           {*** Determine resource data type ***}
           if (TByteArray(NodeData^)[NodePos] and $80) <> $80 then
           begin
              DataTag:= (TByteArray(NodeData^)[NodePos] and $78) shr 3;
              DataSize:= (TByteArray(NodeData^)[NodePos] and $07);
              Inc(NodePos);

              if not DFTagFound then
              begin
                   case DataTag of

                                  PnP_Version_Num : WritePnPVersion;
                             Compatible_Device_ID : WriteCompatibleID;
                                Logical_Device_ID : begin
                                                         LogicalFound:= true;
                                                         Break;
                                                    end;
                                       IRQ_Format : WriteIRQ;
                                       DMA_Format : WriteDMA;
                               IO_Port_Descriptor : WriteIOPort;
                         Fixed_IO_Port_Descriptor : WriteFixedIOPort;
                         Start_Dependent_Function : DFTagFound:= true;
                                          End_Tag : EndTagFound:= true;

                   end;
                   Inc(NodePos, DataSize);
              end
              else
              begin
                  if DataTag = End_Dependent_Function then DFTagFound:= false;
                  Inc(NodePos, DataSize);
              end
           end
           else
           begin
              DataTag:= TByteArray(NodeData^)[NodePos] and $7F;
              Inc(NodePos);
              WordRec(DataSize).Lo:= TByteArray(NodeData^)[NodePos];
              Inc(NodePos);
              WordRec(DataSize).Hi:= TByteArray(NodeData^)[NodePos];
              Inc(NodePos);
              if not DFTagFound then
              begin
                   case DataTag of
                           Memory_Range_Descriptor : WriteMemoryRange;
                                Memory_Range_32Bit : WriteMemoryRange32;
                          Fixed_Memory_Range_32Bit : WriteFixedMemoryRange32;
                            Identifier_String_ANSI : WriteANSI;
                   end;
                   Inc(NodePos, DataSize);
              end;
           end;
        end;

        {*** Query PnP registers for additional information ***}
        WriteISARegisters(LDN);

        {*** Move on to next logical device ***}
        if LogicalFound then
        begin
             {*** Next device please... ***}
             Inc(wDeviceNum);

             Inc(LDN);
             LogicalFound:= false;
        end;

     until (EndTagFound);
 end;


 (* WriteISADevices - call ISA PnP functions to detect ISA PnP cards and
                      output to configuration file.
 *)

 procedure WriteISADevices;

 var SerialID     : TSerialKey; {ISA Serial ID}
     ProductID    : longint;    {Used to convert Serial ID to EISA ID}
     LogicalFound : boolean;    {True if logical device exists}

 begin
     {*** Setup variables ***}
     EndTagFound:= false;
     DFTagFound:= false;
     LogicalFound:= false;

     {*** Set NodeHandle to one (NodeHandle = Card Select Number) ***}
     NodeHandle:= 1;

     repeat
        {*** Obtain information about first card ***}
        RetCode:= PnP_ISA_ReadCardData(NodeHandle, NodeData, NodeSize, SerialID);

        {*** If an error occurs then exit ***}
        if RetCode <> PNP_SUCCESS then
        begin
             {*** Free memory ***}
             if NodeData <> nil then FreeMem(NodeData, NodeSize);
             Exit;
        end;

        {*** Write device header ***}
        writeln(hFile, '[DEVICE_', wDeviceNum, ']');
        writeln(hFile);
        writeln(hFile, ';Device header');
        writeln(hFile, 'DEVICE_TYPE=ISAPNP');
        writeln(hFile, 'DEVICE_CSN=', NodeHandle);

        {*** Convert serial ID ***}
        Move(SerialID, ProductID, 4);
        writeln(hFile, 'DEVICE_ID=', PnP_UnpackID(ProductID));
        writeln(hFile);
        writeln(hFile, ';Configuration information');

        {*** Initialise count variables ***}
        PnPVersionCount:= 0;
        LogicalIDCount:= 0;
        CompatibleIDCount:= 0;
        IRQCount:= 0;
        DMACount:= 0;
        IOCount:= 0;
        MemCount:= 0;
        ANSICount:= 0;
        Mem32Count:= 0;

        {*** Start at beginning of buffer ***}
        NodePos:= 0;

        {*** Analyse node ***}
        while (NodePos < NodeSize) and (not EndTagFound) do
        begin
           {*** Determine resource data type ***}
           if (TByteArray(NodeData^)[NodePos] and $80) <> $80 then
           begin
              DataTag:= (TByteArray(NodeData^)[NodePos] and $78) shr 3;
              DataSize:= (TByteArray(NodeData^)[NodePos] and $07);
              Inc(NodePos);

              if not DFTagFound then
              begin
                   case DataTag of

                                  PnP_Version_Num : WritePnPVersion;
                             Compatible_Device_ID : WriteCompatibleID;
                                Logical_Device_ID : begin
                                                         LogicalFound:= true;
                                                         Break;
                                                    end;
                                       IRQ_Format : WriteIRQ;
                                       DMA_Format : WriteDMA;
                               IO_Port_Descriptor : WriteIOPort;
                         Fixed_IO_Port_Descriptor : WriteFixedIOPort;
                         Start_Dependent_Function : DFTagFound:= true;
                                          End_Tag : EndTagFound:= true;

                   end;
                   Inc(NodePos, DataSize);
              end
              else
              begin
                  if DataTag = End_Dependent_Function then DFTagFound:= false;
                  Inc(NodePos, DataSize);
              end
           end
           else
           begin
              DataTag:= TByteArray(NodeData^)[NodePos] and $7F;
              Inc(NodePos);
              WordRec(DataSize).Lo:= TByteArray(NodeData^)[NodePos];
              Inc(NodePos);
              WordRec(DataSize).Hi:= TByteArray(NodeData^)[NodePos];
              Inc(NodePos);
              if not DFTagFound then
              begin
                   case DataTag of
                           Memory_Range_Descriptor : WriteMemoryRange;
                                Memory_Range_32Bit : WriteMemoryRange32;
                          Fixed_Memory_Range_32Bit : WriteFixedMemoryRange32;
                            Identifier_String_ANSI : WriteANSI;
                   end;
                   Inc(NodePos, DataSize);
              end;
           end;
        end;

        {*** Get Logical Device information if necessary ***}
        if LogicalFound then WriteLogicalISADevices;

        {*** Free memory ***}
        if NodeData <> nil then FreeMem(NodeData, NodeSize);

        EndTagFound:= false;
        DFTagFound:= false;

        {*** Next device please... ***}
        Inc(wDeviceNum);
        Inc(NodeHandle);

        {*** Add spacing in configuration file ***}
        writeln(hFile);
        writeln(hFile, ';****************************************************************************');
        writeln(hFile);

     until (NodeHandle > ISA_LAST_CSN);

     Port[ISA_ADDRESS_PORT]:= $02;
     Port[ISA_WRITE_DATA_PORT]:= $02;
 end;

 (* WritePCIDevices - call PCI BIOS functions to detect PCI cards and output
                      to configuration file.
 *)

 procedure WritePCIDevices;

 const BaseAddrReg : array[0..5] of byte = ($10, $14, $18, $1C, $20, $24);

 var VendorStr  : string;  {Vendor ID as a string}
     DeviceStr  : string;  {Device ID as a string}
     VendorID   : word;    {Vendor ID as a number for BIOS function}
     DeviceID   : word;    {Device ID as a number for BIOS function}
     DeviceInfo : word;    {Combined device/function/bus number of PCI card}
     Temp       : longint; {Temporary PCI register data}
     ConfigAddr : longint; {PCI register to read from}

     ConfigData : TPCIConfigData; {Holds contents of PCI registers}
     BusLoop    : word;           {Loop control variable}

     Addr32     : longint;                {32 bit address}
     Addr64     : array[1..2] of longint; {64 bit address}
     Mem64Count : word;                   {Used for output}


 {*** Actual output procedure ***}

 procedure OutputPCIData;

 var Loop : word; {Loop control variable}

 begin
      if (VendorID <> $FFFF) and (DeviceID <> $FFFF) then
      begin
           for Loop:= 0 to 63 do
           begin
                RetCode:= PnP_PCI_ReadConfigByte(DeviceInfo, Loop,
                                                 ConfigData[Loop]);

                {*** Exit on error ***}
                if RetCode <> PNP_SUCCESS then Exit;
           end;

           {*** Write device header ***}
           writeln(hFile, '[DEVICE_', wDeviceNum, ']');
           writeln(hFile);
           writeln(hFile, ';Device header');
           writeln(hFile, 'DEVICE_TYPE=PCI');
           writeln(hFile, 'DEVICE_BUS=', DeviceInfo);

           {*** Get Vendor and Device ID's as strings ***}
           VendorStr:= U_WordToHex(VendorID, true);
           Move(VendorStr[5], VendorStr[1], 4);
           VendorStr[0]:= #4;
           DeviceStr:= U_WordToHex(DeviceID, true);
           Move(DeviceStr[5], DeviceStr[1], 4);
           DeviceStr[0]:= #4;

           writeln(hFile, 'VENDOR_ID=', VendorStr);
           writeln(hFile, 'DEVICE_ID=', DeviceStr);
           writeln(hFile, 'DEVICE_BASETYPE=', ConfigData[$0B]);
           writeln(hFile, 'DEVICE_SUBTYPE=', ConfigData[$0A]);
           writeln(hFile, 'DEVICE_IFTYPE=', ConfigData[$09]);
           writeln(hFile, 'DEVICE_REVISION=', ConfigData[$08]);
           writeln(hFile);
           writeln(hFile, ';Configuration information');

           {*** Initialise count variables ***}
           IRQCount:= 0;
           IOCount:= 0;
           Mem32Count:= 0;
           Mem64Count:= 0;

           {*** Get configuration information ***}
           case (ConfigData[$0E] and not $80) of

                $00 : begin
                         {*** Base addresses ***}
                         Loop:= 0;
                         while Loop <= 5 do
                         begin
                            {*** Get DWORD from config data ***}
                            Move(ConfigData[BaseAddrReg[Loop]], Addr32, 4);

                            case (Addr32 and 1) of

                                  0 : begin
                                         if (Addr32 and $04) <> $04 then
                                         begin
                                            Addr32:= Addr32 and not $0F;
                                            if Addr32 <> $0000 then
                                            begin
                                               writeln(hFile, 'MEM_BASE32_', Mem32Count, '=', Addr32);
                                               Inc(Mem32Count);
                                            end;
                                         end
                                         else
                                         begin
                                            Move(ConfigData[BaseAddrReg[Loop]], Addr64, 8);
                                            Addr64[1]:= Addr64[1] and not $0F;
                                            writeln(hFile, 'MEM_BASE64_', Mem64Count, '=',
                                            Addr64[2] shl 32 + Addr64[1]);
                                            Inc(Mem64Count);
                                            Inc(Loop);
                                         end;
                                      end;

                                  1 : begin
                                         Addr32:= Addr32 and not $03;
                                         if Addr32 <> $0000 then
                                         begin
                                            writeln(hFile, 'IO_BASE_', IOCount, '=', Addr32);
                                            Inc(IOCount);
                                         end;
                                      end;
                            end;
                            Inc(Loop);
                         end;

                         {*** IRQ information ***}
                         if ConfigData[$3C] <> 0 then
                         begin
                              writeln(hFile, 'IRQ_', IRQCount, '=', ConfigData[$3C]);
                              Inc(IRQCount);
                         end;
                      end;

                $01 : begin
                      end;

                $02 : begin
                      end;
           end;

           {*** Add spacing in configuration file ***}
           writeln(hFile);
           writeln(hFile, ';****************************************************************************');
           writeln(hFile);

           Inc(wDeviceNum);
      end;
 end;

 {*** Main PCI detection routine ***}

 begin
      {*** Scan for PCI devices (bypass BIOS for speed) ***}
      case PnP_PCI_Config of
           1 : begin
                 for BusLoop:= 0 to 511 do
                 begin
                    {*** Address of configuration registers ***}
                    ConfigAddr:= $80000000 + BusLoop * LONGINT(2048);
                    {*** Get first DWORD (vendor and device ID) ***}
                    U_OutpLong($CF8, ConfigAddr);
                    Temp:= U_InpLong($CFC);
                    {*** Calculate Vendor and Device ID ***}
                    VendorID:= WORD(Temp);
                    DeviceID:= WORD(Temp shr 16);
                    {*** Calculate bus/device/function number ***}
                    DeviceInfo:= WORD(ConfigAddr shr 8);
                    {*** Write data to configuration file ***}
                    OutputPCIData;
                 end;
               end;

           2 : begin
                 {*** Allow access to PCI configuration registers ***}
                 Port[$CF8]:= $80;
                 Port[$CFA]:= 0;

                 for BusLoop:= 0 to 15 do
                 begin
                    {*** Read configuration registers ***}
                    Temp:= U_InpLong($C000 + BusLoop * 256);
                    {*** Calculate vendor and device ID ***}
                    VendorID:= WORD(Temp);
                    DeviceID:= WORD(Temp shr 16);
                    {*** Calculate bus/device/function number ***}
                    DeviceInfo:= (BusLoop shl 8);
                    {*** Write data to configuration file ***}
                    OutputPCIData;
                 end;
                 Port[$CF8]:= 0;
               end;
      end;
 end;

{*** Main function ***}

begin
     {*** Create file ***}
     Assign(hFile, PNP_CONFIG_FILENAME + '.INI');
     Rewrite(hFile);

     {*** Write header to file ***}
     writeln(hFile, ';****************************************************************************');
     writeln(hFile, ';');
     writeln(hFile, '; Information');
     writeln(hFile, '; ');
     writeln(hFile, ';');
     writeln(hFile, ';  Title         : Plug and Play Device Configuration File');

     {*** Get date for header ***}
     GetDate(y, m, d, s);
     writeln(hFile, ';  Creation date : ', d:0, '/', m:0, '/', y:0);
     writeln(hFile, ';  Description   : Configuration information for all devices');
     writeln(hFile, ';                  detected in the system.');
     writeln(hFile, ';');
     writeln(hFile, ';****************************************************************************');
     writeln(hFile);

     {*** Set device number to zero ***}
     wDeviceNum:= 0;

     {*** Call PnP BIOS to get system devices ***}
     WriteSystemDevices;

     {*** Call ISA PnP functions to get ISA PnP devices ***}
     if PnP_ISA_Present then WriteISADevices;

     {*** Call PCI BIOS to get PCI devices ***}
     if PnP_PCI_Present then WritePCIDevices;

     {*** Close file ***}
     Close(hFile);
end;

{******}

(* PnP_Init - main initialisation routine called by the unit at program
              startup.
*)

procedure PnP_Init;

begin
     {*** Detect PnP BIOS ***}
     if PnP_BIOS_Detect(PnP_BIOS_Header) then
     begin
          {*** Setup entry point ***}
          PnP_BIOS_Entry:= Ptr(PnP_BIOS_Header.RealEntrySeg,
                               PnP_BIOS_Header.RealEntryOfs);

          {*** Set BIOS present flag to true ***}
          PnP_BIOS_Present:= true;

          {*** Detect PnP ISA cards ***}
          if (PnP_BIOS_GetISAConfig(PnP_ISA_Config) <> PNP_SUCCESS) or
             (PnP_ISA_Config.ReadPort < ISA_MIN_READ_PORT) or
             (PnP_ISA_Config.ReadPort > ISA_MAX_READ_PORT) or
             (PnP_ISA_Config.Reserved <> 0) then
          begin
               {*** Double check - the BIOS maybe faulty ***}
               if PnP_ISA_Discover = PNP_SUCCESS then
               begin
                    {*** Initialise ISA cards ***}
                    PnP_ISA_Initiate;
                    PnP_ISA_SetReadDataPort(ISA_READ_DATA_PORT);
                    {*** Fill in ISA configuration structure ***}
                    with PnP_ISA_Config do
                    begin
                         Revision:= $01;
                         TotalCSN:= ISA_LAST_CSN;
                         ReadPort:= ISA_READ_DATA_PORT;
                         Reserved:= 0;
                    end;
                    PnP_ISA_Present:= true;
               end
               else
               begin
                    {*** No devices found ***}
                    PnP_ISA_Present:= false;
                    FillChar(PnP_ISA_Config, SizeOf(PnP_ISA_Config), 0);
               end;
          end
          else if PnP_ISA_Config.TotalCSN = 0 then PnP_ISA_Present:= false;

          {*** Detect PCI BIOS ***}
          PnP_PCI_Init;

          {*** If neccessary create config file ***}
          if PNP_CONFIG_DETECT = true then
             PnP_WriteConfigFile
          else if not U_FileExists(PNP_CONFIG_FILENAME + '.INI') then
                  PnP_WriteConfigFile;

          {*** Free any memory used by device list ***}
          INI_DisposeData;
     end
     else
     begin
          {*** Fill header structure with zeros ***}
          FillChar(PnP_BIOS_Header, SizeOf(TPnPHeader), 0);
          {*** Set entry point to nil ***}
          PnP_BIOS_Entry:= nil;
          {*** Set BIOS present flag to false ***}
          PnP_BIOS_Present:= false;
          {*** No PnP ISA device access ***}
          PnP_ISA_Present:= false;
          FillChar(PnP_ISA_Config, SizeOf(PnP_ISA_Config), 0);
          {*** No PCI device access ***}
          PnP_PCI_Present:= false;
          PnP_PCI_Config:= 0;
     end;
end;

{*** Startup code ***}

begin
     {*** Initialise PnP structures and variables ***}
     PnP_Init;
end.