unit SPNPBIOS;

(* Information
   

   Program Title : Plug and Play support.
   External name : SPNPBIOS.TPU
   Version       : 1.0
   Start date    : 16/7/97
   Last update   : 7/9/97
   Author        : Rob Anderton.
   Description   : Plug and Play BIOS interface routines
                   (PNP BIOS 1.0A specification)

*)

interface

{******}

uses SPNPTYPE;

{*** PnP BIOS functions and procedures ***}

function PnP_BIOS_Detect(var rPNPHeader : TPNPHeader) : boolean;

function PnP_BIOS_GetNumSysDevNodes(var bNodes     : byte;
                                    var wNodeSize  : word) : byte;

function PnP_BIOS_GetSysDevNode(var bNode       : byte;
                                var pNodeBuffer : pointer;
                                    wControl    : word) : byte;

function PnP_BIOS_GetISAConfig(var rISAConfig : TISAConfig) : byte;

function PnP_BIOS_GetESCDInfo(var wESCDSize : word;
                              var lNVSBase  : longint) : byte;

function PnP_BIOS_ReadESCD(    lNVSBase    : longint;
                           var pESCDBuffer : pointer) : byte;

{******}

implementation

{******}

uses DOS;

{******}

(* PnP_BIOS_Detect - scans ROM address F0000h - FFFFFh for $PnP signature
                     then performs checksum operation to ensure a valid
                     PnP header has been found.
                     If all is OK then the header is returned in the
                     variable rPNPHeader and the function returns true.
                     If not OK then false is returned.
*)

function PnP_BIOS_Detect(var rPnPHeader : TPnPHeader) : boolean;

type THdrArray = array[1..33] of byte; (* Used to typecast header
                                          to get checksum         *)

var Found      : boolean;               {True when PnP signature found}
    TempSig    : array[1..4] of char;   {Holds PnP signature}
    CurOfs     : longint;               {Offset into BIOS memory}
    TempHdr    : TPnPHeader;            {Used to read PnP header from BIOS}
    Check      : byte;                  {Used to calculate checksum}
    Loop       : byte;                  {Used to calculate checksum}

begin
     {*** Setup search variables ***}
     Found:= false;
     CurOfs:= 0;

     repeat
          {*** Get DWORD at current memory address ***}
          LONGINT(TempSig):= MemL[$F000:WORD(CurOfs)];

          if TempSig = '$PnP' then
             {*** We've found a match ***}
             Found:= true
          else
             {*** Move on 16 bytes and repeat search ***}
             Inc(CurOfs, 16);

     {*** End search when found or all memory searched***}
     until (CurOfs > $FFFF) or (Found);

     {*** If signature found, read and check header ***}
     if Found then
     begin
          {*** Copy the header into temporary record ***}
          Move(Mem[$F000:CurOfs], TempHdr, $21);

          {*** Calculate the checksum to ensure header is valid ***}
          Check:= 0;
          for Loop:= 1 to 33 do
              Inc(Check, THdrArray(TempHdr)[Loop]);

          if Check = 0 then
          begin
               {*** Header is all OK ***}
               rPnPHeader:= TempHdr;
               PnP_BIOS_Detect:= true;
          end
          else
             {*** Header failed checksum ***}
             PnP_BIOS_Detect:= false;
     end
     else PnP_BIOS_Detect:= false;
end;

(* PnP_BIOS_GetNumSysDevNodes - obtains the number of nodes the system BIOS
   BIOS function 00h            returns information for. Sytem device nodes
                                describe resources on the motherboard.

                                bNodes    - equals the number of nodes to be
                                            returned
                                wNodeSize - equals the size in bytes of the
                                            largest node

                                The function returns an error/success code.
*)

function PnP_BIOS_GetNumSysDevNodes(var bNodes : byte; var wNodeSize : word) : byte;

var NodesSeg     : word;    {Holds segment address of bNodes variable}
    NodesOfs     : word;    {Holds offset address of bNodes variable}
    NodeSizeSeg  : word;    {Holds segment address of wNodeSize variable}
    NodeSizeOfs  : word;    {Holds offset address of wNodeSize variable}
    BIOS_Select  : word;    {Holds BIOS data segment, taken from PnPHeader}
    FunctionNum  : integer; {Contains BIOS function number}
    RetCode      : byte;    {Error/success code returned by BIOS}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_BIOS_GetNumSysDevNodes:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Calculate segments and offsets required by function ***}
     NodesSeg:= Seg(bNodes);
     NodesOfs:= Ofs(bNodes);
     NodeSizeSeg:= Seg(wNodeSize);
     NodeSizeOfs:= Ofs(wNodeSize);
     BIOS_Select:= PnP_BIOS_Header.RealDataSeg;

     {*** BIOS function 00h - Get Number of System Device Nodes ***}
     FunctionNum:= 0;

     {*** Call using 'C' calling convention ***}
     asm
        push  BIOS_Select       {Push parameters onto stack}
        push  NodeSizeSeg
        push  NodeSizeOfs
        push  NodesSeg
        push  NodesOfs
        push  FunctionNum
        call  PnP_BIOS_Entry    {FAR call to the BIOS entry point}
        add   sp, 12            {Clean up the stack}
        mov   [RetCode], al     {Copy error/success code}
     end;

     {*** Return error/success code ***}
     PnP_BIOS_GetNumSysDevNodes:= RetCode;
end;


(* PnP_BIOS_GetSysDevNode - obtains the system device node specified by the
   BIOS function 01h        bNode parameter.

                            The handle of the next node is returned in bNode.

                            wControl specifies what node data is required:
                                01b = info on current config
                                10b = info on config after next boot

                            pNodeBuffer holds the returned node data.
*)

function PnP_BIOS_GetSysDevNode(var bNode       : byte;
                                var pNodeBuffer : pointer;
                                    wControl    : word) : byte;


var NodeSeg      : word;    {Holds segment address of bNode variable}
    NodeOfs      : word;    {Holds offset address of bNode variable}
    NodeBufSeg   : word;    {Holds segment address of pNodeBuffer variable}
    NodeBufOfs   : word;    {Holds offset address of pNodeBuffer variable}
    BIOS_Select  : word;    {Holds BIOS data segment, taken from PnPHeader}
    ControlFlag  : word;    {Holds the control flag, wControl}
    FunctionNum  : integer; {Contains BIOS function number}
    RetCode      : byte;    {Error/success code returned by BIOS}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_BIOS_GetSysDevNode:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Check buffer ***}
     if pNodeBuffer = nil then
     begin
          {*** Return error code ***}
          PnP_BIOS_GetSysDevNode:= PNP_NO_BUFFER_MEMORY;
          Exit;
     end;

     {*** Calculate segments and offsets required by function ***}
     NodeSeg:= Seg(bNode);
     NodeOfs:= Ofs(bNode);
     NodeBufSeg:= Seg(pNodeBuffer^);
     NodeBufOfs:= Ofs(pNodeBuffer^);
     BIOS_Select:= PnP_BIOS_Header.RealDataSeg;
     ControlFlag:= wControl;

     {*** BIOS function 01h - Get System Device Node ***}
     FunctionNum:= 1;

     {*** Call using 'C' calling convention ***}
     asm
        push  BIOS_Select       {Push parameters onto stack}
        push  ControlFlag
        push  NodeBufSeg
        push  NodeBufOfs
        push  NodeSeg
        push  NodeOfs
        push  FunctionNum
        call  PnP_BIOS_Entry    {FAR call to the BIOS entry point}
        add   sp, 14            {Clean up the stack}
        mov   [RetCode], al     {Copy error/success code}
     end;

     {*** Return error/success code ***}
     PnP_BIOS_GetSysDevNode:= RetCode;
end;

(* PnP_BIOS_GetISAConfig - gets Plug and Play ISA configuration structure.
   (function 40h)
                           rISAConfig - ISA configuration structure returned
                                        by PnP BIOS

*)

function PnP_BIOS_GetISAConfig(var rISAConfig : TISAConfig) : byte;

var ConfigSeg    : word;    {Holds segment address of rISAConfig variable}
    ConfigOfs    : word;    {Holds offset address of rISAConfig variable}
    BIOS_Select  : word;    {Holds BIOS data segment, taken from PnPHeader}
    FunctionNum  : integer; {Contains BIOS function number}
    RetCode      : byte;    {Error/success code returned by BIOS}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_BIOS_GetISAConfig:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Calculate segments and offsets required by function ***}
     ConfigSeg:= Seg(rISAConfig);
     ConfigOfs:= Ofs(rISAConfig);
     BIOS_Select:= PnP_BIOS_Header.RealDataSeg;

     {*** BIOS function 40h - Get ISA Configuration Structure ***}
     FunctionNum:= $40;

     {*** Call using 'C' calling convention ***}
     asm
        push  BIOS_Select       {Push parameters onto stack}
        push  ConfigSeg
        push  ConfigOfs
        push  FunctionNum
        call  PnP_BIOS_Entry    {FAR call to the BIOS entry point}
        add   sp, 8             {Clean up the stack}
        mov   [RetCode], al     {Copy error/success code}
     end;

     {*** Return error/success code ***}
     PnP_BIOS_GetISAConfig:= RetCode;
end;

(* PnP_BIOS_GetESCDInfo - returns the size of ESCD data and the 32bit
   (function 41h)         base address of non-volatile storage for ESCD
                          information.

                          wESCDSize - size of ESCD in bytes

                          lNVSBase  - 32bit address of NVS

                          An error code is returned if ESCD is not
                          supported by the BIOS.
*)

function PnP_BIOS_GetESCDInfo(var wESCDSize : word;
                              var lNVSBase  : longint) : byte;

var NVSBaseSeg   : word;    {Holds segment address of lNVSBase variable}
    NVSBaseOfs   : word;    {Holds offset address of lNVSBase variable}
    ESCDSizeSeg  : word;    {Holds segment address of wESCDSize variable}
    ESCDSizeOfs  : word;    {Holds offset address of wESCDSize variable}
    Temp         : word;    {Temporary variable for holding unused data}
    TempSeg      : word;    {Holds segment address of Temp variable}
    TempOfs      : word;    {Holds offset address of Temp variable}
    BIOS_Select  : word;    {Holds BIOS data segment, taken from PnPHeader}
    FunctionNum  : integer; {Contains BIOS function number}
    RetCode      : byte;    {Error/success code returned by BIOS}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_BIOS_GetESCDInfo:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Calculate segments and offsets required by function ***}
     NVSBaseSeg:= Seg(lNVSBase);
     NVSBaseOfs:= Ofs(lNVSBase);
     ESCDSizeSeg:= Seg(wESCDSize);
     ESCDSizeOfs:= Ofs(wESCDSize);
     TempSeg:= Seg(Temp);
     TempOfs:= Ofs(Temp);
     BIOS_Select:= PnP_BIOS_Header.RealDataSeg;

     {*** BIOS function 41h - Get Extended System Configuration Data
                              (ESCD) Information                          ***}
     FunctionNum:= $41;

     {*** Call using 'C' calling convention ***}
     asm
        push  BIOS_Select       {Push parameters onto stack}
        push  NVSBaseSeg
        push  NVSBaseOfs
        push  ESCDSizeSeg
        push  ESCDSizeOfs
        push  TempSeg
        push  TempOfs
        push  FunctionNum
        call  PnP_BIOS_Entry    {FAR call to the BIOS entry point}
        add   sp, 16            {Clean up the stack}
        mov   [RetCode], al     {Copy error/success code}
     end;

     {*** Return error/success code ***}
     PnP_BIOS_GetESCDInfo:= RetCode;
end;

(* PnP_BIOS_ReadESCD - returns all Extended System Configuration Data.
   (function 42h)
                       pESCDBuffer - buffer containing ESCD information

                       An error code is returned if ESCD is not
                       supported by the BIOS.
*)

function PnP_BIOS_ReadESCD(    lNVSBase    : longint;
                           var pESCDBuffer : pointer) : byte;

var NVSBaseSeg   : word;    {Holds segment address of lNVSBase variable}
    ESCDBufSeg   : word;    {Holds segment address of pESCDBuffer variable}
    ESCDBufOfs   : word;    {Holds offset address of pESCDBuffer variable}
    BIOS_Select  : word;    {Holds BIOS data segment, taken from PnPHeader}
    FunctionNum  : integer; {Contains BIOS function number}
    RetCode      : byte;    {Error/success code returned by BIOS}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_BIOS_ReadESCD:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Check buffer allocated ***}
     if pESCDBuffer = nil then
     begin
          {*** Return error code ***}
          PnP_BIOS_ReadESCD:= PNP_NO_BUFFER_MEMORY;
          Exit;
     end;

     {*** Calculate segments and offsets required by function ***}
     NVSBaseSeg:= lNVSBase div 16;
     ESCDBufSeg:= Seg(pESCDBuffer^);
     ESCDBufOfs:= Ofs(pESCDBuffer^);
     BIOS_Select:= PnP_BIOS_Header.RealDataSeg;

     {*** BIOS function 42h - Read Extended System Configuration Data ***}
     FunctionNum:= $42;

     {*** Call using 'C' calling convention ***}
     asm
        push  BIOS_Select       {Push parameters onto stack}
        push  NVSBaseSeg
        push  ESCDBufSeg
        push  ESCDBufOfs
        push  FunctionNum
        call  PnP_BIOS_Entry    {FAR call to the BIOS entry point}
        add   sp, 10            {Clean up the stack}
        mov   [RetCode], al     {Copy error/success code}
     end;

     {*** Return error/success code ***}
     PnP_BIOS_ReadESCD:= RetCode;
end;

{******}

begin
end.