unit SPNPISA;

(* Information
   

   Program Title : Plug and Play ISA support.
   External name : SPNPISA.TPU
   Version       : 1.0
   Start date    : 2/9/97
   Last update   : 7/9/97
   Author        : Rob Anderton.
   Description   : Plug and Play ISA routines
                   (PNP ISA 1.0A specification)
                   Portions based on Linux PnP source code
                   (See documentation)
*)

interface

{******}

uses SPNPTYPE;

{*** Functions/procedures ***}

function  PnP_ISA_Initiate : byte;

function  PnP_ISA_CheckSerialKey(aKey : TSerialKey) : byte;

function  PnP_ISA_DiscoverNext : byte;

function  PnP_ISA_Discover : byte;

function  PnP_ISA_SetReadDataPort(wReadPort : word) : byte;

function  PnP_ISA_Wake(bCSN : byte) : byte;

function  PnP_ISA_GetResourceData(var bTimeOut : boolean) : byte;

function  PnP_ISA_GetStatus : byte;

function  PnP_ISA_ReadCardData(    bCSN      : byte;
                               var pBuffer   : pointer;
                               var wBufSize  : word;
                               var aSerialID : TSerialKey) : byte;

{******}

implementation

{******}

uses CRT, SUTILS, SBITS;

{******}

(* PnP_ISA_Initiate - sends the initiation key required to enable Plug and
                      Play logic on ISA Plug and Play cards. This only needs
                      to be done at startup, or if a faulty BIOS has failed
                      to setup PnP ISA cards.
*)

function PnP_ISA_Initiate : byte;

var Loop : integer; {Loop control variable}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_ISA_Initiate:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Reset LFSR to initial state ready for Initiation Key ***}
     Port[ISA_ADDRESS_PORT]:= 0;
     Port[ISA_ADDRESS_PORT]:= 0;

     {*** Send the Initiation Key slowly ***}
     for Loop:= 1 to ISA_INITKEY_LENGTH do
     begin
          Delay(1);
          Port[ISA_ADDRESS_PORT]:= ISA_INITKEY_DATA[Loop];
     end;

     {*** Delay ***}
     Delay(1);

     {*** Return success code ***}
     PnP_ISA_Initiate:= PNP_SUCCESS;
end;

(* PnP_ISA_CheckSerialKey - checksums the serial key returned by PnP ISA
                            devices to identify themselves.

                            aKey - serial key returned by the PnP device

                            Returns the calculated checksum.
*)

function PnP_ISA_CheckSerialKey(aKey : TSerialKey) : byte;

var Check1 : byte;    {Check digit calculation variables}
    Check2 : byte;
    Loop1  : integer; {Loop control variables}
    Loop2  : integer;

begin
     {*** Initialise variable ***}
     Check1:= $6A;

     {*** Process bytes 0 to 7 of key (byte 8 is the checksum) ***}
     for Loop2:= 0 to 7 do
     begin
          {*** Process bits ***}
          for Loop1:= 0 to 7 do
          begin
               {*** Calculate checksum ***}
               if Bit_Test(Loop1, aKey[Loop2]) then Check2:= $01
                                               else Check2:= $00;
               Check2:= Check2 xor Check1;
               Check1:= Check1 shr 1;
               Check2:= Check2 xor Check1;
               Check1:= Check1 and $7F;
               Check1:= Check1 or ((Check2 shl 7) and $80);
          end;
     end;

     {*** Return calculated checksum ***}
     PnP_ISA_CheckSerialKey:= Check1;
end;

(* PnP_ISA_DiscoverNext - detects the next ISA PnP card in the system
                          and assigns a Card Select Number (CSN) to it.

                          Returns an error/success code.
*)

function PnP_ISA_DiscoverNext : byte;

var Check1  : byte;        {Used to decode serial identifier}
    Check2  : byte;
    Key     : TSerialKey;  {Holds PnP card's serial identifier}
    Loop1   : integer;     {Loop control variables}
    Loop2   : integer;
    Temp    : longint;     {Temporary variable for type conversion}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_ISA_DiscoverNext:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Put all cards into 'SLEEP' state ***}
     PnP_ISA_Initiate;

     {*** Send WAKE(0) to isolate all undiscovered cards ***}
     PnP_ISA_Wake(0);

     {*** Set read data port ***}
     PnP_ISA_SetReadDataPort(ISA_READ_DATA_PORT);
     Delay(1);

     {*** Read the serial key ***}
     Port[ISA_ADDRESS_PORT]:= $01; {Serial isolation port}
     Delay(1);

     for Loop2:= 0 to 8 do
     begin
          for Loop1:= 0 to 7 do
          begin
               {*** Read next pair, if $FFFF the clear bit, if $55AA set bit ***}
               Delay(1);
               Check1:= Port[ISA_READ_DATA_PORT];
               Delay(1);
               Check2:= Port[ISA_READ_DATA_PORT];

               Temp:= Key[Loop2];
               if ((Check1 <> $55) or (Check2 <> $AA)) then Bit_Clear(Loop1, Temp)
                                                       else Bit_Set(Loop1, Temp);
               Key[Loop2]:= Temp;
          end;
     end;

     {*** Checksum key to ensure it is correct ***}
     if (PnP_ISA_CheckSerialKey(Key) <> Key[8]) then
     begin
          PnP_ISA_DiscoverNext:= PNP_INVALID_SERIAL_ID;
          Exit;
     end;

     {*** Send a CSN to the card ***}
     Inc(ISA_LAST_CSN);
     Port[ISA_ADDRESS_PORT]:= $06;
     Port[ISA_WRITE_DATA_PORT]:= ISA_LAST_CSN;

     {*** Return success code ***}
     PnP_ISA_DiscoverNext:= PNP_SUCCESS;
end;

(* PnP_ISA_Discover - identifies and initialises all PnP ISA cards in the
                      system, returning appropriate error/success code.
*)

function PnP_ISA_Discover : byte;

const Valid_Address : word = 0;

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_ISA_Discover:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Determine first possible read port address ***}
     ISA_READ_DATA_PORT:= ISA_MIN_READ_PORT + 8;

     {*** Put all cards into 'SLEEP' state ***}
     PnP_ISA_Initiate;

     {*** Reset all CSN's ***}
     Port[ISA_ADDRESS_PORT]:= $02;      {Config control register}
     Port[ISA_WRITE_DATA_PORT]:= $04;   {Reset CSN command}
     Delay(1);

     {*** Put cards into 'SLEEP' state again ***}
     PnP_ISA_Initiate;
     Delay(1);

     {*** Look for a good read port ***}
     while (ISA_READ_DATA_PORT < ISA_MAX_READ_PORT) and
           (ISA_LAST_CSN < $FF) do
     begin
          if (PnP_ISA_DiscoverNext = PNP_SUCCESS) then
          begin
               {*** Remember working port address ***}
               Valid_Address:= ISA_READ_DATA_PORT;
               Continue;
          end;

          {*** Try a different port ***}
          if (Valid_Address <> 0) then Break;
          Inc(ISA_READ_DATA_PORT, 8);
     end;

     {*** Return cards to 'WAIT FOR KEY' state ***}
     Port[ISA_ADDRESS_PORT]:= $02;
     Port[ISA_WRITE_DATA_PORT]:= $02;

     {*** Indicate if cards were found ***}
     if (Valid_Address = 0) then
     begin
          PnP_ISA_Discover:= PNP_NO_ISA_PNP_CARDS;
          exit;
     end;

     {*** Set read port ***}
     ISA_READ_DATA_PORT:= Valid_Address;

     {*** Return success code ***}
     PnP_ISA_Discover:= PNP_SUCCESS;
end;

(* PnP_ISA_SetReadDataPort - sets the ISA read data port (which is a
                             relocatable port in the range 0203h-03FFh).

                             wReadPort - address of read data port.

                             NOTE: no range checking is performed on the
                                   value of wReadPort.
*)


function PnP_ISA_SetReadDataPort(wReadPort : word) : byte;

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_ISA_SetReadDataPort:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Send Read Data Port address ***}
     Port[ISA_ADDRESS_PORT]:= $00;
     {*** Send new port value byte ***}
     Port[ISA_WRITE_DATA_PORT]:= wReadPort shr 2;
     {*** Update global variable ***}
     ISA_READ_DATA_PORT:= wReadPort or 3;
     {*** Return success code ***}
     PnP_ISA_SetReadDataPort:= PNP_SUCCESS;
end;

(* PnP_ISA_Wake - sets the card identified by bCSN to the 'CONFIG' state, or
                  if bCSN is zero, sets all cards to the 'ISOLATION' state.

                  bCSN - card select number to 'WAKE'

*)

function PnP_ISA_Wake(bCSN : byte) : byte;

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_ISA_Wake:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Send WAKE command to address port ***}
     Port[ISA_ADDRESS_PORT]:= $03;
     {*** Send Card Select Number to write data port ***}
     Port[ISA_WRITE_DATA_PORT]:= bCSN;
     {*** Return success code ***}
     PnP_ISA_Wake:= PNP_SUCCESS;
end;

(* PnP_ISA_GetResourceData - returns one byte of resource data.

                             bTimeOut - a boolean value, true if the
                                        function timed out waiting for
                                        resource data, false otherwise
*)

function PnP_ISA_GetResourceData(var bTimeOut : boolean) : byte;

var Loop  : integer; {Time out loop variable}

begin
     {*** Wait for status bit to indicate data ready ***}
     Loop:= 32767; {Time out loop}

     while ((Loop > 0) and
            ((PnP_ISA_GetStatus and $01) <> 1)) do Dec(Loop);

     {*** Signal a time out if one occurred ***}
     if Loop <= 0 then
     begin
          bTimeOut:= true;
          PnP_ISA_GetResourceData:= 0;
          Exit;
     end;

     {*** Read and return ISA Data ***}
     bTimeOut:= false;

     {*** Send Resource Data address ***}
     Port[ISA_ADDRESS_PORT]:= $04;
     {*** Return next data byte ***}
     PnP_ISA_GetResourceData:= Port[ISA_READ_DATA_PORT];
end;

(* PnP_ISA_GetStatus - return the status register contents.
                       Bit 0 is set when resource data can be read from
                       the resource data register.
*)

function PnP_ISA_GetStatus : byte;

begin
     {*** Send Status address ***}
     Port[ISA_ADDRESS_PORT]:= $05;
     {*** Return status byte ***}
     PnP_ISA_GetStatus:= Port[ISA_READ_DATA_PORT];
end;

(* PnP_ISA_ReadCardData - returns all available resource data from the
                          resource data register.

                          bCSN      - Card Select Number of card to return
                                      data about

                          pBuffer   - pointer to returned data

                          wBufSize  - size of data in bytes

                          aSerialID - serial ID of card

                          Returns an error/success code.
*)

function PnP_ISA_ReadCardData(    bCSN      : byte;
                              var pBuffer   : pointer;
                              var wBufSize  : word;
                              var aSerialID : TSerialKey) : byte;

var Loop    : integer;    {Loop control variable}
    TimeOut : boolean;    {Holds result of PnP_ISA_GetResourceData}
    Buf     : PByteArray; {Temporary buffer for holding card data}
    Count   : word;       {Temporary store for size of buffer}
    Key     : TSerialKey; {Temporary store for serial ID}

begin
     {*** Check PnP BIOS present ***}
     if not PnP_BIOS_Present then
     begin
          {*** Return error code ***}
          PnP_ISA_ReadCardData:= PNP_BIOS_NOT_PRESENT;
          Exit;
     end;

     {*** Allocate maximum available buffer (64K) ***}
     GetMem(Buf, 64000);
     if Buf = nil then
     begin
          PnP_ISA_ReadCardData:= PNP_NO_BUFFER_MEMORY;
          Exit;
     end;

     {*** Send 'WAKE(CSN)' command to access specified card ***}
     PnP_ISA_Wake(bCSN);

     {*** Read the card's serial key ***}
     Loop:= 0;
     TimeOut:= false;
     while (Loop < 9) and (not TimeOut) do
     begin
          Key[Loop]:= PnP_ISA_GetResourceData(TimeOut);
          Inc(Loop);
     end;

     {*** Set return value ***}
     aSerialID:= Key;

     {*** Initialise variables ***}
     Loop:= 0;
     Count:= 0;
     TimeOut:= false;

     {*** Read from the card until the end tag is found ***}
     while (not TimeOut) do
     begin
          Buf^[Loop]:= PnP_ISA_GetResourceData(TimeOut);
          Inc(Count);

          {*** Stop when end tag is found ***}
          if (Buf^[Loop] = $79) then
          begin
               Inc(Loop);
               Buf^[Loop]:= PnP_ISA_GetResourceData(TimeOut);
               Inc(Count);
               Inc(Loop);
               {*** Exit loop immediately ***}
               Break;
          end;

          Inc(Loop);
     end;

     {*** Allocate return buffer ***}
     GetMem(pBuffer, Count);

     {*** Transfer data ***}
     Move(Buf^, pBuffer^, Count);

     {*** Free temporary buffer ***}
     FreeMem(Buf, 64000);

     {*** Return size of data ***}
     wBufSize:= Count;

     {*** Return success code ***}
     PnP_ISA_ReadCardData:= PNP_SUCCESS;
end;

{******}

begin
end.