unit SPNPPCI;

(* Information
   

   Program Title : Plug and Play PCI support.
   External name : SPNPPCI.TPU
   Version       : 1.0
   Start date    : 11/9/97
   Last update   : 14/9/97
   Author        : Rob Anderton.
   Description   : Plug and Play PCI interface routines
                   (PCI 2.0c specification)

*)

interface

{******}

type
     {*** Array of bytes containing PCI registers 0000h-0040h ***}
     PPCIConfigData = ^TPCIConfigData;
     TPCIConfigData = array[0..63] of byte;

{******}

function PnP_PCI_Init : byte;

function PnP_PCI_FindDevice(    wVendorID   : word;
                                wDeviceID   : word;
                                wDeviceIDX  : word;
                            var wDeviceInfo : word) : byte;

function PnP_PCI_ReadConfigByte(    wDeviceInfo : word;
                                    wRegister   : word;
                                var bConfigData : byte) : byte;

function PnP_PCI_ReadConfigWord(    wDeviceInfo : word;
                                    wRegister   : word;
                                var wConfigData : word) : byte;

function PnP_PCI_ReadConfigLong(    wDeviceInfo : word;
                                    wRegister   : word;
                                var lConfigData : longint) : byte;

function PnP_PCI_ReadConfigData(    wDeviceInfo : word;
                                var rConfigData : TPCIConfigData) : byte;

{******}

implementation

{******}

uses SPNPTYPE, SUTILS, DOS;

{*** Link in PCI assembly module ***}

{$L SPNPPCI.OBJ}

(* PnP_PCI_DetectBIOS - external assembly routine. NASM is used to provide
                        easy access to 32 bit registers used by PCI BIOS call.

*)

procedure PnP_PCI_DetectBIOS; external;

(* PnP_PCI_ReadConfigLongint - external assembly routine. NASM is used to
                               provide easy access to 32 bit registers
                               used by PCI BIOS call.
*)

procedure PnP_PCI_ReadConfigLongint; external;

{******}

(* PnP_PCI_Init - detects if a PCI BIOS is present, if one is found then
                  initialises variables, and PCI support is made available
                  for the PnP package.

                  Returns an error code.
*)


function PnP_PCI_Init : byte;

var MajVer, MinVer : byte;                {Temporary store for version number}
    Signature      : array[1..4] of char; {Stores PCI signature 'PCI '}
    TempSig1       : word;                {Used for processing signature}
    TempSig2       : word;
    ConfigType     : byte;                {Temporary store for configuration
                                           method}
    ErrorCode      : byte;                {Temporary store for error code}

begin
     asm
        call    PnP_PCI_DetectBIOS;  {Call external routine}
        mov     MinVer, bl           {Copy registers into variables}
        mov     MajVer, bh
        mov     TempSig1, di
        mov     TempSig2, dx
        mov     ConfigType, al
     end;

     {*** Build PCI signature ***}
     LONGINT(Signature):= TempSig1 + (LONGINT(TempSig2) shl 16);

     {*** Check signature is valid and version is greater than or equal to
          2.10                                                            ***}

     if (Signature = 'PCI ') and
        (MajVer >= $02) then
     begin
          {*** Set variables ***}
          PnP_PCI_Present:= true;
          ErrorCode:= PNP_SUCCESS;

          {*** Get configuration mechanism supported ***}
          case (ConfigType and $03) of

                           1 : PnP_PCI_Config:= 1;
                           2 : PnP_PCI_Config:= 2;
                           3 : PnP_PCI_Config:= 1;

                          else
                          begin
                               {*** Unknown mechanisim type ***}
                               PnP_PCI_Config:= 0;
                               PnP_PCI_Present:= false;

                               {*** Return error code ***}
                               ErrorCode:= PNP_INVALID_PCI_CONFIG;
                          end;
               Exit;
          end;
     end
     else
     begin
          {*** Update variables ***}
          PnP_PCI_Present:= false;
          PnP_PCI_Config:= 0;
          ErrorCode:= PNP_PCI_BIOS_NOT_PRESENT;
     end;

     {*** Return error/success code ***}
     PnP_PCI_Init:= ErrorCode;
end;

(* PnP_PCI_FindDevice - uses PCI BIOS function B102h to located a PCI
                        device using it's Vendor and Device ID.

                        wVendorID  - device's Vendor ID
                        wDeviceID  - device's ID
                        wDeviceIDX - used to distinguish between multiple
                                     cards with same Vendor and Device ID's

                        wDeviceInfo - word returned by function containing
                                      device's bus number in the upper byte,
                                      and it's device/function number in the
                                      lower byte. This information is used
                                      by other BIOS functions.

                        This function returns an error/status code.

*)

function PnP_PCI_FindDevice(    wVendorID   : word;
                                wDeviceID   : word;
                                wDeviceIDX  : word;
                            var wDeviceInfo : word) : byte;

var Regs      : Registers; {Used to call PCI BIOS function}
    ErrorCode : word;      {Temporary store for error/success code}

begin
     with Regs do
     begin
          AX:= $B102;       {Function number}
          CX:= wDeviceID;   {Device ID}
          DX:= wVendorID;   {Vendor ID}
          SI:= wDeviceIDX;  {Device index (for multiple devices)}
     end;
     Intr($1A, Regs);       {Call function}

     if ((Regs.Flags and fCarry) = 0) and (Regs.AH = $00) then {No error}
     begin
          wDeviceInfo:= Regs.BX; {Return device information}
          ErrorCode:= PNP_SUCCESS;
     end
     else case Regs.AH of {Deal with error code}
                    $83 : ErrorCode:= PNP_PCI_BAD_VENDOR_ID;
                    $86 : ErrorCode:= PNP_PCI_DEVICE_NOT_FOUND;
                   else   ErrorCode:= PNP_GENERIC_ERROR;
          end;

     {*** Return error code ***}
     PnP_PCI_FindDevice:= ErrorCode;
end;


(* PnP_PCI_ReadConfigByte - reads a byte from a specified configuration
                            register of a PCI device.

                            wDeviceInfo - combined device, function and bus
                                          number of required device (this
                                          information is returned by
                                          PnP_PCI_FindDevice)

                            wRegister   - the PCI register to read from
                                          (0000h-00FFh)

                            bConfigData - the returned configuration data

                            This function returns an error/success code.
*)

function PnP_PCI_ReadConfigByte(    wDeviceInfo : word;
                                    wRegister   : word;
                                var bConfigData : byte) : byte;

var Regs      : Registers; {Used to call PCI BIOS function}
    ErrorCode : word;      {Temporary store for error/success code}

begin
     with Regs do
     begin
          AX:= $B108;       {Function number}
          BX:= wDeviceInfo; {Device, function and bus number}
          DI:= wRegister;   {PCI configuration register number}
     end;
     Intr($1A, Regs);       {Call function}

     bConfigData:= 0;

     if ((Regs.Flags and fCarry) = 0) and (Regs.AH = $00) then {No error}
     begin
          bConfigData:= Regs.CL; {Return configuration data}
          ErrorCode:= PNP_SUCCESS;
     end
     else case Regs.AH of {Deal with error code}
                    $87 : ErrorCode:= PNP_PCI_INVALID_REGISTER;
                   else   ErrorCode:= PNP_GENERIC_ERROR;
          end;

     {*** Return error code ***}
     PnP_PCI_ReadConfigByte:= ErrorCode;
end;


(* PnP_PCI_ReadConfigWord - reads a word from a specified configuration
                            register of a PCI device.

                            wDeviceInfo - combined device, function and bus
                                          number of required device (this
                                          information is returned by
                                          PnP_PCI_FindDevice)

                            wRegister   - the PCI register to read from
                                          (0000h-00FFh)

                            wConfigData - the returned configuration data

                            This function returns an error/success code.
*)

function PnP_PCI_ReadConfigWord(    wDeviceInfo : word;
                                    wRegister   : word;
                                var wConfigData : word) : byte;

var Regs      : Registers; {Used to call PCI BIOS function}
    ErrorCode : word;      {Temporary store for error/success code}

begin
     with Regs do
     begin
          AX:= $B109;       {Function number}
          BX:= wDeviceInfo; {Device, function and bus number}
          DI:= wRegister;   {PCI configuration register number}
     end;
     Intr($1A, Regs);       {Call function}

     wConfigData:= 0;

     if ((Regs.Flags and fCarry) = 0) and (Regs.AH = $00) then {No error}
     begin
          wConfigData:= Regs.CX;   {Return configuration data}
          ErrorCode:= PNP_SUCCESS;
     end
     else case Regs.AH of {Deal with error code}
                    $87 : ErrorCode:= PNP_PCI_INVALID_REGISTER;
                   else   ErrorCode:= PNP_GENERIC_ERROR;
          end;

     {*** Return error code ***}
     PnP_PCI_ReadConfigWord:= ErrorCode;
end;

(* PnP_PCI_ReadConfigLong - reads a dword from a specified configuration
                            register of a PCI device.

                            wDeviceInfo - combined device, function and bus
                                          number of required device (this
                                          information is returned by
                                          PnP_PCI_FindDevice)

                            wRegister   - the PCI register to read from
                                          (0000h-00FFh)

                            lConfigData - the returned configuration data

                            This function returns an error/success code.
*)

function PnP_PCI_ReadConfigLong(    wDeviceInfo : word;
                                    wRegister   : word;
                                var lConfigData : longint) : byte;

var ErrorCode : word;      {Temporary store for error/success code}
    TempHi    : word;      {Used to get 32bit data from 16bit registers}
    TempLo    : word;
    Carry     : byte;      {Holds carry flag}
    Status    : byte;      {Holds status (AH register)}

begin
     Carry:= 0;
     asm
        mov  bx, wDeviceInfo  {Device, function and bus number}
        mov  di, wRegister    {PCI configuration register number}

        call PnP_PCI_ReadConfigLongint

        mov  Status, ah       {Store status code}
        mov  TempLo, dx       {Store low word of data}
        mov  TempHi, cx       {Store high word}
        jnc  @EXIT

        mov  Carry, 1         {Indicate carry flag set}

     @EXIT:
     end;

     lConfigData:= 0;

     if (Carry = 0) and (Status = $00) then {No error}
     begin
          {*** Return configuration data ***}
          lConfigData:= (LONGINT(TempHi) shl 16) + TempLo;
          ErrorCode:= PNP_SUCCESS;
     end
     else case Status of {Deal with error code}
                    $87 : ErrorCode:= PNP_PCI_INVALID_REGISTER;
                   else   ErrorCode:= PNP_GENERIC_ERROR;
          end;

     {*** Return error code ***}
     PnP_PCI_ReadConfigLong:= ErrorCode;
end;

(* PnP_PCI_ReadConfigData - reads all configuration registers from 0000h
                            up to (but not including 00040h) of a PCI device.

                            wDeviceInfo - combined device, function and bus
                                          number of required device (this
                                          information is returned by
                                          PnP_PCI_FindDevice)

                            rConfigData - the returned configuration data

                            This function returns an error/success code.
*)

function PnP_PCI_ReadConfigData(    wDeviceInfo : word;
                                var rConfigData : TPCIConfigData) : byte;

var Loop    : byte; {Loop control variable}
    RetCode : byte; {Return code of called functions}

begin
     {*** Initialise variables ***}
     RetCode:= PNP_SUCCESS;
     Loop:= 0;
     FillChar(rConfigData, sizeof(rConfigData), 0);

     while (Loop <= 63) and (RetCode = PNP_SUCCESS) do
     begin
          RetCode:= PnP_PCI_ReadConfigByte(wDeviceInfo, Loop,
                                           rConfigData[Loop]);
          Inc(Loop);
     end;

     {*** Return error code ***}
     PnP_PCI_ReadConfigData:= RetCode;
end;

{******}

end.