unit SCDROM;

(* Information
   

   Program Title : MSCDEX CD-ROM support.
   External name : SCDROM.TPU
   Version       : 1.2.
   Start date    : 22/8/96
   Last update   : 29/8/97.
   Author        : Rob Anderton.
   Description   : A unit to access CD-ROM driver functions (to open/close
                   the tray, etc.).
                   Based on the CD-ROM programming FAQ version 1.00 by
                   Marcus W. Johnson and PC Intern by Michael Tischer.
*)

interface

{******}

uses OBJECTS;

{*** RedBook address data ***}

type TRedBookAddress = record
                             Frame  : byte;
                             Second : byte;
                             Minute : byte;
                             Dummy  : byte;  {Not used}
                       end;

{*** CD-ROM track information ***}

type TCDROMTrackInfo = record
                             RedBookAddress : TRedBookAddress; {Start position
                                                                of current
                                                                track in Red
                                                                Book format}
                             TrackControl   : word; {Track control information}
                       end;

{*** CD-ROM disc sector information ***}

type TCDROMSectorInfo = record
                              Raw             : boolean; {True if raw mode,
                                                          false if cooked mode}
                              SectorSize      : word;    {Sector size in bytes}
                              NumberOfSectors : longint; {Size of volume in
                                                          sectors}
                        end;

{*** CD-ROM read head position information ***}

type TCDROMPositionInfo = record
                                ControlADR : byte;  {CONTROL and ADR byte}
                                TrackNum   : byte;  {Track number (BCD)}
                                CurrentIDX : byte;  {Current index (BCD)}
                                TrackInfo  : record {Time within current track}
                                                   Minute : byte;
                                                   Second : byte;
                                                   Frame  : byte;
                                             end;
                                Reserved   : byte;  {Always zero}
                                TotalInfo  : record {Time within total playing time}
                                                   Minute : byte;
                                                   Second : byte;
                                                   Frame  : byte;
                                             end;
                          end;

{*** CD-ROM drive object ***}

type TCDROMDrive = object(TObject)
                         wError       : word;
                         hDevice      : text;
                         cDrive       : char;
                         wDriveNum    : word;
                         rDriveInfo   : record
                                              SubUnitNum : byte;
                                              Name       : string[8];
                                        end;

                         {*** Object procedures ***}
                         constructor Init(cDriveLetter : char);
                         destructor  Done; virtual;

                         {*** CD-ROM drive procedures ***}
                         procedure   Reset;
                         procedure   OpenDoor;
                         procedure   CloseDoor;
                         procedure   LockDoor;
                         procedure   UnlockDoor;

                         {*** CD-ROM drive/disc information functions ***}
                         function    GetDriveStatus : word;
                         procedure   GetSectorInfo(bRaw : boolean; var rSectorInfo : TCDROMSectorInfo);
                         procedure   GetPosition(var rPosition : TCDROMPositionInfo);

                         {*** CD-Audio functions ***}
                         function    GetNumberOfAudioTracks : byte;
                         function    GetFirstTrack : byte;
                         function    GetLastTrack : byte;
                         function    GetTrackLength(bTrack : byte) : longint;
                         procedure   GetTrackInfo(bTrack : byte; var rTrackInfo : TCDROMTrackInfo);
                         procedure   PlayTrack(bTrack : byte);
                         procedure   PausePlay;
                         procedure   ResumePlay;

                      private

                         function DeviceCommand(iCommand : integer; pControlBlock : pointer;
                                                iControlBlockSize : integer) : word;

                         function RedBookToHSG(bMinute, bSecond, bFrame : byte) : longint;

     end;

{*** MSCDEX information structure ***}

type TMSCDEXInfo = record
                         Installed   : boolean; {True if MSCDEX is installed}
                         Version     : word;    {High byte = major version
                                                 Low byte  = minor version}
                         TotalDrives : byte;    {The number of CD-ROM drives
                                                 installed in the system}
                         DriveLetters: string;  {Drive letters assigned to
                                                 CD-ROM drives}
                   end;

{******}

implementation

{******}

uses SUTILS, DOS;

{*** Unit variables ***}

var MSCDEX_Info : TMSCDEXInfo;

{******}

(* TCDROMDrive.Init - initialise CD-ROM object for specified drive.

                      cDriveLetter - the letter of the CD-ROM drive to use
                                     (e.g. D, E or Z)

*)

constructor TCDROMDrive.Init(cDriveLetter : char);

var DriveNum   : word; {Drive number (converted from cDriveLetter parameter)}
    DriveRes   : word; {Result of drive letter test}

    {*** Record to hold driver information for all CD-ROM drives ***}
    DeviceInfo : array[0..25] of record
                                       SubUnitNum : byte;
                                       NamePtr    : PCharArray;
                                 end;

    DeviceIDX  : word; {Index into DeviceIDX array for current drive}

    DeviceSeg,
    DeviceOfs  : word; {Segment and offset of DeviceInfo array}

    Loop       : word; {Loop control variable}

begin
     inherited Init;

     {*** Check MSCDEX information to ensure initialisation is successful ***}
     with MSCDEX_Info do
     begin
          {*** MSCDEX installed ? ***}
          if not Installed then Fail;

          {*** MSCDEX version greater than 2.10 ? ***}
          if WORDREC(Version).Hi < 2 then Fail;
          if WORDREC(Version).Lo < 10 then Fail;

          {*** At least one CD-ROM drive installed ? ***}
          if TotalDrives = 0 then Fail;

          {*** Has a valid drive letter been given ? ***}
          DriveNum:= Ord(cDriveLetter) - 65;
          asm
             mov  ax, $150B
             mov  cx, DriveNum
             int  $2F
             mov  DriveRes, ax
          end;
          if DriveRes = 0 then Fail;
     end;

     {*** Get device driver name ***}
     for Loop:= 0 to 25 do
     begin
          DeviceInfo[Loop].SubUnitNum:= 0;
          DeviceInfo[Loop].NamePtr:= nil;
     end;

     {*** Set object device information record to 0's ***}
     rDriveInfo.SubUnitNum:= 0;
     rDriveInfo.Name:= '';

     DeviceSeg:= Seg(DeviceInfo);
     DeviceOfs:= Ofs(DeviceInfo);

     asm
        mov  ax, $1501
        mov  bx, DeviceSeg
        mov  es, bx
        mov  bx, DeviceOfs
        int  $2F
     end;

     {*** Calculate DeviceIDX for chosen drive ***}
     DeviceIdx:= Pos(cDriveLetter, MSCDEX_Info.DriveLetters);

     {*** If not found then fail ***}
     if DeviceIdx = 0 then Fail;

     Dec(DeviceIdx); {Array is zero based}

     {*** Fail if driver name not returned ***}
     if DeviceInfo[DeviceIdx].NamePtr = nil then Fail;

     {*** Process driver name ***}
     Loop:= 10;
     while (Loop <= 17) and (DeviceInfo[DeviceIdx].NamePtr^[Loop] <> ' ') do
     begin
          rDriveInfo.Name:= rDriveInfo.Name + DeviceInfo[DeviceIdx].NamePtr^[Loop];
          Inc(Loop);
     end;
     rDriveInfo.SubUnitNum:= DeviceInfo[DeviceIdx].SubUnitNum;

     {*** Open the driver file ***}
     Assign(hDevice, rDriveInfo.Name);
     SYSTEM.Reset(hDevice);
     cDrive:= cDriveLetter;
     wDriveNum:= DriveNum;

     {*** Reset the drive ***}
     Reset;
end;


(* TCDROMDrive.Done - dispose of the CD-ROM object *)

destructor TCDROMDrive.Done;

begin
     {*** Close device driver ***}
     Close(hDevice);
     inherited Done;
end;


(* TCDROMDrive.Reset - resets and reinitialises the CD-ROM drive *)

procedure TCDROMDrive.Reset;

var Data : byte;

begin
     Data:= 2;
     wError:= DeviceCommand($0C, @Data, 1);
end;

(* TCDROMDrive.OpenDoor - ejects the CD-ROM *)

procedure TCDROMDrive.OpenDoor;

var Data : byte;

begin
     Data:= 0;
     wError:= DeviceCommand($0C, @Data, 1);
end;

(* TCDROMDrive.CloseDoor - closes the CD-ROM drive tray (not all drives
                           support this command)
*)

procedure TCDROMDrive.CloseDoor;

var Data : byte;

begin
     Data:= 5;
     wError:= DeviceCommand($0C, @Data, 1);
end;

(* TCDROMDrive.LockDoor - locks the CD-ROM door (on some drives the door
                          is physically locked)
*)

procedure TCDROMDrive.LockDoor;

var Data : word;

begin
     Data:= $0001;
     wError:= DeviceCommand($0C, @Data, 2);
end;

(* TCDROMDrive.UnLockDoor - unlocks the CD-ROM door *)

procedure TCDROMDrive.UnlockDoor;

var Data : word;

begin
     Data:= $0101;
     wError:= DeviceCommand($0C, @Data, 2);
end;

(* TCDROMDrive.GetDriveStatus - returns drive status (NOTE: drive status
                                is NOT the same as the error information
                                stored in the wError field).
*)

function TCDROMDrive.GetDriveStatus : word;

var Data : record
                 Command : byte;  {Device driver command}
                 Status  : word;  {Drive status word}
                 Dummy   : word;  {The device driver returns a DWORD but
                                   only the lowest 13 bits are used, so
                                   upper word can be ignored}
           end;

begin
     Data.Command:= 6;
     wError:= DeviceCommand($03, @Data, 5);
     GetDriveStatus:= Data.Status and 2047;   {Mask off unused bits}
end;

(* TCDROMDrive.GetSectorInfo - return information about sector size and
                               total sectors for current CD-ROM disc.

                               bRaw - true if Raw sector information
                                      required, false for cooked.

                               rSectorInfo - record containing returned
                                             information.
*)

procedure TCDROMDrive.GetSectorInfo(bRaw : boolean; var rSectorInfo : TCDROMSectorInfo);

var Data1 : record
                  Command    : byte;  {Device driver command}
                  Raw        : byte;  {Raw or cooked mode}
                  SectorSize : word;  {Size of sector in bytes}
            end;

    Data2 : record
                  Command         : byte;     {Device driver command}
                  NumberOfSectors : longint;  {Number of sectors on disc}
            end;

begin
     Data1.Command:= 7;
     Data1.Raw:= BYTE(not bRaw);
     wError:= DeviceCommand($03, @Data1, 4);
     if wError <> 256 then exit; {Exit if error}

     Data2.Command:= 8;
     wError:= DeviceCommand($03, @Data2, 5);

     {*** Return sector information ***}
     rSectorInfo.Raw:= (Data1.Raw = 0);
     rSectorInfo.SectorSize:= Data1.SectorSize;
     rSectorInfo.NumberOfSectors:= Data2.NumberOfSectors;
end;

(* TCDROMDrive.GetPosition - returns information about the current position
                             of the CD-ROM read head.

                             rPosition - contains returned position info.
*)

procedure TCDROMDrive.GetPosition(var rPosition : TCDROMPositionInfo);

var Data : record
                 Command : byte;               {Device driver command}
                 Info    : TCDROMPositionInfo; {Position information}
           end;

begin
     Data.Command:= 12;
     wError:= DeviceCommand($03, @Data, 129);
     rPosition:= Data.Info;
end;

(* TCDROMDrive.GetNumberOfAudioTracks - returns the number of audio tracks
                                        on the current CD-ROM disc.
*)

function TCDROMDrive.GetNumberOfAudioTracks : byte;

var Data : record
                 Command : byte;     {Device driver command}
                 First   : byte;     {Number of first audio track}
                 Last    : byte;     {Number of last audio track}
                 Dummy   : longint;  {Not used by this function}
           end;

begin
     Data.Command:= $0A;
     wError:= DeviceCommand($03, @Data, 7);
     GetNumberOfAudioTracks:= Data.Last - Data.First + 1;
end;

(* TCDROMDrive.GetFirstTrack - returns the number of the first track *)

function TCDROMDrive.GetFirstTrack : byte;

var Data : record
                 Command : byte;     {Device driver command}
                 First   : byte;     {Number of first audio track}
                 Last    : byte;     {Number of last audio track}
                 Dummy   : longint;  {Not used by this function}
           end;

begin
     Data.Command:= $0A;
     wError:= DeviceCommand($03, @Data, 7);
     GetFirstTrack:= Data.First;
end;

(* TCDROMDrive.GetLastTrack - returns the number of the last track *)

function TCDROMDrive.GetLastTrack : byte;

var Data : record
                 Command : byte;     {Device driver command}
                 First   : byte;     {Number of first audio track}
                 Last    : byte;     {Number of last audio track}
                 Dummy   : longint;  {Not used by this function}
           end;

begin
     Data.Command:= $0A;
     wError:= DeviceCommand($03, @Data, 7);
     GetLastTrack:= Data.Last;
end;

(* TCDROMDrive.GetTrackLength - returns length of specified track in frames.

                                bTrack - the track to return length of.
*)

function TCDROMDrive.GetTrackLength(bTrack : byte) : longint;

var TrackData        : TCDROMTrackInfo;  {Holds results of GetTrackInfo calls}
    HSGStart         : longint;          {Track start address in HSG format}
    HSGEnd           : longint;          {Track end address in HSG format}
    MaxTrack         : byte;             {Highest track number}
    SectorInfo       : TCDROMSectorInfo; {Used to get size of CD volume}

begin
     {*** Check bTrack is valid track number ***}
     MaxTrack:= GetLastTrack; {This information is needed later on}
     if (bTrack < GetFirstTrack) or (bTrack > MaxTrack) then
     begin
          GetTrackLength:= 0;
          Exit;
     end;

     {*** Get bTrack start address ***}
     GetTrackInfo(bTrack, TrackData);
     if (wError <> 256) then Exit;

     {*** Convert RedBook address to HSG format ***}
     HSGStart:= RedBookToHSG(TrackData.RedBookAddress.Minute,
                             TrackData.RedBookAddress.Second,
                             TrackData.RedBookAddress.Frame);

     if (bTrack < MaxTrack) then
     begin
          {*** As it is not the last track we can calculate the length
               using the start address of the next track               ***}

          GetTrackInfo(bTrack + 1, TrackData);
          if (wError <> 256) then Exit;
          {*** Convert RedBook address to HSG format ***}
          HSGEnd:= RedBookToHSG(TrackData.RedBookAddress.Minute,
                                TrackData.RedBookAddress.Second,
                                TrackData.RedBookAddress.Frame);
     end
     else
     begin
          {*** If bTrack is the last track then calculate length of track
               using size of CD volume                                    ***}
          GetSectorInfo(True, SectorInfo);
          HSGEnd:= SectorInfo.NumberOfSectors - 150;
     end;

     GetTrackLength:= HSGEnd - HSGStart;
end;

(* TCDROMDrive.GetTrackInfo - returns information about the start time of
                              the track specified in bTrack.

                              bTrack - track number to retrieve information
                                       about

                              rTrackInfo - returned track information

*)

procedure TCDROMDrive.GetTrackInfo(bTrack : byte; var rTrackInfo : TCDROMTrackInfo);

var Data : record
                 Command        : byte;  {Device driver command}
                 Track          : byte;  {Track number to get information about}
                 RedBookAddress : TRedBookAddress; {Start address of track}
                 TrackControl   : word;  {Track control information}
           end;

begin
     Data.Command:= $0B;
     Data.Track:= bTrack;
     wError:= DeviceCommand($03, @Data, 8);
     rTrackInfo.RedBookAddress:= Data.RedBookAddress;
     rTrackInfo.TrackControl:= Data.TrackControl;
end;

(* TCDROMDrive.PlayTrack - starts playing the track specified in bTrack
                           and stops at the end of the track.

                           bTrack - track number to play.

*)

procedure TCDROMDrive.PlayTrack(bTrack : byte);

var Data : record
                Bytes          : array[0..13] of byte; {Contains control data}
                HSGAddress     : longint; {HSG address to start playing from}
                NumberOfFrames : longint; {Number of frames to play}
           end;

    TrackData : TCDROMTrackInfo;  {Holds results of GetTrackInfo calls}
    HSGStart  : longint;   {Start address to play from in HSG format}
    Regs      : Registers; {Used for function call}

begin
     PausePlay;

     {*** Check bTrack is valid track number ***}
     if (bTrack < GetFirstTrack) or (bTrack > GetLastTrack) then Exit;

     {*** Get bTrack start address ***}
     GetTrackInfo(bTrack, TrackData);
     if (wError <> 256) then Exit;

     {*** Convert RedBook address to HSG format ***}
     HSGStart:= RedBookToHSG(TrackData.RedBookAddress.Minute,
                             TrackData.RedBookAddress.Second,
                             TrackData.RedBookAddress.Frame);

     {*** Check that the track is an Audio track ***}
     if TrackData.TrackControl and 16384 <> 0 then Exit;

     {*** Send play command ***}
     Data.Bytes[0]:= $16;
     Data.Bytes[1]:= rDriveInfo.SubUnitNum;
     Data.Bytes[2]:= $84;
     Data.Bytes[13]:= $0;
     Data.HSGAddress:= HSGStart;
     Data.NumberOfFrames:= GetTrackLength(bTrack);

     Regs.AX:= $1510;
     Regs.ES:= Seg(Data);
     Regs.BX:= Ofs(Data);
     Regs.CX:= wDriveNum;
     Intr($2F, Regs);
end;

(* TCDROMDrive.PausePlay - stops playback of Audio CD. *)

procedure TCDROMDrive.PausePlay;

var Data : array[0..12] of byte;
    Regs : Registers;

begin
     Data[0]:= $0D;
     Data[1]:= rDriveInfo.SubUnitNum;
     Data[2]:= $85;
     Regs.AX:= $1510;
     Regs.CX:= wDriveNum;
     Regs.ES:= Seg(Data);
     Regs.BX:= Ofs(Data);
     Intr($2F, Regs);
end;

(* TCDROMDrive.ResumePlay - restarts audio playback after a PausePlay call *)

procedure TCDROMDrive.ResumePlay;

var Data : array[0..12] of byte;
    Regs : Registers;

begin
     Data[0]:= $0D;
     Data[1]:= rDriveInfo.SubUnitNum;
     Data[2]:= $88;
     Regs.AX:= $1510;
     Regs.CX:= wDriveNum;
     Regs.ES:= Seg(Data);
     Regs.BX:= Ofs(Data);
     Intr($2F, Regs);
end;

(* TCDROMDrive.DeviceCommand - sends a command to the CD-ROM device driver
                               using the DOS IOCTL system.

                               Command          - $4402 = read from device
                                                  $4403 = write to device

                               ControlBlock     - IOCTL control block data

                               ControlBlockSize - size of ControlBlock

*)

function TCDROMDrive.DeviceCommand(iCommand : integer; pControlBlock : pointer;
                                   iControlBlockSize : integer) : word;

{*** DOS device interface types ***}

type TRequestHeader = record
                            bLength   : byte;  {Length of request structure}
                            bSubUnit  : byte;  {Number of sub-unit}
                            bCommand  : byte;  {Command to be executed}
                            wStatus   : word;  {Error status}
                            {Array holds command specific data}
                            bReserved : array[0..7] of byte;
                      end;

     TIOCTLInfo = record
                        rReqHdr            : TRequestHeader;
                        bMediaDescriptor   : byte;    {always 0}
                        lpTransferAddress  : pointer; {address of control block}
                        wTransferCount     : word;    {size of control block}
                        wStartingSectorNo  : word;    {always 0}
                        lpVolumeID         : pointer; {always nil}
                  end;

{******}

var IOCTLInfo : TIOCTLInfo;
    ReqHdr    : ^TRequestHeader;
    Regs      : Registers;

begin
     {*** Setup IOCTL information ***}
     IOCTLInfo.rReqHdr.bLength:= SizeOf(TRequestHeader);
     IOCTLInfo.rReqHdr.bSubUnit:= 0;
     IOCTLInfo.rReqHdr.bCommand:= BYTE(iCommand);
     IOCTLInfo.bMediaDescriptor:= 0;
     IOCTLInfo.lpTransferAddress:= pControlBlock;
     IOCTLInfo.wTransferCount:= iControlBlockSize;
     IOCTLInfo.wStartingSectorNo:= 0;
     IOCTLInfo.lpVolumeID:= nil;

     ReqHdr:= @IOCTLInfo.rReqHdr;
     Regs.AX:= $1510;        {MSCDEX Send Request to Driver function}
     Regs.CX:= wDriveNum;
     Regs.ES:= LONGINT(ReqHdr) shr 16;
     Regs.BX:= LONGINT(ReqHdr) and $FFFF;
     Intr($2F, Regs);

     {*** Return error information ***}
     DeviceCommand:= ReqHdr^.wStatus;
end;

(* TCDROMDrive.RedBookToHSG - converts CD disc position in minutes, seconds
                              and frames (RedBook format) into a longint
                              (HSG format) for use by CD-Audio functions.
*)

function TCDROMDrive.RedBookToHSG(bMinute, bSecond, bFrame : byte) : longint;

begin
     RedBookToHSG:= (LONGINT(bMinute) * 4500) +
                    (LONGINT(bSecond) * 75) + bFrame - 150;
end;

{******}

(* CD_Detect - internal procedure automatically called when the unit is
               initialised. Checks MSCDEX is installed and gets information
               about available drives.                                      *)

procedure CD_Detect;

var Install_Test1 : byte;  {Holds al register after install check}
    Install_Test2 : word;  {Holds stack contents after install check}
    Version       : word;  {Temporary store for MSCDEX version}
    NumDrives     : byte;  {Temporary store for the number of CD-ROM drives}
    DriveLetters  : string;{Temporary store for CD-ROM drive letters}
    DLSeg, DLOfs  : word;  {Segment and offset of DriveLetters variable}
    Loop          : word;  {Loop control variable}

begin
     asm
        mov  ax, $1100          {MSCDEX installation check function}
        mov  bx, $DADA          {Sub-function $DADA}
        push bx
        int  $2F
        mov  Install_Test1, al  {if al = $FF then MSCDEX is installed}
        pop  ax
        mov  Install_Test2, ax  {if ax <> $DADA then MSCDEX is installed}
     end;

     {*** Set unit variable to indicate result ***}
     MSCDEX_Info.Installed:= (Install_Test1 = $FF) and
                             (Install_Test2 <> $DADA);

     {*** If no MSCDEX then don't go any further ***}
     if not MSCDEX_Info.Installed then
     begin
          {*** Blank out record ***}
          FillChar(MSCDEX_Info, SizeOf(TMSCDEXInfo), 0);

          {*** Set installed field to false ***}
          MSCDEX_Info.Installed:= false;

          {*** Exit procedure ***}
          Exit;
     end;

     {*** Get MSCDEX version ***}
     asm
        mov  ax, $150C
        int  $2F
        mov  Version, bx
     end;

     {*** NOTE: versions less than 2.10 are not supported ***}
     MSCDEX_Info.Version:= Version;

     {*** Get number of CD-ROM drives ***}
     asm
        mov  ax, $1500
        mov  bx, 0
        int  $2F
        mov  NumDrives, bl
     end;

     {*** NOTE: the unit requires at least one CD-ROM drive to be present ***}
     MSCDEX_Info.TotalDrives:= NumDrives;

     {*** If no drives installed then don't go any further ***}
     if NumDrives = 0 then
     begin
          DriveLetters:= '';
          Exit;
     end;

     {*** Get list of drive letters assigned to CD-ROM drives ***}
     DLSeg:= Seg(DriveLetters[1]);
     DLOfs:= Ofs(DriveLetters[1]);
     asm
        mov  ax, $150D
        mov  bx, DLSeg
        mov  es, bx
        mov  bx, DLOfs
        int  $2F
     end;
     DriveLetters[0]:= Chr(NumDrives);

     {*** Convert returned data into ASCII characters ***}
     for Loop:= 1 to Length(DriveLetters) do
         DriveLetters[Loop]:= Chr(Ord(DriveLetters[Loop]) + 65);

     MSCDEX_Info.DriveLetters:= DriveLetters;
end;

{*** Unit initialisation code ***}

begin
     {*** Check MSCDEX and get drive information ***}
     CD_Detect;
end.