unit SUTILS;

(* Information
   

   Program Title : S-Library general routines.
   External name : SUTILS.TPU
   Version       : 1.1
   Start date    : 27/08/1996
   Last update   : 06/11/1997
   Author        : Rob Anderton.
   Description   : Miscellaneous routines used by various S components.

*)

interface

{*** Useful data types (for typecasting etc.) ***}

type PByteArray = ^TByteArray;
     TByteArray = array[0..65534] of byte;

     PCharArray = ^TCharArray;
     TCharArray = array[0..65534] of char;

     PWordArray = ^TWordArray;
     TWordArray = array[0..32766] of word;

     {*** Equivalent to a QWORD or COMP data type ***}
     PQuadRec   = ^TQuadRec;
     TQuadRec   = record
                        Lo : longint;
                        Hi : longint;
                  end;

{*** Functions/procedures ***}

function  U_HexToDec(sHex : string) : longint;

function  U_ByteToHex(bVal         : byte;
                      bLeadingZero : boolean) : string;

function  U_WordToHex(wVal         : word;
                      bLeadingZero : boolean) : string;

function  U_LongToHex(lVal         : longint;
                      bLeadingZero : boolean) : string;

function  U_IntToStr(lVal : longint): string;

function  U_StrUpper(sText : string) : string;

function  U_StrLower(sText : string) : string;

function  U_LoCase(cUpper : char) : char;

function  U_GetLinearAddr(pPtr : pointer) : longint;

function  U_NormalisePtr(pPtr : pointer) : pointer;

function  U_Min(lValA, lValB : longint) : longint;

function  U_Max(lValA, lValB : longint) : longint;

procedure U_Swap(var lValA, lValB : longint);

function  U_Power(lN, lP : longint) : longint;

function  U_InpLong(wReg : word) : longint;

procedure U_OutpLong(wReg : word; lVal : longint);

function  U_FileExists(sFilename : string) : boolean;

function  U_GetTime : longint;

{******}

implementation

{******}

uses DOS;

{******}

(* U_HexToDec - converts a string of hexadecimal characters into a decimal
                representation.

                sHex - the hexadecimal string to convert.

                Returns a longint containing the decimal value, conversion
                is stopped when a non-hexadecimal character is detected, and
                the current working value is returned.

*)

function U_HexToDec(sHex : string) : longint;

var NumChars : integer; {Number of characters processed}
    Value    : longint; {Temporary value of conversion}
    i        : word;    {Loop control variable}

begin
     {*** Initialise variables ***}
     NumChars:= 0;
     Value:= 0;

     {*** Only process non-empty strings ***}
     if sHex <> '' then
     begin
          {*** Skip any leading spaces ***}
          i:= 1;
          while (sHex[i] = ' ') and (i <= Length(sHex)) do Inc(i);

          {*** Process remaining characters ***}
          while i <= Length(sHex) do
          begin
               case UpCase(sHex[i]) of
                    '0'..'9' : begin
                                    {*** Convert value ***}
                                    Value:= Value * $10;
                                    Value:= Value + INTEGER(sHex[i]) -
                                                    INTEGER('0');
                                    NumChars:= Succ(NumChars);
                               end;

                    'A'..'F' : begin
                                    {*** Convert value ***}
                                    Value:= Value * $10;
                                    Value:= Value + INTEGER(UpCase(sHex[i])) -
                                                    INTEGER('A') + 10;
                                    NumChars:= Succ(NumChars);
                               end;

                          else begin
                                    {*** Stop processing when invalid
                                         character detected ***}

                                    if NumChars <> 0 then U_HexToDec:= Value
                                                     else U_HexToDec:= 0;
                                    Exit;
                               end;
               end;
               Inc(i);
          end;
          {*** Return converted value ***}
          U_HexToDec:= Value;
     end
     else U_HexToDec:= 0; {Return zero if sHex is empty}
end;

(* U_ByteToHex - converts a byte value into a string of hexadecimal
                 characters.

                 bVal         - the decimal value to convert
                 bLeadingZero - if true the returned string is padded
                                with zeros to always make the string
                                have a length of 2 characters

                 Returns the converted string.
*)

function U_ByteToHex(bVal : byte; bLeadingZero : boolean) : string;

{*** Hexadecimal characters ***}
const HexChars : array [0..$F] of char = '0123456789ABCDEF';

var HexStr : string; {Temporary string}

begin
     {*** Convert to string ***}
     HexStr:= (HexChars[bVal shr 4] + HexChars[bVal and $F]);

     {*** Remove leading zero ***}
     if not bLeadingZero then
        if HexStr[1] = '0' then
        begin
             HexStr[1]:= HexStr[2];
             HexStr[0]:= Chr(1);
        end;

     {*** Return string ***}
     U_ByteToHex:= HexStr;
end;

(* U_WordToHex - converts a word value into a string of hexadecimal
                 characters.

                 wVal         - the decimal value to convert
                 bLeadingZero - if true the returned string is padded
                                with zeros to always make the string
                                have a length of 4 characters

                 Returns the converted string.
*)

function U_WordToHex(wVal : word; bLeadingZero : boolean) : string;

{*** Hexadecimal characters ***}
const HexChars : array [0..$F] of char = '0123456789ABCDEF';

var HexStr : string; {Temporary string}
    i, j   : byte;   {Loop control variables}

begin
     {*** Convert to string ***}
     HexStr:= (HexChars[Hi(wVal) shr 4] + HexChars[Hi(wVal) and $F] +
               HexChars[Lo(wVal) shr 4] + HexChars[Lo(wVal) and $F]);

     {*** Remove leading zeros ***}
     if not bLeadingZero then
     begin
          i:= 1;
          {*** Find position of first non-zero character ***}
          while HexStr[i] = '0' do Inc(i);

          {*** Remove zeros if necessary ***}
          if i > 1 then
          begin
               Move(HexStr[i], HexStr[1], 5 - i);
               HexStr[0]:= Chr(5 - i);
          end;
     end;

     {*** Return string ***}
     U_WordToHex:= HexStr;
end;


(* U_LongToHex - converts a longint value into a string of hexadecimal
                 characters.

                 lVal         - the decimal value to convert
                 bLeadingZero - if true the returned string is padded
                                with zeros to always make the string
                                have a length of 8 characters

                 Returns the converted string.
*)

function U_LongToHex(lVal : longint; bLeadingZero : boolean) : string;

{*** Hexadecimal characters ***}
const HexChars : array [0..$F] of char = '0123456789ABCDEF';

var HexStr : string; {Temporary string}
    w1, w2 : word;   {Store high and low words of DWORD}
    i      : byte;   {Loop control variable}

begin
     {*** Get high and low words ***}
     w1:= WORD(lVal);
     w2:= WORD(lVal shr 16);

     {*** Convert to string ***}
     HexStr:= (HexChars[Hi(w2) shr 4] + HexChars[Hi(w2) and $F] +
               HexChars[Lo(w2) shr 4] + HexChars[Lo(w2) and $F]) +
              (HexChars[Hi(w1) shr 4] + HexChars[Hi(w1) and $F] +
               HexChars[Lo(w1) shr 4] + HexChars[Lo(w1) and $F]);

     {*** Remove leading zeros ***}
     if not bLeadingZero then
     begin
          i:= 1;
          {*** Find position of first non-zero character ***}
          while HexStr[i] = '0' do Inc(i);

          {*** Remove zeros if necessary ***}
          if i > 1 then
          begin
               Move(HexStr[i], HexStr[1], 9 - i);
               HexStr[0]:= Chr(9 - i);
          end;
     end;

     {*** Return string ***}
     U_LongToHex:= HexStr;
end;


(* U_IntToStr - converts a decimal value into a string. This is a functional
                wrapper for the STR procecedure.

                lVal - the number to convert to a string.

                Returns a string containing the converted value.
*)

function U_IntToStr(lVal : longint): string;

var S : string[11]; {Temporary string for conversion}

begin
     {*** Call standard STR procedure ***}
     Str(lVal, S);
     {*** Return string ***}
     U_IntToStr:= S;
end;


(* U_StrUpper - converts an entire string to upper case.

                sText - the string to convert.

                Returns the upper case string.
*)

function U_StrUpper(sText : string) : string;

var Loop : byte;    {Used to process characters in a string}
    Temp : string;  {Holds converted string}

begin
     {*** Set length ***}
     Temp[0]:= sText[0];

     {*** Convert individual characters to upper case ***}
     for Loop:= 1 to Length(sText) do Temp[Loop]:= Upcase(sText[Loop]);

     {*** Return the converted string ***}
     U_StrUpper:= Temp;
end;


(* U_StrLower - converts an entire string to lower case.

                sText - the string to be converted.

                Returns the converted string.
*)

function U_StrLower(sText : string) : string;

var Loop : byte;   {Used to process individual characters}
    Temp : string; {Holds the converted string}

begin
     {*** Set string length ***}
     Temp[0]:= sText[0];

     {*** Convert individual characters to lower case ***}
     for Loop:= 1 to Length(sText) do Temp[Loop]:= U_LoCase(sText[Loop]);

     {*** Return the converted string ***}
     U_StrLower:= Temp;
end;


(* U_LoCase - converts an alphabetical character to lower case. Does not
              alter lower case, numeric or other characters.

              cUpper - the character to process.

              Returns the converted character.
*)

function U_LoCase(cUpper : char) : char;

begin
     {*** If cUpper is an upper case letter then convert ***}
     if cUpper in ['A'..'Z'] then U_LoCase:= Chr(Ord(cUpper) + 32)
                             else U_LoCase:= cUpper; {Otherwise do nothing}
end;

(* U_GetLinearAddr - converts a segment:offset pointer into a 32 bit linear
                     address.

                     pPtr - the pointer to convert.

                     Returns the 32 bit linear address.
*)

function U_GetLinearAddr(pPtr : pointer) : longint;

begin
     {*** Convert pointer and return ***}
     U_GetLinearAddr:= LONGINT(Seg(pPtr^)) shl 4 + LONGINT(Ofs(pPtr^));
end;


(* U_NormalisePtr - normalises a pointer so that the offset address is always
                    between 0 and 15.

                    pPtr - the pointer to normalise.

                    Returns the normalised pointer.
*)

function U_NormalisePtr(pPtr : pointer) : pointer;

var LinearAddr : longint;  {Holds the 32 bit linear address of pPtr}

begin
     {*** Convert pointer to a linear address ***}
     LinearAddr:= U_GetLinearAddr(pPtr);

     {*** Normalise and return the pointer ***}
     U_NormalisePtr:= Ptr(LinearAddr div 16, LinearAddr mod 16);
end;


(* U_Min - compares two values and returns the lowest.
           If lValA and lValB are equal then lValB is returned.

           lValA, lValB - the two numbers to be compared.

           Returns the minimum value of the two.
*)

function U_Min(lValA, lValB : longint) : longint;

begin
     if lValA < lValB then U_Min:= lValA
                      else U_Min:= lValB;
end;


(* U_Max - compares two values and returns the highest.
           If lValA and lValB are equal then lValB is returned.

           lValA, lValB - the two numbers to be compared.

           Returns the maximum value of the two.
*)

function U_Max(lValA, lValB : longint) : longint;

begin
     if lValA > lValB then U_Max:= lValA
                      else U_Max:= lValB;
end;


(* U_Swap - swaps the values of two numbers, so that lValA becomes the value
            of lValB and lValB takes on the value of lValA.

            lValA, lValB - the two numbers to be swapped.
*)

procedure U_Swap(var lValA, lValB : longint);

var t : longint; {Temporary store for lValA}

begin
     t:= lValA;         {Store lValA}
     lValA:= lValB;     {Set lValA to lValB}
     lValB:= t;         {Set lValB to the original value of lValA}
end;


(* U_Power - raises lN to the power lP.
             For example: lN = 2, lP = 3, returns 2^3 = 8.

             lN - the number to be raised to the power.
             lP - the power.

             Returns lN ^ lP.
*)

function U_Power(lN, lP : longint) : longint;

begin
     {*** if lP = 0 then lN ^ lP is always 1 ***}
     if lP = 0 then U_Power:= 1
               {*** if lN is 0 then lN ^ lP is always 0 ***}
               else if lN = 0 then U_Power:= 0
                              {*** Otherwise recursively call U_Power ***}
                              else U_Power:= lN * U_Power(lN, (lP - 1))
end;


(* U_InpLong - inputs a 32 bit value from the specified register. This is
               equivalent to a single IN EAX, DX instruction and requires
               a 386 or better processor.

               wReg - the register to read from.

               Returns the value read from wReg.

               This function is taken from the VGADOC 4.0 package - refer
               to the documentation for more information.
*)

function U_InpLong(wReg : word) : longint;

var l : longint; {Temporary register value}

begin
     inline($8B/$56/<wReg/$66/$ED/$66/$89/$46/<l);
     U_InpLong:= l;
end;

{******}

(* U_OutpLong - outputs a 32 bit value to the specified register. This is
                equivalent to a single OUT EAX, DX instruction and requires
                a 386 or better processor.

                wReg - the register to write to.
                lVal - the value to write.

                This procedure is taken from the VGADOC 4.0 package - refer
                to the documentation for more information.
*)

procedure U_OutpLong(wReg : word; lVal : longint);

begin
     (* mov dx,  [BP + wReg]
        mov eax, [BP + lVal]
        out dx,  eax
     *)
     inline($8B/$56/<wReg/$66/$8B/$46/<lVal/$66/$EF);
end;


(* U_FileExists - checks a given file name, returning true if the file exists
                  and false if it does not.

                  sFilename - the name of the file to check.

                  Returns result of check.
*)

function U_FileExists(sFilename : string) : boolean;

var f     : file;    {File variable used in attempt to open file}

begin
     {*** Attempt to open file, with I/O checking OFF ***}
     {$I-}
     Assign(f, sFilename);
     FileMode:= 0;
     Reset(f);
     Close(f);
     {$I+}

     {*** If IOResult is zero then no error occurred ***}
     U_FileExists:= (IOResult = 0) and (sFilename <> '');
end;

{******}

function U_GetTime : longint;

var Hour, Min, Sec, Sec100 : word;

begin
     GetTime(Hour, Min, Sec, Sec100);
     U_GetTime:= (LONGINT(Hour) * 3600) + (LONGINT(Min) * 60) + LONGINT(Sec);
end;

{******}

end.