PROGRAM CarEdit3;

USES Dos,     {FindFirst}
     Drivers, {EgaVga driver}
     Graph,   {VGA output}
     KeyPro,  {Keyboard input}
     NumPro,  {Number utilities}
     Strings, {StrL}
     StrPro,  {String utilities}
     Crt,     {Text routines}
     VgaPro;  {VGA/Graph utilities}

CONST
    MAX_CARS      = 32;
    MAX_SHORTNAME = 4;
    MAX_LONGNAME  = 20;
    MAX_DESCLINE  = 32;
    MAX_DATA      = 2048;
    MAX_SPARECARS = 42;
    MAX_SPAREFOLDERS = 41;
    def_carDir    = ''; {Fixed to current directory (4DS-Driving / Stunts root)}
    def_spareCarDir = '.\SpareCars\'; {Customizable "Spare Cars" directory}

CONST
    CHR_GEARS: Array[0..6] of Char = ('N', '1', '2', '3', '4', '5', '6');

TYPE
    CarDescriptor = RECORD
        baseName: String[4];            {As used in the resp. file names.}
        longName: String[MAX_LONGNAME]; {The long name (end of RES file.)}
        unpacked: Boolean;              {unpacked version of shape and image files.}
    END;

TYPE
    CarDetails = RECORD
        raw:       Array[0..$32D] of Byte;
        fullDesc:  Array[1..9] of String[MAX_DESCLINE];
        shortName: String[MAX_SHORTNAME];
        longName:  String[MAX_LONGNAME];
    END;

TYPE
    Settings = RECORD
        video:              String[8];
        carRelPath:         String;
        spareCarRelPath:    String;
    END;

VAR
    carCount: Integer;  {car count to render}
    carACount: Integer; {car count "all" in path }
    carSlots: Array[1..MAX_SPARECARS] of CarDescriptor; {same as spare slots}
    spareCarCount: Integer;      {spare car count to render}
    spareCarACount: Integer;     {spare car count "all" in path}
    spareItemsCount:	Integer; {spare car and folder count to render}
    spareCarSlots: Array[1..255] of CarDescriptor;
    spareCarFoldersCount: Integer;     {spare car folders count to render}
    spareCarFolders: Array[1..MAX_SPARECARS] of String;
    spareCarSelFolderPath: String; {selected spare car folder path}
    spareCarSelFolder: String; {spare car folder name}
    carOpen:  Integer;
    carData:  CarDetails;
    spareCarOpen: Boolean;
    selView:  Integer;
    selItem:  Integer;
    selItemOld:  Integer;
    selHold:  Integer;
    selHoldOld:  Integer;
    selHoldScroll: Integer; {paging purposes}
    backIntX:  Integer;
    backIntY:  Integer;
    curSettings:  Settings;
    settingsChanged: Array[1..3] of Boolean;
    carHasChanges: Boolean;
    configHasChanges: Boolean;
    restartRequested: Boolean;
    quit:  Boolean;

FUNCTION GetResWord(offset: Word): Word;
BEGIN
    GetResWord := (carData.raw[offset+1] * 256) + carData.raw[offset];
END;

PROCEDURE SetResWord(offset, value: Word);
BEGIN
   carData.raw[offset+1] := (value DIV 256);
   carData.raw[offset] := (value MOD 256);
   carHasChanges := True;
END;

PROCEDURE GetResMaxPower(
    var maxNm: Word;
    var maxNmAt: Word;
    var maxHP: Word;
    var maxHPAt: Word);
VAR
    curNm: Word;
    curHP: Word;
    tmp:   Double;
    i:     Integer;
    rpm:   Integer;
BEGIN
    maxNm := 0;
    maxHP := 0;
    FOR i := 0 TO 102 DO
    BEGIN
        rpm := 128*i;
        curNm := Round(7.457 * carData.raw[$61+i]);
        IF (curNm > maxNm) THEN
        BEGIN
            maxNm := curNm;
            maxNmAt := rpm;
        END;
        { (2pi/60) = 0.104719755, 1 PS = 735.49875 W }
        tmp := (0.104719755/735.49875) * curNm * rpm;
        curHP := Round(tmp);
        IF (curHP > maxHP) THEN
        BEGIN
            maxHP := curHP;
            maxHPAt := rpm;
        END
    END;
END;

FUNCTION RaisePowerCurveArea(ml, mr, by: Integer): Boolean;
VAR
    i: Integer;
BEGIN
    IF (mr >= ml) THEN
    BEGIN
        FOR i := ml TO mr DO
            IncByte(carData.raw[$60+i], by);
        RaisePowerCurveArea := True;
    END ELSE
        RaisePowerCurveArea := False;
    carHasChanges := True;
END;

FUNCTION LowerPowerCurveArea(ml, mr, by: Integer): Boolean;
VAR
    i: Integer;
BEGIN
    IF (mr >= ml) THEN
    BEGIN
        FOR i := ml TO mr DO
            DecByte(carData.raw[$60+i], by);
        LowerPowerCurveArea := True;
    END ELSE
        LowerPowerCurveArea := False;
    carHasChanges := True;
END;

FUNCTION StretchPowerCurveArea(ml, mr: Integer; factor: Double): Boolean;
VAR
    mm:   Integer;
    i:    Integer;
    prox: Double;
    pfac: Double;
    rslt: Word;
BEGIN
    IF (mr >= (ml+2)) THEN
    BEGIN
        mm := (ml+mr) DIV 2;
        FOR i := ml TO mm DO
        BEGIN
            prox := (i-ml) / (mm-ml);  {0.0..1.0}
            pfac := (1.0-prox) + (factor*prox);  {1.0..factor}
            pfac := Sqrt(pfac);  {raise intermediate values}
            rslt := Round(carData.raw[$60+i] * pfac);
            carData.raw[$60+i] := MinInt(rslt, $FF);
        END;
        FOR i := (mm+1) TO mr DO
        BEGIN
            prox := (mr-i) / (mr-mm);  {~1.0..0.0}
            pfac := (1.0-prox) + (factor*prox);  {~factor..0.0}
            pfac := Sqrt(pfac);  {raise intermediate values}
            rslt := Round(carData.raw[$60+i] * pfac);
            carData.raw[$60+i] := MinInt(rslt, $FF);
        END;
        StretchPowerCurveArea := True;
    END ELSE
        StretchPowerCurveArea := False;
    carHasChanges := True;
END;

FUNCTION EqualizeGears: Boolean;
VAR
    numGr:  Integer;
    rf, rl: Double;
    factor: Double;
    ratio:  Double;
    i:      Integer;
BEGIN
    EqualizeGears := False;
    numGr := carData.raw[$26];
    IF ((numGr >= 2) AND (numGr <= 6)) THEN
    BEGIN
        rf := GetResWord($34+(2*1));
        rl := GetResWord($34+(2*numGr));
        IF ((rf > 0.0) AND (rl > 0.0)) THEN
        BEGIN
            factor := Pow((rl/rf), 1.0/(numGr-1));
            ratio := rf;
            FOR i := 2 TO numGr DO
            BEGIN
                ratio := ratio * factor;
                SetResWord($34+(2*i), Round(ratio));
            END;
            EqualizeGears := True;
        END;
    END;
END;

FUNCTION ValidWheelCoordinates: Boolean;
VAR x_fl, x_fr, x_rr, x_rl: Word;
    y_fl, y_fr, y_rr, y_rl: Word;
    z_fl, z_fr, z_rr, z_rl: Word;
BEGIN
    x_fl := GetResWord($F8);
    y_fl := GetResWord($FA);
    z_fl := GetResWord($FC);

    x_fr := GetResWord($FE);
    y_fr := GetResWord($100);
    z_fr := GetResWord($102);

    x_rr := GetResWord($104);
    y_rr := GetResWord($106);
    z_rr := GetResWord($108);

    x_rl := GetResWord($10A);
    y_rl := GetResWord($10C);
    z_rl := GetResWord($10E);

    IF ((x_fl + x_fr + x_rr + x_rl = 0) AND
        (y_fl = 0) AND (y_fr = 0) AND (y_rr = 0) AND (y_rl = 0) AND
        (z_fl + z_fr + z_rr + z_rl = 0)) THEN
        ValidWheelCoordinates := True
    ELSE
        ValidWheelCoordinates := False;
END;

PROCEDURE SaveSettingsFile(newSettings: Settings);
VAR
    f:      File OF Settings;

BEGIN
    Assign(f, 'CAREDIT3.CFG');
    Rewrite(f);
    curSettings := newSettings;
    Write(f,curSettings);
    Close(f);
    configHasChanges := False;
END;

PROCEDURE LoadSettingsFile;
VAR
    newSettings: Settings;
    f:      File OF Settings;
BEGIN
    {$I-}
    Assign(f, 'CAREDIT3.CFG');
    FileMode := 0;  { Set file access to read only }
    Reset(f);
    {$I+}
    IF (IOResult <> 0) THEN { If not found,create file with default settings }
    BEGIN
       curSettings.video := 'VGA';
       curSettings.carRelPath := def_carDir;
       curSettings.spareCarRelPath := def_spareCarDir;
       SaveSettingsFile(curSettings);
    END
    ELSE
    BEGIN
        WHILE NOT Eof(f) DO
        BEGIN
              Read(f,newSettings);
              curSettings := newSettings;
              configHasChanges := False;
        END;
        Close(f);
    END;
END;

PROCEDURE LoadResFile(fn: String);
VAR
    f:      File;
    cnt:    Integer;
    buffer: Array[0..MAX_DATA] of Byte;
    i, j:   Integer;
    declaredSize: Integer;
    simd, edes, gsna, gnam: Integer;
    resourceId: String[4];
    resourceOffset: Integer;
    standardIdsString: String[16];
    finalByteOffset, standardOffset, newEofOffset: Integer;
BEGIN
    Assign(f, fn);
    {$I-} Reset(f, 1); {$I+}
    Assert(IoResult = 0, 'Could not open RES file for reading.');
    BlockRead(f, buffer, MAX_DATA, cnt);
    Assert(cnt < MAX_DATA, 'RES file is too large.');
    Assert(cnt > $32D, 'RES file is too small');
    Close(f);
    { Reading the header. }
    declaredSize := buffer[0] + buffer[1] * 256;
    Assert((buffer[2] = 0) AND (buffer[3] = 0) AND (declaredSize = cnt), 'Declared and actual size of RES file differ');
    Assert((buffer[4] = 4) AND (buffer[5] = 0), 'RES file does not have exactly 4 resources');
    simd := -1;
    edes := -1;
    gsna := -1;
    gnam := -1;
    FOR i := 0 TO 3 DO
    BEGIN
        resourceId := '';
        FOR j := 0 TO 3 DO
            resourceId := resourceId + chr(buffer[6 + 4*i + j]);
        resourceOffset := buffer[$16 + 4*i] + buffer[$16 + 4*i + 1] * 256;
        Assert((buffer[$16 + 4*i + 2] = 0) AND (buffer[$16 + 4*i + 3] = 0) AND (resourceOffset < cnt - $26),
            'Bad resource size in RES header');
        IF resourceId = 'simd' THEN
            simd := resourceOffset;
        IF resourceId = 'edes' THEN
            edes := resourceOffset;
        IF resourceId = 'gsna' THEN
            gsna := resourceOffset;
        IF resourceId = 'gnam' THEN
            gnam := resourceOffset;
    END;
    Assert(simd >= 0, 'simd missing from RES header');
    Assert(edes >= 0, 'edes missing from RES header');
    Assert(gsna >= 0, 'gsna missing from RES header');
    Assert(gnam >= 0, 'gnam missing from RES header');
    Assert((simd + $32D) < cnt, 'simd declared to be past the end of the RES file');
    { Copy the raw data. Mostly words. }
    { First, write the beginning of the header. }
    FOR i := 0 TO 5 DO
        carData.raw[i] := buffer[i];
    { The ids go in a fixed order. }
    standardIdsString := 'edesgnamgsnasimd';
    FOR i := 1 TO 16 DO
        carData.Raw[5 + i] := Ord(standardIdsString[i]);
    { Initialize the offsets to zero. }
    FOR i := $16 TO $25 DO
        carData.Raw[i] := 0;
    { Then, copy simd. }
    FOR i := $26 TO $32D DO
        carData.raw[i] := buffer[simd + i];
    { The offsets will be set so that the resources are in standard order. }
    { The standard simd offset is 0, so nothing needs to be done for it. }
    { Standard offset for edes. }
    standardOffset := $308;
    carData.raw[$16] := standardOffset MOD 256;
    carData.raw[$17] := standardOffset DIV 256;
    { Copy the full description. }
    FOR j := 1 TO 9 DO
        carData.fullDesc[j] := '';
    finalByteOffset := simd + $32D;
    i := $26 + edes;
    j := 1;
    WHILE (True) DO
    BEGIN
       CASE buffer[i] OF
           $00: BREAK;
           $5D: Inc(j);
           ELSE BEGIN
                    Assert(j <= 9, 'Car description has too many lines');
                    carData.fullDesc[j] := carData.fullDesc[j] + chr(buffer[i]);
                END;
       END;
       Inc(i);
    END;
    IF i > finalByteOffset THEN
        finalByteOffset := i;
    { Standard offset for gsna. }
    standardOffset := standardOffset + i - $26 - edes + 1;
    carData.raw[$1E] := standardOffset MOD 256;
    carData.raw[$1F] := standardOffset DIV 256;
    { Copy the short high-score name. }
    i := $26 + gsna;
    carData.shortName := '';
    WHILE (True) DO
    BEGIN
       CASE buffer[i] OF
           $00: BREAK;
           $5D: Assert(False, 'Unexpected linebreak in short car name');
           ELSE carData.shortName := carData.shortName + chr(buffer[i]);
       END;
       Inc(i);
    END;
    if i > finalByteOffset THEN
        finalByteOffset := i;
    { Standard offset for gnam. }
    standardOffset := standardOffset + i - $26 - gsna + 1;
    carData.raw[$1A] := standardOffset MOD 256;
    carData.raw[$1B] := standardOffset DIV 256;
    { Copy the long high-score name. }
    i := $26 + gnam;
    carData.longName := '';
    WHILE (True) DO
    BEGIN
       CASE buffer[i] OF
           $00: BREAK;
           $5D: Assert(False, 'Unexpected linebreak in long car name');
           ELSE carData.longName := carData.longName + chr(buffer[i]);
       END;
       Inc(i);
    END;
    IF i > finalByteOffset THEN
        finalByteOffset := i;
    { Perform concluding checks. }
    newEofOffset := standardOffset + i - gnam + 1;
    Assert(newEofOffset = cnt, 'Early termination of RES file text resource');
    Assert(finalByteOffset + 1 = cnt, 'End of RES file at unexpected position');
    carHasChanges := False;
END;

PROCEDURE SaveResFile(fn: String);
VAR
    f:       File;
    cnt:     Integer;
    buffer:  Array[0..MAX_DATA] of Byte;
    i, j, k: Integer;
BEGIN
    { Copy the raw data. Mostly words. }
    FOR i := 0 TO $32D DO
        buffer[i] := carData.raw[i];
    { Copy the full description. }
    i := $32E;
    buffer[$16] := (i-$26) MOD 256;
    buffer[$17] := (i-$26) DIV 256;
    FOR j := 1 TO 9 DO
    BEGIN
        FOR k := 1 TO Length(carData.fullDesc[j]) DO
        BEGIN
            buffer[i] := ord(carData.fullDesc[j][k]);
            Inc(i); Assert(i < MAX_DATA, 'RES data too large.');
        END;
        buffer[i] := $5D;
        Inc(i); Assert(i < MAX_DATA, 'RES data too large.');
    END;
    buffer[i] := $00;
    Inc(i); Assert(i < MAX_DATA, 'RES data too large.');
    { Copy the short high-score name. }
    buffer[$1E] := (i-$26) MOD 256;
    buffer[$1F] := (i-$26) DIV 256;
    FOR k := 1 TO Length(carData.shortName) DO
    BEGIN
        buffer[i] := ord(carData.shortName[k]);
        Inc(i); Assert(i < MAX_DATA, 'RES data too large.');
    END;
    buffer[i] := $00;
    Inc(i); Assert(i < MAX_DATA, 'RES data too large.');
    { Copy the long high-score name. }
    buffer[$1A] := (i-$26) MOD 256;
    buffer[$1B] := (i-$26) DIV 256;
    FOR k := 1 TO Length(carData.longName) DO
    BEGIN
        buffer[i] := ord(carData.longName[k]);
        Inc(i); Assert(i < MAX_DATA, 'RES data too large.');
    END;
    buffer[i] := $00;
    Inc(i); Assert(i < MAX_DATA, 'RES data too large.');
    { Update the file-size entry in the header. }
    buffer[$00] := i MOD 256;
    buffer[$01] := i DIV 256;
    { Dump the buffer to the target file. }
    Assign(f, fn);
    {$I-} Rewrite(f, 1); {$I+}
    Assert(IoResult = 0, 'Could not open RES file for writing.');
    BlockWrite(f, buffer, i, cnt);
    Assert(cnt = i, 'Incomplete write on RES file.');
    Close(f);
    carHasChanges := False;
END;

PROCEDURE CopyFile(srcName, dstName: String);
VAR
    srcF, dstF: File;
    numR, numW: Word;
    buffer:     Array[0..2047] of Byte;
BEGIN
    Assign(srcF, srcName);
    Reset(srcF, 1);
    Assign(dstF, dstName);
    Rewrite(dstF, 1);
    REPEAT
        BlockRead(srcF, buffer, SIZEOF(buffer), numR);
        BlockWrite(dstF, buffer, numR, numW);
    UNTIL ((numR = 0) OR (numW <> numR));
    Close(srcF);
    Close(dstF);
END;

PROCEDURE DeleteFile(name: String);
VAR
    f: File;
BEGIN
    Assign(f, name);
    Erase(f);
END;

FUNCTION IsBaseNameInUse(name, dir: String): Boolean;
VAR
    i: Integer;
    dirItem: SearchRec;
BEGIN
    IsBaseNameInUse := False;
    FindFirst(dir+'CAR'+name+'.RES', Archive, dirItem);
    IF (DosError = 0) THEN IsBaseNameInUse := True;
END;

FUNCTION IsValidBaseName(name: String): Boolean;
VAR
    i: Integer;
BEGIN
    IF (Length(name) = 4) THEN
    BEGIN
        IsValidBaseName := True;
        FOR i := 1 TO 4 DO
        BEGIN
            IF (NOT (
                ((name[i] >= 'A') AND (name[i] <= 'Z')) OR
                ((name[i] >= 'a') AND (name[i] <= 'z')) OR
                ((name[i] >= '0') AND (name[i] <= '9'))
               )) THEN
            BEGIN
                IsValidBaseName := False;
                BREAK;
            END;
        END;
    END ELSE BEGIN
        IsValidBaseName := False;
    END;
END;


FUNCTION GetCarFileName(fileType, base, dir, newBase: String ): String;
VAR
   dirInfo     : SearchRec;
   strC        : Array [0..12] Of Char;
BEGIN
     IF ((fileType = 'ST') OR (fileType = 'STDA') OR (fileType = 'STDB')) THEN
     BEGIN
         FindFirst(dir+fileType+base+'.*', Archive, dirInfo);
         IF (DosError = 0) THEN BEGIN
             Str_Array(dirInfo.name,strC);
             IF (StrPos(strC, '.3SH') <> nil) THEN GetCarFileName := fileType+newBase+'.3SH';
             IF (StrPos(strC, '.VSH') <> nil) THEN GetCarFileName := fileType+newBase+'.VSH';
             IF (StrPos(strC, '.P3S') <> nil) THEN GetCarFileName := fileType+newBase+'.P3S';
             IF (StrPos(strC, '.PVS') <> nil) THEN GetCarFileName := fileType+newBase+'.PVS';
         END
         ELSE ShowMessageBox('Error: Get Car File Name','Code'+Byte2Dec(DosError));
     END;
END;

PROCEDURE CopyCarFiles(srcBase, dstBase, srcDir, dstDir: String);
VAR
    srcFileName, dstFileName    : String;
BEGIN
    CopyFile(srcDir+'CAR' + srcBase + '.RES',
             dstDir+'CAR' + dstBase + '.RES');
    srcFileName:= GetCarFileName('ST', srcBase, srcDir, srcBase);
    dstFileName:= GetCarFileName('ST', srcBase, srcDir, dstBase);
    CopyFile(srcDir+srcFileName, dstDir+dstFileName);
    srcFileName:= GetCarFileName('STDA', srcBase, srcDir, srcBase);
    dstFileName:= GetCarFileName('STDA', srcBase, srcDir, dstBase);
    CopyFile(srcDir+srcFileName, dstDir+dstFileName);
    srcFileName:= GetCarFileName('STDB', srcBase, srcDir, srcBase);
    dstFileName:= GetCarFileName('STDB', srcBase, srcDir, dstBase);
    CopyFile(srcDir+srcFileName, dstDir+dstFileName);
END;

PROCEDURE DeleteCarFiles(base, dir: String);
VAR
   fileName : String;
BEGIN
     DeleteFile(dir+'CAR' + base + '.RES');
     fileName:= GetCarFileName('ST', base, dir, base);
     DeleteFile(dir+fileName);
     fileName:= GetCarFileName('STDA', base, dir, base);
     DeleteFile(dir+fileName);
     fileName:= GetCarFileName('STDB', base, dir, base);
     DeleteFile(dir+fileName);
END;

FUNCTION HasUnpackedCarFiles(shortName, dir: String): Boolean;
VAR
   unpackFound : Boolean;
   dirInfo     : SearchRec;
   strC        : Array [0..12] Of Char;
   pFound      : pChar;
BEGIN
     unpackFound := False;
     FindFirst(dir+'ST'+shortName+'.*', Archive, dirInfo);
     WHILE ((DosError = 0) AND (unpackFound = False)) DO BEGIN
        IF (unpackFound = False) THEN BEGIN
           Str_Array(dirInfo.name,strC);
           IF (StrPos(strC, '.3SH') = nil) THEN unpackFound := False
           ELSE unpackFound := True;
        END;
        FindNext(dirInfo);
     END;
     FindFirst(dir+'STD?'+shortName+'.*', Archive, dirInfo);
     WHILE ((DosError = 0) AND (unpackFound = False)) DO BEGIN
        IF (unpackFound = False) THEN BEGIN
           Str_Array(dirInfo.name,strC);
           IF (StrPos(strC, '.VSH') = nil) THEN unpackFound := False
           ELSE unpackFound := True;
        END;
        FindNext(dirInfo);
     END;
     HasUnpackedCarFiles := unpackFound;
END;

FUNCTION FindSpareCarFolders(dir : string): Integer;
VAR
    dirItem: SearchRec;
BEGIN
     spareCarFoldersCount := 0;
     FindFirst(dir+'*', Directory, dirItem);
     WHILE ((DosError = 0) AND (spareCarFoldersCount < MAX_SPAREFOLDERS)) DO
     BEGIN
         IF ((dirItem.Name <> '.') AND (dirItem.Name <> '..') AND
             (spareCarFoldersCount < MAX_SPARECARS)) THEN
         BEGIN
             Inc(spareCarFoldersCount);
             spareCarFolders[spareCarFoldersCount] := dirItem.Name;
         END;
         FindNext(dirItem);
     END;
     FindSpareCarFolders := DosError;
END;

FUNCTION FindCarFiles(dir : string): Integer;
VAR
    dirItem: SearchRec;
    iSkip:   Integer;
BEGIN
    IF (dir = curSettings.carRelPath) THEN
    BEGIN
       carCount := 0;
       carACount := 0;
    END
    ELSE BEGIN
       spareCarCount := 0;
       spareCarACount := 0;
    END;
    iSkip := (selHoldScroll*(MAX_SPARECARS-spareCarFoldersCount));
    FindFirst(dir+'CAR????.RES', Archive, dirItem);
    WHILE (DosError = 0) DO
    BEGIN
        IF (Length(dir)+Length(dirItem.name) = Length(dir)+11) THEN
        BEGIN
            LoadResFile(dir+dirItem.name);
            IF (dir = curSettings.carRelPath) THEN { car dir }
            BEGIN
               Inc (carACount);
               IF ((carCount < MAX_SPARECARS)) THEN
               BEGIN
                 Inc(carCount);
                 carSlots[carCount].baseName := Copy(dirItem.name, 4, 4);
                 carSlots[carCount].longName := carData.longName;
                 carSlots[carCount].unpacked := HasUnpackedCarFiles(carSlots[carCount].baseName, dir);
               END
            END
            ELSE { spare car dirs }
            BEGIN
               IF (spareCarACount >= 512) THEN BREAK; { FAT16 files in folder limit }
               Inc (spareCarACount);
               IF (spareCarCount+spareCarFoldersCount < MAX_SPARECARS) THEN
               BEGIN
                 IF (iSkip > 0) THEN
                 BEGIN
                      Dec(iSkip);
                      FindNext(dirItem);
                      CONTINUE;
                 END;
                 Inc(spareCarCount);
                 spareCarSlots[spareCarCount].baseName := Copy(dirItem.name, 4, 4);
                 spareCarSlots[spareCarCount].longName := carData.longName;
                 spareCarSlots[spareCarCount].unpacked := HasUnpackedCarFiles(spareCarSlots[spareCarCount].baseName, dir);
               END;
            END;
        END;
        FindNext(dirItem);
    END;
    FindCarFiles := DosError;
    {IF ((dir = curSettings.carRelPath)) THEN
        Assert(carCount > 0, 'No car files found in car directory ('+ curSettings.carRelPath+')');}
END;

FUNCTION CreateSpareFolderDialog: Boolean;
VAR newFolderName: String;
BEGIN
    CreateSpareFolderDialog := False;
    newFolderName := EditString('', 'New Spare Folder Name', 8);
    newFolderName := Str_ToUpper(newFolderName);
    {$I-}
    MkDir(spareCarSelFolderPath + newFolderName);
    IF (IOResult = 0) THEN
    BEGIN
        FindSpareCarFolders(spareCarSelFolderPath);
        CreateSpareFolderDialog := True;
    END
    {$I+}
    ELSE BEGIN
        ShowMessageBox('Error: Make Folder','That is not a valid name.');
    END;
END;

FUNCTION CopyCarFilesDialog(eraseOld: Boolean; dir: string): Boolean;
VAR
    base: String[4];
BEGIN
    CopyCarFilesDialog := False;
    base := EditString('', 'New Four-Letter Base Name', 4);
    base := Str_ToUpper(base);
    IF (NOT IsValidBaseName(base)) THEN
    BEGIN
        ShowMessageBox('Error: Bad Input',
            'That is not a valid car base name.');
    END ELSE BEGIN
        IF (IsBaseNameInUse(base, dir)) THEN
        BEGIN
            ShowMessageBox('Error: Bad Input',
                'That car base name is already in use.');
        END
        ELSE BEGIN
            CopyCarFiles(carSlots[selItem].baseName, base, dir, dir);
            IF (eraseOld) THEN
                DeleteCarFiles(carSlots[selItem].baseName, dir);
            FindCarFiles(dir);
            CopyCarFilesDialog := True;
        END;
    END;
END;

FUNCTION DeleteCarFilesDialog (dir : string): Boolean;
VAR
    base: String[4];
BEGIN
    DeleteCarFilesDialog := False;
    base := carSlots[selItem].baseName;
    IF (ShowQuestion('Confirmation of Delete Operation',
            'Are you sure you want to erase "'+base+'"?')) THEN
    BEGIN
        DeleteCarFiles(base, dir);
        FindCarFiles(dir);
        DeleteCarFilesDialog := True;
    END;
END;

PROCEDURE EditResWord(offset: Word; cap: String);
VAR
    old, new: Word;
BEGIN
    old := GetResWord(offset);
    new := EditWordValue(old, cap);
    SetResWord(offset, new);
END;

PROCEDURE EditSignedWord(offset: Word; cap: String; isNeg: Boolean; ratio: Integer);
VAR
    old, new: Word;
BEGIN
    old := GetResWord(offset);
    IF isNeg THEN old := 0 - old; {signed word as two's complement}
    new := EditSignedWordValue(old, cap, isNeg, ratio);
    IF isNeg THEN new := 0 - new;
    SetResWord(offset, new);
END;


PROCEDURE EditResByte(offset: Word; cap: String);
VAR
    old, new: Byte;
BEGIN
    old := carData.raw[offset];
    new := EditWordValue(old, cap);
    carData.raw[offset] := new;
    carHasChanges := True;
END;

PROCEDURE EditResColorByte(offset: Word; cap: String);
VAR
    old, new: Byte;
BEGIN
    old := carData.raw[offset];
    new := EditColorByteValue(old, cap);
    carData.raw[offset] := new;
    carHasChanges := True;
END;

PROCEDURE RenderSection(x, y, max: Integer; cap: String);
BEGIN
    SetColor(15);
    OutTextXY(x, y, cap);
    Line(x, y+11, x+((8*max)-1), y+11);
END;

PROCEDURE RenderStringValue(x, y, max: Integer; cap, value: String; idx: Word);
BEGIN
    IF (idx = selItem) THEN
        SetColor(15)
    ELSE
        SetColor(8);
    OutTextXY(x, y, cap);
    OutTextXY(x+((8*max)+2), y, value);
END;

PROCEDURE RenderResWordValue(x, y, max: Integer; cap: String; idx, off: Word);
BEGIN
     RenderStringValue(x, y, max, cap, Word2Dec(GetResWord(off)), idx);
END;

PROCEDURE RenderResByteValue(x, y, max: Integer; cap: String; idx, off: Word);
BEGIN
     RenderStringValue(x, y, max, cap, Byte2Dec(GetResWord(off)), idx);
END;

PROCEDURE RenderCarListItem(isSpare: Boolean);
BEGIN
   IF (isSpare = False) THEN BEGIN
     IF (carCount > 0) THEN BEGIN
        {previous selection clearing}
        IF (selItemOld > MAX_CARS) THEN SetColor(6) ELSE SetColor(8);
        OutTextXY(60, (8*selItemOld)+48, carSlots[selItemOld].baseName+'  '+carSlots[selItemOld].longName);
        IF (carSlots[selItemOld].unpacked = True) THEN
           OutTextXY(276, (8*selItemOld)+48, '(*)');

        {current select marking}
        SetColor(15);
        OutTextXY(60, (8*selItem)+48, carSlots[selItem].baseName+'  '+carSlots[selItem].longName);
        IF (carSlots[selItem].unpacked = True) THEN
           OutTextXY(276, (8*selItem)+48, '(*)');
     END;
   END

   ELSE BEGIN
       {spare folder selection clearing and marking}
       IF (spareCarFoldersCount > 0) THEN BEGIN
          IF (selHoldOld <= spareCarFoldersCount) THEN BEGIN
             SetColor(8);
             OutTextXY(336, (8*selHoldOld)+48, '['+ spareCarFolders[selHoldOld] +']');
          END;
          IF (selHold <= spareCarFoldersCount) THEN BEGIN
             SetColor(15);
             OutTextXY(336, (8*selHold)+48, '['+ spareCarFolders[selHold] +']');
          END;
       END;

       {spare car list}
       IF (spareCarCount > 0) THEN BEGIN
          {previous selection clearing}
          IF (selHoldOld-spareCarFoldersCount > 0) THEN BEGIN
             SetColor(8);
             OutTextXY(336, (8*(selHoldOld))+48,
               spareCarSlots[selHoldOld-spareCarFoldersCount].baseName+'  '
               +spareCarSlots[selHoldOld-spareCarFoldersCount].longName);
             IF (spareCarSlots[selHoldOld-spareCarFoldersCount].unpacked = True) THEN
                OutTextXY(552, (8*(selHoldOld))+48, '(*)');
          END;
          {current select marking}
          IF (selHold-spareCarFoldersCount > 0) THEN BEGIN
             SetColor(15);
             OutTextXY(336, (8*(selHold))+48,
               spareCarSlots[selHold-spareCarFoldersCount].baseName+'  '
               +spareCarSlots[selHold-spareCarFoldersCount].longName);
             IF (spareCarSlots[selHold-spareCarFoldersCount].unpacked = True) THEN
                OutTextXY(552, (8*(selHold))+48, '(*)');
          END;
       END;
   END;
END;

PROCEDURE RenderGarageScreen(fully: Boolean);
VAR
    i:  Integer;
    listedUnpacked:     Boolean;
BEGIN
  { Fix improper selections (may occur after "Delete"): }
  listedUnpacked := False;
  IF (selItem > carCount) THEN selItem := 1;
  spareItemsCount := spareCarCount+spareCarFoldersCount;
  IF (selHold > spareItemsCount) THEN selHold := 1;

  IF (fully) THEN BEGIN
    { Draw the header: }
    SetFilLStyle(SolidFill, 7);
    Bar(576, 1, 631, 8);
    SetColor(15);
    OutTextXY(580, 1, 'Garage');
    { Draw the car list: }
    RenderSection(60, 40, 26, 'Cars');
    FOR i := 1 TO carCount DO BEGIN
        IF (i = selItem) THEN BEGIN
            IF (spareCarOpen) THEN SetColor(4) ELSE SetColor(15);
        END
        ELSE BEGIN
            IF ((i <= MAX_CARS)) THEN SetColor(8)
            ELSE BEGIN
                 SetColor(6);
                 OutTextXY(56, 432, 'Only first'+Byte2Dec(MAX_CARS)+' cars are playable in game');
            END
        END;
        OutTextXY(60, (8*i)+48, carSlots[i].baseName+'  '+carSlots[i].longName);
        IF (carSlots[i].unpacked = True) THEN BEGIN
            OutTextXY(276, (8*i)+48, '(*)');
            listedUnpacked := True;
        END;
        IF (i = carCount) THEN BEGIN
            SetColor(8);
            OutTextXY(92, (8*i)+72, 'Number of cars: ');
            IF (carCount > MAX_CARS) THEN BEGIN
               IF (carACount > MAX_SPARECARS) THEN BEGIN
                  SetColor(6); OutTextXY(96, (8*i)+60, '(Only showing'+Byte2Dec(carCount)+')');
                  SetColor(12);
               END
               ELSE SetColor(6);
            END
            ELSE BEGIN
                IF (spareCarOpen) THEN BEGIN
                   IF (carCount < MAX_CARS) THEN SetColor(8)
                   ELSE SetColor(12);
                END
                ELSE SetColor(2);
            END;
            OutTextXY(212, (8*i)+72, Byte2Dec(carACount));

            Break;
        END;
    END;
    IF ((carCount = 0)) THEN BEGIN
       SetColor(8);
       OutTextXY(60, 56, '(No cars found)');
    END;

    IF (spareCarOpen) THEN BEGIN
    { Draw the spare car list: }
      RenderSection(336, 40, 26, 'Spare Cars');
      {IF (spareCarSelFolder <> '') THEN
         OutTextXY(422, 40, 'in ['+spareCarSelFolder+']');}

      { start by folders }
      IF (spareCarFoldersCount > 0) THEN BEGIN
         FOR i := 1 TO spareCarFoldersCount DO BEGIN
           IF (i = selHold) THEN SetColor(15) ELSE SetColor(8);
           OutTextXY(336, (8*i)+48, '['+ spareCarFolders[i] +']');
         END;
      END;
      { complete with spare cars }
      IF ((spareCarFoldersCount + spareCarACount) > MAX_SPARECARS) THEN BEGIN
         { check paging }
         IF (selHoldScroll > 0) THEN BEGIN
            SetColor(4); OutTextXY(336, (8*(spareItemsCount))+60,'PGUP');
            SetColor(8); OutTextXY(376, (8*(spareItemsCount))+60,'<<');
         END;
         SetColor(8);
         OutTextXY(404, (8*(spareItemsCount))+60,
            Byte2Dec((selHoldScroll*(MAX_SPARECARS-spareCarFoldersCount))+1)
            +' - ');
         IF ((spareCarFoldersCount + spareCarACount) > {if full page range}
             (((selHoldScroll+1)*(MAX_SPARECARS-spareCarFoldersCount))+spareCarFoldersCount)) THEN
             OutTextXY(452, (8*(spareItemsCount))+60,
                Byte2Dec( ((selHoldScroll+1)*(MAX_SPARECARS-spareCarFoldersCount)) ))
         ELSE {remaining page range}
             OutTextXY(452, (8*(spareItemsCount))+60,Byte2Dec( spareCarACount) ) ;
         IF ((spareCarFoldersCount + spareCarACount) >
             (((selHoldScroll+1)*(MAX_SPARECARS-spareCarFoldersCount))+spareCarFoldersCount)) THEN BEGIN
            SetColor(8); OutTextXY(488, (8*(spareItemsCount))+60,'>>');
            SetColor(4); OutTextXY(512, (8*(spareItemsCount))+60,'PGDN');
         END;
      END;

      FOR i := 1 TO spareCarCount DO BEGIN
        IF ((i+spareCarFoldersCount) = selHold) THEN SetColor(15) ELSE SetColor(8);
        OutTextXY(336, (8*(i+spareCarFoldersCount))+48, spareCarSlots[i].baseName+'  '+ spareCarSlots[i].longName);
        IF (spareCarSlots[i].unpacked = True) THEN BEGIN
           OutTextXY(552, (8*(i+spareCarFoldersCount))+48, '(*)');
           listedUnpacked := True;
        END;
        IF (i= spareCarCount) THEN BEGIN
           SetColor(8);
           OutTextXY(344, (8*(i+spareCarFoldersCount))+72, 'Number of spare cars: ');
           SetColor(2);
           OutTextXY(512, (8*(i+spareCarFoldersCount))+72, Byte2Dec(spareCarACount));
        END;
      END;
      IF (spareCarCount > 0) THEN BEGIN
         SetColor(8);
         OutTextXY(336, 424, '(Loaded from:');
         IF (Length(spareCarSelFolderPath) > 21) THEN
            OutTextXY(344, 432, '"'+spareCarSelFolderPath+ '")')
         ELSE OutTextXY(440, 424, '"'+spareCarSelFolderPath+ '")');

      END
      ELSE BEGIN
         IF (spareCarFoldersCount = 0 ) THEN BEGIN
            SetColor(8);
            OutTextXY(336, 56, '(No spare cars found)');
         END;
      END;
    END;

    IF (listedUnpacked) THEN BEGIN
       SetColor(8);
       OutTextXY(48, 424, 'NOTES:');
       OutTextXY(56, 440, '(*) car with modified or unpacked files');
    END;

    { Draw the help text: }
    SetColor(15);
    Line(5, 451, 634, 451);

    IF (spareCarOpen = False) THEN BEGIN
       IF (carCount > 0) THEN BEGIN
          SetColor(4); OutTextXY(8, 456, 'Up/Down');
          SetColor(8); OutTextXY(72, 456, 'Select Car');
          SetColor(4); OutTextXY(168, 456, 'Enter/F4');
          SetColor(8); OutTextXY(240, 456, 'Edit');
          IF (carCount < MAX_CARS ) THEN BEGIN
             SetColor(4); OutTextXY(8, 464, 'F5');
             SetColor(8); OutTextXY(32, 464, 'Copy');
          END;
          SetColor(4); OutTextXY(80, 464, 'F6');
          SetColor(8); OutTextXY(104, 464, 'Rename');
          SetColor(4); OutTextXY(168, 464, 'F8');
          SetColor(8); OutTextXY(192, 464, 'Delete');
       END;
       SetColor(4); OutTextXY(256, 464, 'F9/Right');
       SetColor(8); OutTextXY(328, 464, 'Open Spare Cars');
    END
    ELSE
    BEGIN
         IF (spareItemsCount > 0) THEN BEGIN
            SetColor(4); OutTextXY(8, 456, 'Up/Down');
            SetColor(8); OutTextXY(72, 456, 'Select Car/Folder');
         END;
         IF (spareCarFoldersCount > 0) THEN BEGIN
            SetColor(4); OutTextXY(8, 464, 'Enter');
            SetColor(8); OutTextXY(54, 464, 'Load Folder');
         END;
         IF (carCount > 0) THEN BEGIN
            SetColor(4); OutTextXY(216, 456, 'F2');
            SetColor(8); OutTextXY(240, 456, 'Move Car into Spares');
         END;
         IF ((carCount < MAX_CARS) AND (spareCarCount > 0)) THEN BEGIN
            SetColor(4); OutTextXY(416, 456, 'F3');
            SetColor(8); OutTextXY(440, 456, 'Copy Spare into Cars');
         END;
         IF ((spareCarSelFolderPath = curSettings.spareCarRelPath) AND
             (spareCarFoldersCount < MAX_SPAREFOLDERS-1))THEN BEGIN
            SetColor(4); OutTextXY(152, 464, 'F7');
            SetColor(8); OutTextXY(176, 464, 'Make Folder');
         END;
         SetColor(4); OutTextXY(272, 464, 'F9/Left');
         SetColor(8); OutTextXY(336, 464, 'Close Spare Cars');
    END;
    SetColor(4); OutTextXY(472, 464, 'Home');
    SetColor(8); OutTextXY(512, 464, 'Config');
    SetColor(4); OutTextXY(568, 464, 'Esc');
    SetColor(8); OutTextXY(600, 464, 'Quit');
  END;
END;

FUNCTION HandleGarageScreen(key: Word): Boolean;
VAR spareCarFolderExists: Boolean;
    createDirString:      String;
BEGIN
    HandleGarageScreen := False;
    CASE key OF
        KEY_F2:
            BEGIN {Move car to spares}
                IF (spareCarOpen AND (selItem > 0)) THEN
                BEGIN
                    IF (IsBaseNameInUse(carSlots[selItem].baseName, spareCarSelFolderPath)) THEN
                       ShowMessageBox('Error: Move Car '+carSlots[selItem].baseName+
                            ' to Spares','Car files colision detected.')
                    ELSE BEGIN
                       CopyCarFiles(carSlots[selItem].baseName, carSlots[selItem].baseName,
                             curSettings.carRelPath, spareCarSelFolderPath);
                       DeleteCarFiles(carSlots[selItem].baseName, curSettings.carRelPath);
                       FindCarFiles(curSettings.carRelPath);
                       FindCarFiles(spareCarSelFolderPath);
                    END;
                    HandleGarageScreen := True;
                END;
            END;
        KEY_F3:
            BEGIN {Copy spare to cars}
               IF (spareCarOpen AND (spareCarCount > 0)
                                AND (selHold > spareCarFoldersCount)
                                AND (carCount < MAX_CARS)
                                AND (selHold > spareCarFoldersCount)) THEN
               BEGIN
                    IF (IsBaseNameInUse(spareCarSlots[selHold-spareCarFoldersCount].baseName,
                           curSettings.carRelPath)) THEN
                       ShowMessageBox('Error: Copy Spare Car '+
                           spareCarSlots[selHold-spareCarFoldersCount].baseName+
                           ' to Cars', 'Car files colision detected.')
                    ELSE BEGIN
                         CopyCarFiles(spareCarSlots[selHold-spareCarFoldersCount].baseName,
                             spareCarSlots[selHold-spareCarFoldersCount].baseName,
                             spareCarSelFolderPath, curSettings.carRelPath);
                         FindCarFiles(curSettings.carRelPath);
                    END;
                    HandleGarageScreen := True;
               END;
            END;
        KEY_ENTER, { Edit }
        KEY_F4:
            BEGIN
                IF ((spareCarOpen = False) AND (carCount > 0)) THEN
                BEGIN
                   carOpen := selItem;
                   LoadResFile(curSettings.carRelPath+'CAR' + carSlots[carOpen].baseName + '.RES');
                   selView := 2;
                   selItem := 1;
                   HandleGarageScreen := True;
                END;
                IF (spareCarOpen AND (key = KEY_ENTER) AND
                   (selHold <= spareCarFoldersCount)) THEN
                BEGIN
                    spareCarSelFolderPath := spareCarSelFolderPath
                       + spareCarFolders[selHold] + '\';
                    spareCarSelFolder := spareCarFolders[selHold];
                    selHold := 1;
                    selHoldScroll := 0;
                    FindSpareCarFolders(spareCarSelFolderPath);
                    FindCarFiles(spareCarSelFolderPath);
                    HandleGarageScreen := True;
                END;
            END;
        KEY_F5: { Copy }
            BEGIN
               IF ((spareCarOpen = False) AND (carCount > 0) AND (carCount < MAX_CARS)) THEN
               BEGIN
                    CopyCarFilesDialog(False, curSettings.carRelPath);
                    HandleGarageScreen := True;
               END;
            END;
        KEY_F6: { Rename }
            BEGIN
               IF ((spareCarOpen = False) AND (carCount > 0)) THEN
               BEGIN
                    CopyCarFilesDialog(True, curSettings.carRelPath);
                    HandleGarageScreen := True;
               END;
            END;
        KEY_F7: { Make Spare Directory - only on root }
            BEGIN
               IF (spareCarOpen AND
                   (spareCarSelFolderPath = curSettings.spareCarRelPath)) THEN
               BEGIN
                    CreateSpareFolderDialog;
                    HandleGarageScreen := True;
               END;
            END;
        KEY_F8: { Delete }
            BEGIN
               IF ((spareCarOpen = False) AND (carCount > 0)) THEN
               BEGIN
                    DeleteCarFilesDialog(curSettings.carRelPath);
                    HandleGarageScreen := True;
               END;
            END;
        KEY_LEFT: { Close spare car list}
            BEGIN
                 IF (spareCarOpen) THEN BEGIN
                     spareCarOpen:= False;
                     selHold := 1;
                     HandleGarageScreen := True;
                 END
            END;
        KEY_RIGHT,
        KEY_F9: { Open spare car list }
            BEGIN
                 IF (spareCarOpen = False) THEN
                 BEGIN
                     spareCarFolderExists := { DosError: 3 = PathNotFound }
                        (FindSpareCarFolders(curSettings.spareCarRelPath) <> 3);
                     IF (spareCarFolderExists = False) THEN
                     BEGIN
                        IF (ShowQuestion('Info: No Spare Cars Folder Found',
                             'Create folder '+curSettings.spareCarRelPath+'?')) THEN
                        BEGIN
                            {$I-}
                            MkDir(curSettings.spareCarRelPath+'.');
                            {$I+}
                            spareCarOpen := True;
                            spareCarSelFolderPath := curSettings.spareCarRelPath;
                            FindCarFiles(curSettings.spareCarRelPath);
                        END;
                     END
                     ELSE
                     BEGIN
                          spareCarOpen := True;
                          selHold := 1;
                          selHoldScroll := 0;
                          spareCarSelFolder :='';
                          spareCarSelFolderPath := curSettings.spareCarRelPath;
                          FindCarFiles(curSettings.spareCarRelPath);
                          spareItemsCount := spareCarCount+spareCarFoldersCount;
                          IF (spareItemsCount = 0) THEN
                          BEGIN
                               ShowMessageBox('Info: Empty Spare Cars',
                                 'Move car into spares or Make folder.');
                          END;
                     END;
                     HandleGarageScreen := True;
                 END
                 ELSE IF (key = KEY_F9) THEN
                 BEGIN
                     spareCarOpen:= False;
                     selHold := 1;
                     HandleGarageScreen := True;
                 END;
            END;
        KEY_DOWN:
            BEGIN
               IF (spareCarOpen) THEN
               BEGIN
                  IF (spareItemsCount > 1) THEN BEGIN
                     selHoldOld := selHold;
                     IF (selHold < spareItemsCount) THEN Inc(selHold)
                     ELSE selHold := 1;
                     RenderCarListItem(True);
                  END;
               END
               ELSE BEGIN
                  selItemOld := selItem;
                  IF (selItem < carCount) THEN Inc(selItem)
                  ELSE selItem := 1;
                  RenderCarListItem(False);
               END;
            END;
        KEY_UP:
            BEGIN
               IF (spareCarOpen) THEN
               BEGIN
                  IF (spareItemsCount > 0) THEN BEGIN
                     selHoldOld := selHold;
                     IF (selHold > 1) THEN Dec(selHold)
                     ELSE
                     BEGIN
                         IF (spareItemsCount = 0) THEN selHold := 1
                         ELSE selHold := spareItemsCount;
                     END;
                     RenderCarListItem(True);
                  END;
               END
               ELSE BEGIN
                  selItemOld := selItem;
                  IF (selItem > 1) THEN Dec(selItem)
                  ELSE selItem := carCount;
                  RenderCarListItem(False);
               END;
            END;
        KEY_HOME:
            BEGIN
                 selView := 11;
                 selItem := 1;
                 HandleGarageScreen := True;
            END;
        KEY_PGUP:
            BEGIN
                 IF (spareCarOpen AND ((spareCarACount+spareCarFoldersCount) > MAX_SPARECARS)) THEN
                 BEGIN
                      IF (selHoldScroll > 0) THEN
                      BEGIN
                           Dec(selHoldScroll);
                           FindCarFiles(spareCarSelFolderPath);
                           HandleGarageScreen := True;
                      END;
                 END;
            END;
        KEY_PGDN:
            BEGIN
                 IF (spareCarOpen AND ((spareCarACount+spareCarFoldersCount) > MAX_SPARECARS)) THEN
                 BEGIN
                      IF ((((selHoldScroll+1)*(MAX_SPARECARS-spareCarFoldersCount))+spareCarFoldersCount
                         < spareCarACount+spareCarFoldersCount)) THEN
                      BEGIN
                           Inc(selHoldScroll);
                           FindCarFiles(spareCarSelFolderPath);
                           HandleGarageScreen := True;
                      END;
                 END;
            END;
    END;
END;

PROCEDURE RenderGenericHelp;
BEGIN
    SetColor(4); OutTextXY(8, 464, 'F1');
    IF (selView = 1) THEN SetColor(15) ELSE SetColor(8);
    OutTextXY(32, 464, 'Garage');
    SetColor(4); OutTextXY(96, 464, 'F2');
    IF (selView = 2) THEN SetColor(15) ELSE SetColor(8);
    OutTextXY(120, 464, 'Engine');
    SetColor(4); OutTextXY(184, 464, 'F3');
    IF (selView = 3) THEN SetColor(15) ELSE SetColor(8);
    OutTextXY(208, 464, 'Gears');
    SetColor(4); OutTextXY(264, 464, 'F4');
    IF (selView = 4) THEN SetColor(15) ELSE SetColor(8);
    OutTextXY(288, 464, 'Misc');
    SetColor(4); OutTextXY(336, 464, 'F5');
    IF (selView = 5) THEN SetColor(15) ELSE SetColor(8);
    OutTextXY(360, 464, 'Text');
    SetColor(4); OutTextXY(408, 464, 'F6');
    IF (selView = 6) THEN SetColor(15) ELSE SetColor(8);
    OutTextXY(432, 464, 'GFX');
    IF carHasChanges THEN
    BEGIN
        SetColor(4); OutTextXY(488, 464, 'F10');
        SetColor(8); OutTextXY(520, 464, 'Save');
    END;
    SetColor(4); OutTextXY(568, 464, 'Esc');
    SetColor(8); OutTextXY(600, 464, 'Quit');
END;

FUNCTION SwitchToView(view: Integer): Boolean;
BEGIN
   selView := view;
   selItem := 1;
   selHold := 1;
   SwitchToView := True;
END;

FUNCTION HandleGenericScreen(key: Word): Boolean;
VAR
    s: String[4];
BEGIN
    HandleGenericScreen := False;
    CASE key OF
        KEY_F1:
            BEGIN
                IF carHasChanges = True THEN
                BEGIN
                    IF (ShowQuestion('Returning to Garage',
                        'Changes will be lost. Proceed?')) THEN
                    BEGIN
                         SwitchToView(1);
                    END
                END
                ELSE
                    SwitchToView(1);
                HandleGenericScreen := True;
            END;
        KEY_F2,
        KEY_F3,
        KEY_F4,
        KEY_F5,
        KEY_F6:
            HandleGenericScreen := SwitchToView((key - KEY_F1) + 1);
        KEY_F10:
            BEGIN
                s := carSlots[carOpen].baseName;
                IF (ShowQuestion('Confirmation of Save Operation',
                    'Are you sure you want to save "'+s+'"?')) THEN
                BEGIN
                    SaveResFile(curSettings.carRelPath + 'CAR' + s + '.RES');
                END;
                HandleGenericScreen := True;
            END;
    END;
END;

PROCEDURE GetPowerCurveArea(var ml, mr: Integer);
BEGIN
    IF ((selItem >= 5) AND (selItem <= 108)) THEN
    BEGIN
        IF ((selHold >= 5) AND (selHold <= 108)) THEN
        BEGIN
            ml := MinInt(selItem-5, selHold-5);
            mr := MaxInt(selItem-5, selHold-5);
        END ELSE BEGIN
            ml := selItem-5;
            mr := ml;
        END;
    END;
END;

PROCEDURE RenderEngineScreenPowerCurve;
VAR
    ml, mr: Integer;
    i:      Integer;
    x:      Integer;
BEGIN
    { Fill the power graph area: }
    RenderSection(60, 40, 65, 'Power Curve');
    SetFillStyle(SolidFill, 1);
    Bar(60, 56, 60+(104*5)-1, 311);
    { Highlight the edited range, if any: }
    IF ((selItem >= 5) AND (selItem <= 108)) THEN
    BEGIN
        GetPowerCurveArea(ml, mr);
        SetFillStyle(SolidFill, 15);
        FOR i := ml TO mr DO
        BEGIN
            x := (i*5)+60;
            Bar(x, 311-carData.raw[$60+i], x+4, 311);
        END;
        { Draw power rpm cursor value }
        RenderSection(220, 320, 20, 'Selected Parameter');
        OutTextXY(220, 336, 'Index ' + Byte2Dec(i) + ' ('+Word2Dec(i*128) +'rpm)');
        OutTextXY(220, 344, 'Value ' + Byte2Dec(carData.raw[$60+(selItem-5)]))
    END;
    { Draw the torque power curve: }
    SetColor(14);
    MoveTo(60+2, 311-carData.raw[$60]);
    FOR i := 1 TO 103 DO
    BEGIN
        x := (i*5)+60+2;
        LineTo(x, 311-carData.raw[$60+i]);
    END;
    { Draw the RPM markers: }
    SetColor(7);
    x := Round(5.0 * (GetResWord($2C)/128.0)) + 65;
    Line(x, 56, x, 311);
    x := Round(5.0 * (GetResWord($2E)/128.0)) + 65;
    Line(x, 56, x, 311);
    x := Round(5.0 * (GetResWord($30)/128.0)) + 65;
    Line(x, 56, x, 311);
    x := Round(5.0 * (GetResWord($32)/128.0)) + 65;
    Line(x, 56, x, 311);
END;

PROCEDURE RenderEngineScreen(fully: Boolean);
VAR
    i, j:    Integer;
    x:       Integer;
    maxNm:   Word;
    maxNmAt: Word;
    maxHP:   Word;
    maxHPAt: Word;
BEGIN
    { Draw the header: }
    SetFillStyle(SolidFill, 7);
    Bar(576, 1, 631, 8);
    SetColor(15);
    OutTextXY(580, 1, 'Engine');
    { Draw the power curve: }
    IF (fully) THEN
        RenderEngineScreenPowerCurve;
    { Draw the torque/power summary: }
    RenderSection(404, 320, 22, 'Power Summary');
    SetColor(8);
    GetResMaxPower(maxNm, maxNmAt, maxHP, maxHPAt);
    OutTextXY(404, 336, 'Max. torque');
    OutTextXY(508, 336, Word2Dec(maxNm) + ' Nm');
    OutTextXY(404, 344, '  ... at');
    OutTextXY(508, 344, Word2Dec(maxNmAt) + ' rpm');
    OutTextXY(404, 352, 'Max. power');
    OutTextXY(508, 352, Word2Dec(maxHP) + ' HP');
    OutTextXY(404, 360, '  ... at');
    OutTextXY(508, 360, Word2Dec(maxHPAt) + ' rpm');
    OutTextXY(412, 376, '(approximate values)');
    { Draw the four RPM parameters: }
    RenderSection(60, 320, 17, 'RPM Bounds');
    RenderResWordValue(60, 336, 12, 'Idle', 1, $2C);
    RenderResWordValue(60, 344, 12, 'Shift down', 2, $2E);
    RenderResWordValue(60, 352, 12, 'Shift up', 3, $30);
    RenderResWordValue(60, 360, 12, 'Maximum', 4, $32);
    { Draw the help text: }
    SetColor(15);
    IF (selItem <= 4) THEN
    BEGIN
        Line(5, 451, 634, 451);
        SetColor(4); OutTextXY(8, 456, 'Tab');
        SetColor(8); OutTextXY(40, 456, 'Edit curve');
        SetColor(4); OutTextXY(136, 456, 'Up/Down');
        SetColor(8); OutTextXY(200, 456, 'Select');
        SetColor(4); OutTextXY(264, 456, 'Enter');
        SetColor(8); OutTextXY(312, 456, 'Edit');
    END ELSE BEGIN
        Line(5, 443, 634, 443);
        SetColor(4); OutTextXY(8, 448, 'Tab');
        SetColor(8); OutTextXY(40, 448, 'Edit bounds');
        SetColor(4); OutTextXY(144, 448, 'Left/Right');
        SetColor(8); OutTextXY(232, 448, 'Select');
        SetColor(4); OutTextXY(296, 448, 'Enter');
        SetColor(8);
        IF ((selHold < 5) OR (selHold > 108)) THEN
            OutTextXY(344, 448, 'Hold area selection')
        ELSE
            OutTextXY(344, 448, 'Release area selection');
        SetColor(4); OutTextXY(8, 456, 'Up/Dn');
        SetColor(8); OutTextXY(56, 456, '+1/-1');
        SetColor(4); OutTextXY(112, 456, 'PgUp/PgDn');
        SetColor(8); OutTextXY(192, 456, '+10/-10');
        SetColor(4); OutTextXY(264, 456, 'Home/End');
        SetColor(8); OutTextXY(336, 456, '+50/-50');
        IF ((selHold >= 5) AND (selHold <= 108)) THEN
        BEGIN
            SetColor(4); OutTextXY(408, 456, 'Ins/Del');
            SetColor(8); OutTextXY(472, 456, 'Stretch +/-20%');
        END;
    END;
    RenderGenericHelp;
END;

FUNCTION HandleEngineScreen(key: Word): Boolean;
VAR
    ml, mr: Integer;
    i:      Integer;
BEGIN
    HandleEngineScreen := False;
    IF (selItem <= 4) THEN
    BEGIN
        CASE key OF
            KEY_TAB:
                BEGIN
                    selItem := 5;
                    HandleEngineScreen := True;
                END;
            KEY_UP:
                IF (selItem > 1) THEN
                    Dec(selItem)
                ELSE
                    selItem := 4;
            KEY_DOWN:
                IF (selItem < 4) THEN
                    Inc(selItem)
                ELSE
                    selItem := 1;
            KEY_ENTER:
                BEGIN
                    HandleEngineScreen := True;
                    CASE selItem OF
                        1:   EditResWord($2C, 'RPM (idle)');
                        2:   EditResWord($2E, 'RPM (shift down)');
                        3:   EditResWord($30, 'RPM (shift up)');
                        4:   EditResWord($32, 'RPM (maximum)');
                        ELSE HandleEngineScreen := False;
                    END;
                END;
        END;
    END ELSE BEGIN
        GetPowerCurveArea(ml, mr);
        CASE key OF
            KEY_TAB:
                BEGIN
                    selItem := 1;
                    HandleEngineScreen := True;
                END;
            KEY_LEFT:
                BEGIN
                    IF (selItem > 5) THEN
                        Dec(selItem)
                    ELSE
                        selItem := 108;
                    HandleEngineScreen := True;
                END;
            KEY_RIGHT:
                BEGIN
                    IF (selItem < 108) THEN
                        Inc(selItem)
                    ELSE
                        selItem := 5;
                    HandleEngineScreen := True;
                END;
            KEY_ENTER:
                BEGIN
                    IF ((selHold < 5) OR (selHold > 108)) THEN
                        selHold := selItem
                    ELSE
                        selHold := 1;
                    HandleEngineScreen := True;
                END;
            KEY_UP:
                HandleEngineScreen := RaisePowerCurveArea(ml, mr, 1);
            KEY_DOWN:
                HandleEngineScreen := LowerPowerCurveArea(ml, mr, 1);
            KEY_PGUP:
                HandleEngineScreen := RaisePowerCurveArea(ml, mr, 10);
            KEY_PGDN:
                HandleEngineScreen := LowerPowerCurveArea(ml, mr, 10);
            KEY_HOME:
                HandleEngineScreen := RaisePowerCurveArea(ml, mr, 50);
            KEY_END:
                HandleEngineScreen := LowerPowerCurveArea(ml, mr, 50);
            KEY_INS:
                HandleEngineScreen := StretchPowerCurveArea(ml, mr, 1.2);
            KEY_DEL:
                HandleEngineScreen := StretchPowerCurveArea(ml, mr, 0.8);
        END;
    END;
END;

PROCEDURE RenderGearsScreenVelGraph(numGr: Integer);
VAR
    rpmSdn: Word;
    rpmSup: Word;
    rpmMax: Word;
    rpm:    Word;
    gr:     Integer;
    ratio:  Word;
    speed:  Word;
    bx, by: Integer;
    x1, y1: Integer;
    x2, y2: Integer;
    fix:    Double;
BEGIN
    RenderSection(60, 40, 40, 'RPM/Velocity Graph');
    SetFillStyle(SolidFill, 1);
    Bar(60, 56, 60+270-1, 56+(104*2)-1);
    rpmSdn := GetResWord($2E);
    rpmSup := GetResWord($30);
    rpmMax := GetResWord($32);
    IF ((rpmSup >= rpmSdn) AND (rpmMax >= rpmSup) AND
        (rpmMax > 0) AND (rpmMax < (104*128))) THEN
    BEGIN
        bx := 60;
        by := 56+(104*2)-1;
        SetColor(8);
        SetLineStyle(DottedLn, 0, NormWidth);
        y1 := rpmSdn DIV 64; Line(bx, by-y1, bx+269, by-y1);
        y1 := rpmSup DIV 64; Line(bx, by-y1, bx+269, by-y1);
        y1 := rpmMax DIV 64; Line(bx, by-y1, bx+269, by-y1);
        SetLineStyle(SolidLn, 0, NormWidth);
        FOR y1 := 0 TO 13 DO
        BEGIN
            rpm := y1 * 1000;
            y2 := rpm DIV 64;
            Line(bx+270, by-y2, bx+275, by-y2);
            OutTextXY(bx+280, (by-y2)-3, Word2Dec(rpm));
        END;
        FOR x1 := 0 TO 5 DO
        BEGIN
            x2 := x1 * 50;
            Line(bx+x2, by+1, bx+x2, by+4);
            OutTextXY((bx+x2)-36, by+9, Word2Dec(x2));
        END;
        FOR gr := 1 TO numGr DO
        BEGIN
            ratio := GetResWord($34+(2*gr));
            IF (ratio > 0) THEN
            BEGIN
                IF (gr = 1) THEN
                BEGIN
                    x1 := 0;
                    y1 := 0;
                END ELSE BEGIN
                    x1 := Round((rpmSdn * 256.0) / ratio);
                    y1 := rpmSdn DIV 64;
                END;
                x2 := Round((rpmMax * 256.0) / ratio);
                y2 := rpmMax DIV 64;
                IF (x2 >= 270) THEN
                BEGIN
                    fix := x2/269;
                    x2 := 269;
                    y2 := Round(y2 / fix);
                END;
                SetColor(8);
                SetLineStyle(DashedLn, 0, NormWidth);
                Line(bx+x1, by-y1, bx+x1, by);
                Line(bx+x2, by-y2, bx+x2, by);
                SetLineStyle(SolidLn, 0, NormWidth);
                IF (gr = 1) THEN
                    SetColor(10)
                ELSE IF (gr = NumGr) THEN
                    SetColor(12)
                ELSE
                    SetColor(15);
                Line(bx+x1, by-y1, bx+x2, by-y2);
            END;
        END;
    END ELSE BEGIN
        SetColor(2);
        OutTextXY(60, 56, 'Cannot render the RPM/velocity graph:');
        OutTextXY(60, 64, 'Bad RPM and/or gear ratio values.');
    END;
END;

PROCEDURE RenderGearsScreenKnobBox(numGr, boxX, boxY: Integer);
VAR
    i:    Integer;
    x, y: Integer;
BEGIN
    RenderSection(boxX, boxY, 10, 'Knob Pos.');
    SetFillStyle(SolidFill, 1);
    Bar(boxX, boxY+16, boxX+79, boxY+95);
    SetColor(15);
    FOR i := 0 TO numGr DO
    BEGIN
        x := GetResWord($42+(4*i)) + boxX;
        y := GetResWord($44+(4*i)) + boxY+16;
        OutTextXY(x, y, CHR_GEARS[i]);
    END;
END;

PROCEDURE RenderGearsScreen(fully: Boolean);
VAR
    i, j:  Integer;
    numGr: Byte;
    x, y:  Integer;
    ratio: Word;
    rpm:   Word;
BEGIN
    { Draw the header: }
    SetFillStyle(SolidFill, 7);
    Bar(584, 1, 631, 8);
    SetColor(15);
    OutTextXY(588, 1, 'Gears');
    { Draw the RPM/velocity graph: }
    numGr := carData.raw[$26];
    IF (fully) THEN
        RenderGearsScreenVelGraph(numGr);
    { Draw the shifting knob position indicator: }
    IF (fully) THEN
        RenderGearsScreenKnobBox(numGr, 444, 104);
    { Draw the table of editable gear values: }
    SetColor(15);
    OutTextXY(60, 328, 'Gear');
    OutTextXY(108, 328, 'Ratio');
    OutTextXY(164, 328, 'Pos-X');
    OutTextXY(220, 328, 'Pos-Y');
    Line(60, 339, 259, 339);
    FOR i := 0 TO 6 DO
    BEGIN
        y := (8*i)+344;
        IF ((selItem >= ((3*i)+1)) AND (selItem <= ((3*i)+3))) THEN
            SetColor(15)
        ELSE
            SetColor(8);
        OutTextXY(60, y, CHR_GEARS[i]);
        IF (selItem = ((3*i)+1)) THEN
            SetColor(15)
        ELSE
            SetColor(8);
        OutTextXY(108, y, Word2Dec(GetResWord($34+(2*i))));
        IF (selItem = ((3*i)+2)) THEN
            SetColor(15)
        ELSE
            SetColor(8);
        OutTextXY(164, y, Word2Dec(GetResWord($42+(4*i))));
        IF (selItem = ((3*i)+3)) THEN
            SetColor(15)
        ELSE
            SetColor(8);
        OutTextXY(220, y, Word2Dec(GetResWord($44+(4*i))));
    END;
    SetColor(8); OutTextXY(88, 408, 'Number of gears:');
    SetColor(2); OutTextXY(224, 408, chr($30 + numGr));
    SetColor(2);
    Line(55, 344, 55, 344+(numGr*8)+7);
    Line(264, 344, 264, 344+(GetResWord($26)*8)+7);
    SetColor(4);
    Line(55, 344+(numGr*8)+8, 55, 400);
    Line(264, 344+(numGr*8)+8, 264, 400);

    { Draw the table of non-editable gear/speed values: }
    SetColor(15);
    OutTextXY(356, 320, 'Velocities[mph]');
    Line(356, 347, 579, 347);
    OutTextXY(356, 336, 'Gear');
    OutTextXY(396, 328, 'rpmi@');
    OutTextXY(444, 328, 'rpm-@');
    OutTextXY(492, 328, 'rpm+@');
    OutTextXY(540, 328, 'rpm!@');
    OutTextXY(396, 336, Word2Dec(GetResWord($2C)));
    OutTextXY(444, 336, Word2Dec(GetResWord($2E)));
    OutTextXY(490, 336, Word2Dec(GetResWord($30)));
    OutTextXY(540, 336, Word2Dec(GetResWord($32)));

    { Add Direct Drive ratio data approximation}
    OutTextXY(278, 328, 'Approx.' );
    OutTextXY(278, 336, 'Ratio(*)' );
    Line(278, 347, 338, 347);
    SetColor(8);
    OutTextXY(278, 432, '(*) Direct Drive relation (128000 = 1:1)' );

    FOR i := 1 TO numGr DO
    BEGIN
        y := (8*i)+344;
        OutTextXY(356, y, CHR_GEARS[i]);
        ratio := GetResWord($34+(2*i));
        OutTextXY(274, y, Word2Dec(ratio DIV 128) );
        OutTextXY(294, y+1, '.' );
        IF (ratio > 0) THEN
        BEGIN
            FOR j := 0 TO 3 DO
            BEGIN
                rpm := GetResWord($2C+(2*j));
                OutTextXY(396+(j*48), y,
                    Word2Dec(Round((rpm * 256.0) / ratio)));
            END;
        END;
    END;
    { Draw the help text: }
    SetColor(15);
    Line(5, 451, 634, 451);
    SetColor(4); OutTextXY(8, 456, '+/-');
    SetColor(8); OutTextXY(40, 456, 'Change #gears');
    SetColor(4); OutTextXY(160, 456, 'Up/Down/Left/Right');
    SetColor(8); OutTextXY(312, 456, 'Select');
    SetColor(4); OutTextXY(376, 456, 'Enter');
    SetColor(8); OutTextXY(424, 456, 'Edit');
    SetColor(4); OutTextXY(472, 456, 'PgDn');
    SetColor(8); OutTextXY(512, 456, 'Equalize ratios');
    RenderGenericHelp;
END;

FUNCTION HandleGearsScreen(key: Word): Boolean;
VAR
    numGr: Integer;
    x, y: Integer;
BEGIN
    HandleGearsScreen := False;
    numGr := carData.raw[$26];
    x := ((selItem-1) MOD 3);  {0..2}
    y := ((selItem-1) DIV 3);  {N=0..6}
    CASE key OF
        KEY_PLUS:
            IF (numGr < 6) THEN
            BEGIN
                Inc(carData.raw[$26]);
                HandleGearsScreen := True;
            END;
        KEY_MINUS:
            IF (numGr > 1) THEN
            BEGIN
                Dec(carData.raw[$26]);
                HandleGearsScreen := True;
            END;
        KEY_DOWN:
            IF (y < numGr) THEN
                Inc(selItem, 3)
            ELSE
                selItem := x + 1;
        KEY_UP:
            IF (y > 0) THEN
                Dec(selItem, 3)
            ELSE
                selItem := (3 * numGr) + x + 1;
        KEY_LEFT:
            IF (x > 0) THEN
               Dec(selItem)
            ELSE
               Inc(selItem, 2);
        KEY_RIGHT:
            IF (x < 2) THEN
               Inc(selItem)
            ELSE
               Dec(selItem, 2);
        KEY_ENTER:
            BEGIN
                HandleGearsScreen := True;
                CASE x OF
                    0: EditResWord($34+(2*y),
                           'Ratio for Gear ' + CHR_GEARS[y]);
                    1: EditResWord($42+(4*y),
                           'Shifting Knob X for Gear ' + CHR_GEARS[y]);
                    2: EditResWord($44+(4*y),
                           'Shifting Knob Y for Gear ' + CHR_GEARS[y]);
                END;
            END;
        KEY_PGDN:
            BEGIN
                HandleGearsScreen := True;
                IF (NOT EqualizeGears) THEN
                BEGIN
                    ShowMessageBox('Error: Bad Data',
                        'Cannot equalize gear ratios.');
                END;
            END;
    END;
END;

PROCEDURE RenderMiscScreen;
VAR
    i,j,i2:   Integer;
    w:   Word;
    tmp: Longint;
    maxSpd,maxSpdGear, maxSpdRpm: Word;
BEGIN
    { Draw the header: }
    SetFillStyle(SolidFill, 7);
    Bar(592, 1, 631, 8);
    SetColor(15);
    OutTextXY(596, 1, 'Misc');
    { Draw the core physics: }
    RenderSection(60,40, 23, 'Core Physics');
    RenderResWordValue(60, 56, 18, 'Car mass', 1, $28);
    RenderResWordValue(60, 64, 18, 'Braking power', 2, $2A);
    RenderResWordValue(60, 72, 18, 'Aerodyn. resist.', 3, $5E);
    RenderResWordValue(60, 80, 18, 'Oversteering', 4, $DA);

    { Draw the Car information: }
    RenderSection(384, 48, 24, 'Real Specs Approximation');
    SetColor(8);
    OutTextXY(384, 64, 'Mass');
    OutTextXY(448, 64, Word2Dec(Round((GetResWord($28)*104))) + 'lb /'+
                       Word2Dec(Round((GetResWord($28)*47))) + 'kg');
    OutTextXY(384, 72, 'Width');
    OutTextXY(448, 72, Word2Dec(Round((GetResWord($EE)*2*12*(205/1024)*(0.75)))) + 'in /'+
                       Word2Dec(Round((GetResWord($EE)*2*304.8*(205/1024)*(0.75)))) + 'mm');
    OutTextXY(384, 80, 'Height');
    OutTextXY(448, 80, Word2Dec(Round((GetResWord($F0)*12*(205/1024)))) + 'in /'+
                       Word2Dec(Round((GetResWord($F0)*304.8*(205/1024)))) + 'mm');
    OutTextXY(384, 88, 'Length');
    OutTextXY(448, 88, Word2Dec(Round((GetResWord($F2)*2*12*(205/1024)*(0.75)))) + 'in /'+
                       Word2Dec(Round((GetResWord($F2)*2*304.8*(205/1024)*(0.75)))) + 'mm');
    OutTextXY(384, 96, 'Wheelbase');
    OutTextXY(448, 96, Word2Dec(Round(((GetResWord($FC)+(0-GetResWord($10E)))/64)*12*(205/1024)*(0.75)))+ 'in /'+
                       Word2Dec(Round(((GetResWord($FC)+(0-GetResWord($10E)))/64)*304.8*(205/1024)*(0.75))) + 'mm');

    { Draw the car dimensions: }
    RenderSection(60, 128, 23, 'Car Dimensions');
    RenderResWordValue(60, 144, 18, 'Width (half)', 5, $EE);
    RenderResWordValue(60, 152, 18, 'Height', 6, $F0);
    RenderResWordValue(60, 160, 18, 'Length (half)', 7, $F2);
    RenderResWordValue(60, 168, 18, 'Colision radius', 8, $F4);
    RenderResWordValue(60, 176, 18, 'Onboard car height', 9, $F6);

    { Draw the car dimensions symmary: }
    RenderSection(400, 128, 22, 'Car Dimensions Summary');
    SetColor(8);
    OutTextXY(400, 144, 'width');
    OutTextXY(464, 144, 'height');
    OutTextXY(528, 144, 'length');
    OutTextXY(408, 152, Byte2Dec((GetResWord($EE))*2 ));
    OutTextXY(472, 152, Byte2Dec((GetResWord($F0)) ));
    OutTextXY(536, 152, Byte2Dec((GetResWord($F2))*2 ));
    OutTextXY(400, 168, '(CAR1 related values)');

    { Draw the wheel parameters: }
    RenderSection(60, 196, 23, 'Wheel Coordinates');
    SetColor(8); OutTextXY(60, 212, 'Front Left:');
    RenderResWordValue(164, 212, 5, 'X', 10, $F8);
    RenderResWordValue(164, 220, 5, 'Y', 11, $FA);
    RenderResWordValue(164, 228, 5, 'Z', 12, $FC);
    SetColor(8); OutTextXY(60, 236, 'Front Right:');
    RenderResWordValue(164, 236, 5, 'X', 13, $FE);
    RenderResWordValue(164, 244, 5, 'Y', 14, $100);
    RenderResWordValue(164, 252, 5, 'Z', 15, $102);
    SetColor(8); OutTextXY(60, 260, 'Rear Right:');
    RenderResWordValue(164, 260, 5, 'X', 16, $104);
    RenderResWordValue(164, 268, 5, 'Y', 17, $106);
    RenderResWordValue(164, 276, 5, 'Z', 18, $108);
    SetColor(8); OutTextXY(60, 284, 'Rear Left:');
    RenderResWordValue(164, 284, 5, 'X', 19, $10A);
    RenderResWordValue(164, 292, 5, 'Y', 20, $10C);
    RenderResWordValue(164, 300, 5, 'Z', 21, $10E);

    { Draw the wheel summary: }
    RenderSection(408, 196, 21, 'Wheels Summary');
    SetColor(8);
    OutTextXY(456, 212, 'X');
    OutTextXY(504, 212, 'Y');
    OutTextXY(552, 212, 'Z');
    { Wheel FL}
    OutTextXY(408, 220, 'FL');
    OutTextXY(448, 220, '-'+Byte2Dec(($0-GetResWord($F8)) DIV 64));
    OutTextXY(496, 220, Byte2Dec((GetResWord($FA)) DIV 64));
    OutTextXY(552, 220, Byte2Dec((GetResWord($FC)) DIV 64));
    { Wheel FR}
    OutTextXY(408, 228, 'FR');
    OutTextXY(456, 228, Byte2Dec((GetResWord($FE)) DIV 64));
    OutTextXY(496, 228, Byte2Dec((GetResWord($100)) DIV 64));
    OutTextXY(552, 228, Byte2Dec((GetResWord($102)) DIV 64));
    { Wheel RR}
    OutTextXY(408, 236, 'RR');
    OutTextXY(456, 236, Byte2Dec((GetResWord($104)) DIV 64));
    OutTextXY(496, 236, Byte2Dec((GetResWord($106)) DIV 64));
    OutTextXY(544, 236, '-'+Byte2Dec(($0-GetResWord($108)) DIV 64));
    { Wheel RL}
    OutTextXY(408, 244, 'RL');
    OutTextXY(448, 244, '-'+Byte2Dec(($0-GetResWord($10A)) DIV 64));
    OutTextXY(496, 244, Byte2Dec((GetResWord($10C)) DIV 64));
    OutTextXY(544, 244, '-'+Byte2Dec(($0-GetResWord($10E)) DIV 64));
    OutTextXY(408, 260, '(CAR1 related values)');
    { Valid Coordinates }
    IF (ValidWheelCoordinates = False) THEN
    BEGIN
       SetColor(12);
       OutTextXY(258, 284, 'Invalid Coordinates, please check values.');
       SetColor(8);
       OutTextXY(258, 292, 'Tip: the sum of all X, Y and Z must be 0');
       OutTextXY(258, 300, '     to avoid car misbehaviour.');
    END;
    { Draw the grip: }
    RenderSection(60, 320, 23, 'Grip');
    RenderResWordValue(60, 336, 18, 'Base grip', 22, $CA);
    RenderResWordValue(60, 344, 18, 'Asphalt modifier', 23, $DC);
    RenderResWordValue(60, 352, 18, 'Dirt modifier', 24, $DE);
    RenderResWordValue(60, 360, 18, 'Ice modifier', 25, $E0);
    RenderResWordValue(60, 368, 18, 'Grass modifier', 26, $E2);

    { Draw the grip summary: }
    RenderSection(448, 320, 16, 'Grip Summary');
    SetColor(8);
    OutTextXY(448, 336, 'Base grip');
    OutTextXY(448, 344, 'Asphalt');
    OutTextXY(448, 352, 'Dirt');
    OutTextXY(448, 360, 'Ice');
    OutTextXY(448, 368, 'Grass');
    OutTextXY(536, 336, Word2Dec(GetResWord($CA)));
    FOR i := 0 TO 3 DO
    BEGIN
        tmp := GetResWord($CA);
        w := (tmp * GetResWord($DC+(2*i))) DIV 256;
        OutTextXY(536, 344+(i*8), Word2Dec(w));
    END;

    { Draw the custom parameters: }
    RenderSection(60, 392, 23, 'Custom Mod Parameters(*)');
    RenderResByteValue(60, 408, 20, 'Speedo needle color', 27, $D4); {213}
    RenderResByteValue(60, 416, 20, 'Tacho needle color', 28, $D5); {214}
    SetColor(8);
    OutTextXY(60,432,'(*) No effect on stock game');

    { Draw the help text: }
    SetColor(15);
    Line(5, 451, 634, 451);
    RenderGenericHelp;
    SetColor(4); OutTextXY(8, 456, 'Tab');
    SetColor(8); OutTextXY(40, 456, 'Cycle section');
    SetColor(4); OutTextXY(160, 456, 'Up/Down/Left/Right');
    SetColor(8); OutTextXY(312, 456, 'Select');
    SetColor(4); OutTextXY(376, 456, 'Enter');
    SetColor(8); OutTextXY(424, 456, 'Edit');
    IF (selItem > 9) AND (selItem < 22) THEN BEGIN
        SetColor(4); OutTextXY(464, 456, 'Ins');
        SetColor(8); OutTextXY(496, 456, 'Edit CAR1 value');
    END
    ELSE BEGIN
        SetColor(7);
        OutTextXY(464, 456, 'Ins');
        OutTextXY(496, 456, 'Edit CAR1 value');
    END;
END;

FUNCTION HandleMiscScreen(key: Word): Boolean;
BEGIN
    HandleMiscScreen := False;
    CASE key OF
        KEY_TAB:
            BEGIN
                IF (selItem < 5) THEN { Core Physics }
                    selItem := 5
                ELSE IF (selItem > 4) AND (selItem < 10) THEN { Car Dimensions }
                        selItem := 10
                ELSE IF (selItem > 9) AND (selItem < 22) THEN { Wheel Coordinates }
                        selItem := 22
                ELSE IF (selItem > 21) AND (selItem < 27) THEN { Grip }
                        selItem := 27
                ELSE IF (selItem > 26) AND (selItem < 29) THEN { Custom Params }
                        selItem := 1
                ELSE selItem := 1
            END;
        KEY_UP:
            IF (selItem > 1) THEN
                Dec(selItem)
            ELSE
                selItem := 28;
        KEY_DOWN:
            IF (selItem < 28) THEN
                Inc(selItem)
            ELSE
                selItem := 1;
        KEY_INS:
            BEGIN
                HandleMiscScreen := True;
                CASE selItem OF
                    10:   EditSignedWord($F8, 'Front Left X (CAR1 relative value)', True, 64);
                    11:   EditSignedWord($FA, 'Front Left Y (CAR1 relative value)', False, 64);
                    12:   EditSignedWord($FC, 'Front Left Z (CAR1 relative value)', False, 64);
                    13:   EditSignedWord($FE, 'Front Right X (CAR1 relative value)', False, 64);
                    14:   EditSignedWord($100, 'Front Right Y (CAR1 relative value)', False, 64);
                    15:   EditSignedWord($102, 'Front Right Z (CAR1 relative value)', False, 64);
                    16:   EditSignedWord($104, 'Rear Right X (CAR1 relative value)', False, 64);
                    17:   EditSignedWord($106, 'Rear Right Y (CAR1 relative value)', False, 64);
                    18:   EditSignedWord($108, 'Rear Right Z (CAR1 relative value)', True, 64);
                    19:   EditSignedWord($10A, 'Rear Left X (CAR1 relative value)', True, 64);
                    20:   EditSignedWord($10C, 'Rear Left Y (CAR1 relative value)', False, 64);
                    21:   EditSignedWord($10E, 'Rear Left Z (CAR1 relative value)', True, 64);
                    ELSE HandleMiscScreen := False;
                END;
            END;
        KEY_ENTER:
            BEGIN
                HandleMiscScreen := True;
                CASE selItem OF
                    1:   EditResWord($28, 'Car Mass');
                    2:   EditResWord($2A, 'Braking Power');
                    3:   EditResWord($5E, 'Aerodynamic Resistance');
                    4:   EditResWord($DA, 'Oversteering');
                    5:   EditResWord($EE, 'Width (half)');
                    6:   EditResWord($F0, 'Height');
                    7:   EditResWord($F2, 'Length (half)');
                    8:   EditResWord($F4, 'Colision Radius');
                    9:   EditResWord($F6, 'Onboard Car Height');
                    10:   EditSignedWord($F8, 'Front Left X (negative convertion)', True, 0);
                    11:   EditSignedWord($FA, 'Front Left Y', False, 0);
                    12:   EditSignedWord($FC, 'Front Left Z', False, 0);
                    13:   EditSignedWord($FE, 'Front Right X', False, 0);
                    14:   EditSignedWord($100, 'Front Right Y', False, 0);
                    15:   EditSignedWord($102, 'Front Right Z', False, 0);
                    16:   EditSignedWord($104, 'Rear Right X', False, 0);
                    17:   EditSignedWord($106, 'Rear Right Y', False, 0);
                    18:   EditSignedWord($108, 'Rear Right Z (negative convertion)', True, 0);
                    19:   EditSignedWord($10A, 'Rear Left X (negative convertion)', True, 0);
                    20:   EditSignedWord($10C, 'Rear Left Y', False, 0);
                    21:   EditSignedWord($10E, 'Rear Left Z (negative convertion)', True, 0);
                    22:   EditResWord($CA, 'Base Grip');
                    23:   EditResWord($DC, 'Grip Modifier for Asphalt');
                    24:   EditResWord($DE, 'Grip Modifier for Dirt');
                    25:   EditResWord($E0, 'Grip Modifier for Ice');
                    26:   EditResWord($E2, 'Grip Modifier for Grass');
                    27:   EditResColorByte($D4, 'Speedo Needle Color');
                    28:   EditResColorByte($D5, 'Tacho Needle Color (0 = speedo color)');
                    ELSE HandleMiscScreen := False;
                END;
            END;
    END;
END;

PROCEDURE RenderTextScreen;
VAR
    i: Integer;
    y: Integer;
BEGIN
    { Draw the header: }
    SetFillStyle(SolidFill, 7);
    Bar(592, 1, 631, 8);
    SetColor(15);
    OutTextXY(596, 1, 'Text');
    { Remove old markers: }
    SetColor(7);
    Line(189, 56, 189, 191);
    Line(451, 56, 451, 191);
    { Draw the full description: }
    RenderSection(192, 40, MAX_DESCLINE, 'Description');
    FOR i := 1 TO 9 DO
    BEGIN
        y := (8*i)+48;
        IF (i = selItem) THEN
        BEGIN
            SetColor(15);
            Line(189, y, 189, y+7);
            Line(451, y, 451, y+7);
        END ELSE
            SetColor(8);
        OutTextXY(192, y, carData.fullDesc[i]);
    END;
    { Draw the short name: }
    RenderSection(192, 136, MAX_DESCLINE, 'Short Scoreboard Name');
    IF (10 = selItem) THEN
    BEGIN
        SetColor(15);
        Line(189, 152, 189, 159);
        Line(451, 152, 451, 159);
    END ELSE
        SetColor(8);
    OutTextXY(192, 152, carData.shortName);
    { Draw the long name: }
    RenderSection(192, 168, MAX_DESCLINE, 'Long Scoreboard Name');
    IF (11 = selItem) THEN
    BEGIN
        SetColor(15);
        Line(189, 184, 189, 191);
        Line(451, 184, 451, 191);
    END ELSE
        SetColor(8);
    OutTextXY(192, 184, carData.longName);
    { Draw the help text: }
    SetColor(15);
    Line(5, 451, 634, 451);
    RenderGenericHelp;
    SetColor(4); OutTextXY(8, 456, 'Up/Down');
    SetColor(8); OutTextXY(72, 456, 'Select');
    SetColor(4); OutTextXY(136, 456, 'Enter');
    SetColor(8); OutTextXY(184, 456, 'Edit');
END;

FUNCTION HandleTextScreen(key: Word): Boolean;
BEGIN
    HandleTextScreen := False;
    CASE key OF
        KEY_DOWN:
            IF (selItem < 11) THEN
                Inc(selItem)
            ELSE
                selItem := 1;
        KEY_UP:
            IF (selItem > 1) THEN
                Dec(selItem)
            ELSE
                selItem := 11;
        KEY_ENTER:
            BEGIN
                Assert(selItem >= 1, 'Invalid text item selection');
                Assert(selItem <= 11, 'Invalid text item selection');
                CASE selItem OF
                    10:  BEGIN
                         carData.shortName :=
                             EditString(carData.shortName,
                                 'Short Scoreboard Name', MAX_SHORTNAME);
                         carHasChanges := True;
                         END;
                    11:  BEGIN
                         carData.longName :=
                             EditString(carData.longName,
                                 'Long Scoreboard Name', MAX_LONGNAME);
                         carHasChanges := True;
                         END;
                    ELSE BEGIN
                         carData.fullDesc[selItem] :=
                             EditString(carData.fullDesc[selItem],
                                 'Full Description - Line ' + chr($30+selItem),
                                 MAX_DESCLINE);
                         carHasChanges := True;
                         END;
                END;
                HandleTextScreen := True;
            END;
    END;
END;

PROCEDURE RenderGfxScreenWheel;
VAR
    j,k,l,m:    Integer;
    i:    Integer;
    x,xs, y: Integer;
    xCenter, yCenter: Integer;
BEGIN
    RenderSection(160, 40, 40, 'Steering Wheel Dots');
    xCenter:=carData.raw[$110] + 160;
    yCenter:=carData.raw[$111] + 56;

    FOR i := 0 TO 30 DO
    BEGIN
       x := carData.raw[$110+(2*i)] + 160;
       y := carData.raw[$111+(2*i)] + 56;

       IF (selHold > 62) THEN selHold := 62;
       IF (selHold < 1) THEN selHold := 1;

       IF (i = ((selHold-1) DIV 2)) THEN BEGIN
           IF ((i MOD 5) = 0) THEN SetColor(12) ELSE SetColor(10);
           Rectangle(x-1, y-1, x+1, y+1);
           SetColor(15);
           xs := xCenter - (carData.raw[$110+(2*i)] +160 - xCenter);
           Rectangle(xs-1, y-1, xs+1, y+1);

           { draw selected coordinate values: }
           SetColor(8);
           OutTextXY(244, 344, Byte2Dec(i));
           IF (selHold MOD 2 =1) THEN SetColor(15) ELSE SetColor(8);
           OutTextXY(296, 344, Byte2Dec(carData.raw[$110+(2*i)]));
           IF (selHold MOD 2 =0) THEN SetColor(15) ELSE SetColor(8);
           OutTextXY(336, 344, Byte2Dec(carData.raw[$111+(2*i)]));
           SetColor(8);
           IF (selHold <= 2) THEN OutTextXY(372, 344, 'Center');
           IF (selHold > 60) THEN OutTextXY(380, 344, 'Max.');
       END
       ELSE BEGIN { draw all other coordinates:}
           IF ((i MOD 5) = 0) THEN SetColor(4) ELSE SetColor(2);
           Rectangle(x, y, x, y);
           { draw the "shadowed" left-hand dots: }
           xs := xCenter - (carData.raw[$110+(2*i)] +160 - xCenter);
           SetColor(7);
           Rectangle(xs, y, xs, y);
       END;

    END;

    SetColor(15);
    OutTextXY(236, 320, 'Selected Parameter');
    RenderSection(236, 328, 23, 'Index    X    Y    ');

    SetColor(8);
    OutTextXY(160, 280, '- All positions relative to dashboard "WHL2"');
    OutTextXY(160, 288, '- Only the right-hand dots are stored');
    OutTextXY(160, 296, '- Left-hand dots are drawn symmetrically');
END;

PROCEDURE RenderGfxScreenSpeedo;
VAR
    i:    Integer;
    x, y: Integer;
    x0, y0: Integer;
    spd, spdLim: Integer;
BEGIN
    IF (GetResWord($150) = 0) THEN
    BEGIN
        RenderSection(160, 40, 40, 'Digital Speedometer');
        FOR i := 0 TO 2 DO
        BEGIN
            x := carData.raw[$154+(2*i)] + 160;
            y := carData.raw[$155+(2*i)] + 56;

            IF (selHold > 6) THEN selHold := 6;
            IF (selHold < 1) THEN selHold := 1;

            IF (i = ((selHold-1) DIV 2)) THEN BEGIN
               Rectangle(x-1, y-1, x+1, y+1);
               { draw selected coordinate values: }
               SetColor(8);
               OutTextXY(244, 344, Byte2Dec(i));
               IF (selHold MOD 2 =1) THEN SetColor(15) ELSE SetColor(8);
               OutTextXY(296, 344, Byte2Dec(carData.raw[$154+(2*i)]));
               IF (selHold MOD 2 =0) THEN SetColor(15) ELSE SetColor(8);
               OutTextXY(336, 344, Byte2Dec(carData.raw[$155+(2*i)]));
               SetColor(8);
               OutTextXY(380, 344, Word2Dec(TenToThe(2-i)));
            END
            ELSE BEGIN { draw all other coordinates:}
                SetColor(15);
                Rectangle(x, y, x, y);
            END;
        END;

        SetColor(15);
        OutTextXY(236, 320, 'Selected Parameter');
        RenderSection(236, 328, 23, 'Index    X    Y    Magn.');

        SetColor(8);
        OutTextXY(160, 280, '- This is where the digits are drawn');
        OutTextXY(160, 288, '- All positions relative to dashboard "INS2"');

    END ELSE BEGIN
        RenderSection(160, 40, 40, 'Analog Speedometer');
        x0 := GetResWord($14E) + 160;
        y0 := GetResWord($150) + 56;
        spd := 0;
        spdLim := GetResWord($152);

        IF (selHold < 1) THEN BEGIN { center position: }
           SetFillStyle(SolidFill, 14);
           Bar(x0-1, y0-1, x0+1, y0+1);
           SetColor(8);
           OutTextXY(238, 344, 'Center');
           IF (selHold=-1) THEN SetColor(15) ELSE SetColor(8);
           OutTextXY(280, 344, Word2Dec(carData.raw[$14E]));
           IF (selHold=0) THEN SetColor(15) ELSE SetColor(8);
           OutTextXY(320, 344, Word2Dec(carData.raw[$150]));
        END
        ELSE BEGIN { needle positions: }
             SetFillStyle(SolidFill, 14);
             Bar(x0, y0, x0, y0);
             IF (selHold >= spdLim*2) THEN selHold:=spdLim*2;
             FOR i := 0 TO (spdLim-1) DO BEGIN
                x := carData.raw[$154+(2*i)] + 160;
                y := carData.raw[$155+(2*i)] + 56;
                { draw selected coordinate: }
                IF (i = ((selHold-1) DIV 2)) THEN BEGIN
                   IF ((i MOD 5) = 0) THEN SetColor(12) ELSE SetColor(10);
                   Rectangle(x-1, y-1, x+1, y+1);
                   SetColor(15);
                   Line(x,y,x0,y0);
                   { draw selected coordinate values: }
                   SetColor(8);
                   OutTextXY(244, 344, Byte2Dec(i));
                   IF (selHold MOD 2 =1) THEN SetColor(15) ELSE SetColor(8);
                   OutTextXY(296, 344, Byte2Dec(carData.raw[$154+(2*i)]));
                   IF (selHold MOD 2 =0) THEN SetColor(15) ELSE SetColor(8);
                   OutTextXY(336, 344, Byte2Dec(carData.raw[$155+(2*i)]));
                   IF ((i MOD 5) = 0) THEN SetColor(4) ELSE SetColor(2);
                   IF (i MOD 2 = 0) THEN
                      OutTextXY(364, 344, Word2Dec(spd)+'.0')
                   ELSE
                      OutTextXY(364, 344, Word2Dec(spd)+'.5');
                END
                ELSE BEGIN { draw all other coordinates:}
                     IF ((i MOD 5) = 0) THEN SetColor(4) ELSE SetColor(2);
                     Rectangle(x, y, x, y);
                END;
                { calc next speed integer value}
                Inc (spd,(2+ (i MOD 2)));
             END;
        END;

        SetColor(15);
        OutTextXY(236, 320, 'Selected Parameter');
        RenderSection(236, 328, 23, 'Index    X    Y     MPH');
        SetColor(8);
        OutTextXY(236, 360, 'Number of positions:');
        SetColor(2);
        OutTextXY(396, 360, Byte2Dec(carData.raw[$152]));

        SetColor(8);
        OutTextXY(160, 280, '- All positions relative to dashboard "INS2"');
        OutTextXY(160, 288, '- Edit center coordinates selecting below index "0"');
        OutTextXY(160, 296, '- Disable speedometer by set center to x=255,y=255');
    END;
END;

PROCEDURE RenderGfxScreenRevMeter;
VAR
    i:    Integer;
    x, y: Integer;
    x0, y0: Integer;
    rpm, rpmLim : Integer;
BEGIN
    RenderSection(160, 40, 40, 'Rev Meter');
    x0 := GetResWord($224) + 160;
    y0 := GetResWord($226) + 56;
    rpm := 0;
    rpmLim := GetResWord($228);

    IF (selHold < 1) THEN BEGIN { center position: }
       SetFillStyle(SolidFill, 14);
       Bar(x0-1, y0-1, x0+1, y0+1);
       SetColor(8);
       OutTextXY(238, 344, 'Center');
       IF (selHold=-1) THEN SetColor(15) ELSE SetColor(8);
       OutTextXY(280, 344, Word2Dec(carData.raw[$224]));
       IF (selHold=0) THEN SetColor(15) ELSE SetColor(8);
       OutTextXY(320, 344, Word2Dec(carData.raw[$226]));
    END
    ELSE BEGIN { needle positions: }
         SetFillStyle(SolidFill, 14);
         Bar(x0, y0, x0, y0);
         IF (selHold >= rpmLim * 2) THEN selHold := rpmLim*2;
         FOR i := 0 TO (rpmLim-1) DO BEGIN
             x := carData.raw[$22A+(2*i)] + 160;
             y := carData.raw[$22B+(2*i)] + 56;
             { draw selected coordinate: }
             IF (i = ((selHold-1) DIV 2)) THEN BEGIN
                IF ((i MOD 5) = 0) THEN SetColor(12) ELSE SetColor(10);
                Rectangle(x-1, y-1, x+1, y+1);
                SetColor(15);
                Line(x,y,x0,y0);
                { draw selected coordinate values: }
                SetColor(8);
                OutTextXY(244, 344, Byte2Dec(i));
                IF (selHold MOD 2 =1) THEN SetColor(15) ELSE SetColor(8);
                OutTextXY(296, 344, Byte2Dec(carData.raw[$22A+(2*i)]));
                IF (selHold MOD 2 =0) THEN SetColor(15) ELSE SetColor(8);
                OutTextXY(336, 344, Byte2Dec(carData.raw[$22B+(2*i)]));
                IF ((i MOD 5) = 0) THEN SetColor(4) ELSE SetColor(2);
                OutTextXY(380, 344, Word2Dec(rpm));
             END
             ELSE BEGIN { draw all other coordinates:}
                 IF ((i MOD 5) = 0) THEN SetColor(4) ELSE SetColor(2);
                 Rectangle(x, y, x, y);
             END;
             { calc next rpm value}
             Inc(rpm,128);
         END;
    END;

    SetColor(15);
    OutTextXY(236, 320, 'Selected Parameter');
    RenderSection(236, 328, 23, 'Index    X    Y     RPM');
    SetColor(8);
    OutTextXY(236, 360, 'Number of positions:');
    SetColor(2);
    OutTextXY(396, 360, Byte2Dec(carData.raw[$228]));

    SetColor(8);
    OutTextXY(160, 280, '- All positions relative to dashboard "INS2"');
    OutTextXY(160, 288, '- Edit center coordinates selecting below index "0"');
END;

PROCEDURE RenderGfxScreen;
BEGIN
    { Draw the header: }
    SetFillStyle(SolidFill, 7);
    Bar(600, 1, 631, 8);
    SetColor(15);
    OutTextXY(604, 1, 'GFX');
    { Draw the selected set of positions: }
    SetFillStyle(SolidFill, 1);
    Bar(160, 56, 479, 256);
    SetColor(7);
    Line(160, 256, 479, 256);
    CASE selItem OF
        1:   RenderGfxScreenWheel;
        2:   RenderGfxScreenSpeedo;
        3:   RenderGfxScreenRevMeter;
        ELSE Assert(False, 'Invalid item selection');
    END;
    { Draw the help text: }
    SetColor(15);
    Line(5, 443, 634, 443);
    RenderGenericHelp;
    SetColor(4); OutTextXY(8, 448, 'Tab');
    SetColor(8); OutTextXY(40, 448, 'Cycle objects');
    IF ((selItem = 2)) THEN BEGIN
       SetColor(4); OutTextXY(160, 448, 'PgUp/PgDn');
       SetColor(8); OutTextXY(240, 448, 'Convert Analog/Digital');
    END;
    IF ((selItem > 1) AND (carData.raw[$150] > 0)) THEN BEGIN
       SetColor(4); OutTextXY(416, 456, '+/-');
       SetColor(8); OutTextXY(448, 456, 'Change #pos');
    END;
    SetColor(4); OutTextXY(8, 456, 'Up/Down');
    SetColor(8); OutTextXY(72, 456, 'Select');
    SetColor(4); OutTextXY(136, 456, 'Left/Right');
    SetColor(8); OutTextXY(224, 456, 'Coordinate');
    SetColor(4); OutTextXY(320, 456, 'Enter');
    SetColor(8); OutTextXY(368, 456, 'Edit');
END;

FUNCTION HandleGfxScreen(key: Word): Boolean;
BEGIN
    HandleGfxScreen := False;
    CASE key OF
        KEY_TAB:
            BEGIN
                IF (selItem < 3) THEN Inc(selItem)
                ELSE selItem := 1;
                HandleGfxScreen := True;
            END;
        KEY_ENTER:
            BEGIN
                CASE selItem OF
                    1: BEGIN
                       IF (selHold > 0) THEN BEGIN
                          IF (selHold MOD 2 = 1) THEN
                             EditResByte($110+(selHold-1),'Position Coordinate X')
                          ELSE
                             EditResByte($110+(selHold-1),'Position Coordinate Y');
                       END
                    END;
                    2: BEGIN
                       IF (selHold > 0) THEN BEGIN
                          IF (selHold MOD 2 = 1) THEN
                             EditResByte($154+(selHold-1),'Position Coordinate X')
                          ELSE
                             EditResByte($154+(selHold-1),'Position Coordinate Y');
                       END
                       ELSE BEGIN
                           IF (selHold = -1) THEN
                               EditResWord($14E,'Center Coordinate X')
                           ELSE
                               EditResWord($150,'Center Coordinate Y');
                       END
                    END;
                    3: BEGIN
                       IF (selHold > 0) THEN BEGIN
                          IF (selHold MOD 2 = 1) THEN
                             EditResByte($22A+(selHold-1),'Position Coordinate X')
                          ELSE
                             EditResByte($22A+(selHold-1),'Position Coordinate Y');
                       END
                       ELSE BEGIN
                           IF (selHold = -1) THEN
                               EditResWord($224,'Center Coordinate X')
                           ELSE
                               EditResWord($226,'Center Coordinate Y');
                       END
                    END;
                END;
                HandleGfxScreen := True;
            END;
        KEY_UP:
            BEGIN
                 CASE selItem OF
                    1: IF (selHold < 256) THEN BEGIN
                          Inc(selHold,2);
                          HandleGfxScreen := True;
                       END;
                    2: IF ((carData.raw[$152] >= 1) AND (carData.raw[$150] >= 0)) THEN
                       BEGIN
                          IF (selHold < 256) THEN Inc(selHold,2);
                          HandleGfxScreen := True;
                       END;
                    3: IF (carData.raw[$228] >= 1) THEN
                       BEGIN
                          IF (selHold < 256) THEN Inc(selHold,2);
                          HandleGfxScreen := True;
                       END;
                END;
            END;
        KEY_DOWN:
            BEGIN
                 CASE selItem OF
                    1: BEGIN
                          IF (selHold > 0 ) THEN Dec(selHold,2)
                          ELSE selHold := -1;
                          HandleGfxScreen := True;
                       END;
                    2: IF ((carData.raw[$152] >= 1) AND (carData.raw[$150] >= 0)) THEN
                       BEGIN
                          IF (selHold > 0 ) THEN Dec(selHold,2)
                          ELSE selHold := -1;
                          HandleGfxScreen := True;
                       END;
                    3: IF (carData.raw[$228] >= 1) THEN
                       BEGIN
                          IF (selHold > 0 ) THEN Dec(selHold,2)
                          ELSE selHold := -1;
                          HandleGfxScreen := True;
                       END;
                 END;
            END;
        KEY_LEFT:
            IF (((selHold MOD 2 = 0) AND (selHold > 0)) OR (selHold = 0)) THEN
                Dec(selHold);
        KEY_RIGHT:
            IF (((selHold MOD 2 = 1) AND (selHold < 256)) OR (selHold =-1)) THEN
                Inc(selHold);
        KEY_MINUS:
            BEGIN
                CASE selItem OF
                    2: IF ((carData.raw[$152] >= 1) AND (carData.raw[$150] >= 0)) THEN
                       BEGIN
                          Dec(carData.raw[$152]);
                          IF (carData.raw[$152] < 1) THEN selHold := -1;
                          HandleGfxScreen := True;
                       END;
                    3: IF (carData.raw[$228] >= 1) THEN
                       BEGIN
                          Dec(carData.raw[$228]);
                          IF (carData.raw[$228] < 1) THEN selHold := -1;
                          HandleGfxScreen := True;
                       END;
                END;
            END;
        KEY_PLUS:
            BEGIN
                CASE selItem OF
                    2: IF ((carData.raw[$152] < 104) AND (carData.raw[$150] >= 0)) THEN
                       BEGIN
                          Inc(carData.raw[$152]);
                          HandleGfxScreen := True;
                       END;
                    3: IF (carData.raw[$228] < 128) THEN
                       BEGIN
                          Inc(carData.raw[$228]);
                          HandleGfxScreen := True;
                       END;
                END;
            END;
        KEY_PGUP:
            BEGIN
                 IF ((selItem = 2) AND (carData.raw[$150] >= 0)) THEN BEGIN
                      backIntX := GetResWord($14E);
                      backIntY := GetResWord($150);
                      SetResWord($14E, 0);
                      SetResWord($150, 0);
                      HandleGfxScreen := True;
                 END;
            END;
        KEY_PGDN:
            BEGIN
                 IF ((selItem = 2) AND (carData.raw[$150] = 0)) THEN BEGIN
                      IF ((backIntX = 0) AND (backIntY = 0)) THEN BEGIN
                         backIntX := 1;
                         backIntY := 1;
                      END;
                      SetResWord($14E, backIntX);
                      SetResWord($150, backIntY);
                      HandleGfxScreen := True;
                 END;
            END;
    END;
END;

FUNCTION HandleConfigScreen(key: Word): Boolean;
VAR tempSetValue : String;
    strC         : Array [0..4] Of Char;
BEGIN
    HandleConfigScreen := False;
    CASE key OF
        KEY_F1:
            BEGIN
                IF (configHasChanges) THEN
                BEGIN
                    IF (ShowQuestion('Returning to Garage',
                        'Unsaved changes will be discarted. Ok?')) THEN
                    BEGIN
                         LoadSettingsFile;
                         spareCarOpen := False;
                         SwitchToView(1);
                    END;
                END
                ELSE BEGIN
                     spareCarOpen := False;
                     SwitchToView(1);
                END;
                HandleConfigScreen := True;
            END;
        KEY_F10:
            BEGIN
                IF (configHasChanges) THEN
                BEGIN
                   IF (ShowQuestion('Confirmation of Save Operation',
                          'Are you sure you want to save"?')) THEN
                   BEGIN
                     SaveSettingsFile(curSettings);
                     IF ((settingsChanged[1] OR settingsChanged[2])) THEN
                     BEGIN
                        IF (ShowQuestion('Settings Saved',
                           'Restart with new saved settings?')) THEN
                        BEGIN
                           settingsChanged[1] := False;
                           settingsChanged[2] := False;
                           settingsChanged[3] := False;
                           restartRequested := True;
                           quit := True;
                        {END
                        ELSE
                             ShowMessageBox('Warning: Applying Settings',
                             '"#" marked settings requires restart.');}
                        END;
                     END;
                   END;
                END;
                HandleConfigScreen := True;
            END;
        KEY_UP:
            BEGIN
                 IF (selItem > 1) THEN
                     Dec(selItem)
                 ELSE
                     selItem := 3;
            END;
        KEY_DOWN:
            BEGIN
                 IF (selItem < 3) THEN
                    Inc(selItem)
                 ELSE
                     selItem := 1;
            END;
        KEY_ENTER:
            BEGIN
                CASE selItem OF
                    1:  BEGIN
                             tempSetValue := curSettings.video;
                             curSettings.video :=
                             EditString(curSettings.video,
                                 'Video Driver (Use "VGA" or "SVGA")', 4 );
                             curSettings.video := Str_ToUpper(curSettings.video);
                             IF (((curSettings.video = 'VGA') OR
                                 (curSettings.video = 'SVGA')) = False) THEN BEGIN
                                curSettings.video := tempSetValue;
                                ShowMessageBox('Info: Bad Input', 'Use "VGA" or "SVGA" as Video Driver.');
                             END
                             ELSE BEGIN
                                IF (tempSetValue <> curSettings.video) THEN
                                BEGIN
                                     settingsChanged[selItem] := True;
                                     configHasChanges := True;
                                END;
                             END;
                        END;
                    2:  BEGIN
                             tempSetValue := curSettings.carRelPath;
                             curSettings.carRelPath :=
                             EditString(curSettings.carRelPath,
                                 'Cars Directory Relative Path', 255);
                             IF (curSettings.carRelPath = curSettings.spareCarRelPath) THEN BEGIN
                                curSettings.carRelPath := tempSetValue;
                                ShowMessageBox('Info: Bad Input', 'Use a diffent Path.');
                             END
                             ELSE BEGIN
                                IF (tempSetValue <> curSettings.carRelPath) THEN
                                BEGIN
                                   settingsChanged[selItem] := True;
                                   configHasChanges := True;
                                END;
                             END;
                        END;
                    3:  BEGIN
                             tempSetValue := curSettings.spareCarRelPath;
                             curSettings.spareCarRelPath :=
                             EditString(curSettings.spareCarRelPath,
                                 'Spare Cars Directory Relative Path', 255);
                             IF (curSettings.spareCarRelPath = curSettings.carRelPath) THEN BEGIN
                                curSettings.spareCarRelPath := tempSetValue;
                                ShowMessageBox('Info: Bad Input', 'Use a diffent Path.');
                             END
                             ELSE BEGIN
                                IF (tempSetValue <> curSettings.spareCarRelPath) THEN
                                BEGIN
                                   settingsChanged[selItem] := True;
                                   configHasChanges := True;
                                END;
                             END;
                        END;
                END;
                HandleConfigScreen := True;
            END;
    END;
END;


PROCEDURE RenderConfigScreen;
VAR i:    Integer;
    s:    String;

BEGIN
    { Draw the header: }
    SetFillStyle(SolidFill, 7);
    Bar(520, 1, 631, 8);
    SetColor(15);
    OutTextXY(524, 1, 'Configuration');

    { Draw the settings: }
    RenderSection(60, 40, 64, 'Settings');
    RenderStringValue(60, 56, 28, 'Video Driver', curSettings.video, 1);
    RenderStringValue(60, 64, 28, 'Cars Directory Path', curSettings.carRelPath ,2);
    RenderStringValue(60, 72, 28, 'Spare Cars Directory Path', curSettings.spareCarRelPath ,3);

    { Test Font Chars and Colors
    FOR i:=0 TO 63 DO
    BEGIN
        SetColor(i);    OutTextXY(64+8*i, 300, Chr(i));
        SetColor(i+64); OutTextXY(64+8*i, 312, Chr(i+64));
        SetColor(i+128);OutTextXY(64+8*i, 324, Chr(i+128));
        SetColor(i+192);OutTextXY(64+8*i, 336, Chr(i+192));
    END;
    }

    { Draw Info and notes }
    SetColor(8);
    GetDir(0,s); { 0 = Current drive }
    OutTextXY(68, 112, '(Current directory path: "'+s+'")');

    FOR i:=1 TO 2 DO { mark settings that requires restart }
    BEGIN
         IF (settingsChanged[i] = True) THEN BEGIN
            IF ((i = 1) AND (curSettings.video = 'SVGA')) THEN BEGIN
               SetColor(12);
               OutTextXY(320, 56, '(*)');
               OutTextXY(48, 440, '(*) If error after restart, restore defaults deleting "CAREDIT3.CFG".');
            END;
            SetColor(8);
            OutTextXY(48, 424, 'NOTES:');
            OutTextXY(48, 432, '(#) Requires restart to be applied');
            OutTextXY(272,48+(i*8),'#');
        END;
    END;

    { Draw the help text: }
    SetColor(15);
    Line(5, 451, 634, 451);
    SetColor(4); OutTextXY(8, 456, 'Up/Down');
    SetColor(8); OutTextXY(72, 456, 'Select');
    SetColor(4); OutTextXY(136, 456, 'Enter');
    SetColor(8); OutTextXY(184, 456, 'Edit');
    SetColor(4); OutTextXY(8, 464, 'F1');
    SetColor(8); OutTextXY(32, 464, 'Garage');
    IF (configHasChanges) THEN BEGIN
       SetColor(4); OutTextXY(488, 464, 'F10');
       SetColor(8); OutTextXY(520, 464, 'Save');
    END;
    SetColor(4); OutTextXY(568, 464, 'Esc');
    SetColor(8); OutTextXY(600, 464, 'Quit');
END;

PROCEDURE RenderScreen(fully: Boolean);
BEGIN
    IF (fully) THEN
    BEGIN
        SetBkColor(7);
        ClearDevice;
        SetFillStyle(SolidFill, 7);
        SetLineStyle(SolidLn, 15, NormWidth);
        SetWriteMode(NormalPut);
        SetColor(15);
        Rectangle(4, 4, 635, 475);
        Bar(8, 1, 286, 8);
        OutTextXY(12, 1, '4D Sports Driving Car Editor 3.4.2');
        IF (selView > 1) AND (selView <= 6) THEN
        BEGIN
            Bar(300, 1, 339, 8);
            OutTextXY(304, 1, carSlots[carOpen].baseName);
        END;
    END;
    CASE selView OF
        1:   RenderGarageScreen(fully);
        2:   RenderEngineScreen(fully);
        3:   RenderGearsScreen(fully);
        4:   RenderMiscScreen;
        5:   RenderTextScreen;
        6:   RenderGfxScreen;
        11:   RenderConfigScreen;
        ELSE Assert(False, 'Invalid view mode');
    END;
    { variable watch: }
    {Bar(304, 16, 420, 32);
    SetColor(8);
    OutTextXY(304,16, 'selItem:'+ Word2Dec(selItem));
    OutTextXY(304,24, 'selHold:'+ Word2Dec(selHold) + ' selHoldScroll:'+ Word2Dec(selHoldScroll));
    OutTextXY(304,32, 'spareCarSelFolder:'+ spareCarSelFolderPath);}
END;

PROCEDURE Mainloop;
VAR
    key:   Word;
    fully: Boolean;
BEGIN
    fully := True;
    quit := False;
    restartRequested := False;
    REPEAT
        RenderScreen(fully);
        fully := False;
        REPEAT
            key := ReadKeyPro;
            IF (key = KEY_ESC) THEN
            BEGIN
                IF (ShowQuestion('Quit',
                        'Do you really want to leave?')) THEN
                BEGIN
                   quit := True;
                END;
                fully := True;
            END ELSE BEGIN
                IF ((selView > 1) AND (selView <= 6) AND
                    (key >= KEY_F1) AND (key <= KEY_F10)) THEN
                BEGIN
                    fully := HandleGenericScreen(key);
                END ELSE BEGIN
                    CASE selView OF
                        1:   fully := HandleGarageScreen(key);
                        2:   fully := HandleEngineScreen(key);
                        3:   fully := HandleGearsScreen(key);
                        4:   fully := HandleMiscScreen(key);
                        5:   fully := HandleTextScreen(key);
                        6:   fully := HandleGfxScreen(key);
                        11:  fully := HandleConfigScreen(key);
                        ELSE Assert(False, 'Invalid view mode');
                    END;
                END;
            END;
        UNTIL (NOT KeyPressedPro);
    UNTIL (quit);
    { Consume all remaining keyboard input. }
    WHILE (KeyPressedPro) DO
        ReadKeyPro;
END;

BEGIN
    restartRequested := False;
    REPEAT
    LoadSettingsFile;
    spareCarOpen := False;
    spareCarSelFolderPath := curSettings.spareCarRelPath;
    FindCarFiles(curSettings.spareCarRelPath);
    FindCarFiles(curSettings.carRelPath);
    selView := 1;
    selItem := 1;
    InitGraphics(curSettings.video);
    MainLoop;
    ExitGraphics;
    UNTIL (restartRequested = False);
END.
