'Engine 3. Started 31 January 2021
'Base completed on 13 February 2021
'Drew some track (partially) on 16 February 2021
'Finished setting up all original Stunts objects on 28 February 2021
'Added the ability to load car shapes on 15 March 2023, for ReplDump fusion
'17 March 2023, I was able to read the coordinates and yaw of the car to properly display
'18 March 2023, fixed rotation errors for mobs
'23 Decemeber 2023, changed horizon model to square island
'9 Febrary 2024, adapted this version to serve as renderer for Pretty Garage

'Engine issues to fix:
'- Replace textured line subs to escape FB's bug
'- Develop a random ellipse rendering routine to produce wheels
'- Resolve track drawing bias when directly on a hill or water
'- Create a car shape sorting mechanism so cars can be occluded
'- Understand and improve near clipping for F1 views
'- Solve general horizon rendering

#include "file.bi"
#include "fbgfx.bi"
#include "targalib.bi"	'Not part of the engine

#ifdef __FB_LINUX__
	#define DIRSEP "/"
#else
	#define DIRSEP "\"
#endif

#define RGBA_R( c ) ( CUInt( c ) Shr 16 And 255 )
#define RGBA_G( c ) ( CUInt( c ) Shr  8 And 255 )
#define RGBA_B( c ) ( CUInt( c )        And 255 )
#define RGBA_A( c ) ( CUInt( c ) Shr 24 

Dim Shared As Short SWIDTH = 480, SHEIGHT = 320, HALFSW, HALFSH
Dim Shared As Byte draw_wheels = -1

HALFSW = SWIDTH \ 2
HALFSH = SHEIGHT \ 2

#define PPM (76800 / 23)	'Pixels per meter
#define MAX_VERTICES 100000
#define MAX_PRIMITIVES 100000
#define MAX_WORKING_VERTICES 15000
#define MAX_WORKING_PRIMITIVES 15000
#define MAX_TRACK_OBJECTS 512
#define MAX_MATERIALS 256
#define MAX_ACTIVE_OBJECTS 1000
#define MAX_MOVING_OBJECTS 30
#define PI 3.1415925


Type Vector3D
	x As Double
	y As Double
	z As Double
End Type

Type Camera
	x As Double
	y As Double
	z As Double
	azm As Double	'Azimuth, Z rotation, first
	alt As Double	'Altitude, X rotation, second
	bank As Double	'Bank, Y rotation, last
End Type

Type TrackObject
	fver As Long		'First vertex in pool
	vers As Short		'Number of vertices
	fprim As Long		'First primitive in pool
	prims As Short		'Number of primitives
	zshift As Double	'Added to the Z to make a good 3D centre for sorting
End Type

Type Primitive
	kind As Byte		'Primitive type (polygons will be decomposed to triangles)
	v(1 To 6) As Long	'Pointers to vertices
	matf As Short		'Front material
	matb As Short		'Back material (-1 for one-sided)
	layer As Byte		'Layer number (lower drawn first)
	shade As Byte		'Shade to use in static mode
	side As Byte		'Temporal to store PFront
End Type

Type Material
	c(-8 To 8) As ULong	'Shades. Base colour is c(0)
	texture As Byte		'Texture type (0 for solid)
End Type

Type ActiveObject
	x As Double			'Centre X, Y and Z
	y As Double
	z As Double
	d As Double			'Distance to camera: used during rendering
	rot As Double		'Rotation in Z
	kind As Short		'Object type
End Type

Type MovingObject
	x As Double
	y As Double
	z As Double
	d As Double
	azm As Double
	alt As Double
	bank As Double
	kind As Short
	id As Short			'Used to identify objects after sorting
End Type

Dim Shared fakescreen As Any Pointer = 0

NameSpace GEngine
	Declare Function PFront(t As Primitive) As Boolean
	
	'Pool of vertices, primitives, objects and materials
	Dim Shared pv(1 To MAX_VERTICES) As Vector3D, pvs As Long
	Dim Shared pp(1 To MAX_PRIMITIVES) As Primitive, pps As Long
	Dim Shared tob(1 To MAX_TRACK_OBJECTS) As TrackObject, tobs As Short
	Dim Shared mat(0 To MAX_MATERIALS) As Material, mats As Short
	Dim Shared aob(1 To MAX_ACTIVE_OBJECTS) As ActiveObject, aobs As Short
	Dim Shared mob(1 To MAX_MOVING_OBJECTS) As MovingObject, mobs As Short

	'Working vertices and primitives (currently working object)
	Dim Shared wv(1 To MAX_WORKING_VERTICES) As Vector3D, wvs As Short
	Dim Shared wp(1 To MAX_WORKING_PRIMITIVES) As Primitive, wps As Short

	'Global parameters
	Dim Shared cam As Camera, visibility As Short = 5000
	Dim Shared As Double SMWIDTH, SMHEIGHT, DTS = .3, CLIP_DISTANCE = .02 '.7
	Dim Shared RotM(1 To 3, 1 To 3) As Double	'Rotation Matrix
	Dim Shared horizonob As Short	'Track object number for horizon box

	'Convert to screen/window coordinates
	Sub Screenise
		For i As Short = 1 To wvs
			wv(i).x = wv(i).x * PPM + SWIDTH / 2
			wv(i).z = SHEIGHT / 2 - wv(i).z * PPM
			Swap wv(i).y, wv(i).z
		Next i
	End Sub
	
	'Load an object from the object pool as the working object
	Sub TakeObject (n As Short)
		Dim i As Short
		
		wvs = tob(n).vers
		For i = tob(n).fver To tob(n).fver + tob(n).vers - 1
			wv(i - tob(n).fver + 1) = pv(i)
		Next i
		
		wps = tob(n).prims
		For i = tob(n).fprim To tob(n).fprim + tob(n).prims - 1
			wp(i - tob(n).fprim + 1) = pp(i)
		Next i
	End Sub
	
	Sub Matrix_to_Identity
		'Generate identity matrix for the rotation matrix
		'(that is, no rotation)
		
		For i As Byte = 1 To 3
			For j As Byte = 1 To 3
				If i = j Then RotM(i, j) = 1 Else RotM(i, j) = 0
			Next j
		Next i
	End Sub
	
	'Move working object
	Sub MoveObject (dx As Double, dy As Double, dz As Double)
		For i As Short = 1 To wvs
			wv(i).x += dx
			wv(i).y += dy
			wv(i).z += dz
		Next i
	End Sub
	
	Sub RotateX (angle As Double, tomatrix As Boolean = False)
		If tomatrix Then
			Dim As Double c, s
			Dim tempM(1 To 3, 1 To 3) As Double
			
			For i As Byte = 1 To 3
				For j As Byte = 1 To 3
					tempM(i, j) = RotM(i, j)
				Next j
			Next i
			
			c = Cos(angle) : s = Sin(angle)
			'RotM(1, 1) = tempM(1, 1)
			'RotM(1, 2) = tempM(1, 2)
			'RotM(1, 3) = tempM(1, 3)
			RotM(2, 1) = tempM(2, 1) * c - tempM(3, 1) * s
			RotM(2, 2) = tempM(2, 2) * c - tempM(3, 2) * s
			RotM(2, 3) = tempM(2, 3) * c - tempM(3, 3) * s
			RotM(3, 1) = tempM(2, 1) * s + tempM(3, 1) * c
			RotM(3, 2) = tempM(2, 2) * s + tempM(3, 2) * c
			RotM(3, 3) = tempM(2, 3) * s + tempM(3, 3) * c
		Else
			Dim v As Vector3D
			
			For i As Short = 1 To wvs
				v = wv(i)
				wv(i).y = v.y * Cos(angle) - v.z * Sin(angle)
				wv(i).z = v.y * Sin(angle) + v.z * Cos(angle)
			Next i
		End If
	End Sub
	
	Sub RotateY (angle As Double, tomatrix As Boolean = False)
		If tomatrix Then
			Dim As Double c, s
			Dim tempM(1 To 3, 1 To 3) As Double
			
			For i As Byte = 1 To 3
				For j As Byte = 1 To 3
					tempM(i, j) = RotM(i, j)
				Next j
			Next i
			
			'I don't know why the angle has to be flipped
			c = Cos(-angle) : s = Sin(-angle)
			RotM(1, 1) = tempM(1, 1) * c - tempM(3, 1) * s
			RotM(1, 2) = tempM(1, 2) * c - tempM(3, 2) * s
			RotM(1, 3) = tempM(1, 3) * c - tempM(3, 3) * s
			'RotM(2, 1) = tempM(2, 1)
			'RotM(2, 2) = tempM(2, 2)
			'RotM(2, 3) = tempM(2, 3)
			RotM(3, 1) = tempM(1, 1) * s + tempM(3, 1) * c
			RotM(3, 2) = tempM(1, 2) * s + tempM(3, 2) * c
			RotM(3, 3) = tempM(1, 3) * s + tempM(3, 3) * c
		Else
			Dim v As Vector3D
			
			For i As Short = 1 To wvs
				v = wv(i)
				wv(i).x = v.x * Cos(angle) + v.z * Sin(angle)
				wv(i).z = -v.x * Sin(angle) + v.z * Cos(angle)
			Next i
		End If
	End Sub

	Sub RotateZ (angle As Double, tomatrix As Boolean = False)
		If tomatrix Then
			Dim As Double c, s
			Dim tempM(1 To 3, 1 To 3) As Double
			
			For i As Byte = 1 To 3
				For j As Byte = 1 To 3
					tempM(i, j) = RotM(i, j)
				Next j
			Next i
			
			'I don't know why the angle has to be flipped
			c = Cos(angle) : s = Sin(angle)
			RotM(1, 1) = tempM(1, 1) * c - tempM(2, 1) * s
			RotM(1, 2) = tempM(1, 2) * c - tempM(2, 2) * s
			RotM(1, 3) = tempM(1, 3) * c - tempM(2, 3) * s
			RotM(2, 1) = tempM(1, 1) * s + tempM(2, 1) * c
			RotM(2, 2) = tempM(1, 2) * s + tempM(2, 2) * c
			RotM(2, 3) = tempM(1, 3) * s + tempM(2, 3) * c
			'RotM(3, 1) = tempM(3, 1)
			'RotM(3, 2) = tempM(3, 2)
			'RotM(3, 3) = tempM(3, 3)
		Else
			Dim v As Vector3D
			
			For i As Short = 1 To wvs
				v = wv(i)
				wv(i).x = v.x * Cos(-angle) + v.y * Sin(-angle)
				wv(i).y = -v.x * Sin(-angle) + v.y * Cos(-angle)
			Next i
		End If
	End Sub
	
	Sub RotateByMatrix
		Dim v As Vector3D
		
		For i As Short = 1 to wvs
			v = wv(i)
			wv(i).x = v.x * RotM(1, 1) + v.y * RotM(1, 2) + v.z * RotM(1, 3)
			wv(i).y = v.x * RotM(2, 1) + v.y * RotM(2, 2) + v.z * RotM(2, 3)
			wv(i).z = v.x * RotM(3, 1) + v.y * RotM(3, 2) + v.z * RotM(3, 3)
		Next i
	End Sub
	
	'Apply perspective to working object's X and Z coordinates over Y
	Sub ApplyPerspective
		For i As Short = 1 To wvs
			If wv(i).y > .05 Then
				wv(i).x = DTS * wv(i).x / wv(i).y
				wv(i).z = DTS * wv(i).z / wv(i).y
			End If
		Next i
	End Sub
	
	Sub drawtriangle(ByVal x1 As Short, ByVal y1 As Short, _
					ByVal x2 As Short, ByVal y2 As Short, _
					ByVal x3 As Short, ByVal y3 As Short, col As ULong)
		
		'STEPS TO FOLLOW:
		'Step 1: Sort point by Y
		'Step 2: Drop all triangles out of Y bounds
		'Step 3: Classify as top, bottom or mixed
		'Step 4: If mixed, split the triangle and call the sub twice and exit
		'Step 5: Obtain initial Y and X1 and X2 for first line, plus deltas
		'Step 6: Apply deltas till Y is onscreen
		'Step 7: Draw lines till triangle is complete or out of screen

		Dim currenty As Long
		Dim As Long deltaxi, deltaxf, currentxi, currentxf
		
		'Step 1: Sort point by Y
		If y1 > y2 Then Swap y1, y2 : Swap x1, x2
		If y2 > y3 Then Swap y2, y3 : Swap x2, x3
		If y1 > y2 Then Swap y1, y2 : Swap x1, x2
		
		'Step 2: Drop all triangles out of Y bounds
		If y3 < 0 Or y1 >= SHEIGHT Then Exit Sub
		
		'Step 3: Classify as top, bottom or mixed
		If y1 = y2 AndAlso y2 = y3 Then
			'Just a horizontal line
			'... not drawing anything so far
		ElseIf y2 = y3 Then			'Top
			'Step 5: Obtain initial Y and X1 and X2 for first line, plus deltas
			If x2 > x3 Then Swap x2, x3 : Swap y2, y3	'Sort by X
			currenty = y1
			currentxi = x1 ShL 8 : currentxf = x1 ShL 8
			deltaxi = ((x2 - x1) ShL 8) \ (y2 - y1)	'Add some precision
			deltaxf = ((x3 - x1) ShL 8) \ (y3 - y1)
		ElseIf y1 = y2 Then		'Bottom
			'Step 5: Obtain initial Y and X1 and X2 for first line, plus deltas
			If x1 > x2 Then Swap x1, x2 : Swap y1, y2	'Sort by X
			currenty = y1
			currentxi = x1 ShL 8 : currentxf = x2 ShL 8
			deltaxi = ((x3 - x1) ShL 8) \ (y3 - y1)	'Add some precision
			deltaxf = ((x3 - x2) ShL 8) \ (y3 - y2)
		Else					'Mixed
			'Step 4: If mixed, split the triangle and call the sub twice and exit
			Dim otherx As Short
			
			otherx = (y2 - y1) * (x3 - x1) \ (y3 - y1) + x1
			drawtriangle x1, y1, x2, y2, otherx, y2, col
			drawtriangle x2, y2, otherx, y2, x3, y3, col
			Exit Sub
		End If
		
		'Step 6: Apply deltas till Y is onscreen
		If currenty < 0 Then
			currentxi += Abs(currenty) * deltaxi
			currentxf += Abs(currenty) * deltaxf
			currenty = 0
		End If
		
		'Step 7: Draw lines till triangle is complete or out of screen
		Dim As Short tempxi, tempxf
		Do
			'Only draw if onscreen
			If currentxf >= 0 And (currentxi ShR 8) < SWIDTH Then
				'Clip the line
				If currentxi >= 0 Then tempxi = currentxi ShR 8 Else tempxi = 0
				If (currentxf ShR 8) < SWIDTH Then tempxf = currentxf ShR 8 Else tempxf = SWIDTH - 1
				'Draw it
				If tempxi <> 0 Or tempxf <> 0 Then _ 'Don't know why this IF is necessary to avoid a horrible vertical line
					Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col
			End If
			
			'If triangle is done or we go offscreen, then exit
			If currenty >= y3 OrElse currenty >= SHEIGHT - 1 Then Exit Do
		
			'Update coordinates
			currentxi += deltaxi
			currentxf += deltaxf
			currenty += 1
		Loop
		
		'Complete triangle sides
		Line fakescreen, (x1, y1)-(x2, y2), col
		Line fakescreen, -(x3, y3), col
		Line fakescreen, -(x1, y1), col
	End Sub
	
	Sub drawtranstriangle(ByVal x1 As Short, ByVal y1 As Short, _
					ByVal x2 As Short, ByVal y2 As Short, _
					ByVal x3 As Short, ByVal y3 As Short, col As ULong, texture As Byte = 1)

		If texture = 7 Then Exit Sub	'Transparent material
		
		'STEPS TO FOLLOW:
		'Step 1: Sort point by Y
		'Step 2: Drop all triangles out of Y bounds
		'Step 3: Classify as top, bottom or mixed
		'Step 4: If mixed, split the triangle and call the sub twice and exit
		'Step 5: Obtain initial Y and X1 and X2 for first line, plus deltas
		'Step 6: Apply deltas till Y is onscreen
		'Step 7: Draw lines till triangle is complete or out of screen

		Dim currenty As Long
		Dim As Long deltaxi, deltaxf, currentxi, currentxf
		
		'Step 1: Sort point by Y
		If y1 > y2 Then Swap y1, y2 : Swap x1, x2
		If y2 > y3 Then Swap y2, y3 : Swap x2, x3
		If y1 > y2 Then Swap y1, y2 : Swap x1, x2
		
		'Step 2: Drop all triangles out of Y bounds
		If y3 < 0 Or y1 >= SHEIGHT Then Exit Sub
		
		'Step 3: Classify as top, bottom or mixed
		If y1 = y2 AndAlso y2 = y3 Then
			'Just a horizontal line
			'... not drawing anything so far
		ElseIf y2 = y3 Then			'Top
			'Step 5: Obtain initial Y and X1 and X2 for first line, plus deltas
			If x2 > x3 Then Swap x2, x3 : Swap y2, y3	'Sort by X
			currenty = y1
			currentxi = x1 ShL 8 : currentxf = x1 ShL 8
			deltaxi = ((x2 - x1) ShL 8) \ (y2 - y1)	'Add some precision
			deltaxf = ((x3 - x1) ShL 8) \ (y3 - y1)
		ElseIf y1 = y2 Then		'Bottom
			'Step 5: Obtain initial Y and X1 and X2 for first line, plus deltas
			If x1 > x2 Then Swap x1, x2 : Swap y1, y2	'Sort by X
			currenty = y1
			currentxi = x1 ShL 8 : currentxf = x2 ShL 8
			deltaxi = ((x3 - x1) ShL 8) \ (y3 - y1)	'Add some precision
			deltaxf = ((x3 - x2) ShL 8) \ (y3 - y2)
		Else					'Mixed
			'Step 4: If mixed, split the triangle and call the sub twice and exit
			Dim otherx As Short
			
			otherx = (y2 - y1) * (x3 - x1) \ (y3 - y1) + x1
			drawtranstriangle x1, y1, x2, y2, otherx, y2, col
			drawtranstriangle x2, y2, otherx, y2, x3, y3, col
			Exit Sub
		End If
		
		'Step 6: Apply deltas till Y is onscreen
		If currenty < 0 Then
			currentxi += Abs(currenty) * deltaxi
			currentxf += Abs(currenty) * deltaxf
			currenty = 0
		End If
		
		'Step 7: Draw lines till triangle is complete or out of screen
		Dim As Short tempxi, tempxf
		Do
			'Only draw if onscreen
			If currentxf >= 0 And (currentxi ShR 8) < SWIDTH Then
				'Clip the line
				If currentxi >= 0 Then tempxi = currentxi ShR 8 Else tempxi = 0
				If (currentxf ShR 8) < SWIDTH Then tempxf = currentxf ShR 8 Else tempxf = SWIDTH - 1
				
				'Draw it
				Select Case texture
					Case 1	'Grate
						If currenty And 1 Then
							If ((currenty ShR 1) XOr tempxi) And 1 Then
								Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &HAAAA
							Else
								Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &H5555
							End If
						Else
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col
						End If
					Case 2	'Inverted grate
						If currenty And 1 Then
							If ((currenty ShR 1) XOr tempxi) And 1 Then
								Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &H5555
							Else
								Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &HAAAA
							End If
						End If
					Case 3	'Grillé
						If currenty And 1 Then
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &H3333
						Else
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &HCCCC
						End If
					Case 4	'Inverted grillé
						If currenty And 1 Then
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &HCCCC
						Else
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &H3333
						End If
					Case 5	'Glass
						If currenty And 1 Then
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &H6B6B
						Else
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &HD6D6
						End If
					Case 6	'Inverted glass
						If currenty And 1 Then
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &HD6D6
						Else
							Line fakescreen, (tempxi, currenty)-(tempxf, currenty), col, , &H6B6B
						End If
				End Select
			End If
			
			'If triangle is done or we go offscreen, then exit
			If currenty >= y3 OrElse currenty >= SHEIGHT - 1 Then Exit Do
		
			'Update coordinates
			currentxi += deltaxi
			currentxf += deltaxf
			currenty += 1
		Loop
	End Sub
	
	Sub KillInvisiblePrimitives
		'This process will fail if applied before clipping
		'against the front plane, because perspective can
		'make triangles flip. Strangely enough, it also
		'fails to draw triangles that are entirely on one
		'side of the plane. I don't know why this is so
		Dim i As Short = 1
		
		Do While i <= wps
			'Store side for later quick access
			wp(i).side = PFront(wp(i))
			
			'If it won't be visible, remove the primitive
			If wp(i).side = 0 AndAlso wp(i).matb = -1 Then
				Swap wp(i), wp(wps)
				i -= 1 : wps -= 1
			End If
			
			i += 1
		Loop
	End Sub
	
	Sub SortPrimitives
		Dim As Short i, j
		Dim As Double zmax(1 To wps), zmin(1 To wps), ddist(1 To wps)
				
		For i = 1 To wps
			Select Case wp(i).kind
				Case 3	'Triangles
					If wv(wp(i).v(1)).z > wv(wp(i).v(2)).z Then
						zmax(i) = wv(wp(i).v(1)).z
						zmin(i) = wv(wp(i).v(2)).z
					Else
						zmin(i) = wv(wp(i).v(1)).z
						zmax(i) = wv(wp(i).v(2)).z
					End If
					If wv(wp(i).v(3)).z > zmax(i) Then
						zmax(i) = wv(wp(i).v(3)).z
					ElseIf wv(wp(i).v(3)).z < zmin(i) Then
						zmin(i) = wv(wp(i).v(3)).z
					End If
				Case 2	'Line segments
					If wv(wp(i).v(1)).z > wv(wp(i).v(2)).z Then
						zmax(i) = wv(wp(i).v(1)).z
						zmin(i) = wv(wp(i).v(2)).z
					Else
						zmin(i) = wv(wp(i).v(1)).z
						zmax(i) = wv(wp(i).v(2)).z
					End If
				Case 1, 11	'Point and sphere
					zmax(i) = wv(wp(i).v(1)).z
					zmin(i) = zmax(i)
				Case 12		'Wheel
					If wv(wp(i).v(1)).z > wv(wp(i).v(4)).z Then
						zmax(i) = wv(wp(i).v(1)).z
						zmin(i) = wv(wp(i).v(4)).z
					Else
						zmin(i) = wv(wp(i).v(1)).z
						zmax(i) = wv(wp(i).v(4)).z
					End If
			End Select	
			
			'Calculate double-distance (faster than distance)
			ddist(i) = Abs(zmin(i) + zmax(i))
		Next i
		
		'Bubble sort
		Dim sorted As Byte
		Do
			sorted = -1
			For i = 1 To wps - 1
				If zmax(i) < zmin(i + 1) Then	'No conflict, but inverted
					Swap wp(i), wp(i + 1)
					Swap zmin(i), zmin(i + 1)
					Swap zmax(i), zmax(i + 1)
					Swap ddist(i), ddist(i + 1)
					sorted = 0
				ElseIf zmax(i + 1) < zmin(i) Then	'No problem
					'-----------
				Else	'Conflict
					If wp(i).layer = wp(i + 1).layer Then	'Sort by distance
						If ddist(i) < ddist(i + 1) Then
							Swap wp(i), wp(i + 1)
							Swap zmin(i), zmin(i + 1)
							Swap zmax(i), zmax(i + 1)
							Swap ddist(i), ddist(i + 1)
							sorted = 0
						End If
					ElseIf wp(i).layer > wp(i + 1).layer Then	'Sort by layer
						Swap wp(i), wp(i + 1)
						Swap zmin(i), zmin(i + 1)
						Swap zmax(i), zmax(i + 1)
						Swap ddist(i), ddist(i + 1)
						sorted = 0
					End If
				End If
			Next i
		Loop Until sorted
	End Sub
	
	'See if the primitive is facing front
	Function PFront(t As Primitive) As Boolean
		Dim As Double r, r2
		
		'Polygons are decomposed to triangles. Other
		'primitives don't have "sides", so they always
		'test positive
		If t.kind <> 3 Then Return True
		
		r = wv(t.v(1)).x * wv(t.v(2)).y
		r += wv(t.v(2)).x * wv(t.v(3)).y
		r += wv(t.v(3)).x * wv(t.v(1)).y
		
		r2 = wv(t.v(2)).x * wv(t.v(1)).y
		r2 += wv(t.v(3)).x * wv(t.v(2)).y
		r2 += wv(t.v(1)).x * wv(t.v(3)).y
		
		Return r > r2
	End Function
	
	Sub DrawObject
		For i As Short = 1 To wps
			Select Case wp(i).kind
				Case 3	'Triangle
					Dim col As ULong
					If wp(i).side Then
						col = mat(wp(i).matf).c(wp(i).shade)
						If mat(wp(i).matf).texture Then
							drawtranstriangle wv(wp(i).v(1)).x, wv(wp(i).v(1)).y, _
								wv(wp(i).v(2)).x, wv(wp(i).v(2)).y, _
								wv(wp(i).v(3)).x, wv(wp(i).v(3)).y, col, mat(wp(i).matf).texture
						Else
							drawtriangle wv(wp(i).v(1)).x, wv(wp(i).v(1)).y, _
								wv(wp(i).v(2)).x, wv(wp(i).v(2)).y, _
								wv(wp(i).v(3)).x, wv(wp(i).v(3)).y, col
						End If
					ElseIf wp(i).matb > 0 Then
						col = mat(wp(i).matb).c(wp(i).shade)
						If mat(wp(i).matf).texture Then
							drawtranstriangle wv(wp(i).v(1)).x, wv(wp(i).v(1)).y, _
								wv(wp(i).v(2)).x, wv(wp(i).v(2)).y, _
								wv(wp(i).v(3)).x, wv(wp(i).v(3)).y, col, mat(wp(i).matf).texture

						Else
							drawtriangle wv(wp(i).v(1)).x, wv(wp(i).v(1)).y, _
								wv(wp(i).v(2)).x, wv(wp(i).v(2)).y, _
								wv(wp(i).v(3)).x, wv(wp(i).v(3)).y, col
						End If
					End If
				Case 2	'Segment
					Line fakescreen, (wv(wp(i).v(1)).x, wv(wp(i).v(1)).y)-_
						(wv(wp(i).v(2)).x, wv(wp(i).v(2)).y), mat(wp(i).matf).c(0)
				Case 11	'Sphere
					Dim r As Double
					
					r = Sqr((wv(wp(i).v(1)).x - wv(wp(i).v(2)).x) ^ 2 + _
							(wv(wp(i).v(1)).y - wv(wp(i).v(2)).y) ^ 2 + _
							(wv(wp(i).v(1)).z - wv(wp(i).v(2)).z) ^ 2)
							
					Circle fakescreen, (wv(wp(i).v(1)).x, wv(wp(i).v(1)).y), r, mat(wp(i).matf).c(0), , , , F
				Case 12	'Wheel
					'To be developed
					Dim r As Double
					
					If draw_wheels Then
						r = Sqr((wv(wp(i).v(1)).x - wv(wp(i).v(2)).x) ^ 2 + _
								(wv(wp(i).v(1)).y - wv(wp(i).v(2)).y) ^ 2 + _
								(wv(wp(i).v(1)).z - wv(wp(i).v(2)).z) ^ 2)
								
						Circle fakescreen, (wv(wp(i).v(1)).x, wv(wp(i).v(1)).y), r, mat(wp(i).matf).c(0), , , , F
						r = .67 * r
						Circle fakescreen, (wv(wp(i).v(1)).x, wv(wp(i).v(1)).y), r, mat(wp(i).matf + 2).c(0), , , , F
					End If
				Case 1	'Point
					PSet fakescreen, (wv(wp(i).v(1)).x, wv(wp(i).v(1)).y), mat(wp(i).matf).c(0)
			End Select
		Next i
	End Sub
	
	'Clip primitives against the near plane in game space
	Sub ClipNear
		Dim As Short i, n
		Dim isok(1 To 3) As Byte, areok As Byte
		
		n = 1
		Do While wps >= n
			If wp(n).kind = 3 Then	'If it's a triangle...
				areok = 0
				For i = 1 To 3
					isok(i) = (wv(wp(n).v(i)).y >= CLIP_DISTANCE)
					If isok(i) Then areok += 1
				Next i
				
				Select Case areok
					Case 3
						'Good boy!  Keep this triangle as is. Next triangle
						n += 1
					Case 0
						'Don't draw this triangle. It's behind the fustrum
						Swap wp(n), wp(wps)
						wps -= 1
					Case 1
						'Only one good vertex. Transform the other two
						Dim temp As Short
						
						'First, reorder so that point 1 is the good one
						If isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						Dim a As Double
						
						'Create the two new points
						a = (CLIP_DISTANCE - wv(wp(n).v(1)).y) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wvs += 1
						wv(wvs).x = a * (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x) + wv(wp(n).v(1)).x
						wv(wvs).y = CLIP_DISTANCE
						wv(wvs).z = a * (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) + wv(wp(n).v(1)).z
						
						a = (CLIP_DISTANCE - wv(wp(n).v(1)).y) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wvs += 1
						wv(wvs).x = a * (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x) + wv(wp(n).v(1)).x
						wv(wvs).y = CLIP_DISTANCE
						wv(wvs).z = a * (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) + wv(wp(n).v(1)).z
						
						'Replace points 2 and 3
						wp(n).v(2) = wvs - 1
						wp(n).v(3) = wvs
						
						'Debugging marker
						'wt(n).col = RGB(255, 255, 0)
						
						n += 1
					Case 2
						'Two vertices are OK. We need to split the triangle
						Dim temp As Short
						
						'First, reorder so that point 1 is the bad one
						If Not isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf Not isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						Dim a As Double
						
						'Create the two new points
						a = (CLIP_DISTANCE - wv(wp(n).v(1)).y) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wvs += 1
						wv(wvs).x = a * (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x) + wv(wp(n).v(1)).x
						wv(wvs).y = CLIP_DISTANCE
						wv(wvs).z = a * (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) + wv(wp(n).v(1)).z
						
						a = (CLIP_DISTANCE - wv(wp(n).v(1)).y) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wvs += 1
						wv(wvs).x = a * (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x) + wv(wp(n).v(1)).x
						wv(wvs).y = CLIP_DISTANCE
						wv(wvs).z = a * (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) + wv(wp(n).v(1)).z
						
						'Update point 1 of current triangle
						wp(n).v(1) = wvs - 1
						
						'Create new triangle and assign the points
						wps += 1
						wp(wps) = wp(n)	'Copy all other properties (inc. colour)
						wp(wps).v(1) = wvs		'But change the points
						wp(wps).v(2) = wvs - 1
						wp(wps).v(3) = wp(n).v(3)
										
						'Debugging markers
						'wt(n).col = RGB(255, 128, 0)
						'wt(wts).col = RGB(250, 0, 250)
						
						'Skip the newly created triangle
						Swap wp(n + 1), wp(wps)
						n += 2
				End Select
			ElseIf wp(n).kind = 2 Then	'If it's a line segment...
				areok = 0
				For i = 1 To 2
					isok(i) = (wv(wp(n).v(i)).y >= CLIP_DISTANCE)
					If isok(i) Then areok += 1
				Next i
				
				Select Case areok
					Case 2
						'Good boy!  Keep this segment as is
						n += 1
					Case 0, 1
						'Don't draw this segment. It's behind the fustrum
						'NOTE: For the time being, we're not drawing partial segments either!!
						Swap wp(n), wp(wps)
						wps -= 1
				End Select
			Else
				n += 1	'OK, draw it. Whatever it is
			End If
		Loop
	End Sub
	
	'Clip primitives against screen borders in screen space
	Sub ClipSides
		Dim As Short i, n
		Dim isok(1 To 3) As Byte, areok As Byte
		
		'Clip against left side ==================================
		n = 1
		Do While wps >= n
			If wp(n).kind = 3 Then	'If it's a triangle
				areok = 0
				For i = 1 To 3
					isok(i) = (wv(wp(n).v(i)).x >= 0)
					If isok(i) Then areok += 1
				Next i
				
				Select Case areok
					Case 3
						'Good boy!  Keep this triangle as is. Next triangle
						n += 1
					Case 0
						'Don't draw this triangle. It's outside the screen
						Swap wp(n), wp(wps)
						wps -= 1
					Case 1
						'Only one good vertex. Transform the other two
						Dim temp As Short
						
						'First, reorder so that point 1 is the good one
						If isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = 0
						wv(wvs).y = (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= -wv(wp(n).v(1)).x
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= -wv(wp(n).v(1)).x
						wv(wvs).z += wv(wp(n).v(1)).z
						
						wvs += 1
						wv(wvs).x = 0
						wv(wvs).y = (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= -wv(wp(n).v(1)).x
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= -wv(wp(n).v(1)).x
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Replace points 2 and 3
						wp(n).v(2) = wvs - 1
						wp(n).v(3) = wvs
						
						'Debugging marker
						'wt(n).col = RGB(255, 255, 0)
						
						n += 1
					Case 2
						'Two vertices are OK. We need to split the triangle
						Dim temp As Short
						
						'First, reorder so that point 1 is the bad one
						If Not isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf Not isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = 0
						wv(wvs).y = (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= -wv(wp(n).v(1)).x
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= -wv(wp(n).v(1)).x
						wv(wvs).z += wv(wp(n).v(1)).z
						
						wvs += 1
						wv(wvs).x = 0
						wv(wvs).y = (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= -wv(wp(n).v(1)).x
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= -wv(wp(n).v(1)).x
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Update point 1 of current triangle
						wp(n).v(1) = wvs - 1
						
						'Create new triangle and assign the points
						wps += 1
						wp(wps) = wp(n)	'Copy all other properties (inc. colour)
						wp(wps).v(1) = wvs		'But change the points
						wp(wps).v(2) = wvs - 1
						wp(wps).v(3) = wp(n).v(3)
										
						'Debugging markers
						'wt(n).col = RGB(255, 128, 0)
						'wt(wts).col = RGB(250, 0, 250)
						
						'Skip the newly created triangle
						Swap wp(n + 1), wp(wps)
						n += 2
				End Select
			Else
				n += 1	'Just draw it
			End If
		Loop
		
		'Clip against right side ==================================
		n = 1
		Do While wps >= n
			If wp(n).kind = 3 Then	'If it's a triangle
				areok = 0
				For i = 1 To 3
					isok(i) = (wv(wp(n).v(i)).x <= SWIDTH - 1)
					If isok(i) Then areok += 1
				Next i
				
				Select Case areok
					Case 3
						'Good boy!  Keep this triangle as is. Next triangle
						n += 1
					Case 0
						'Don't draw this triangle. It's outside the screen
						Swap wp(n), wp(wps)
						wps -= 1
					Case 1
						'Only one good vertex. Transform the other two
						Dim temp As Short
						
						'First, reorder so that point 1 is the good one
						If isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = SWIDTH - 1
						wv(wvs).y = (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).z += wv(wp(n).v(1)).z
						
						wvs += 1
						wv(wvs).x = SWIDTH - 1
						wv(wvs).y = (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Replace points 2 and 3
						wp(n).v(2) = wvs - 1
						wp(n).v(3) = wvs
						
						'Debugging marker
						'wt(n).col = RGB(255, 255, 0)
						
						n += 1
					Case 2
						'Two vertices are OK. We need to split the triangle
						Dim temp As Short
						
						'First, reorder so that point 1 is the bad one
						If Not isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf Not isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = SWIDTH - 1
						wv(wvs).y = (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).z += wv(wp(n).v(1)).z
						
						wvs += 1
						wv(wvs).x = SWIDTH - 1
						wv(wvs).y = (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).y *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).y += wv(wp(n).v(1)).y
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x)
						wv(wvs).z *= (SWIDTH - 1 - wv(wp(n).v(1)).x)
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Update point 1 of current triangle
						wp(n).v(1) = wvs - 1
						
						'Create new triangle and assign the points
						wps += 1
						wp(wps) = wp(n)	'Copy all other properties (inc. colour)
						wp(wps).v(1) = wvs		'But change the points
						wp(wps).v(2) = wvs - 1
						wp(wps).v(3) = wp(n).v(3)
										
						'Debugging markers
						'wt(n).col = RGB(255, 128, 0)
						'wt(wts).col = RGB(250, 0, 250)
						
						'Skip the newly created triangle
						Swap wp(n + 1), wp(wps)
						n += 2
				End Select
			Else
				n += 1	'Just draw it
			End If
		Loop

		'Clip against top border ==================================
		n = 1
		Do While wps >= n
			If wp(n).kind = 3 Then	'If it's a triangle
				areok = 0
				For i = 1 To 3
					isok(i) = (wv(wp(n).v(i)).y >= 0)
					If isok(i) Then areok += 1
				Next i
				
				Select Case areok
					Case 3
						'Good boy!  Keep this triangle as is. Next triangle
						n += 1
					Case 0
						'Don't draw this triangle. It's outside the screen
						Swap wp(n), wp(wps)
						wps -= 1
					Case 1
						'Only one good vertex. Transform the other two
						Dim temp As Short
						
						'First, reorder so that point 1 is the good one
						If isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= -wv(wp(n).v(1)).y
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = 0
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
														
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= -wv(wp(n).v(1)).y
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = 0
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Replace points 2 and 3
						wp(n).v(2) = wvs - 1
						wp(n).v(3) = wvs
						
						'Debugging marker
						'wt(n).col = RGB(255, 255, 0)
						
						n += 1
					Case 2
						'Two vertices are OK. We need to split the triangle
						Dim temp As Short
						
						'First, reorder so that point 1 is the bad one
						If Not isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf Not isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= -wv(wp(n).v(1)).y
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = 0
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
														
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= -wv(wp(n).v(1)).y
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = 0
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Update point 1 of current triangle
						wp(n).v(1) = wvs - 1
						
						'Create new triangle and assign the points
						wps += 1
						wp(wps) = wp(n)	'Copy all other properties (inc. colour)
						wp(wps).v(1) = wvs		'But change the points
						wp(wps).v(2) = wvs - 1
						wp(wps).v(3) = wp(n).v(3)
										
						'Debugging markers
						'wt(n).col = RGB(255, 128, 0)
						'wt(wts).col = RGB(250, 0, 250)
						
						'Skip the newly created triangle
						Swap wp(n + 1), wp(wps)
						n += 2
				End Select
			Else
				n += 1	'Just draw it
			End If
		Loop

		'Clip against bottom border ==================================
		n = 1
		Do While wps >= n
			If wp(n).kind = 3 Then	'If it's a triangle
				areok = 0
				For i = 1 To 3
					isok(i) = (wv(wp(n).v(i)).y <= SHEIGHT - 1)
					If isok(i) Then areok += 1
				Next i
				
				Select Case areok
					Case 3
						'Good boy!  Keep this triangle as is. Next triangle
						n += 1
					Case 0
						'Don't draw this triangle. It's outside the screen
						Swap wp(n), wp(wps)
						wps -= 1
					Case 1
						'Only one good vertex. Transform the other two
						Dim temp As Short
						
						'First, reorder so that point 1 is the good one
						If isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= (SHEIGHT - 1 - wv(wp(n).v(1)).y)
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = SHEIGHT - 1
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
														
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= (SHEIGHT - 1 - wv(wp(n).v(1)).y)
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = SHEIGHT - 1
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Replace points 2 and 3
						wp(n).v(2) = wvs - 1
						wp(n).v(3) = wvs
						
						'Debugging marker
						'wt(n).col = RGB(255, 255, 0)
						
						n += 1
					Case 2
						'Two vertices are OK. We need to split the triangle
						Dim temp As Short
						
						'First, reorder so that point 1 is the bad one
						If Not isok(2) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(2)
							wp(n).v(2) = wp(n).v(3)
							wp(n).v(3) = temp
						ElseIf Not isok(3) Then
							temp = wp(n).v(1)
							wp(n).v(1) = wp(n).v(3)
							wp(n).v(3) = wp(n).v(2)
							wp(n).v(2) = temp
						End If
						
						'Create the two new points
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(2)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= (SHEIGHT - 1 - wv(wp(n).v(1)).y)
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = SHEIGHT - 1
						wv(wvs).z = (wv(wp(n).v(2)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(2)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
														
						wvs += 1
						wv(wvs).x = (wv(wp(n).v(3)).x - wv(wp(n).v(1)).x) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).x *= (SHEIGHT - 1 - wv(wp(n).v(1)).y)
						wv(wvs).x += wv(wp(n).v(1)).x
						wv(wvs).y = SHEIGHT - 1
						wv(wvs).z = (wv(wp(n).v(3)).z - wv(wp(n).v(1)).z) / (wv(wp(n).v(3)).y - wv(wp(n).v(1)).y)
						wv(wvs).z *= -wv(wp(n).v(1)).y
						wv(wvs).z += wv(wp(n).v(1)).z
						
						'Update point 1 of current triangle
						wp(n).v(1) = wvs - 1
						
						'Create new triangle and assign the points
						wps += 1
						wp(wps) = wp(n)	'Copy all other properties (inc. colour)
						wp(wps).v(1) = wvs		'But change the points
						wp(wps).v(2) = wvs - 1
						wp(wps).v(3) = wp(n).v(3)
										
						'Debugging markers
						'wt(n).col = RGB(255, 128, 0)
						'wt(wts).col = RGB(250, 0, 250)
						
						'Skip the newly created triangle
						Swap wp(n + 1), wp(wps)
						n += 2
				End Select
			Else
				n += 1	'Just draw it
			End If
		Loop
	End Sub
	
	Sub SortActiveObjects
		'Calculate taxicab distances
		For i As Short = 1 To aobs
			aob(i).d = Abs(aob(i).x - cam.x) + Abs(aob(i).y - cam.y) _
				+ Abs(aob(i).z - cam.z + tob(aob(i).kind).zshift)
		Next i
		
		'Sort by selection
		For i As Short = 1 To aobs
			Dim max As Double, who As Short
			
			max = 0 : who = i
			For j As Short = i To aobs
				If aob(j).d > max Then
					max = aob(j).d
					who = j
				End If
			Next j
			
			If who <> i Then Swap aob(i), aob(who)
		Next i
		
		'Calculate taxicab distances (for moving objects)
		For i As Short = 1 To mobs
			mob(i).d = Abs(mob(i).x - cam.x) + Abs(mob(i).y - cam.y) _
				+ Abs(mob(i).z - cam.z + tob(mob(i).kind).zshift)
		Next i
		
		'Sort by selection (for moving objects)
		For i As Short = 1 To mobs
			Dim max As Double, who As Short
			
			max = 0 : who = i
			For j As Short = i To mobs
				If aob(j).d > max Then
					max = mob(j).d
					who = j
				End If
			Next j
			
			If who <> i Then Swap mob(i), mob(who)
		Next i
	End Sub

	Sub CreateHorizonBox
		'Set danielwalls to 0 for a square-island horizon or
		'set it to non-zero for inifinite horizon using Daniël walls
		Dim danielwalls As Byte = -1
				
		'Create square island
		tobs += 1
		tob(tobs).vers = 4
		tob(tobs).fver = pvs + 1
		tob(tobs).prims = 2
		tob(tobs).fprim = pps + 1
		tob(tobs).zshift = 0
		horizonob = tobs
		
		pvs += 1 : pv(pvs).x = -1 : pv(pvs).y = -1 : pv(pvs).z = 0
		pvs += 1 : pv(pvs).x = 59 : pv(pvs).y = -1 : pv(pvs).z = 0
		pvs += 1 : pv(pvs).x = 59 : pv(pvs).y = 59 : pv(pvs).z = 0
		pvs += 1 : pv(pvs).x = -1 : pv(pvs).y = 59 : pv(pvs).z = 0
		
		pps += 1
		pp(pps).kind = 3
		pp(pps).v(1) = 1 : pp(pps).v(2) = 2 : pp(pps).v(3) = 3
		
		pps += 1
		pp(pps).kind = 3
		pp(pps).v(1) = 3 : pp(pps).v(2) = 4 : pp(pps).v(3) = 1
		
		If danielwalls Then
			'Create Daniël walls
			tob(horizonob).vers += 4
			tob(horizonob).prims += 8
			
			pvs += 1 : pv(pvs).x = -1 : pv(pvs).y = -1 : pv(pvs).z = 1
			pvs += 1 : pv(pvs).x = 59 : pv(pvs).y = -1 : pv(pvs).z = 1
			pvs += 1 : pv(pvs).x = 59 : pv(pvs).y = 59 : pv(pvs).z = 1
			pvs += 1 : pv(pvs).x = -1 : pv(pvs).y = 59 : pv(pvs).z = 1
			
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 2 : pp(pps).v(2) = 6 : pp(pps).v(3) = 3
			
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 6 : pp(pps).v(2) = 7 : pp(pps).v(3) = 3
			
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 3 : pp(pps).v(2) = 7 : pp(pps).v(3) = 4
					
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 7 : pp(pps).v(2) = 8 : pp(pps).v(3) = 4
					
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 4 : pp(pps).v(2) = 8 : pp(pps).v(3) = 1
					
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 8 : pp(pps).v(2) = 5 : pp(pps).v(3) = 1
					
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 1 : pp(pps).v(2) = 2 : pp(pps).v(3) = 6
					
			pps += 1 : pp(pps).kind = 3
			pp(pps).v(1) = 6 : pp(pps).v(2) = 5 : pp(pps).v(3) = 1
		End If
				
		For i As Short = tob(horizonob).fprim To tob(horizonob).fprim + tob(horizonob).prims - 1
			pp(i).matf = 16: pp(i).matb = 16
			pp(i).layer = 0 : pp(i).shade = 0 : pp(i).side = 0
		Next i
	End Sub

	Sub DrawHorizon
		Line fakescreen, (0, 0)-(SWIDTH - 1, SHEIGHT - 1), &H00FF00FF, BF
		Exit Sub
		If horizonob = 0 Then
			Line (0, 0)-(SWIDTH - 1, HALFSH - 1), &HFF5CFCFC, BF
			Line (0, HALFSH)-(SWIDTH - 1, SHEIGHT - 1), &HFF246820, BF
		Else
			Line (0, 0)-(SWIDTH - 1, SHEIGHT - 1), &HFF5CFCFC, BF
			
			If tob(horizonob).vers = 8 Then	'There are Daniël walls
				For i As Short = tob(horizonob).fver + 4 To tob(horizonob).fver + 7
					pv(i).z = cam.z
				Next i
			End If
			
			TakeObject horizonob
			MoveObject -cam.x, -cam.y, -cam.z
			RotateByMatrix
					
			ClipNear
			ApplyPerspective
			Screenise
			KillInvisiblePrimitives
			ClipSides
			SortPrimitives
			DrawObject
		End If
	End Sub

	Sub Render
		SortActiveObjects
				
		ScreenLock
		'Set up matrix for current world orientation
		Matrix_to_Identity
		RotateZ -cam.azm, True
		RotateX -cam.alt, True
		RotateY -cam.bank, True
		
		DrawHorizon
		
		'Draw active (static) objects
		For i As Short = 1 To aobs
			If aob(i).d <= visibility Then
				TakeObject aob(i).kind
				RotateZ aob(i).rot
				MoveObject aob(i).x - cam.x, aob(i).y - cam.y, aob(i).z - cam.z
				
				'These three can be packed in one matrix multiplication
				'RotateZ -cam.azm
				'RotateX -cam.alt
				'RotateY -cam.bank
				RotateByMatrix
				
				ClipNear
				ApplyPerspective
				Screenise
				KillInvisiblePrimitives
				ClipSides
				SortPrimitives
				DrawObject
			End If
		Next i
		
		'Draw moving objects
		For i As Short = 1 To mobs
			If mob(i).d <= visibility Then
				TakeObject mob(i).kind
				RotateY mob(i).bank
				RotateX mob(i).alt
				RotateZ mob(i).azm
								
				MoveObject mob(i).x - cam.x, mob(i).y - cam.y, mob(i).z - cam.z
				
				RotateByMatrix
				
				ClipNear
				ApplyPerspective
				Screenise
				KillInvisiblePrimitives
				ClipSides
				SortPrimitives
				DrawObject
			End If
		Next i
		ScreenUnlock
	End Sub
	
	Sub Load3SH(filename As String, paintjob As Byte = 1)
		Dim f As Integer, i As Short, j As Short, k As Short
		Dim ofs(1 To 100) As Long, id(1 To 100) As String
		Dim objs As Short, start As Long, length As Long
		
		If Not FileExists(filename) Then Exit Sub
		f = FreeFile
		Open filename For Binary Access Read As f
		Get #f, 5, objs
		For i = 1 To objs
			id(i) = Space(4)
			Get #f, , id(i)
		Next i
		For i = 1 To objs
			Get #f, , ofs(i)
		Next i
		
		For i = 1 To objs
			start = ofs(i) + 7 + 8 * objs
			'Find next object offset to calculate length
			length = 2000000000	'A number large enough
			For j = 1 To objs
				If ofs(j) > ofs(i) AndAlso ofs(j) < length Then
					length = ofs(j)
				End If
			Next j
			If length = 2000000000 Then
				length = LoF(f) + 1 - start
			Else
				length = ofs(j) - ofs(i)	'<-- BUG!!!!
			End If
			
			Seek #f, start
			Dim As UByte vs, ps, pjs, thispj
			Dim c As Short, s As String
			
			Get #f, , vs
			Get #f, , ps
			Get #f, , pjs
			thispj = IIf(paintjob > pjs, 1, paintjob)
			Seek #f, Seek(f) + 1
			
			tobs += 1
			tob(tobs).vers = vs
			tob(tobs).prims = ps
			tob(tobs).fver = pvs + 1
			tob(tobs).fprim = pps + 1
			
			'Check!  First 8 vertices are bounding box!!
			'Right now reading all vertices, but the first
			'eight should probably be skipped and perhaps
			'used to calculate the centre of the object
			Dim As Double maxz, minz
			
			maxz = 0 : minz = 10
			For j = 1 To vs
				pvs += 1
				Get #f, , c
				pv(pvs).x = c / 512
				Get #f, , c
				pv(pvs).z = c / 512
				If pv(pvs).z > maxz Then maxz = pv(pvs).z
				If pv(pvs).z < minz Then minz = pv(pvs).z
				Get #f, , c
				pv(pvs).y = c / 512
			Next j
			tob(tobs).zshift = (maxz + minz) / 2
			
			Seek #f, Seek(f) + 8 * ps	'Skip culling data
			For j = 1 To ps
				pps += 1
				Get #f, , pp(pps).kind
				Get #f, , pp(pps).layer
								
				s = Space(pjs)
				Get #f, , s
				pp(pps).matf = ASC(Mid(s, thispj, 1))
				If pp(pps).layer And 1 Then
					pp(pps).matb = ASC(Mid(s, thispj, 1))
				Else
					pp(pps).matb = -1
				End If
				pp(pps).layer ShR= 1
							
				'Calculate number of vertices for this
				'primitive and store it in c
				Select Case pp(pps).kind
					Case Is > 12 : c = 0
					Case 11 : c = 2
					Case 12 : c = 6
					Case Else : c = pp(pps).kind
				End Select
				
				If pp(pps).kind > 3 AndAlso pp(pps).kind <= 10 Then
					'Non-triangular polygons
					Dim As UByte n1, n2, n3
					Get #f, , n1
					Get #f, , n2
					For k = 1 To c - 2
						Get #f, , n3
						pp(pps).v(1) = n1 + 1
						pp(pps).v(2) = n2 + 1
						pp(pps).v(3) = n3 + 1
						pp(pps).kind = 3
						n2 = n3
						If k < c - 2 Then
							pps += 1
							pp(pps).matf = pp(pps - 1).matf
							pp(pps).matb = pp(pps - 1).matb
							pp(pps).layer = pp(pps - 1).layer
						End If
					Next k
					tob(tobs).prims += c - 3
				Else
					For k = 1 To c
						Dim n As UByte
						Get #f, , n
						pp(pps).v(k) = n + 1
					Next k
				End If
			Next j
		Next i
		Close f
	End Sub
	
	Sub LoadCarShape(filename As String, paintjob As Byte = 1)
		Dim f As Integer, i As Short, j As Short, k As Short
		Dim ofs As Long, id As String
		Dim objs As Short, start As Long
		Dim wheel(1 To 4) As Short, wheels As Byte = 0
		
		If Not FileExists(filename) Then Exit Sub
		f = FreeFile
		Open filename For Binary Access Read As f
		Get #f, 5, objs
		id = Space(4)
		For i = 1 To objs
			Get #f, , id
			If id = "car1" Then Exit For
		Next i
		Get #f, 7 + 4 * (objs + i - 1), ofs
				
		start = ofs + 7 + 8 * objs
		Seek #f, start
		
		Dim As UByte vs, ps, pjs, thispj
		Dim c As Short, s As String
		
		Get #f, , vs
		Get #f, , ps
		Get #f, , pjs
		thispj = IIf(paintjob > pjs, 1, paintjob)
		Seek #f, Seek(f) + 1
			
		tobs += 1
		tob(tobs).vers = vs
		tob(tobs).prims = ps
		tob(tobs).fver = pvs + 1
		tob(tobs).fprim = pps + 1
			
		'Check!  First 8 vertices are bounding box!!
		'Right now reading all vertices, but the first
		'eight should probably be skipped and perhaps
		'used to calculate the centre of the object
		Dim As Double maxz, minz
			
		maxz = 0 : minz = 10
		For j = 1 To vs
			pvs += 1
			Get #f, , c
			pv(pvs).x = c / 512
			Get #f, , c
			pv(pvs).z = c / 512
			If pv(pvs).z > maxz Then maxz = pv(pvs).z
			If pv(pvs).z < minz Then minz = pv(pvs).z
			Get #f, , c
			pv(pvs).y = c / 512
		Next j
		tob(tobs).zshift = (maxz + minz) / 2
			
		Seek #f, Seek(f) + 8 * ps	'Skip culling data
		For j = 1 To ps
			pps += 1
			Get #f, , pp(pps).kind
			Get #f, , pp(pps).layer
							
			s = Space(pjs)
			Get #f, , s
			pp(pps).matf = ASC(Mid(s, thispj, 1))
			If pp(pps).layer And 1 Then
				pp(pps).matb = ASC(Mid(s, thispj, 1))
			Else
				pp(pps).matb = -1
			End If
			pp(pps).layer ShR= 1
						
			'Calculate number of vertices for this
			'primitive and store it in c
			Select Case pp(pps).kind
				Case Is > 12 : c = 0
				Case 11 : c = 2
				Case 12
					c = 6
					If wheels < 4 Then
						wheels += 1			'Add a wheel
						wheel(wheels) = pps
					End If
				Case Else : c = pp(pps).kind
			End Select
			
			If pp(pps).kind > 3 AndAlso pp(pps).kind <= 10 Then
				'Non-triangular polygons
				Dim As UByte n1, n2, n3
				Get #f, , n1
				Get #f, , n2
				For k = 1 To c - 2
					Get #f, , n3
					pp(pps).v(1) = n1 + 1
					pp(pps).v(2) = n2 + 1
					pp(pps).v(3) = n3 + 1
					pp(pps).kind = 3
					n2 = n3
					If k < c - 2 Then
						pps += 1
						pp(pps).matf = pp(pps - 1).matf
						pp(pps).matb = pp(pps - 1).matb
						pp(pps).layer = pp(pps - 1).layer
					End If
				Next k
				tob(tobs).prims += c - 3
			Else
				For k = 1 To c
					Dim n As UByte
					Get #f, , n
					pp(pps).v(k) = n + 1
				Next k
			End If
		Next j
		
		'~ Dim curver As Long, curprim As Long
		'~ curver = tob(tobs).fver + tob(tobs).vers
		'~ curprim = tob(tobs).fprim + tob(tobs).prims
		'~ For i = 1 To wheels
			'~ Dim As Double xsep, ysep, r
			
			'~ xsep = pv(pp(wheel(i)).v(1) + tob(tobs).fver - 1).x
			'~ ysep = pv(pp(wheel(i)).v(1) + tob(tobs).fver - 1).y
			'~ r = Abs(pv(pp(wheel(i)).v(1) + tob(tobs).fver - 1).z)
			'~ pv(curver).x = xsep : pv(curver).y = ysep : pv(curver).z = r
			'~ curver += 1
			'~ pv(curver).x = xsep : pv(curver).y = ysep + r : pv(curver).z = r
			'~ curver += 1
			'~ pv(curver).x = xsep : pv(curver).y = ysep : pv(curver).z = 2 * r
			'~ tob(tobs).vers += 3 : pvs += 3
			'~ pp(curprim).kind = 3
			'~ pp(curprim).v(1) = tob(tobs).vers - 2
			'~ pp(curprim).v(2) = tob(tobs).vers - 1
			'~ pp(curprim).v(3) = tob(tobs).vers
			'~ pp(curprim).matf = 9
			'~ pp(curprim).matb = 21
		'~ Next i
		Close f
	End Sub
	
	Sub LoadMaterials
		Dim f As Integer, i As Short, s As String
		Dim j As Byte
		
		f = FreeFile
		Open "defmat.txt" For Input As f
		For i = 0 To 128
			Line Input #f, s
			For j = -8 To 8
				mat(i).c(j) = ValULng("&H" & s)
				mat(i).c(j) Or= &HFF000000&
			Next j
			Select Case Trim(LCase(Mid(s, 7)))
				Case "" : mat(i).texture = 0
				Case "grate" : mat(i).texture = 1
				Case "igrate" : mat(i).texture = 2	'This one does not exist in original Stunts
				Case "grill" : mat(i).texture = 3
				Case "igrill" : mat(i).texture = 4
				Case "glass" : mat(i).texture = 5
				Case "iglass" : mat(i).texture = 6
				Case "trans" : mat(i).texture = 7
				Case Else : mat(i).texture = 0
			End Select
		Next i
		Close f
		mats = 128
	End Sub
	
	Sub LoadTrack (trackfile As String)
		Dim f As Integer
		Dim t(1 To 30, 1 to 30) As UByte
		Dim l(1 To 30, 1 To 30) As UByte
		
		f = FreeFile
		Open trackfile For Binary Access Read As f
		For j As Byte = 30 To 1 Step -1
			For i As Byte = 1 To 30
				Get #1, , t(i, j)
			Next i
		Next j
		Seek #f, Seek(f) + 1
		For j As Byte = 1 To 30
			For i As Byte = 1 To 30
				Get #1, , l(i, j)
			Next i
		Next j
		Close f
		
		Dim s As String, cc(0 To 255) As String
		Dim n As Short
		Open "stxlat.cfg" For Input As f
		While Not EoF(f)
			Line Input #f, s
			n = InStr(s, "#")
			If n Then s = Left(s, n - 1)
			s = Trim(s)
			If Len(s) Then
				n = ValInt(s)
				cc(n) = LTrim(Mid(s, InStr(s, ":") + 1))
			End If
		Wend
		Close f
		
		For j As Short = 1 To 30
			For i As Short = 1 To 30
				If l(i, j) Then
					aobs += 1
					aob(aobs).x = 2 * (i - 1)
					aob(aobs).y = 2 * (30 - j)
					aob(aobs).z = 0
					aob(aobs).rot = 0
				End If
				
				Select Case l(i, j)
					Case 1
						aob(aobs).kind = 82
					Case 2 To 5
						aob(aobs).kind = 81
						If l(i, j) Mod 2 Then
							aob(aobs).rot = l(i, j) * PI / 2 + PI
						Else
							aob(aobs).rot = (l(i, j) - 2) * PI / 2
						End If
					Case 6
						aob(aobs).kind = 80
						aob(aobs).z = .88
					Case 7 To 10
						aob(aobs).kind = 76
						If l(i, j) Mod 2 Then
							aob(aobs).rot = (l(i, j) + 1) * PI / 2
						Else
							aob(aobs).rot = (l(i, j) - 1) * PI / 2 + PI
						End If
					Case 11 To 14
						aob(aobs).kind = 75
						If l(i, j) Mod 2 Then
							aob(aobs).rot = (l(i, j) + 1) * PI / 2
						Else
							aob(aobs).rot = (l(i, j) - 1) * PI / 2 + PI
						End If
					Case 15 To 18
						aob(aobs).kind = 74
						If l(i, j) Mod 2 Then
							aob(aobs).rot = (l(i, j) + 1) * PI / 2
						Else
							aob(aobs).rot = (l(i, j) - 1) * PI / 2 + PI
						End If
				End Select
			Next i
		Next j
		
		For j As Short = 1 To 30
			For i As Short = 1 To 30
				If Len(cc(t(i, j))) Then
					s = cc(t(i, j))
					Do
						aobs += 1
						aob(aobs).kind = ValInt(s)
						aob(aobs).x = 2 * (i - 1)
						aob(aobs).y = 2 * (30 - j)
						If InStr(s, "w") Then aob(aobs).x += 1
						If InStr(s, "h") Then aob(aobs).y -= 1
						n = InStr(s, "z")
						If n Then
							aob(aobs).z = Val(Mid(s, n + 1))
						Else
							aob(aobs).z = 0
						End If

						n = InStr(s, "r")
						If n Then
							aob(aobs).rot = PI * ValInt(Mid(s, n + 1)) / 2
						Else
							aob(aobs).rot = 0
						End If
						
						Select Case l(i, j)
							Case 6 : aob(aobs).z = .88
							Case 7
								Select Case t(i, j)
									Case 4 : aob(aobs).kind = 89 : aob(aobs).rot = 0
									Case 14 : aob(aobs).kind = 133 : aob(aobs).rot = 0
									Case &H18 : aob(aobs).kind = 134 : aob(aobs).rot = 0
									Case &H3B : aob(aobs).kind = 118
									Case &H27 : aob(aobs).kind = 119
									Case &H62 : aob(aobs).kind = 120
								End Select														
							Case 8
								Select Case t(i, j)
									Case 5 : aob(aobs).kind = 89 : aob(aobs).rot = PI / 2
									Case 15 : aob(aobs).kind = 133 : aob(aobs).rot = PI / 2
									Case &H19 : aob(aobs).kind = 134 : aob(aobs).rot = PI / 2
									Case &H38 : aob(aobs).kind = 118
									Case &H24 : aob(aobs).kind = 119
									Case &H5F : aob(aobs).kind = 120
								End Select							
							Case 9
								Select Case t(i, j)
									Case 4 : aob(aobs).kind = 89 : aob(aobs).rot = PI
									Case 14 : aob(aobs).kind = 133 : aob(aobs).rot = PI
									Case &H18 : aob(aobs).kind = 134 : aob(aobs).rot = PI
									Case &H3A : aob(aobs).kind = 118
									Case &H26 : aob(aobs).kind = 119
									Case &H61 : aob(aobs).kind = 120
								End Select
							Case 10
								Select Case t(i, j)
									Case 5 : aob(aobs).kind = 89 : aob(aobs).rot = -PI / 2
									Case 15 : aob(aobs).kind = 133 : aob(aobs).rot = -PI / 2
									Case &H19 : aob(aobs).kind = 134 : aob(aobs).rot = -PI / 2
									Case &H39 : aob(aobs).kind = 118
									Case &H25 : aob(aobs).kind = 119
									Case &H60 : aob(aobs).kind = 120
								End Select														
						End Select
												
						n = InStr(s, ";")
						If n = 0 Then Exit Do
						s = Mid(s, n + 1)
					Loop
				End If
			Next i
		Next j
	End Sub
	
	Sub TiltRamp(n As Short)
		'Create new track object from object n
		tobs += 1
		tob(tobs).fver = pvs + 1
		tob(tobs).vers = tob(n).vers
		tob(tobs).fprim = tob(n).fprim	'Share primitives
		tob(tobs).prims = tob(n).prims
				
		'Generate new vertices
		pvs += tob(n).vers
		For i As Short = 0 To tob(n).vers - 1
			pv(tob(tobs).fver + i) = pv(tob(n).fver + i)
			pv(tob(tobs).fver + i).z += (1 - pv(tob(n).fver + i).y) * .44
		Next i
		
		tob(tobs).zshift = tob(n).zshift
	End Sub
	
	Sub ChangeMaterial(n As Short, disp As Short)
		'Create new track object from object n
		tobs += 1
		tob(tobs).fver = tob(n).fver	'Share vertices
		tob(tobs).vers = tob(n).vers
		tob(tobs).fprim = pps + 1
		tob(tobs).prims = tob(n).prims
		
		'Generate new primitives with new material
		pps += tob(n).prims
		For i As Short = 0 To tob(n).prims - 1
			pp(tob(tobs).fprim + i) = pp(tob(n).fprim + i)
			If pp(tob(tobs).fprim + i).matf >= 19 _
				AndAlso pp(tob(tobs).fprim + i).matf <= 21 Then
					
				pp(tob(tobs).fprim + i).matf += disp
			End If
			If pp(tob(tobs).fprim + i).matb >= 19 _
				AndAlso pp(tob(tobs).fprim + i).matb <= 21 Then
					
				pp(tob(tobs).fprim + i).matb += disp
			End If
		Next i
	End Sub
End NameSpace

'------------------ ENGINE ENDS HERE ----------------------------
endofengine:

Type Player
	name As String
	rplfile As String
	binfile As String
	carid As String
	paintjob As UByte
	filenum As Integer
	finame As String
End Type

Type TimePointType
	c As Camera
	ctype As Byte	'Camera type. 0 is just fixed at c
	pl As Byte		'Player number to follow
	frame As Long   'Frame at which this switch occurss
End Type

Dim Shared player(1 To 50) As Player, players As Byte, firstcar As Short
Dim Shared curframe As Long, subdir As String
Dim Shared tp(1 To 100) As TimePointType, tps As Short
Dim Shared totalframes As Long
Dim Shared currentcar As Short = 1, disttocar As Double = 2, f3angle As Double = 0
Dim Shared freerun As Byte = 0, snapon As Byte = 0, followcar As Byte = 0
Dim Shared fastforward As Byte = 0, cswitches As Byte = 0, caminfo As Byte = -1

Dim Shared As Double SMWIDTH, SMHEIGHT
SMWIDTH = SWIDTH / PPM
SMHEIGHT = SHEIGHT / PPM


Sub LoadRacerNames
	Dim f As Long, s As String, n As Short
	Dim thisrpls As String, thisrpln As Byte
	Dim key As String, value As String
	
	f = FreeFile
	If Open(subdir & DIRSEP & "players.cfg" For Input As f) Then Exit Sub
	
	While Not EoF(f)
		Line Input #f, s
		If Right(s, 1) = ":" Then
			thisrpls = Trim(Left(s, Len(s) - 1))
			For i As Short = 1 To players
				If player(i).rplfile = thisrpls Then
					thisrpln = i
					Exit For
				End If
			Next i
		Else
			n = InStr(s, "=")
			key = LCase(Trim(Left(s, n - 1)))
			value = Trim(Mid(s, n + 1))
			
			Select Case key
				Case "name": player(thisrpln).name = value
			End Select
		End If
	WEnd
	
	Close f
End Sub

Sub LoadRaceFiles
	Dim s As String, track As String, f As Integer, i As Short
	
	s = Dir(subdir & DIRSEP & "*")
	Do
		If LCase(Right(s, 4)) = ".trk" Then track = s : Exit Do
		s = Dir
	Loop Until Len(s) = 0
	
	GEngine.LoadTrack subdir & DIRSEP & track
		
	#ifdef __FB_LINUX__
		s = Dir(subdir & DIRSEP & "*")
	#else
		s = Dir(subdir & DIRSEP & "*.*")
	#endif
	Do
		If LCase(Right(s, 4)) = ".rpl" Then
			players += 1
			player(players).name = Left(s, Len(s) - 4)
			player(players).rplfile = s
			f = FreeFile
			Open subdir & DIRSEP & s For Binary Access Read As f
			player(players).carid = Space(4)
			Get #f, , player(players).carid
			Get #f, , player(players).paintjob
			player(players).paintjob += 1
			Close f
		End If
		s = Dir	
	Loop Until Len(s) = 0
	
	#ifdef __FB_LINUX__
		s = Dir(subdir & DIRSEP & "*")
	#else
		s = Dir(subdir & DIRSEP & "*.*")
	#endif
	Do
		If LCase(Right(s, 4)) = ".bin" Then
			For i = 1 To players
				If LCase(player(i).name) = LCase(Left(s, Len(s) - 4)) Then
					player(i).binfile = s
					f = FreeFile
					Open subdir & DIRSEP & s For Binary Access Read As f
					If (LoF(f) - 3) \ 1120 > totalframes Then totalframes = (Lof(f) - 3) \ 1120
					player(i).filenum = f
					'--- new
					player(i).finame = subdir & DIRSEP & s
					Close f
					'---
					Exit For
				End If
			Next i
		End If
		s = Dir
	Loop Until Len(s) = 0
	
	'Create car mobs
	For i = 1 To players
		GEngine.LoadCarShape "ST" & player(i).carid & ".3SH", player(i).paintjob
		GEngine.mobs += 1
		GEngine.mob(GEngine.mobs).kind = firstcar + i - 1
		GEngine.mob(GEngine.mobs).id = 1000 + i
	Next i
	
	LoadRacerNames
End Sub


Sub PullFromDump
	Dim buffer As String, n As Short
	Dim As Short i, j
	Dim f As Long
	
	buffer = Space(1120)
	
	For i = 1 To players
		'Find the mob for this player (car object)
		For j As Short = 1 To GEngine.mobs
			If GEngine.mob(j).id = 1000 + i Then n = j : Exit For
		Next j
		
		f = FreeFile
		Open player(i).finame For Binary As f
		If curframe > (LoF(f) - 3) \ 1120 Then
			Get #f, LoF(f) - 1119, buffer
		Else
			Get #f, 3 + 1120 * curframe, buffer
		End If
		Close f
		'Get #player(i).filenum, 3 + 1120 * curframe, buffer
		
		GEngine.mob(n).x = CvL(Mid(buffer, 339, 4)) / 32768 - 1
		GEngine.mob(n).y = CvL(Mid(buffer, 347, 4)) / 32768 - 1
		GEngine.mob(n).z = CvL(Mid(buffer, 343, 4)) / 32768
		GEngine.mob(n).azm = CvShort(Mid(buffer, 363, 2)) * 3.14159 / 512 - 6.2830
		GEngine.mob(n).alt = CvShort(Mid(buffer, 365, 2)) * 3.14159 / 512
		GEngine.mob(n).bank = CvShort(Mid(buffer, 367, 2)) * 3.14159 / 512
	Next i
End Sub


Sub MoviePanel
	Dim As Short px1, py1, px2, py2
	Dim As Double framefactor
	
	px1 = 8 : py1 = 8 : px2 = SWIDTH - 8: py2 = 128
	framefactor = (px2 - px1 - 32) / totalframes
	
	Dim As Integer xm, ym, wm, bm
	
	GetMouse xm, ym, wm, bm
	If bm = 1 Then
		If ym >= py1 + 12 AndAlso ym <= py1 + 20 AndAlso xm >= px1 + 16 AndAlso xm <= px2 - 16 Then
			curframe = (xm - px1 - 16) / framefactor
			If curframe > totalframes Then curframe = totalframes
		End If
	End If
	
	Line (px1, py1)-(px2, py2), RGB(100, 100, 100), BF
	Line (px1 + 4, py1 + 4)-(px2 - 5, py2 - 5), 0, B
	Line (px1 + 5, py1 + 5)-(px2 - 4, py2 - 4), RGB(200, 200, 200), B

	For i As Short = 1 To tps
		Line (px1 + 16 + tp(i).frame * framefactor, py1 + 12)- Step (0, 8), RGB(0, 0, 200)
	Next i
	Line (px1 + 16, py1 + 16)- Step (px2 - px1 - 32, 0), 0
	Line (px1 + 16, py1 + 12)- Step (0, 8), 0
	Line (px2 - 16, py1 + 12)- Step (0, 8), 0
	Circle (px1 + 16 + curframe * framefactor, py1 + 16), 3, RGB(200, 200, 0), , , , F

	'Draw current lap time
	Dim As String s, s2
	s = curframe \ 1200 & ":"
	s2 = Str((curframe Mod 1200) \ 20)
	If Len(s2) = 1 Then s2 = "0" & s2
	s &= s2 & "."
	s2 = Str(5 * (curframe Mod 20))
	If Len(s2) = 1 Then s2 = "0" & s2
	s &= s2
	Draw String (px1 + 16, py1 + 24), s, RGB(0, 0, 0)
	
	'Draw total lap time
	s = totalframes \ 1200 & ":"
	s2 = Str((totalframes Mod 1200) \ 20)
	If Len(s2) = 1 Then s2 = "0" & s2
	s &= s2 & "."
	s2 = Str(5 * (totalframes Mod 20))
	If Len(s2) = 1 Then s2 = "0" & s2
	s &= s2
	Draw String (px2 - 72, py1 + 24), s, RGB(0, 0, 0)
	
	Select Case followcar
		Case 0, 5: s = "Free View"
		Case 1: s = "F1 view. Following " & player(currentcar).name
		Case 3: s = "F3 view. Following " & player(currentcar).name
		Case Else: s = ""
	End Select
	Draw String (px1 + 16, py1 + 40), s, RGB(0, 0, 0)
End Sub


Function ScaleDown (img As Any Ptr) As Any Ptr
	'Take an image that's 480x320 and scale it down 8 times
	'to a new size of 60x40
	
	Dim newimg As Any Ptr
	Dim As Short x, y, u, v
	Dim As ULong col, red, green, blue, al
	
	newimg = ImageCreate(60, 40)
	
	For y = 0 To 39
		For x = 0 To 59
			red = 0 : green = 0 : blue = 0 : al = 0
			For v = 0 To 7
				For u = 0 To 7
					col = Point(8 * x + u, 8 * y + v, img)
					If (col And &Hffffff) = &Hff00ff Then
						al += 1
					Else
						red += RGBA_R(col)
						green += RGBA_G(col)
						blue += RGBA_B(col)
					End If
				Next u
			Next v
			
			'Divide by 64, because it's 8x8 pixels we'll place in one
			red ShR= 6
			green ShR= 6
			blue ShR= 6
			
			'For alpha, we need to normalize at 256 so multiply by 4
			al ShL= 2
			If al = 256 Then al = 255
			
			'But we actually need it inverted
			al = 255 - al
			
			'Paste the pixel
			PSet newimg, (x, y), RGBA(red, green, blue, al)
		Next x
	Next y
	
	Return newimg
End Function


Function GetCarIcon (shapefile As String, paintjob As Byte) As Any Ptr
	fakescreen = ImageCreate(SWIDTH, SHEIGHT)

	GEngine.LoadMaterials

	GEngine.pvs = 0
	GEngine.pps = 0
	GEngine.tobs = 0
	GEngine.LoadCarShape shapefile, paintjob
	GEngine.aobs = 1
	GEngine.aob(1).kind = 1
	GEngine.aob(1).rot = 3 * PI / 2

	GEngine.cam.y = -.5 * 500 / SWIDTH
	GEngine.cam.z = .05 + .05 * 350 / SHEIGHT
	GEngine.cam.alt = -.15

	GEngine.Render

	'Dim myimg As Any Ptr
	'myimg = ScaleDown(fakescreen)

	'ImageDestroy fakescreen
	Return fakescreen
End Function


If Len(Command) = 0 OrElse Command = "/?" OrElse LCase(Command) = "--help" Then
	Print
	Print "Usage: crender <shape_file> [paintjob] [width [height]]"
	Print
	End
End If

Dim ima As Any Ptr, pj As Short, sf As String, n As Short, of As String

sf = Command(1)
If Not FileExists(sf) Then
	Print
	Print "Could not open file " & sf
	Print
	End
End If

pj = ValInt(Command(2))
SWIDTH = ValInt(Command(3))
SHEIGHT = ValInt(Command(4))

If pj > 16 Then
	pj = 1
	SHEIGHT = SWIDTH
	SWIDTH = pj
End If

If pj = 0 Then pj = 1
If SWIDTH = 0 Then SWIDTH = 320
If SHEIGHT = 0 Then SHEIGHT = 200	
n = InStrRev(sf, ".")
If n Then
	of = Left(sf, n) + "tga"
Else
	of = sf + ".tga"
End If

ScreenRes 640, 480, 32, , fb.GFX_NULL

ima = GetCarIcon(sf, pj)

TargaSave of, ima

Screen 0
